package org.maachang.comet.net.nio ;

import java.io.IOException;
import java.io.InputStream;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.comet.net.HttpReceiveCallback;
import org.maachang.comet.net.HttpdRequestImpl;
import org.maachang.comet.net.HttpdResponseImpl;
import org.maachang.comet.net.ReadHttpdRequest;
import org.maachang.comet.net.RequestBuffer;
import org.maachang.util.SequenceSync;
import org.maachang.util.atomic.AtomicINT;
import org.maachang.util.thread.LoopThread;

/**
 * 処理用プーリングスレッド.
 * 
 * @version 2008/05/28
 * @author  masahito suzuki
 * @since   MaachangComet 1.1B
 */
class NioPoolThread extends LoopThread {
    
    /**
     * ログ.
     */
    private static final Log LOG = LogFactory.getLog( NioPoolThread.class ) ;
    
    /**
     * 受信タイムアウト値.
     */
    private static final long TIMEOUT = 15000L ;
    
    /**
     * コールバックオブジェクト.
     */
    private HttpReceiveCallback call = null ;
    
    /**
     * キューオブジェクト.
     */
    private ReceiveLinkQueue queue = null ;
    
    /**
     * シーケンスID発行オブジェクト.
     */
    private SequenceSync sequence = null ;
    
    /**
     * Active数管理.
     */
    private AtomicINT actionLength = null ;
    
    /**
     * リクエストバッファ.
     */
    private final RequestBuffer requestBuffer = new RequestBuffer() ;
    
    /**
     * コネクション情報.
     */
    private final ConnectionInfoImpl connectionInfo = new ConnectionInfoImpl() ;
    
    /**
     * リクエスト情報.
     */
    private final HttpdRequestImpl request = new HttpdRequestImpl() ;
    
    /**
     * レスポンス情報.
     */
    private final HttpdResponseImpl response = new HttpdResponseImpl() ;
    
    /**
     * コンストラクタ.
     */
    private NioPoolThread() {
        
    }
    
    /**
     * コンストラクタ.
     * @param call コールバックオブジェクトを設定します.
     * @param queue キューオブジェクトを設定します.
     * @param actionLength アクティブ実行数管理長オブジェクトを設定します.
     * @param sequence シーケンスID割り当てオブジェクトを設定します.
     * @param threadId 対象のスレッドIDを設定します.
     * @exception Exception 例外.
     */
    public NioPoolThread( HttpReceiveCallback call,ReceiveLinkQueue queue,
        AtomicINT actionLength,SequenceSync sequence,int threadId )
        throws Exception {
        if( call == null || queue == null || actionLength == null || sequence == null ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        this.call = call ;
        this.queue = queue ;
        this.actionLength = actionLength ;
        this.sequence = sequence ;
        super.threadId.set( threadId ) ;
        //super.setPriority( Thread.NORM_PRIORITY -1 ) ;
        this.startThread() ;
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     */
    public void destroy() {
        super.stopThread() ;
    }
    
    /**
     * オブジェクトクリア.
     */
    protected void clear() {
        queue = null ;
        call = null ;
    }
    
    /**
     * 実行処理.
     * @exception Exception 例外.
     */
    protected boolean execution() throws Exception {
        NioElement element = null ;
        while( true ) {
            if( ( element = queue.getQueue() ) != null ) {
                // 実行開始.
                actionLength.inc() ;
                try {
                    executionServer( element ) ;
                } catch( Exception e ) {
                    if( element != null ) {
                        element.destroy() ;
                    }
                    element = null ;
                    throw e ;
                } catch( Error err ) {
                    if( element != null ) {
                        element.destroy() ;
                    }
                    element = null ;
                    throw err ;
                } finally {
                    // 実行終了.
                    actionLength.dec() ;
                }
            }
            else {
                Thread.sleep( 1 ) ;
            }
        }
    }
    
    /**
     * 例外ハンドラ.
     */
    protected void toException( Throwable e ) {
        //LOG.warn( "error",e ) ;
    }
    
    /**
     * １つのコネクションを処理.
     */
    private void executionServer( NioElement element ) throws Exception {
        String url = null ;
        long time = -1L ;
        int id = -1 ;
        try {
            // pipeline対応.
            // リクエスト情報が存在するまで実行する.
            while(true) {
                if( LOG.isDebugEnabled() ) {
                    time = System.currentTimeMillis() ;
                }
                url = null ;
                id = sequence.getId() ;
                if( connectionInfo.create( element ) == null ) {
                    // コネクション切断.
                    if( element != null ) {
                        element.destroy() ;
                    }
                    return ;
                }
                request.clear() ;
                // HTTPヘッダ＋Body読み込み.
                readHttpdRequest( request,connectionInfo,requestBuffer,id ) ;
                // HTTPレスポンス送信前に情報が残っている場合はPIPELINE.
                boolean pipeline = element.isReceiveBuffer() ;
                if( LOG.isDebugEnabled() ) {
                    url = request.getUrlPath() ;
                    LOG.debug( new StringBuilder().append( " ## ***** start " ).
                        append( "receive(" ).append( id ).append( ") - [" ).
                        append( connectionInfo.getInetAddress().getHostAddress() ).
                        append( ":" ).append( connectionInfo.getPort() ).
                        append( "] - threadId:(" ).append( threadId ).
                        append( ") url:" + url ).toString() ) ;
                }
                // 実行処理.
                this.call.execution( connectionInfo,id,request,response ) ;
                // オブジェクトクリア.
                request.clear() ;
                response.clear() ;
                // コメットフラグがOFFの場合の処理.
                // コメットフラグがONの場合は、何もしない.
                if( connectionInfo.isComet() == false ) {
                    if( connectionInfo.isUse() == true ) {
                        if( connectionInfo.isCloseFlag() == true ) {
                            element.setCloseFlag( true ) ;
                        }
                        else {
                            if( connectionInfo.recyclingConnection() == false ) {
                                element.setCloseFlag( true ) ;
                            }
                            else {
                                element.setCloseFlag( false ) ;
                            }
                        }
                    }
                    else {
                        element.destroy() ;
                    }
                }
                // 終了ログ.
                if( LOG.isDebugEnabled() ) {
                    LOG.debug( new StringBuilder().append( " ## @@@@@ end " ).
                        append( "## id[" ).append( id ).
                        append( "]-threadId:(" ).append( threadId ).append( ") url:" ).append( url ).
                        append( " " ).append( (System.currentTimeMillis() - time) ).append( "msec" ).toString() ) ;
                }
                // pipelineでない場合は処理続行しない.
                if( pipeline == false ) {
                    break ;
                }
                if( LOG.isDebugEnabled() ) {
                    LOG.debug( "#### pipline - next" ) ;
                }
            }
        } catch( Exception e ) {
            if( element != null ) {
                element.destroy() ;
            }
            if( LOG.isDebugEnabled() ) {
                StringBuilder buf = new StringBuilder() ;
                buf.append( " ## $$$$$ error " ).
                    append( "## id[" ).append( id ).
                    append( "]-threadId:(" ).append( threadId ) ;
                if( url != null ) {
                    buf.append( ") url:" ).append( url ).append( " " ) ;
                }
                else {
                    buf.append( ") " ) ;
                }
                buf.append( ( System.currentTimeMillis() - time ) ).append( "msec - message:" ).
                    append( e.getMessage() ) ;
                LOG.debug( buf.toString(),e ) ;
            }
            throw e ;
        } finally {
            connectionInfo.clear() ;
        }
    }
    
    /**
     * 受信データをHttpdRequestに変換して取得.
     */
    private static final boolean readHttpdRequest( HttpdRequestImpl request,ConnectionInfo info,RequestBuffer buf,int seqId )
        throws Exception {
        if( info == null || info.isUse() == false ) {
            throw new IOException( "コネクション情報は利用できません" ) ;
        }
        InputStream in = null ;
        try {
            in = info.getInputStream( TIMEOUT ) ;
            ReadHttpdRequest.receiveHttpRequest( request,info,in,buf,seqId ) ;
            in.close() ;
            in = null ;
            return true ;
        } catch( Exception e ) {
            throw e ;
        } finally {
            if( in != null ) {
                in.close() ;
            }
        }
    }
}
