package org.maachang.comet.net.nio ;

import java.io.IOException;
import java.net.SocketTimeoutException;
import java.nio.ByteBuffer;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.util.atomic.AtomicBOOL;
import org.maachang.util.atomic.AtomicINT;
import org.maachang.util.atomic.AtomicOBJECT;

/**
 * 受信バッファ.
 * 
 * @version 2008/11/27
 * @author  masahito suzuki
 * @since   MaachangComet 1.2A
 */
public class ReceiveBuffer {
    private static final Log LOG = LogFactory.getLog( ReceiveBuffer.class ) ;
    protected static final int MAX_LENGTH = 32 * 0x00100000 ;
    private static final int BUFFER = 8192 ;
    private final AtomicOBJECT<byte[]> binary = new AtomicOBJECT<byte[]>( new byte[ BUFFER ] ) ;
    private final AtomicINT binaryLength = new AtomicINT( BUFFER ) ;
    private final AtomicINT length = new AtomicINT( 0 ) ;
    private final AtomicINT position = new AtomicINT( 0 ) ;
    private final AtomicBOOL useConnectFlag = new AtomicBOOL( true ) ;
    
    /**
     * コンストラクタ.
     */
    public ReceiveBuffer() {
    }
    
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     */
    protected void destroy() {
        this.useConnectFlag.set( false ) ;
        this.binary.set( null ) ;
    }
    
    /**
     * 内容リセット.
     */
    public void reset() {
        if( useConnectFlag.get() ) {
            if( compact() == false ) {
                if( this.binaryLength.get() != BUFFER ) {
                    this.binary.set( new byte[ BUFFER ] ) ;
                    this.binaryLength.set( BUFFER ) ;
                }
                this.length.set( 0 ) ;
                this.position.set( 0 ) ;
            }
        }
    }
    
    /**
     * データセット.
     * @param buf ByteBufferオブジェクトを設定します.
     * @exception Exception 例外.
     */
    public void put( ByteBuffer buf ) throws Exception {
        if( buf.limit() < buf.position() ) {
            buf.flip() ;
        }
        if( useConnectFlag.get() == false ) {
            throw new IOException( "オブジェクトは既にクローズしています" ) ;
        }
        int len = buf.limit() ;
        int binLength = length.get() ;
        if( binLength + len >= MAX_LENGTH ) {
            reset() ;
            LOG.warn( "## 受信データ長が最大値["+MAX_LENGTH+"]を越した内容を確認しました" ) ;
            throw new IOException( "受信データ長が最大値["+MAX_LENGTH+"]を越しています" ) ;
        }
        addLimit( len ) ;
        buf.get( this.binary.get(),binLength,len ) ;
        this.length.add( len ) ;
        buf.clear() ;
    }
    
    /**
     * 受信情報を１バイト読み込む.
     * @param timeout 対象のタイムアウト値を設定します.
     * @return byte 対象のバイナリが返されます.
     * @exception IOException 例外.
     */
    public byte get( long timeout ) throws IOException {
        byte ret ;
        if( timeout <= -1 ) {
            while( true ) {
                if( useConnectFlag.get() == false ) {
                    throw new IOException( "オブジェクトは既にクローズしています" ) ;
                }
                int binLength = length.get() ;
                int binPosition = position.get() ;
                if( binLength > binPosition ) {
                    ret = binary.get()[ binPosition ] ;
                    position.inc() ;
                    return ret ;
                }
                try { Thread.sleep( 1 ) ; } catch( Exception e ) {}
            }
        }
        else {
            long t = System.currentTimeMillis() + timeout ;
            for( int c = 0 ;; c ++ ) {
                if( useConnectFlag.get() == false ) {
                    throw new IOException( "オブジェクトは既にクローズしています" ) ;
                }
                int binLength = length.get() ;
                int binPosition = position.get() ;
                if( binLength > binPosition ) {
                    ret = binary.get()[ binPosition ] ;
                    position.inc() ;
                    return ret ;
                }
                if( c > 100 ) {
                    if( System.currentTimeMillis() >= t ) {
                        throw new SocketTimeoutException( "受信タイムアウトを検知" ) ;
                    }
                    c = -1 ;
                }
                try { Thread.sleep( 1 ) ; } catch( Exception e ) {}
            }
        }
    }
    
    /**
     * 受信情報を読み込む.
     * @param timeout 対象のタイムアウト値を設定します.
     * @param out 受信結果内容格納先の条件を設定します.
     * @param off 対象のオフセット値を設定します.
     * @param len 対象の長さを設定します.
     * @return int 受信長が返されます.
     * @exception IOException 例外.
     */
    public int get( long timeout,byte[] out,int off,int len )
        throws IOException {
        if( timeout <= -1 ) {
            while( true ) {
                if( useConnectFlag.get() == false ) {
                    throw new IOException( "オブジェクトは既にクローズしています" ) ;
                }
                int binLength = length.get() ;
                int binPosition = position.get() ;
                if( binLength > binPosition ) {
                    int readLen = binLength - binPosition ;
                    readLen = ( readLen > len ) ? len : readLen ;
                    System.arraycopy( binary.get(),binPosition,out,off,readLen ) ;
                    position.add( readLen ) ;
                    return readLen ;
                }
                try { Thread.sleep( 1 ) ; } catch( Exception e ) {}
            }
        }
        else {
            long t = System.currentTimeMillis() + timeout ;
            for( int c = 0 ;; c ++ ) {
                if( useConnectFlag.get() == false ) {
                    throw new IOException( "オブジェクトは既にクローズしています" ) ;
                }
                int binLength = length.get() ;
                int binPosition = position.get() ;
                if( binLength > binPosition ) {
                    int readLen = binLength - binPosition ;
                    readLen = ( readLen > len ) ? len : readLen ;
                    System.arraycopy( binary.get(),binPosition,out,off,readLen ) ;
                    position.add( readLen ) ;
                    return readLen ;
                }
                if( c > 100 ) {
                    if( System.currentTimeMillis() >= t ) {
                        throw new SocketTimeoutException( "受信タイムアウトを検知" ) ;
                    }
                    c = -1 ;
                }
                try { Thread.sleep( 1 ) ; } catch( Exception e ) {}
            }
        }
    }
    
    /**
     * ポジション位置からN長を戻す.
     * @param n 戻し対象のN値を設定します.
     * @exception Exception 例外.
     */
    public void reversePosition( int n ) {
        int binPosition = position.get() ;
        if( binPosition < n ) {
            throw new IllegalArgumentException( "指定戻り位置は現在のポジション値を越しています" ) ;
        }
        position.remove( n ) ;
    }
    
    /**
     * 読み込み可能データが存在するかチェック.
     * @return boolean [true]の場合、読み込み可能データが存在します.
     */
    public boolean isData() {
        return ( length.get() > position.get() ) ;
    }
    
    /**
     * データ長を取得.
     * @return int データ長が返されます.
     */
    public int length() {
        int binLength = length.get() ;
        int binPosition = position.get() ;
        if( binLength > binPosition ) {
            return binLength - binPosition ;
        }
        return 0 ;
    }
    
    /**
     * オブジェクトが有効かチェック.
     * @return boolean [true]の場合、有効です.
     */
    public boolean isUse() {
        return useConnectFlag.get() ;
    }
    
    /** リミット時にバイナリ長を増加. **/
    private void addLimit( int len ) {
        int maxLength = binaryLength.get() ;
        int binLength = length.get() ;
        if( maxLength <= binLength + len ) {
            int iLen = ( maxLength * 2 ) + len ;
            byte[] x = new byte[ iLen ] ;
            System.arraycopy( binary.get(),0,x,0,maxLength ) ;
            binary.set( x ) ;
            binaryLength.set( iLen ) ;
            x = null ;
        }
    }
    
    /** POSITION以降の内容を先頭に配置. **/
    private boolean compact() {
        int binLength = length.get() ;
        int binPosition = position.get() ;
        if( binPosition <= 0 || binLength <= binPosition ) {
            return false ;
        }
        int len = binLength - binPosition ;
        byte[] b = binary.get() ;
        for( int i = 0 ; i < len ; i ++ ) {
            b[ i ] = b[ binPosition + i ] ;
        }
        length.set( len ) ;
        position.set( 0 ) ;
        return true ;
    }
}
