package org.maachang.comet.net.nio ;

import java.io.IOException;
import java.nio.ByteBuffer;

import org.maachang.util.atomic.AtomicBOOL;
import org.maachang.util.atomic.AtomicINT;
import org.maachang.util.atomic.AtomicOBJECT;

/**
 * 送信バッファ.
 * 
 * @version 2008/11/27
 * @author  masahito suzuki
 * @since   MaachangComet 1.2A
 */
public class SendBuffer {
    private static final int BUFFER = 32767 ;
    private final AtomicOBJECT<byte[]> binary = new AtomicOBJECT<byte[]>() ;
    private int baseLength = 0 ;
    private final AtomicINT binaryLength = new AtomicINT( 0 ) ;
    private final AtomicINT length = new AtomicINT( 0 ) ;
    private final AtomicINT position = new AtomicINT( 0 ) ;
    private final AtomicBOOL lastFlag = new AtomicBOOL( false ) ;
    private final AtomicBOOL useConnectFlag = new AtomicBOOL( true ) ;
    
    /**
     * コンストラクタ.
     */
    public SendBuffer() {
        this( -1 ) ;
    }
    
    /**
     * コンストラクタ.
     * @param size バッファサイズを設定します.
     */
    public SendBuffer( int size ) {
        if( size <= 0 ) {
            size = BUFFER ;
        }
        this.baseLength = size ;
        this.binaryLength.set( size ) ;
        this.binary.set( new byte[ size ] ) ;
    }
    
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     */
    public void destroy() {
        this.useConnectFlag.set( false ) ;
        this.binary.set( null ) ;
        this.binaryLength.set( 0 ) ;
    }
    
    /**
     * オブジェクトリセット.
     */
    public void reset() {
        if( useConnectFlag.get() ) {
            if( this.binaryLength.get() > baseLength ) {
                this.binary.set( new byte[ baseLength ] ) ;
                this.binaryLength.set( baseLength ) ;
            }
            this.length.set( 0 ) ;
            this.position.set( 0 ) ;
            this.lastFlag.set( false ) ;
        }
    }
    
    /**
     * データ読み込み.
     * @param buf 読み込み対象のバッファを設定します.
     * @exception Exception 例外.
     */
    public boolean read( ByteBuffer buf ) throws Exception {
        if( buf == null ) {
            return false ;
        }
        if( useConnectFlag.get() == false ) {
            throw new IOException( "オブジェクトは既にクローズしています" ) ;
        }
        buf.clear() ;
        int len = buf.limit() ;
        int binPosition = this.position.get() ;
        int nowLen = this.length.get() - binPosition ;
        if( nowLen <= 0 || this.binaryLength.get() <= 0 ) {
            return false ;
        }
        len = ( nowLen > len ) ? len : nowLen ;
        buf.put( this.binary.get(),binPosition,len ) ;
        this.position.add( len ) ;
        buf.flip() ;
        return true ;
    }
    
    /**
     * データ書き込み.
     * @param b 書き込み対象のデータを設定します.
     * @exception IOException I/O例外.
     */
    public void write( int b ) throws IOException {
        if( useConnectFlag.get() == false ) {
            throw new IOException( "オブジェクトは既にクローズしています" ) ;
        }
        addLimit( 1 ) ;
        int binLength = this.length.get() ;
        binary.get()[ binLength ] = ( byte )( b & 0x000000ff ) ;
        this.length.inc() ;
    }
    
    /**
     * データ書き込み.
     * @param bin 対象のバイナリを設定します.
     * @exception IOException I/O例外.
     */
    public void write( byte[] bin ) throws IOException {
        write( bin,-1 ) ;
    }
    
    /**
     * データ書き込み.
     * @param bin 対象のバイナリを設定します.
     * @param len 書き込み対象長を設定します.
     * @exception IOException I/O例外.
     */
    public void write( byte[] bin,int len ) throws IOException {
        if( bin == null || bin.length <= 0 ) {
            return ;
        }
        if( len <= 0 || bin.length <= len ) {
            len = bin.length ;
        }
        if( useConnectFlag.get() == false ) {
            throw new IOException( "オブジェクトは既にクローズしています" ) ;
        }
        addLimit( len ) ;
        int binLength = this.length.get() ;
        System.arraycopy( bin,0,this.binary.get(),binLength,len ) ;
        this.length.add( len ) ;
    }
    
    /**
     * 先頭にデータ書き込み.
     * @param bin 対象のバイナリを設定します.
     * @exception IOException I/O例外.
     */
    public void writeHead( byte[] bin ) throws IOException {
        writeHead( bin,-1 ) ;
    }
    
    /**
     * 先頭にデータ書き込み.
     * @param bin 対象のバイナリを設定します.
     * @param len 書き込み対象長を設定します.
     * @exception IOException I/O例外.
     */
    public void writeHead( byte[] bin,int len ) throws IOException {
        if( bin == null || bin.length <= 0 ) {
            return ;
        }
        if( len <= 0 || bin.length <= len ) {
            len = bin.length ;
        }
        if( useConnectFlag.get() == false ) {
            throw new IOException( "オブジェクトは既にクローズしています" ) ;
        }
        addLimit( len ) ;
        int binLength = this.length.get() ;
        move( this.binary.get(),0,binLength,len ) ;
        System.arraycopy( bin,0,this.binary.get(),0,len ) ;
        this.length.add( len ) ;
    }
    
    /**
     * データ書き込み.
     * @param buf 対象のByteBufferを設定します.
     * @exception IOException I/O例外.
     */
    public void write( ByteBuffer buf ) throws Exception {
        if( buf.limit() < buf.position() ) {
            buf.flip() ;
        }
        int len = buf.limit() ;
        if( useConnectFlag.get() == false ) {
            throw new IOException( "オブジェクトは既にクローズしています" ) ;
        }
        addLimit( len ) ;
        int binLength = this.length.get() ;
        buf.get( this.binary.get(),binLength,len ) ;
        this.length.add( len ) ;
        buf.clear() ;
    }
    
    /**
     * 先頭にデータ書き込み.
     * @param buf 対象のByteBufferを設定します.
     * @exception IOException I/O例外.
     */
    public void writeHead( ByteBuffer buf ) throws Exception {
        if( buf.limit() < buf.position() ) {
            buf.flip() ;
        }
        int len = buf.limit() ;
        if( useConnectFlag.get() == false ) {
            throw new IOException( "オブジェクトは既にクローズしています" ) ;
        }
        addLimit( len ) ;
        int binLength = this.length.get() ;
        move( this.binary.get(),0,binLength,len ) ;
        buf.get( this.binary.get(),0,len ) ;
        this.length.add( len ) ;
        buf.clear() ;
    }
    
    /**
     * 書き込み終了設定.
     */
    public void last() {
        lastFlag.set( true ) ;
    }
    
    /**
     * 書き込みが終了しているかチェック.
     * @return boolean [true]の場合、書き込み終了しています.
     */
    public boolean isLast() {
        return lastFlag.get() ;
    }
    
    /**
     * 読み込み可能データ長を取得.
     * @return int 読み込み可能データ長が返されます.
     */
    public int length() {
        return this.length.get() - this.position.get() ;
    }
    
    /**
     * オブジェクトが有効かチェック.
     * @return boolean [true]の場合、有効です.
     */
    public boolean isUse() {
        return useConnectFlag.get() ;
    }
    
    /** リミット時にバイナリ長を増加. **/
    private void addLimit( int len ) {
        int maxLength = binaryLength.get() ;
        int binLength = length.get() ;
        if( maxLength <= binLength + len ) {
            int iLen = ( maxLength * 2 ) + len ;
            byte[] x = new byte[ iLen ] ;
            System.arraycopy( binary.get(),0,x,0,maxLength ) ;
            binary.set( x ) ;
            binaryLength.set( iLen ) ;
            x = null ;
        }
    }
    
    /** 指定範囲の領域を移動 **/
    private static void move( byte[] bin,int pos,int len,int moveStart )
        throws IOException {
        for( int i = len-1 ; i >= 0 ; i -- ) {
            bin[ moveStart + i ] = bin[ pos + i ] ;
        }
    }
}
