package org.maachang.comet.net.ssl ;

import java.io.IOException;
import java.nio.ByteBuffer;

import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLEngineResult;
import javax.net.ssl.SSLEngineResult.HandshakeStatus;
import javax.net.ssl.SSLEngineResult.Status;

import org.maachang.comet.net.nio.NioElement;
import org.maachang.comet.net.nio.ReceiveBuffer;
import org.maachang.util.atomic.AtomicBOOL;
import org.maachang.util.atomic.AtomicOBJECT;

/**
 * SSLソケット要素.
 *  
 * @version 2008/06/01
 * @author  masahito suzuki
 * @since   MaachangComet 1.1D
 */
public class SslElement {
    
    /**
     * NioElement.
     */
    private final AtomicOBJECT<NioElement> element = new AtomicOBJECT<NioElement>() ;
    
    /**
     * SSLEngine.
     */
    private final AtomicOBJECT<SSLEngine> sslEngine = new AtomicOBJECT<SSLEngine>() ;
    
    /**
     * クライアント受け取りバッファ.
     */
    private final AtomicOBJECT<ByteBuffer> netInBuffer = new AtomicOBJECT<ByteBuffer>() ;
    
    /**
     * クライアント出力バッファ.
     */
    private final AtomicOBJECT<ByteBuffer> netOutBuffer = new AtomicOBJECT<ByteBuffer>() ;
    
    /**
     * 受信バッファ.
     */
    private final AtomicOBJECT<ByteBuffer> recvBuffer = new AtomicOBJECT<ByteBuffer>() ;
    
    /**
     * ハンドシェイクステータス.
     */
    private final AtomicOBJECT<HandshakeStatus> handShakeState = new AtomicOBJECT<HandshakeStatus>() ;
    
    /**
     * ハンドシェイク終了フラグ.
     */
    private final AtomicBOOL isHandShakeFinish = new AtomicBOOL( false ) ;
    
    /**
     * クローズフラグ.
     */
    private final AtomicBOOL isClose = new AtomicBOOL( false ) ;
    
    /**
     * ハンドシェイクデータ送信完了フラグ.
     */
    private final AtomicBOOL isExitHandShakeSend = new AtomicBOOL( false ) ;
    
    /**
     * 基本受信バッファサイズ.
     */
    private int recvBufferSize = -1 ;
    
    /**
     * デフォルトバッファサイズ.
     */
    private int defaultSize = -1 ;
    
    /**
     * コンストラクタ.
     */
    private SslElement() {
        
    }
    
    /**
     * コンストラクタ.
     * @param element 対象のNioElementを設定します.
     * @param sslEngine 対象のSSLエンジンを設定します.
     * @exception Exception 例外.
     */
    public SslElement( NioElement element,SSLEngine engine ) throws Exception {
        if( element == null || engine  == null ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        this.recvBufferSize = engine.getSession().getApplicationBufferSize() ;
        this.defaultSize = engine.getSession().getPacketBufferSize() * 2 ;
        this.recvBuffer.set( ByteBuffer.allocate( this.recvBufferSize ) ) ;
        this.netInBuffer.set( ByteBuffer.allocate( this.defaultSize ) ) ;
        this.netOutBuffer.set( ByteBuffer.allocate( this.defaultSize ) ) ;
        this.netOutBuffer.get().limit( 0 ) ;
        this.sslEngine.set( engine ) ;
        this.element.set( element ) ;
        this.isHandShakeFinish.set( false ) ;
        this.isExitHandShakeSend.set( false ) ;
        this.isClose.set( false ) ;
        engine.beginHandshake();
        String[] supportCiphers = engine.getSupportedCipherSuites() ;
        if( supportCiphers != null ) {
            engine.setEnabledCipherSuites( supportCiphers ) ;
        }
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     */
    public void destroy() {
        if( sslEngine.get() != null ) {
            try {
                sslEngine.get().closeInbound() ;
            } catch( Exception e ) {
            }
            try {
                sslEngine.get().closeOutbound() ;
            } catch( Exception e ) {
            }
        }
        sslEngine.set( null ) ;
        netInBuffer.set( null ) ;
        netOutBuffer.set( null ) ;
        element.set( null ) ;
        recvBuffer.set( null ) ;
        isClose.set( true ) ;
    }
    
    /**
     * オブジェクトクローズ.
     */
    public void close()
        throws Exception {
        if( element.get() != null && sslEngine.get() != null ) {
            element.get().restartWrite() ;
            sslEngine.get().closeInbound() ;
            sslEngine.get().closeOutbound() ;
            SSLEngineResult res = SslUtil.wrap( sslEngine.get(),SslUtil.EMPTY_BUFFER,netOutBuffer.get() ) ;
            if( res.getStatus() == SSLEngineResult.Status.CLOSED ) {
                writeQueue( netOutBuffer.get() ) ;
            }
        }
    }
    
    /**
     * Bufferクリア.
     */
    public void clearBuffer() {
        if( element.get() != null ) {
            this.recvBuffer.set( ByteBuffer.allocate( recvBufferSize ) ) ;
            this.netInBuffer.set( ByteBuffer.allocate( defaultSize ) ) ;
            this.netOutBuffer.set( ByteBuffer.allocate( defaultSize ) ) ;
            this.netOutBuffer.get().limit( 0 ) ;
        }
        else {
            this.recvBuffer.set( null ) ;
            this.netInBuffer.set( null ) ;
            this.netOutBuffer.set( null ) ;
        }
    }
    
    /**
     * 既にクローズ状態であるかチェック.
     * @return boolean [true]の場合、クローズしています.
     */
    public boolean isClosed() {
        return isClose.get() ;
    }
    
    /**
     * HandShake終了チェック.
     * @return boolean [true]の場合、HandShakeが終了しています.
     */
    public boolean isHandShake() {
        return isHandShakeFinish.get() ;
    }
    
    /**
     * ハンドシェイク送信完了の場合の呼び出し.
     */
    public void sendEndHandShake() {
        if( isHandShakeFinish.get() ) {
            isExitHandShakeSend.set( true ) ;
            netInBuffer.set( ByteBuffer.allocate( defaultSize ) ) ;
            netOutBuffer.set( ByteBuffer.allocate( defaultSize ) ) ;
            netOutBuffer.get().limit( 0 ) ;
        }
    }
    
    /**
     * ハンドシェイク送信完了チェック.
     * @return boolean [true]の場合、ハンドシェイク処理は終了しています.
     */
    public boolean isSendEndHandShake() {
        return isExitHandShakeSend.get() ;
    }
    
    /**
     * 受信バッファを取得.
     * @return ByteBuffer 受信バッファを取得します.
     */
    public ByteBuffer getRecvBuffer() {
        return recvBuffer.get() ;
    }
    
    /**
     * HandShake処理.
     * @param recvBuf 受信されたByteBufferを設定します.
     * @exception Exception 例外.
     */
    public void handShake( ByteBuffer recvBuf ) throws Exception {
        if( isClose.get() ) {
            throw new IOException( "オブジェクトは既にクローズしています" ) ;
        }
        // 最初の処理の場合、ハンドシェイクステータスは、unwrapにセットする.
        if( this.handShakeState.get() == null ) {
            sslEngine.get().beginHandshake() ;
            this.handShakeState.set( sslEngine.get().getHandshakeStatus() ) ;
        }
        
        // 読み込みデータ内容が存在しない場合は処理しない.
        SSLEngineResult result = null ;
        if( isUseReadData( recvBuf ) ) {
            if( isHandShakeFinish.get() ) {
                throw new IOException( "ハンドシェイク処理は完了しています" ) ;
            }
            // 読み込み処理.
            result = SslUtil.unwrap( sslEngine.get(),recvBuf,netInBuffer.get() ) ;
            this.handShakeState.set( result.getHandshakeStatus() ) ;
            // unwrapが正常終了するまでループ.
            while(true) {
                // ステータスOKの場合.
                if( result.getStatus() == Status.OK ) {
                    break ;
                }
                // バッファオーバフローの場合は、Inputバッファサイズを増やして
                // 受信処理を行う.
                else if( result.getStatus() == Status.BUFFER_OVERFLOW ) {
                    netInBuffer.set( SslUtil.reallocate( netInBuffer.get() ) ) ;
                    result = SslUtil.unwrap( sslEngine.get(),recvBuf,netInBuffer.get() ) ;
                    this.handShakeState.set( result.getHandshakeStatus() ) ;
                }
                // その他.
                else {
                    throw new IOException( "不明なステータス[" +
                        this.handShakeState.get() + "/"+ result.getStatus() + "]が返されました" ) ;
                }
            }
        }
        else {
            return ;
        }
        
        // ハンドシェイクイベント処理.
        boolean roopEnd = true ;
        while( roopEnd ) {
            switch( this.handShakeState.get() ) {
                
                // wrap.
                case NEED_WRAP :
                    // 書き込み処理.
                    result = SslUtil.wrap( sslEngine.get(),SslUtil.EMPTY_BUFFER,netOutBuffer.get() ) ;
                    this.handShakeState.set( result.getHandshakeStatus() ) ;
                    // ステータスがOKの場合.
                    if( result.getStatus() == Status.OK ) {
                        writeQueue( netOutBuffer.get() ) ;
                        // 書き込み後の読み込み条件の場合は、一端ハンドシェイクループを抜ける.
                        if( this.handShakeState.get() == HandshakeStatus.NEED_UNWRAP ) {
                            roopEnd = false ;
                            break ;
                        }
                        break ;
                    }
                    throw new IOException( "不明なステータス[" +
                        this.handShakeState.get() + "/"+ result.getStatus() + "]が返されました" ) ;
                
                // unwrap.
                case NEED_UNWRAP :
                    // 続けて読み込む場合.
                    if( isUseReadData( recvBuf ) ) {
                        for( ;; ) {
                            result = SslUtil.unwrap( sslEngine.get(),recvBuf,netInBuffer.get() ) ;
                            this.handShakeState.set( result.getHandshakeStatus() ) ;
                            if( result.getStatus() == Status.BUFFER_OVERFLOW ) {
                                netInBuffer.set( SslUtil.reallocate( netInBuffer.get() ) ) ;
                                continue ;
                            }
                            break ;
                        }
                    }
                    else if( this.handShakeState.get() == HandshakeStatus.NEED_UNWRAP ) {
                        if( isUseReadData( recvBuf ) == false ) {
                            roopEnd = false ;
                        }
                    }
                    break ;
                    
                // task.
                case NEED_TASK :
                    this.handShakeState.set( tasks() ) ;
                    break ;
                
                // finish.
                case FINISHED :
                    isHandShakeFinish.set( true ) ;
                    roopEnd = false ;
                    break ;
                
                // その他.
                default :
                    throw new IOException(
                        "不明なステータス[" + this.handShakeState.get() + "]が返されました" ) ;
            }
        }
    }
    
    /**
     * 読み込み可能情報が存在する場合.
     */
    public boolean isUseReadData( ByteBuffer recvBuf ) {
        boolean ret ;
        ret = ( recvBuf.position() != 0 && recvBuf.hasRemaining() &&
            handShakeState.get() == HandshakeStatus.NEED_UNWRAP ) ;
        if( ret == false ) {
            ret = ( recvBuf.position() == recvBuf.capacity() ) ;
        }
        return ret ;
    }
    
    /**
     * アプリケーションデータを読み込み.
     * @exception Exception 例外.
     */
    public boolean read( ByteBuffer recvBuf ) throws Exception {
        checkExitHandShake() ;
        boolean ret = false ;
        int len = element.get().getChannel().read( recvBuf ) ;
        if( len <= 0 ) {
            try {
                sslEngine.get().closeInbound();
            } catch (IOException ex) {
            }
        }
        else {
            readToSslConvert( element.get().getReceiveBuffer(),recvBuf ) ;
            element.get().update() ;
            ret = true ;
        }
        return ret ;
    }
    
    /**
     * ハンドシェイク完了後に対する受信データ残りが存在する場合に、受信処理を行う.
     */
    public boolean readTo( ByteBuffer recvBuf ) throws Exception {
        boolean ret = false ;
        if( recvBuf.position() != 0 && recvBuf.hasRemaining() ) {
            while( recvBuf.position() != 0 && recvBuf.hasRemaining() ) {
                int ln = readToSslConvert( element.get().getReceiveBuffer(),recvBuf ) ;
                ret = true ;
                if( ln <= 0 ) {
                    break ;
                }
            }
        }
        return ret ;
    }
    
    /**
     * アプリケーションデータ書き込み.
     * @param buffer 書き込み対象データを設定します.
     * @exception Exception 例外.
     */
    public void write( ByteBuffer buf ) throws Exception {
        checkExitHandShake() ;
        int written = 0;
        SSLEngineResult result = SslUtil.wrap( sslEngine.get(),buf,netOutBuffer.get() ) ;
        written = result.bytesConsumed();
        if (result.getStatus() == Status.OK) {
            if (result.getHandshakeStatus() == HandshakeStatus.NEED_TASK) {
                tasks() ;
            }
        } else {
            if( result.getStatus() == Status.CLOSED ) {
                throw new CloseSslException( "SSLコネクションクローズ" ) ;
            }
            throw new IOException("不正なステータス[" + result.getStatus() + "]を検知しました" ) ;
        }
        
        // ソケットにデータを書き込む.
        element.get().getChannel().write( netOutBuffer.get() ) ;
        
        // バッファ長と比較して、残りデータが存在する場合は、
        // 次の書き込み処理にまわす.
        if( buf.limit() != written ) {
            buf.position( written ) ;
            buf.compact() ;
            element.get().getSendBuffer().writeHead( buf ) ;
        }
    }
    
    /**
     * HandShake終了チェック.
     */
    private void checkExitHandShake() throws Exception {
        if( isClose.get() ) {
            throw new IOException( "オブジェクトは既にクローズしています" ) ;
        }
        if( isExitHandShakeSend.get() == false ) {
            throw new IOException( "ハンドシェイク処理が終了していません" ) ;
        }
    }
    
    /**
     * タスク処理.
     */
    private HandshakeStatus tasks() {
        Runnable r = null ;
        while ( ( r = sslEngine.get().getDelegatedTask() ) != null ) {
            r.run() ;
        }
        return sslEngine.get().getHandshakeStatus() ;
    }
    
    /**
     * キューに書き込みデータを出力.
     */
    private void writeQueue( ByteBuffer buf ) throws Exception {
        if( element.get() != null && element.get().isUse() ) {
            element.get().getSendBuffer().write( netOutBuffer.get() ) ;
        }
    }
    
    /**
     * SSL処理を解析して受信バッファにセットする.
     */
    private int readToSslConvert( ReceiveBuffer emtBuf,ByteBuffer recvBuf )
        throws Exception {
        SSLEngineResult unwrap ;
        netInBuffer.get().clear() ;
        int recv = 0 ;
        do {
            // SSLデータ解析.
            unwrap = SslUtil.unwrap( sslEngine.get(),recvBuf,netInBuffer.get() ) ;
            // 処理結果を判別.
            if( unwrap.getStatus() == Status.OK || unwrap.getStatus() == Status.BUFFER_UNDERFLOW ) {
                recv += unwrap.bytesProduced() ;
                // OKの場合はデータを対象スレッド宛に送信.
                if( unwrap.getStatus() == Status.OK ) {
                    // 受信データを書き込む.
                    netInBuffer.get().flip() ;
                    emtBuf.put( netInBuffer.get() ) ;
                    netInBuffer.get().clear() ;
                }
                // タスク要求の場合.
                if( unwrap.getHandshakeStatus() == HandshakeStatus.NEED_TASK ) {
                    tasks() ;
                }
                // データ解析失敗.
                if( unwrap.getStatus() == Status.BUFFER_UNDERFLOW ) {
                    break ;
                }
            } else if( ( unwrap.getStatus()==Status.BUFFER_OVERFLOW && recv > 0 ) || unwrap.getStatus()==Status.CLOSED ) {
                break ;
            } else if( unwrap.getStatus()==Status.CLOSED ) {
                break ;
            } else {
                throw new IOException("不正なステータス[" + unwrap.getStatus() + "]を検知しました" ) ;
            }
        } while( ( recv != 0 ) ) ;
        return recv ;
    }
}

