package org.maachang.dao.dbms.replication ;

import java.io.IOException;
import java.io.OutputStream;

import org.maachang.dao.dbms.DbUtil;
import org.maachang.util.ConvertParam;

import com.y_ys.cluster.ha.MqNetUtil;
import com.y_ys.cluster.ha.MqPacket;

/**
 * SQLレプリケーション情報.
 * 
 * @version 2008/11/08
 * @author masahito suzuki
 * @since MaachangDao 1.11
 */
public class ReplicationSQL extends MqPacket {
    
    /**
     * レプリケーションコマンドタイプ : Insert.
     */
    public static final int COMMAND_INSERT = 1 ;
    
    /**
     * レプリケーションコマンドタイプ : Update.
     */
    public static final int COMMAND_UPDATE = 2 ;
    
    /**
     * レプリケーションコマンドタイプ : Delete.
     */
    public static final int COMMAND_DELETE = 3 ;
    
    /**
     * コマンドタイプ.
     */
    private int command = -1 ;
    
    /**
     * テーブル名.
     */
    private String tableName = null ;
    
    /**
     * SQL文.
     */
    private String sql = null ;
    
    /**
     * パラメータ群.
     */
    private Object[] params = null ;
    
    /**
     * コンストラクタ.
     * @param id 対象のシーケンスIDを設定します.
     * @param createTime パケット生成時間が返されます.
     * @param bin 復元対象の要素バイナリを設定します.
     * @param off 対象のオフセット値を設定します.
     */
    public ReplicationSQL( long id,long createTime,byte[] bin,int off ) throws Exception {
        super( id,createTime,bin,off ) ;
        this.convertBinary( bin,off ) ;
    }
    
    /**
     * コンストラクタ.
     * @param command 対象のコマンドタイプを設定します.
     * @param tableName 対象のテーブル名を設定します.
     * @param sql 対象のSQL文を設定します.
     * @param params 対象のパラメータ名を設定します.
     * @exception Exception 例外.
     */
    public ReplicationSQL( int command,String tableName,String sql,Object[] params )
        throws Exception {
        if( ( tableName = DbUtil.convertJavaNameByDBName( tableName ) ) == null || 
            sql == null || ( sql = sql.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        switch( command ) {
            case COMMAND_INSERT:
            case COMMAND_UPDATE:
            case COMMAND_DELETE:
                break ;
            default :
                throw new IllegalArgumentException( "コマンド定義(" + command + ")が不正です" ) ;
        }
        if( params == null || params.length <= 0 ) {
            params = null ;
        }
        this.command = command ;
        this.tableName = tableName ;
        this.sql = sql ;
        this.params = params ;
    }
    
    protected void finalize() throws Exception {
        destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     */
    public void destroy() {
        super.destroy() ;
        this.command = -1 ;
        this.tableName = null ;
        this.sql = null ;
        this.params = null ;
    }
    
    /**
     * レプリケーションタイプを取得.
     * @return int レプリケーションタイプが返されます.
     */
    public int getType() {
        return 0 ;// SQLのレプリケーションを表す.
    }
    
    /**
     * レプリケーションタイプを文字列で取得.
     * @return String レプリケーションタイプが文字列で返されます.
     */
    public String typString() {
        return "DB" ;
    }
    
    /**
     * コマンドタイプを取得.
     * @return int コマンドタイプが返されます.
     */
    public int getCommand() {
        return command ;
    }
    
    /**
     * テーブル名を取得.
     * @return String テーブル名が返されます.
     */
    public String getTableName() {
        return tableName ;
    }
    
    /**
     * 実行SQLを取得.
     * @return String SQL文が返されます.
     */
    public String getSql() {
        return sql ;
    }
    
    /**
     * パラメータを取得.
     * @return Object[] パラメータが返されます.
     */
    public Object[] getParams() {
        return params ;
    }
    
    /**
     * 文字列で出力.
     * @return String 文字列で返されます.
     */
    public String toString() {
        StringBuilder buf = new StringBuilder() ;
        buf.append( " command:" ) ;
        switch( command ) {
            case COMMAND_INSERT: buf.append( "insert" ) ; break ;
            case COMMAND_UPDATE: buf.append( "update" ) ; break ;
            case COMMAND_DELETE: buf.append( "delete" ) ; break ;
            default : buf.append( "unknown" ) ; break ;
        }
        buf.append( " table:" ).append( tableName ) ;
        buf.append( " sql:[" ).append(  sql ).append( "]" ) ;
        buf.append( " params(" ) ;
        if( params != null ) {
            int len = params.length ;
            buf.append( len ).append( "): {" ) ;
            for( int i = 0 ; i < len ; i ++ ) {
                if( i != 0 ) {
                    buf.append( "," ) ;
                }
                if( params[ i ] == null ) {
                    buf.append( " null" ) ;
                }
                else if( params[ i ] instanceof byte[] ) {
                    buf.append( " " ).append( ( ( byte[] )params[ i ] ).length ).append( "@" ).
                    append( "byte[]" ) ;
                }
                else {
                    buf.append( " " ).append( params[ i ] ).append( "@" ).
                        append( params[ i ].getClass().getName() ) ;
                }
            }
            buf.append( " }" ) ;
        }
        else {
            buf.append( "0)" ) ;
        }
        super.toString( buf ) ;
        return buf.toString() ;
    }
    
    // パラメータ型定義.
    private static final int PARAMS_NULL = 0 ;
    private static final int PARAMS_BOOL = 1 ;
    private static final int PARAMS_INTEGER = 2 ;
    private static final int PARAMS_LONG = 3 ;
    private static final int PARAMS_DOUBLE = 4 ;
    private static final int PARAMS_FLOAT = 5 ;
    private static final int PARAMS_DATE = 10 ;
    private static final int PARAMS_TIME = 11 ;
    private static final int PARAMS_TIMESTAMP = 12 ;
    private static final int PARAMS_STRING = 20 ;
    private static final int PARAMS_BINARY = 30 ;
    
    
    /**
     * 要素条件をバイナリ変換.
     * @param out バイナリを格納するOutputStreamを設定します.
     * @exception Exception 例外.
     */
    protected void getBinary( OutputStream out ) throws Exception {
        if( ( tableName = DbUtil.convertJavaNameByDBName( tableName ) ) == null || 
                sql == null || ( sql = sql.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        switch( command ) {
            case COMMAND_INSERT:
            case COMMAND_UPDATE:
            case COMMAND_DELETE:
                break ;
            default :
                throw new IllegalArgumentException( "コマンド定義(" + command + ")が不正です" ) ;
        }
        if( params == null || params.length <= 0 ) {
            params = null ;
        }
        byte[] b = null ;
        //** データ領域開始 **//
        // コマンドタイプ.
        out.write( ConvertParam.convertInt( command ) ) ;
        // テーブル名.
        b = tableName.getBytes( "UTF8" ) ;
        out.write( ConvertParam.convertInt( b.length ) ) ;
        out.write( b ) ;
        // SQL.
        b = sql.getBytes( "UTF8" ) ;
        out.write( ConvertParam.convertInt( b.length ) ) ;
        out.write( b ) ;
        b = null ;
        // パラメータ群.
        int paramLen = ( params == null ) ? 0 : params.length ;
        out.write( ConvertParam.convertInt( paramLen ) ) ;
        for( int i = 0 ; i < paramLen ; i ++ ) {
            Object o = params[ i ] ;
            if( o == null ) {
                out.write( PARAMS_NULL ) ;
            }
            else if( o instanceof Boolean ) {
                out.write( PARAMS_BOOL ) ;
                out.write( ( ( ( Boolean )o ).booleanValue() ) ? 1 : 0 ) ;
            }
            else if( o instanceof Integer ) {
                out.write( PARAMS_INTEGER ) ;
                out.write( ConvertParam.convertInt( ( ( Integer )o ).intValue() ) ) ;
            }
            else if( o instanceof Long ) {
                out.write( PARAMS_LONG ) ;
                out.write( ConvertParam.convertLong( ( ( Long )o ).longValue() ) ) ;
            }
            else if( o instanceof Double ) {
                out.write( PARAMS_DOUBLE ) ;
                out.write( ConvertParam.convertDouble( ( ( Double )o ).doubleValue() ) ) ;
            }
            else if( o instanceof Float ) {
                out.write( PARAMS_FLOAT ) ;
                out.write( ConvertParam.convertFloat( ( ( Float )o ).floatValue() ) ) ;
            }
            else if( o instanceof java.sql.Date ) {
                out.write( PARAMS_DATE ) ;
                out.write( ConvertParam.convertLong( ( ( java.sql.Date )o ).getTime() ) ) ;
            }
            else if( o instanceof java.sql.Time ) {
                out.write( PARAMS_TIME ) ;
                out.write( ConvertParam.convertLong( ( ( java.sql.Time )o ).getTime() ) ) ;
            }
            else if( o instanceof java.sql.Timestamp ) {
                out.write( PARAMS_TIMESTAMP ) ;
                out.write( ConvertParam.convertLong( ( ( java.sql.Timestamp )o ).getTime() ) ) ;
            }
            else if( o instanceof String ) {
                out.write( PARAMS_STRING ) ;
                if( ( ( String )o ).length() <= 0 ) {
                    out.write( ConvertParam.convertInt( 0 ) ) ;
                }
                else {
                    b = ( ( String )o ).getBytes( "UTF8" ) ;
                    out.write( ConvertParam.convertInt( b.length ) ) ;
                    out.write( b ) ;
                    b = null ;
                }
            }
            else if( o instanceof byte[] ) {
                out.write( PARAMS_BINARY ) ;
                if( ( ( byte[] )o ).length <= 0 ) {
                    out.write( ConvertParam.convertInt( 0 ) ) ;
                }
                else {
                    out.write( ConvertParam.convertInt( ( ( byte[] )o ).length ) ) ;
                    out.write( ( byte[] )o ) ;
                }
            }
            else {
                throw new IOException( "不正なオブジェクト型(" + o.getClass().getName() + ")です" ) ;
            }
        }
    }
    
    /**
     * 要素バイナリを要素条件に変換.
     * @param bin 対象の要素バイナリを設定します.
     * @param off 対象のオフセット値を設定します.
     * @exception Exception 例外.
     */
    protected void convertBinary( byte[] bin,int off ) throws Exception {
        //** データ領域開始 **//
        // コマンドタイプを取得.
        int cmd = ConvertParam.convertInt( off,bin ) ;
        off += 4 ;
        // テーブル名を取得.
        int len = ConvertParam.convertInt( off,bin ) ;
        off += 4 ;
        String tblName = new String( bin,off,len,"UTF8" ) ;
        off += len ;
        // SQL文を取得.
        len = ConvertParam.convertInt( off,bin ) ;
        off += 4 ;
        String sql = new String( bin,off,len,"UTF8" ) ;
        off += len ;
        // パラメータ長を取得.
        int paramLen = ConvertParam.convertInt( off,bin ) ;
        off += 4 ;
        // パラメータ群を取得.
        Object[] pms = null ;
        if( paramLen > 0 ) {
            pms = new Object[ paramLen ] ;
            for( int i = 0 ; i < paramLen ; i ++ ) {
                // パラメータタイプを取得.
                int paramType = ( int )( bin[ off ] & 0x000000ff ) ;
                off += 1 ;
                switch( paramType ) {
                    case PARAMS_NULL :
                        break ;
                    case PARAMS_BOOL :
                        pms[ i ] = ( bin[ off ] == 0 ) ? Boolean.FALSE : Boolean.TRUE ;
                        off += 1 ;
                        break ;
                    case PARAMS_INTEGER :
                        pms[ i ] = new Integer( ConvertParam.convertInt( off,bin ) ) ;
                        off += 4 ;
                        break ;
                    case PARAMS_LONG :
                        pms[ i ] = new Long( ConvertParam.convertLong( off,bin ) ) ;
                        off += 8 ;
                        break ;
                    case PARAMS_DOUBLE :
                        pms[ i ] = ConvertParam.convertDoubleObject( off,bin ) ;
                        off += ( int )ConvertParam.convertShort( off,bin ) + 2 ;
                        break ;
                    case PARAMS_FLOAT :
                        pms[ i ] = ConvertParam.convertFloatObject( off,bin ) ;
                        off += ( int )ConvertParam.convertShort( off,bin ) + 2 ;
                        break ;
                    case PARAMS_DATE :
                        pms[ i ] = new java.sql.Date( ConvertParam.convertLong( off,bin ) ) ;
                        off += 8 ;
                        break ;
                    case PARAMS_TIME :
                        pms[ i ] = new java.sql.Time( ConvertParam.convertLong( off,bin ) ) ;
                        off += 8 ;
                        break ;
                    case PARAMS_TIMESTAMP :
                        pms[ i ] = new java.sql.Timestamp( ConvertParam.convertLong( off,bin ) ) ;
                        off += 8 ;
                        break ;
                    case PARAMS_STRING :
                        len = ConvertParam.convertInt( off,bin ) ;
                        off += 4 ;
                        if( len <= 0 ) {
                            pms[ i ] = "" ;
                        }
                        else {
                            pms[ i ] = new String( bin,off,len,"UTF8" ) ;
                            off += len ;
                        }
                        break ;
                    case PARAMS_BINARY :
                        len = ConvertParam.convertInt( off,bin ) ;
                        off += 4 ;
                        if( len <= 0 ) {
                            pms[ i ] = new byte[ 0 ] ;
                        }
                        else {
                            byte[] b = new byte[ len ] ;
                            System.arraycopy( bin,off,b,0,len ) ;
                            pms[ i ] = b ;
                            off += len ;
                        }
                        break ;
                    default :
                        throw new IOException( "[" + i + "]:パラメータタイプ(" + paramType + ")は不正です" ) ;
                }
            }
        }
        this.command = cmd ;
        this.tableName = tblName ;
        this.sql = sql ;
        this.params = pms ;
    }
    
    /** test. **/
    public static final void main( String[] args ) throws Exception {
        String tableName = "TestTable" ;
        String sql = "insert into test_table ( id,name,name_kana,yen,img ) values ( ?,?,?,?,? )" ;
        Object[] params = { new Long(1),"穂下太郎","ホゲタロウ",new Double( 100.234f ),new byte[]{0,1,2,3,4,5} } ;
        ReplicationSQL rep = new ReplicationSQL( COMMAND_INSERT,tableName,sql,params ) ;
        rep.setId( 100L ) ;
        System.out.println( "create:" + rep ) ;
        byte[] bin = rep.toBinary() ;
        //System.out.println( "bin.length:" + bin.length ) ;
        //System.out.println( BinaryUtil.toString( bin ) ) ;
        MqPacket pk = MqNetUtil.convertMqPacket( bin ) ;
        System.out.println( "packet:" + pk ) ;
    }
}
