package org.maachang.session.engine ;

import java.io.IOException;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.util.BaseRandom;
import org.maachang.util.RandomId;
import org.maachang.util.thread.LoopThread;

/**
 * セッション生成オブジェクト.
 *  
 * @version 2008/05/25
 * @author  masahito suzuki
 * @since  PersistenceSession 1.00
 */
class CreateSession {
    
    /**
     * 生成セッションタイムアウト値.
     */
    private static final long TIMEOUT = 30000L ;
    
    /**
     * 管理テーブル.
     */
    private Map<String,Long> map = null ;
    
    /**
     * MDBM.
     */
    private SessionCache cacheSession = null ;
    
    /**
     * タイムアウト監視.
     */
    private CreateSessionMonThread thread = null ;
    
    /**
     * セッション生成オブジェクト.
     */
    private RandomId ssKey = null ;
    
    /**
     * コンストラクタ.
     */
    private CreateSession() {
    }
    
    /**
     * コンストラクタ.
     * @param cacheSession 対象のセッションキャッシュを設定します.
     * @exception Exception 例外.
     */
    protected CreateSession( SessionCache cacheSession ) throws Exception {
        if( cacheSession == null || cacheSession.isUse() == false ) {
            throw new IllegalArgumentException( "SessionCacheは不正です" ) ;
        }
        try {
            this.ssKey = new RandomId( PersistenceDefine.SESSION_KEY_LENGTH ) ;
            this.cacheSession = cacheSession ;
            this.map = new ConcurrentHashMap<String,Long>() ;
            this.thread = new CreateSessionMonThread( map,TIMEOUT ) ;
        } catch( Exception e ) {
            destroy() ;
            throw e ;
        }
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        destroy() ;
    }
    
    /**
     * オブジェクトを破棄.
     */
    public void destroy() {
        if( thread != null ) {
            thread.stopThread() ;
        }
        thread = null ;
        if( map != null ) {
            map.clear() ;
        }
        map = null ;
        cacheSession = null ;
        ssKey = null ;
    }
    
    /**
     * 新しいセッションキーを生成.
     * @param baseRandom ランダムオブジェクトを設定します.
     * @return String 新しいSessionKeyが返されます.
     * @exception Exception 例外.
     */
    public String createSessionKey( BaseRandom random )
        throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは無効な状態です" ) ;
        }
        String ret = null ;
        RandomId randKey = this.ssKey ;
        while( true ) {
            ret = randKey.get( random ) ;
            if( map.containsKey( ret ) || cacheSession.containsKey( ret ) ) {
                try {Thread.sleep( 1 ) ;} catch( Exception e ) {}
                continue ;
            }
            map.put( ret,new Long( System.currentTimeMillis() ) ) ;
            break ;
        }
        return ret ;
    }
    
    /**
     * 指定セッションIDが、存在するかチェック.
     * @param sessionId 対象のセッションIDを設定します.
     * @return boolean [true]の場合、セッションは存在します.
     */
    public boolean containsKey( String sessionId ) {
        if( sessionId == null ||
            ( sessionId = sessionId.trim() ).length() != PersistenceDefine.SESSION_KEY_LENGTH ) {
            return false ;
        }
        if( map.containsKey( sessionId ) ) {
            map.put( sessionId,new Long( System.currentTimeMillis() ) ) ;
            return true ;
        }
        return false ;
    }
    
    /**
     * オブジェクトが有効かチェック.
     * @return boolean [true]の場合、有効です.
     */
    public boolean isUse() {
        return ( map == null || cacheSession == null || thread == null ) ? false : true ;
    }
}

/**
 * タイムアウト監視スレッド.
 */
class CreateSessionMonThread extends LoopThread {
    private static final Log LOG = LogFactory.getLog( CreateSessionMonThread.class ) ;
    private long timeout = -1L ;
    private Map<String,Long> map = null ;
    
    private CreateSessionMonThread() {
        
    }
    
    public CreateSessionMonThread( Map<String,Long> map,long timeout )
        throws Exception {
        this.map = map ;
        this.timeout = timeout ;
        startThread() ;
    }
    
    protected void clear() {
        this.map = null ;
    }
    
    protected boolean execution() throws Exception {
        if( map.size() <= 0 ) {
            Thread.sleep( 500 ) ;
        }
        else {
            for( Iterator<String> keys = map.keySet().iterator() ; keys.hasNext() ; ) {
                Thread.sleep( 50 ) ;
                String key = keys.next() ;
                Long time = map.get( key ) ;
                if( time.longValue() + timeout <= System.currentTimeMillis() ) {
                    if( LOG.isDebugEnabled() ) {
                        LOG.debug( "## 生成セッション管理から[" + key + "]を削除" ) ;
                    }
                    keys.remove() ;
                }
            }
        }
        return false ;
    }
    
}

