package org.junitdoc.core.decisiontable;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jdt.core.dom.AnnotationTypeDeclaration;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jdt.core.dom.MarkerAnnotation;
import org.eclipse.jdt.core.dom.MethodDeclaration;
import org.eclipse.jdt.core.dom.Name;
import org.eclipse.jface.viewers.TreeNode;
import org.junitdoc.core.rewriter.ASTUtils;
import org.junitdoc.ui.Const;

public class DecisionTableModel {

	private String name;

	private String testClassName;

	private List<Type> types;

	private List<TestMethod> testMethods;

	private Name testPackageName;

	public DecisionTableModel(String name) {
		this.name = name;
	}

	public List<Type> getTypes() {
		return types;
	}

	public List<TestMethod> getTestMethods() {
		return testMethods;
	}

	public String getName() {
		return name;
	}

	public Name getTestPackageName() {
		return testPackageName;
	}

	public String getTestClassName() {
		return testClassName;
	}

	public void parse(CompilationUnit ast) {
		this.testPackageName = ast.getPackage().getName();
		this.testClassName = ast.getJavaElement().getElementName();

		AnnotationTypeDeclaration anno = findDecisionTable(ast);
		parseDecisionTable(anno);

		parseTestMethods(ast);
	}

	private void parseTestMethods(CompilationUnit ast) {

		List<MethodDeclaration> testAllMethods = ASTUtils
				.findMethodDeclarations(ast);

		parseTestMethods(testAllMethods);
	}

	private void parseCheckedCount(Type type) {

		type.clearCheckedCount();

		for (TestMethod testMethod : testMethods) {
			boolean checked = testMethod.isChecked(type);
			if (checked) {
				type.addCheckedCount();
			}
			List<Type> children = type.getChildren();
			for (Type child : children) {
				parseCheckedCount(child);
			}
		}
	}

	private void parseCheckedCount() {
		for (Type type : types) {
			parseCheckedCount(type);
		}
	}

	/**
	 * Parse testMethods. this method is called when testMethods are only
	 * changed.
	 * 
	 * @param testAllMethods
	 */
	public void parseTestMethods(List<MethodDeclaration> testAllMethods) {
		testMethods = new ArrayList<TestMethod>();

		int columnIndex = 0;

		for (MethodDeclaration anno : testAllMethods) {

			List modifiers = anno.modifiers();
			for (Object modifier : modifiers) {
				if (!(modifier instanceof MarkerAnnotation)) {
					continue;
				}

				if (ASTUtils.isConditionOrActionAnnotation(name,
						(MarkerAnnotation) modifier)) {

					TestMethod testMethod = new TestMethod(this, anno,
							columnIndex);
					testMethods.add(testMethod);

					columnIndex++;

					break;
				}
			}
		}
		parseCheckedCount();
	}

	private AnnotationTypeDeclaration findDecisionTable(CompilationUnit ast) {
		List<AnnotationTypeDeclaration> annoTypes = ASTUtils
				.getAnnotationTypeDeclarationsWithAnnotation(ast,
						Const.DECISIONTABLE_ANNOTATION);

		for (AnnotationTypeDeclaration annoType : annoTypes) {
			if (annoType.getName().getFullyQualifiedName().equals(name)) {
				return annoType;
			}
		}

		return null;
	}

	public void parseDecisionTable(AnnotationTypeDeclaration def) {

		// initialize
		types = new ArrayList<Type>();

		// no decisionTable
		if (def == null) {
			return;
		}

		List bodyDeclarations = def.bodyDeclarations();
		int currentDepth = 0;
		int nextDepth = currentDepth + 1;

		for (Object body : bodyDeclarations) {
			if (body instanceof AnnotationTypeDeclaration) {
				AnnotationTypeDeclaration anno = (AnnotationTypeDeclaration) body;
				String name = anno.getName().getFullyQualifiedName();

				Type type = createType(anno, currentDepth);

				List declarations = anno.bodyDeclarations();
				createChildType(declarations, type, nextDepth);

				types.add(type);
			}
		}
	}

	private void createChildType(List declarations, Type type, int depth) {
		if (declarations == null) {
			return;
		}

		int currentDepth = depth;
		int nextDepth = depth + 1;

		for (Object declaration : declarations) {
			if (declaration instanceof AnnotationTypeDeclaration) {
				AnnotationTypeDeclaration anno = (AnnotationTypeDeclaration) declaration;
				Type childType = createType(anno, currentDepth);

				type.addChild(childType);

				List bodyDeclarations = anno.bodyDeclarations();

				createChildType(bodyDeclarations, childType, nextDepth);
			}
		}
	}

	private Type createType(AnnotationTypeDeclaration anno, int depth) {
		Type type = new Type(anno, depth);
		return type;
	}

	public List<TestMethod> getTestMethodList() {
		return testMethods;
	}

	public TestMethod getTestMethod(int index) {
		return testMethods.get(index);
	}

	public TreeNode[] toTreeNodeModel() {

		TreeNode[] root = new TreeNode[types.size()];

		for (int i = 0; i < root.length; i++) {
			Type type = types.get(i);
			root[i] = new TreeNode(type);
			setChildren(root[i], type);
		}

		return root;
	}

	private void setChildren(TreeNode parent, Type type) {

		if (type.hasChildren()) {
			List<Type> children = type.getChildren();
			TreeNode[] childrenNode = new TreeNode[children.size()];
			for (int i = 0; i < childrenNode.length; i++) {
				Type child = children.get(i);
				childrenNode[i] = new TreeNode(child);
				childrenNode[i].setParent(parent);

				setChildren(childrenNode[i], child);
			}

			parent.setChildren(childrenNode);
		}
	}

	public List<Type> getTypeList() {
		List<Type> rtn = new ArrayList<Type>();
		for (Type type : types) {
			rtn.add(type);
			rtn.addAll(type.getTypeList());
		}
		return rtn;
	}

	public int getCheckedTotalCount(String typeName) {
		for (Type type : types) {
			if (type.getName().equals(typeName)) {
				return type.getCheckedTotalCount();
			}
		}
		return 0;
	}

}
