#include "stdafx.h"

#include "JavaVMMainThread.hpp"

#include "ExceptionFilter.hpp"

#include <assert.h>
#include <process.h>

namespace
{
	struct ClassInvokeInfo
	{
	public:
		JavaVMMainThread* pThis;
		jclass mainClass;
		jmethodID mainMethodID;
		const StartupInfo* pArgs;
		HANDLE hStarted;
	};
}

StartupInfo::StartupInfo( const std::string v_mainClassName, const std::string v_mainMethodName, bool v_fork )
	: mainClassName_( v_mainClassName )
	, mainMethodName_( v_mainMethodName )
	, fork_( v_fork )
{
	if( v_mainClassName.empty() ) {
		throw std::runtime_error( "CNXw肳Ă܂B" );
	}
	if( v_mainMethodName.empty() ) {
		throw std::runtime_error( "C\bhw肳Ă܂B" );
	}
}

StartupInfo::StartupInfo( const StartupInfo& v_other ) throw()
	: mainClassName_( v_other.mainClassName_ )
	, mainMethodName_( v_other.mainMethodName_ )
	, args_( v_other.args_ )
	, fork_( v_other.fork_ )
{
}

StartupInfo& StartupInfo::operator=( const StartupInfo& v_other ) throw()
{
	if( this != &v_other ) {
		mainClassName_ = v_other.mainClassName_;
		mainMethodName_ = v_other.mainMethodName_;
		args_ = v_other.args_;
		fork_ = v_other.fork_;
	}
	return *this;
}

StartupInfo::~StartupInfo() throw()
{
}

bool StartupInfo::isFork() const
{
	return fork_;
}

const std::string& StartupInfo::getMainClassName() const throw() 
{
	return mainClassName_;
}

const std::string& StartupInfo::getMainMethodName() const throw()
{
	return mainMethodName_;
}

StartupInfo::iterator StartupInfo::begin() throw()
{
	return args_.begin();
}

StartupInfo::const_iterator StartupInfo::begin() const throw()
{
	return args_.begin();
}

StartupInfo::iterator StartupInfo::end() throw()
{
	return args_.end();
}

StartupInfo::const_iterator StartupInfo::end() const throw()
{
	return args_.end();
}

StartupInfo::size_type StartupInfo::size() const throw()
{
	return args_.size();
}

bool StartupInfo::push_back( const std::string& v_arg ) throw()
{
	if( ! v_arg.empty() ) {
		args_.push_back( v_arg );
		return true;
	}
	return false;
}

///////////////////


JavaVMStartupInfo::JavaVMStartupInfo( const bool v_ignoreUnrecognized )
	: ignoreUnrecognized_( v_ignoreUnrecognized )
{
}

JavaVMStartupInfo::JavaVMStartupInfo( const JavaVMStartupInfo& v_other ) throw()
	: ignoreUnrecognized_( v_other.ignoreUnrecognized_ )
	, options_( v_other.options_ )
{
}

JavaVMStartupInfo& JavaVMStartupInfo::operator=( const JavaVMStartupInfo& v_other ) throw()
{
	if( this != &v_other ) {
		ignoreUnrecognized_ = v_other.ignoreUnrecognized_;
		options_ = v_other.options_;
	}
	return *this;
}

JavaVMStartupInfo::~JavaVMStartupInfo() throw()
{
}

bool JavaVMStartupInfo::isIgnoreUnrecognized() const throw()
{
	return ignoreUnrecognized_;
}

JavaVMStartupInfo::iterator JavaVMStartupInfo::begin() throw()
{
	return options_.begin();
}

JavaVMStartupInfo::const_iterator JavaVMStartupInfo::begin() const throw()
{
	return options_.begin();
}

JavaVMStartupInfo::iterator JavaVMStartupInfo::end() throw()
{
	return options_.end();
}

JavaVMStartupInfo::const_iterator JavaVMStartupInfo::end() const throw()
{
	return options_.end();
}

JavaVMStartupInfo::size_type JavaVMStartupInfo::size() const throw()
{
	return options_.size();
}

bool JavaVMStartupInfo::push_back( const std::string& v_arg ) throw()
{
	if( ! v_arg.empty() ) {
		options_.push_back( v_arg );
		return true;
	}
	return false;
}

///////////////

JavaVMMainThread::JavaVMMainThread( const JavaVMCallback& v_callback, const JavaVMStartupInfo& v_jvmStartupInfo )
	: jvm_( NULL )
	, jvmStartupInfo_( v_jvmStartupInfo )
	, callback_( v_callback )
{
}

JavaVMMainThread::~JavaVMMainThread() throw()
{
}

void JavaVMMainThread::start()
{
	//required:
	assert( jvm_ == NULL && "łJVMNĂ܂B" );
	if( jvm_ != NULL ) {
		throw std::logic_error( "łJVMNĂ܂B" );
	}

	//do:
	int callbackCount = 0;
	if( callback_.onJVMAbort    ) callbackCount++;
	if( callback_.onJVMExit     ) callbackCount++;
	if( callback_.onJVMVfprintf ) callbackCount++;

	JavaVMInitArgs vm_args;
	std::vector<JavaVMOption> options( jvmStartupInfo_.size() + callbackCount);
	int idx=0;
	for( StartupInfo::const_iterator p=jvmStartupInfo_.begin(), last=jvmStartupInfo_.end();
		p != last;
		++p, ++idx)
	{
		const std::string& option = *p;
		options[ idx ].optionString = (char*) option.c_str();
		options[ idx ].extraInfo = NULL;
	}

	if( callback_.onJVMExit ) {
		options[ idx ].optionString = "exit";
		options[ idx ].extraInfo = callback_.onJVMExit;
		idx++;
	}
	if( callback_.onJVMAbort ) {
		options[ idx ].optionString = "abort";
		options[ idx ].extraInfo = callback_.onJVMAbort;
		idx++;
	}
	if( callback_.onJVMVfprintf ) {
		options[ idx ].optionString = "vfprintf";
		options[ idx ].extraInfo = callback_.onJVMVfprintf;
		idx++;
	}

	vm_args.version = JNI_VERSION_1_2;
	vm_args.options = &options[0];
	vm_args.nOptions = (int) options.size();
	vm_args.ignoreUnrecognized = jvmStartupInfo_.isIgnoreUnrecognized() ? JNI_TRUE : JNI_FALSE;

	JNIEnv* env;
	if( JNI_CreateJavaVM( &jvm_, (void **)&env, &vm_args) != 0 ) {
		throw std::runtime_error( "JVM̋NɎs܂B" );
	}

	if( callback_.onJVMStarted ) {
		callback_.onJVMStarted();
	}
}

void JavaVMMainThread::invokeClassMethod( const StartupInfo& v_startupInfo, bool v_wait )
{
	//required:
	assert( jvm_ != NULL && "JVMJnĂ܂B" );
	if( jvm_ == NULL ) {
		throw std::logic_error( "JVMJnĂ܂B" );
	}

	//do:
	JNIEnv* env;
	if( jvm_->GetEnv( (void**) &env, JNI_VERSION_1_2 ) != JNI_OK || env == NULL ) {
		throw std::runtime_error( "XbhJVMɃA^b`Ă܂B" );
	}

	ClassInvokeInfo classInvokeInfo;
	classInvokeInfo.pThis = this;
	classInvokeInfo.pArgs = &v_startupInfo;

	const std::string mainClassName = ConvertToJNIClassName( v_startupInfo.getMainClassName() );
	classInvokeInfo.mainClass = env->FindClass( mainClassName.c_str() );
	if( classInvokeInfo.mainClass == NULL ) {
		std::string strErrDescription( "CNX݂܂B" );
		GetExceptionDescription( env, strErrDescription );
		throw std::runtime_error( strErrDescription.c_str() );
	}
	const std::string mainMethodName = v_startupInfo.getMainMethodName();
	classInvokeInfo.mainMethodID = env->GetStaticMethodID( classInvokeInfo.mainClass, mainMethodName.c_str(), "([Ljava/lang/String;)V");
	if( classInvokeInfo.mainMethodID == NULL ) {
		std::string strErrDescription( "static void main( String[] )擾ł܂B" );
		GetExceptionDescription( env, strErrDescription );
		throw std::runtime_error( strErrDescription.c_str() );
	}

	classInvokeInfo.hStarted = CreateEvent( NULL, TRUE, FALSE, NULL );
	if( classInvokeInfo.hStarted == NULL ) {
		throw std::runtime_error( "Cxgnh̍쐬Ɏs܂B" );
	}

	unsigned int threadid = 0;
	HANDLE hThread = (HANDLE) _beginthreadex(
		NULL,
		0,
		invokeClassMethodProc,
		&classInvokeInfo,
		0,
		&threadid
		);
	if( hThread == NULL ) {
		CloseHandle( classInvokeInfo.hStarted );
		throw std::runtime_error( "Xbh쐬ł܂B" );
	}
	WaitForSingleObject( classInvokeInfo.hStarted, INFINITE );
	CloseHandle( classInvokeInfo.hStarted );

	if( v_wait ) {
		WaitForSingleObject( hThread, INFINITE );
	}
	CloseHandle( hThread );
}

void JavaVMMainThread::join() 
{
	assert( jvm_ != NULL && "JVMJnĂ܂B" );
	if( jvm_ == NULL ) {
		throw std::logic_error( "JVMJnĂ܂B" );
	}

	jvm_->DestroyJavaVM();

	if( callback_.onJVMNormalExit ) {
		callback_.onJVMNormalExit();
	}
}

void JavaVMMainThread::stopInterrupt() 
{
	//required:
	assert( jvm_ != NULL && "JVM쐬Ă܂B" );
	if( jvm_ == NULL ) {
		throw std::logic_error( "JVM쐬Ă܂B" );
	}

	//do:
	bool attached = false;
	JNIEnv* env;
	if( jvm_->GetEnv( (void**) &env, JNI_VERSION_1_2 ) != JNI_OK || env == NULL ) {
		JavaVMAttachArgs attach_args;
		attach_args.version = JNI_VERSION_1_2;
		attach_args.name = NULL;
		attach_args.group = NULL;
		if( jvm_->AttachCurrentThread( (void**) &env, &attach_args ) != 0 ) {
			throw std::runtime_error( "XbhɃA^b`ł܂B" );
		}
		attached = true;
	}

	std::string errDescription("System#exit()Ɏs܂B");
	bool raisedException = false;
	jclass systemClass = env->FindClass( "java/lang/System" );
	if( ( raisedException = GetExceptionDescription( env, errDescription ) ) == false ) {
		jmethodID exitMethod = env->GetStaticMethodID( systemClass, "exit", "(I)V");
		if( ( raisedException = GetExceptionDescription( env, errDescription ) ) == false ) {
			env->CallStaticVoidMethod( systemClass, exitMethod, (jint)0 );
			//: System#exitɌĂяoꂽꍇAexitR[obNĂяo邽ߐ͖߂܂B
			raisedException = GetExceptionDescription( env, errDescription );
		}
	}

	if( attached ) {
		jvm_->DetachCurrentThread();
	}

	if( raisedException ) {
		throw std::runtime_error( errDescription.c_str() );
	}
}

std::string JavaVMMainThread::ConvertToJNIClassName( const std::string& v_className ) throw() {
	std::vector<char> buf( v_className.size() + 1 );
	strcpy( &buf[0], v_className.c_str() );
	for( std::vector<char>::iterator p=buf.begin(), last=buf.end();
		p != last;
		++p)
	{
		if( *p == '.' ) {
			*p = '/';
		}
	}
	return std::string( &buf[0] );
}

bool JavaVMMainThread::GetExceptionDescription( JNIEnv* env, std::string& v_description ) throw()
{
	//required:
	assert( env != NULL );

	//do:
	jthrowable err = env->ExceptionOccurred();
	env->ExceptionClear();
	if( err != NULL ) {
		jclass throwableClass = env->GetObjectClass( err );
		jmethodID toStringMethodID = env->GetMethodID( throwableClass, "toString", "()Ljava/lang/String;" );
		assert( toStringMethodID != NULL );
		jstring errDescription = reinterpret_cast<jstring>(env->CallObjectMethod( err, toStringMethodID ));
		const char* szErrDescription = env->GetStringUTFChars( errDescription, NULL );
		v_description += szErrDescription;
		env->ReleaseStringUTFChars( errDescription, szErrDescription );
		return true;
	}
	return false;
}

unsigned __stdcall JavaVMMainThread::invokeClassMethodProc( void* v_pData )
{
	UnhandledExceptionHandler::setUnhandledExceptionHandler();

	//required:
	assert( v_pData != NULL );
	ClassInvokeInfo classInvokeInfo = *reinterpret_cast<ClassInvokeInfo*>( v_pData );
	assert( classInvokeInfo.pThis != NULL );
	JavaVM* jvm = classInvokeInfo.pThis->jvm_;

	assert( jvm != NULL && "JVM쐬Ă܂B" );
	if( jvm == NULL ) {
		throw std::logic_error( "JVM쐬Ă܂B" );
	}
	assert( classInvokeInfo.mainClass != NULL && "CNXw肳Ă܂B" );
	if( classInvokeInfo.mainClass == NULL ) {
		throw std::logic_error( "CNXw肳Ă܂B" );
	}
	assert( classInvokeInfo.mainMethodID != NULL && "C\bhw肳Ă܂B" );
	if( classInvokeInfo.mainMethodID == NULL ) {
		throw std::logic_error( "C\bhw肳Ă܂B" );
	}

	//do:
	try{
		JavaVMAttachArgs attach_args;
		attach_args.version = JNI_VERSION_1_2;
		attach_args.name = NULL;
		attach_args.group = NULL;

		// JVMɃXbhA^b`B
		JNIEnv* env;
		if( jvm->AttachCurrentThread( (void**) &env, &attach_args ) != 0 ) {
			throw std::runtime_error( "XbhɃA^b`ł܂B" );
		}

		// NIvV̍\z
		jclass stringClass = env->FindClass( "java/lang/String" );
		if( stringClass == NULL ) {
			throw std::runtime_error( "java/lang/string̎擾Ɏs܂B" );
		}
		jobjectArray args = env->NewObjectArray( (jsize) classInvokeInfo.pArgs->size(), stringClass, NULL );
		if( args == NULL ) {
			throw std::runtime_error( "java/lang/string[]̎擾Ɏs܂B" );
		}
		int idx=0;
		for( StartupInfo::const_iterator p=classInvokeInfo.pArgs->begin(), last=classInvokeInfo.pArgs->end();
			p != last;
			++p, ++idx)
		{
			std::string errDescription( "main\bh̐ݒɎs܂B" );
			const std::string arg = *p;
			jstring arg1 = env->NewStringUTF( arg.c_str() );
			if( arg1 == NULL ) {
				throw std::runtime_error( errDescription.c_str() );
			}
			env->SetObjectArrayElement( args, idx, arg1 );
			if( GetExceptionDescription( env, errDescription ) ) {
				throw std::runtime_error( errDescription.c_str() );
			}
		}

		// ĂяoƂ̑ҋ@
		SetEvent( classInvokeInfo.hStarted );

		// \bh̋N
		std::string errDescription( "C\bh̎sɗO܂B" );
		env->CallStaticVoidMethod( classInvokeInfo.mainClass, classInvokeInfo.mainMethodID, args );
		const bool raisedException = GetExceptionDescription( env, errDescription );

		jvm->DetachCurrentThread();

		if( raisedException ) {
			throw std::runtime_error( errDescription.c_str() );
		}
	}
	catch( const std::exception& v_exception ) {
		if( classInvokeInfo.pThis->callback_.onJVMNotifyException ) {
			classInvokeInfo.pThis->callback_.onJVMNotifyException( v_exception.what() );
		}
	}
	return 0;
}
