// $Id: BS2TclDevice.cpp,v 1.1.1.1 2002/08/31 04:47:24 fukasawa Exp $

//=============================================================================
/**
 *  @file    BS2TclDevice.cpp
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2001 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#include "BS2TclDevice.h"
#include "BS2Device.h"
#include "BS2Message.h"
#include "BS2MsgControl.h"
#include "tcl.h"

//extern "C" {
//int ThreadSvSetObjCmd(ClientData clientData, Tcl_Interp *interp, int objc,
//                      struct Tcl_Obj * CONST * objv);
//}

extern BS2QueueMessage(BS2Message * msg);

static BS2TclDevice * _tcl_device = NULL;

//-----------------------------------------------------------------------------
// timeout of Delay Time Handler
//-----------------------------------------------------------------------------
void TCLDelayHandler::handle_time_out(const ACE_Time_Value &tv,
                                     const void * arg)
{
    TRACE_FUNCTION(TRL_LOW, "TCLDelayHandler::handle_time_out");
    BS2TclDevice * mngr = (BS2TclDevice *)arg;
    mngr->accept();
    mngr->timerId(INIT_TIMERID);     // Clear timer id
}

//-----------------------------------------------------------------------------
// Constructor/Destructor
//-----------------------------------------------------------------------------
BS2TclDevice::BS2TclDevice(char * name)
        : m_name(name), m_device(NULL), m_interp(NULL), m_threadId(0),
          m_varname("")
{
    TRACE_FUNCTION(TRL_CONSTRUCT, "BS2TclDevice::BS2TclDevice");
    m_tv = ACE_Time_Value::zero;
    m_logmask = ACE_Log_Msg::instance()->priority_mask();
}

//-----------------------------------------------------------------------------
// Open and import database.
//-----------------------------------------------------------------------------
int BS2TclDevice::open(void * parm)
{
    TRACE_FUNCTION(TRL_LOW, "BS2TclDevice::open");

    DeviceParameter * dev_parm = (DeviceParameter *)parm;
    m_device = new BS2Device;
    m_device->setName(m_name.c_str());
    if (m_device->initialize(dev_parm) < 0)
    {
        ACE_ERROR((LM_ERROR,
                   ACE_TEXT("Tcl manager does not initialize device.\n")));
        delete m_device;
        m_device = NULL;
        return BS_ERROR;
    }

    m_tv.set(1, 0);      // set timeout value of message queue (1 sec)

    if (this->activate(THR_NEW_LWP | THR_DETACHED) == -1)
        ACE_ERROR_RETURN((LM_ERROR, "%p\n", "spawn"), BS_ERROR);

    m_device->open();      // start to receive message

    return BS_SUCCESS;
}

//-----------------------------------------------------------------------------
// Close .
//-----------------------------------------------------------------------------
int BS2TclDevice::close(int flag)
{
    TRACE_FUNCTION(TRL_LOW, "BS2TclDevice::close");
    m_device->close();      // start to receive message
    delete m_device;
    return 0;
}

//-----------------------------------------------------------------------------
// Start Delay Timer for Manager.
//-----------------------------------------------------------------------------
int BS2TclDevice::startTimer(long sec)
{
    TRACE_FUNCTION(TRL_LOW, "BS2TclDevice::startTimer");
    ACE_Time_Value tv(sec);
    m_tmid = ACE_Proactor::instance()->schedule_timer(m_timer, this, tv);
    if (m_tmid == ERROR_TIMERID)
    {
        ACE_ERROR_RETURN ((LM_ERROR, "%p\n", "schedule_timer"), BS_ERROR);
    }
    return BS_SUCCESS;
}

//-----------------------------------------------------------------------------
// Start Delay Timer for Manager.
//-----------------------------------------------------------------------------
int BS2TclDevice::cancelTimer()
{
    TRACE_FUNCTION(TRL_LOW, "BS2TclDevice::cancelTimer");

    if (m_tmid != INIT_TIMERID)
    {
        ACE_Proactor::instance()->cancel_timer(m_tmid);
        m_tmid = INIT_TIMERID;
    }
    else
    {
        ACE_ERROR_RETURN ((LM_ERROR, "timer id is not found.\n"), BS_ERROR);
    }
    return BS_SUCCESS;
}

//-----------------------------------------------------------------------------
// Stop loop in thread.
//-----------------------------------------------------------------------------
int BS2TclDevice::stopLoop()
{
    TRACE_FUNCTION(TRL_LOW, "BS2TclDevice::stopLoop");

    ACE_Message_Block * mb;
    ACE_NEW_RETURN(mb, ACE_Message_Block(BUFSIZ + 1), BS_ERROR);

    // Send a shutdown message to the other thread and exit.
    mb->length(0);
    if (this->put(mb) == -1)
        ACE_ERROR((LM_ERROR, "(%t) %p\n", "put"));

    return BS_SUCCESS;
}

//-----------------------------------------------------------------------------
// Accept trigger from applications.
//-----------------------------------------------------------------------------
int BS2TclDevice::accept()
{
    TRACE_FUNCTION(TRL_LOW, "BS2TclDevice::accept");

    ACE_Message_Block * mb;

    int len = 256;
    ACE_NEW_RETURN(mb, ACE_Message_Block(len + 32), BS_ERROR);
    mb->length(len);
    strcpy(mb->rd_ptr(), "Trigger");

    if (this->put(mb) == BS_ERROR)
    {
        ACE_ERROR ((LM_ERROR, "(%t) %p\n", "send trigger that is accepted"));
    }
    return BS_SUCCESS;
}

//-----------------------------------------------------------------------------
// Send message.
//-----------------------------------------------------------------------------
int BS2TclDevice::send(BS2Message * msg)
{
    TRACE_FUNCTION(TRL_LOW, "BS2TclDevice::send");

    int t_id = m_device->send(msg);
    return t_id;
}

//-----------------------------------------------------------------------------
// Receive message thread.
//-----------------------------------------------------------------------------
int BS2TclDevice::svc(void)
{
    ACE_DEBUG((LM_DEBUG, "BS2TclDevice::svc start.\n"));
    int result = 0;
    BS2Message * msg = NULL;
    BS2MsgControl msginfo;
    ACE_Time_Value mb_tv(1);
    int wait = 1;              // async

    for (;;)
    {
        ACE_Message_Block * mb;
        ACE_Log_Msg::instance()->priority_mask(this->logmask());

        result = this->getq(mb, &mb_tv);
        if (result == -1)
        {   // time out
            if ((result = m_device->receive(msginfo)) >= 0)
            {
                msg = msginfo.message();
                if ((msginfo.getResult() == BS2RET_NORMAL) && (msg != NULL))
                {
                    msg->dump();    // for DEBUG

                    BS2QueueMessage(msg);

                    // Notify receipt message to Tcl
                    Tcl_SetVar(m_interp.value(), (char *)m_varname.c_str(), "1", TCL_GLOBAL_ONLY);
                    // sprintf(buf, "set %s 1", m_varname.c_str());
                    // ThreadSend(m_interp.value(), m_threadId, buf, wait);
					//Tcl_Obj * objs[5];
                    //objs[0] = Tcl_NewStringObj("thread::sv_set", -1);
                    //objs[1] = Tcl_NewStringObj("rcv_secs", -1);
                    //objs[2] = Tcl_NewStringObj(m_varname.c_str(), -1);
                    //objs[3] = Tcl_NewStringObj("1", -1);
                    //objs[4] = NULL;
					//ThreadSvSetObjCmd(NULL, m_interp.value(), 4, objs);
                }
                else
                {
                    ACE_ERROR((LM_ERROR, 
                              "JGDevice: Received time out message.\n"));
                }
            }
            else
            {
                ACE_ERROR((LM_ERROR,
                           "JGDevice: Received error.\n"));
            }
        }
        else
        {
            int length = mb->length();
            if (length > 0)
            {
                char * top = mb->rd_ptr();

                ACE_ERROR((LM_ERROR, "BS2TclDevice: Received unexpected message.\n"));
            }
            mb->release ();

            if (length == 0)             // shutdown
                break;
        }
    }

    ACE_DEBUG((LM_DEBUG, "BS2TclDevice is deleted.\n"));
    return 0;
}

