// $Id: b_objtype.cpp,v 1.1.1.1 2002/08/31 04:47:22 fukasawa Exp $

//=============================================================================
/**
 *  @file    b_objtype.cpp
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2001-2002 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#define BEE_BUILD_DLL

#include "b_objtype.h"
#include "b_object.h"
#include "b_attribute.h"

//-----------------------------------------------------------------------------
// Make object idenntifier
//-----------------------------------------------------------------------------
b_objtype::~b_objtype()
{
    TRACE_FUNCTION(TRL_LOW, "b_objtype::~b_objtype");

    // Release all objects
    b_objectMap::iterator iter = m_objs.begin();
    for ( ; iter != m_objs.end(); iter++)
    {
        b_object * obj = (*iter).second;
        delete obj;
        ObjSpec::instance()->remove(obj);
    }
    m_objs.clear();

    // Release all attributes
    for (int i = 0; i < m_attrs.size(); i++)
    {
        b_attribute * attr = m_attrs[i];
        delete attr;
    }
    m_attrs.clear();
    m_dict.clear();

    ObjSpec::instance()->remove(this);

}

//-----------------------------------------------------------------------------
// Make object idenntifier
//-----------------------------------------------------------------------------
int b_objtype::makeObjID(string& retid)
{
    TRACE_FUNCTION(TRL_LOW, "b_objtype::makeObjID");

    BCHAR buf[256+2];
    u_int oid = m_seed++;
    _stprintf(buf, _TX("%s_%u"), m_name.c_str(), oid);
    retid = buf;
    return BEE_SUCCESS;
}

//-----------------------------------------------------------------------------
// Add attribute
//-----------------------------------------------------------------------------
int b_objtype::add(b_attribute * attr)
{
    TRACE_FUNCTION(TRL_LOW, "b_objtype::add");

    m_attrs.push_back(attr);
    attr->position(m_attrs.size() - 1);     // Set position of object
    m_dict.insert(b_attrPair(attr->name(), attr));
    return BEE_SUCCESS;
}

//-----------------------------------------------------------------------------
// Find attribute by name
//-----------------------------------------------------------------------------
const b_attribute * b_objtype::find(const string& name) const
{
    TRACE_FUNCTION(TRL_LOW, "b_objtype::find");

    b_attrMap::const_iterator iter = m_dict.find(name);
    if (iter == m_dict.end())
    {
        return NULL;
    }
    const b_attribute * result = (*iter).second;
    return result;
}

//-----------------------------------------------------------------------------
// Create object
//-----------------------------------------------------------------------------
b_object * b_objtype::instance(const char * name)
{
    TRACE_FUNCTION(TRL_LOW, "b_objtype::instance");

    BEE_GUARD_RETURN(ACE_Thread_Mutex, ace_mon, this->m_lock, NULL,
                     10, 200);

    b_object * obj = new b_object(this, name);
    m_objs.insert(b_objectPair(obj->getObjID(), obj));
    ObjSpec::instance()->addObject(obj);
    return obj;
}

//-----------------------------------------------------------------------------
int b_objtype::entry(b_object * obj)
{
    TRACE_FUNCTION(TRL_LOW, "b_objtype::entry");
    
    BEE_GUARD_RETURN(ACE_Thread_Mutex, ace_mon, this->m_lock, NULL,
                     10, 200);

    m_objs.insert(b_objectPair(obj->getObjID(), obj));
    ObjSpec::instance()->addObject(obj);
    return BEE_SUCCESS;
}

//-----------------------------------------------------------------------------
// Remove object
//-----------------------------------------------------------------------------
int b_objtype::remove(b_object * obj)
{
    TRACE_FUNCTION(TRL_LOW, "b_objtype::remove");

    BEE_GUARD_RETURN(ACE_Thread_Mutex, ace_mon, this->m_lock, NULL,
                     10, 300);

    ObjSpec::instance()->remove(obj);
    size_t q = m_objs.erase(obj->getObjID());
    if (q != 1)
    {
        return BEE_ERROR;
    }
    return BEE_SUCCESS;
}


//-----------------------------------------------------------------------------
// Get instances
//-----------------------------------------------------------------------------
int b_objtype::allObject(vector<b_object *>& objects)
{
    TRACE_FUNCTION(TRL_LOW, "b_objtype::allObject");

    BEE_GUARD_RETURN(ACE_Thread_Mutex, ace_mon, this->m_lock, NULL,
                     2, 400);

    objects.clear();
    b_objectMap::iterator iter = m_objs.begin();
    for ( ; iter != m_objs.end(); iter++)
    {
        b_object * obj = (*iter).second;
        objects.push_back(obj);
    }

    return objects.size();

}

//-----------------------------------------------------------------------------
// Dump
//-----------------------------------------------------------------------------
void b_objtype::dump() const
{
    ACE_DEBUG((LM_DEBUG, ACE_TEXT("\n*** Class: %s ***\n"), m_name.c_str()));
    ACE_DEBUG((LM_DEBUG, ACE_TEXT("  current seed-num is %d.\n"), m_seed.value()));
    ACE_DEBUG((LM_DEBUG, ACE_TEXT("  [ attributes ]\n")));
    b_attributes::const_iterator at_iter = m_attrs.begin();
    for ( ; at_iter != m_attrs.end(); at_iter++)
    {
        const b_attribute * attr = *at_iter;
        attr->dump();
    }

    ACE_DEBUG((LM_DEBUG, ACE_TEXT("  [ objects ]\n")));
    b_objectMap::const_iterator obj_iter = m_objs.begin();
    for ( ; obj_iter != m_objs.end(); obj_iter++)
    {
        const b_object * obj = (*obj_iter).second;
        ACE_DEBUG((LM_DEBUG, ACE_TEXT("    %s\n"), obj->getObjID()));
    }
}

