# Kanbanara Authenticate Component
# Written by Rebecca Shalfield between 2013 and 2018
# Copyright (c) 2013-2018 Rebecca Shalfield and Kanbanara Software Foundation
# Released under the GNU AGPL v3

'''Kanbanara's Authenticate Component'''

import datetime
from email.mime.text import MIMEText
import os
import random
import re
import smtplib
import socket
import sys

import bson
from bson import ObjectId
import cherrypy
from pymongo import MongoClient


class Authenticate:
    '''Kanbanara's Authenticate Component'''
    
    CURRENT_DIR = os.path.dirname(os.path.abspath(__file__))

    def __init__(self):
        self.read_mongodb_ini_file()

        # Connect to MongoDB on given host and port
        connection = MongoClient(self.mongodb_host, self.mongodb_port)

        # Connect to 'vault' database, creating if not already exists
        vault_db = connection['vault']

        # Connect to 'members' collection
        self.members_collection = vault_db['members']

        self.kanbanara_settings = {'instance': "Untitled"}
        self.read_administrator_ini_file()
        self.read_kanbanara_ini_file()
        self.instance = self.kanbanara_settings['instance']

        self.major, self.minor, self.revision, self.build, self.date = self.read_version_ini_file()

    @cherrypy.expose
    def change_password(self, returnurl="", username="", existing_password="", new_password="",
                       new_password_confirm=""):
        content = []
        content.append(self.header(returnurl, title='Change Password'))
        content.append(('<div id="maincontent">'
                        '<h3 class="subheading">Change Password</h3>'))
        if username and existing_password and new_password and new_password_confirm:
            if self.members_collection.count({'username': username,
                                              'password': existing_password
                                             }):
                if new_password == new_password_confirm:
                    self.members_collection.find_one_and_update(
                            {'username': username,
                             'password': existing_password
                            },
                            {'$set': {'password': new_password}})

                    subject = "Kanbanara - Password Changed"
                    message = (f'Hi,\n\n'
                               'Someone, hopefully yourself, has changed your Kanbanara password.\n\n'
                               f"Your old password was '{existing_password}'.\n\n"
                               f"Your new password is '{new_password}'.\n\n"
                               'The Kanbanara Team')
                    self.send_email(lowercase_username, subject, message)    
    
                    message = "Your password has now been changed!"
                    raise cherrypy.HTTPRedirect(f'/authenticate/confirmation?message={message}&returnurl={returnurl}', 302)
                else:
                    content.append(('<div class="warning">'
                                    '<p class="warning">New passwords do not match!</p>'
                                    '</div>'))

            else:
                content.append('<div class="warning"><p class="warning">Access denied!</p></div>')

        content.append(('<form action="/authenticate/change_password" method="post">'
                        f'<input type="hidden" name="returnurl" value="{returnurl}">'
                        '<table border="0"><tr><td align="right">Username (Email Address)</td><td>'
                        '<input type="email" size="40" name="username" '
                        'pattern="[a-z0-9._%+-]+@[a-z0-9.-]+\.[a-z]{2,}$"'))
        if username:
            content.append(f' value="{username}"')

        content.append((' placeholder="Email Address" required>'
                        '</td></tr>'
                        '<tr><td align="right">Existing Password</td><td>'
                        '<input type="password" size="40" name="existing_password" '
                        'pattern=".{4,}"'))
        if existing_password:
            content.append(f' value="{existing_password}"')

        content.append((' placeholder="Existing Password" required>'
                        '</td></tr><tr><td align="right">New Password</td>'
                        '<td><input type="password" size="40" name="new_password" '
                        'pattern=".{4,}" placeholder="New Password" required></td></tr>'
                        '<tr><td align="right">New Password Confirmation</td>'
                        '<td><input type="password" size="40" name="new_password_confirm" '
                        'pattern=".{4,}" placeholder="New Password" required></td></tr>'
                        '<tr><td align="center" colspan="2">'
                        '<button type="submit" value="Change Password">'
                        '<span class="fas fa-unlock-alt fa-lg"></span>&nbsp;Change Password</button>'
                        '</td></tr></table></form>'))
        content.append(self.insert_return_link(returnurl))
        content.append(self.insert_links('change_password', username, returnurl))
        content.append('</div>')
        content.append(self.footer())
        return "".join(content)

    def insert_links(self, page, username, returnurl):
        content = []
        content.append('<hr>')
        socket_hostname = socket.gethostname()
        if (socket_hostname == 'www.kanbanara.com' or
                os.path.exists(os.path.join(self.CURRENT_DIR, '..', 'website'))):
            content.append(('<form class="inline" action="/" method="post">'
                            '<button type="submit" value="Website">'
                            '<span class="fas fa-globe fa-lg"></span>&nbsp;Website</button>'
                            '</form> '))

        if page != 'index':
            content.append('<form class="inline" action="/authenticate/index" method="post">')
            if returnurl:
                content.append(f'<input type="hidden" name="returnurl" value="{returnurl}">')

            content.append(('<button type="submit" value="Logon">'
                            '<span class="fas fa-sign-in-alt fa-lg"></span>&nbsp;Logon</button>'
                            '</form> '))

        if page != 'register':
            content.append('<form class="inline" action="/authenticate/register" method="post">')
            if returnurl:
                content.append(f'<input type="hidden" name="returnurl" value="{returnurl}">')

            content.append(('<button type="submit" value="Register">'
                            '<span class="fas fa-registered fa-lg"></span>&nbsp;Register</button>'
                            '</form> '))

        if page != 'change_password':
            content.append('<form class="inline" action="/authenticate/change_password" method="post">')
            if returnurl:
                content.append(f'<input type="hidden" name="returnurl" value="{returnurl}">')

            if username:
                content.append(f'<input type="hidden" name="username" value="{username}">')

            content.append(('<button type="submit" value="Change Password">'
                            '<span class="fas fa-unlock-alt fa-lg"></span>&nbsp;Change Password</button>'
                            '</form> '))

        if page != 'reset_password':
            content.append('<form class="inline" action="/authenticate/reset_password" method="post">')
            if returnurl:
                content.append(f'<input type="hidden" name="returnurl" value="{returnurl}">')

            if username:
                content.append(f'<input type="hidden" name="username" value="{username}">')

            content.append(('<button type="submit" value="Reset Password">'
                            '<span class="fas fa-unlock-alt fa-lg">'
                            '</span>&nbsp;Reset Password</button>'
                            '</form>'))

        return "".join(content)

    @cherrypy.expose
    def confirmation(self, message, returnurl=""):
        content = []
        content.append(self.header('', title='Confirmation'))
        content.append(('<div id="maincontent">'
                        '<h3 class="subheading">Confirmation</h3>'
                        f'<p>{message}</p>'))
        if returnurl:
            if returnurl != '/kanban':
                content.append(('<p class="returnurl">'
                                f'You\'ll be redirected to {returnurl} in a few moments...</p>'))

        content.append(self.insert_links('logoff', '', ''))
        content.append('</div>')
        content.append(self.footer())
        return "".join(content)

    @cherrypy.expose
    def delete_member(self, username, password):
        content = []
        content.append(self.header('', title='Delete Member'))
        content.append(('<div id="maincontent">'
                        '<h3 class="subheading">Delete Member</h3>'))
        if self.members_collection.count({'username': username, 'password': password}):
            self.members_collection.find_one_and_delete({'username': username,
                                                         'password': password
                                                        })
            content.append(f"<p>Member '{username}' has now been deleted!</p>")
        else:
            content.append('<p class="warning">Username and/or password are incorrect!</p>')

        content.append('</div>')
        content.append(self.footer())
        return "".join(content)

    def footer(self):
        content = []
        content.append(('<div id="footer_container">'
                        '<div id="footer_contents"><h3>Copyright &copy; Rebecca Shalfield and '
                        'Kanbanara Software Foundation 2013-2018</h3></div>'
                        '</div>'
                        '</body></html>'))
        return "".join(content)

    def header(self, returnurl, title):
        content = []
        content.append(f'<html><head><title>Kanbanara - {title}</title>')
        for css_path in ['/authenticate/css/authenticate.css',
                         '/css/rootmenuwebapps.css',
                         '/fontawesome-free-5.0.10/web-fonts-with-css/css/fontawesome-all.min.css']:
            content.append(f'<link rel="stylesheet" type="text/css" href="{css_path}">')
        
        if title == 'Confirmation' and returnurl:
            content.append(f'<meta http-equiv="refresh" content="5;URL={returnurl}">')

        if title == 'Registration':
            for css_path in ['/jquery-ui-1.12.1.custom/jquery-ui.css',
                             '/jquery-ui-1.12.1.custom/jquery-ui.structure.css',
                             '/jquery-ui-1.12.1.custom/jquery-ui.theme.css']:
                content.append(f'<link rel="stylesheet" type="text/css" href="{css_path}">')
            
            for script_path in ['/jquery-ui-1.12.1.custom/external/jquery/jquery.js',
                                '/jquery-ui-1.12.1.custom/jquery-ui.min.js',
                                '/authenticate/scripts/authenticate.js']:
                content.append(f'<script type="text/javascript" src="{script_path}"></script>')

        slides = os.listdir(os.path.join(self.CURRENT_DIR, 'images', 'slideshow'))
        if slides:
            random_index = random.randrange(0, len(slides))
            slide = slides[random_index]
            content.append("<style>body {background-image : url('/authenticate/images/slideshow/"+slide+"')}</style>")

        content.append(('</head><body>'
                        '<h1 class="header">Kanbanara</h1>'
                        '<h2>A Kanban-Based Project Management System</h2>'
                        f'<h2>Version {self.major}.{self.minor}.{self.revision}.{self.build} ({self.date})</h2>'))
        return "".join(content)

    def read_kanbanara_ini_file(self):
        """Reads the kanbanara.ini file and extracts instance information"""
        # TODO - We can switch to using configParser in Kanbanara version 2
        with open(os.path.join(self.CURRENT_DIR, '..', 'kanbanara.ini'), 'r') as handle:
            data = handle.read()
            for (setting, regex) in [('instance', r'(?i)instance\s?=\s?([\w\. ]+)')]:
                pattern = re.compile(regex)
                results = pattern.findall(data)
                if results:
                    self.kanbanara_settings[setting] = results[0]

    @cherrypy.expose
    def index(self, returnurl="", username="", password=""):
        if username and password:
            if self.members_collection.count({'username': username, 'password': password}):
                member_document = self.members_collection.find_one({'username': username})
                cherrypy.session['username']  = username
                cherrypy.session['firstname'] = member_document['firstname']
                cherrypy.session['lastname']  = member_document['lastname']
                if returnurl:
                    raise cherrypy.HTTPRedirect(returnurl, 302)
                else:
                    raise cherrypy.HTTPRedirect('/kanban', 302)

            else:
                raise cherrypy.HTTPError(401, 'Unauthorised')

        content = []
        content.append(self.header(returnurl, title='Logon'))
        content.append(('<div id="maincontent">'
                        '<h3 class="subheading">Logon</h3>'
                        '<table class="outer"><tr><td class="outer">'
                        '<p>If you\'re already registered, you may logon here<br>'
                        'by entering your username and password</p>'
                        '<form action="/authenticate/index" method="post">'))
        if returnurl:
            content.append(f'<input type="hidden" name="returnurl" value="{returnurl}">')

        content.append(('<table class="logon">'
                        f'<tr><td><p>{self.instance}</p></td></tr>'
                        '<tr><td><p><span class="fas fa-user fa-lg" title="Username"></span> '
                        '<input type="email" size="40" name="username" '
                        'pattern="[a-z0-9._%+-]+@[a-z0-9.-]+\.[a-z]{2,}$"'))
        if username:
            content.append(f' value="{username}"')

        content.append((' placeholder="Username" required></p></td></tr>'
                        '<tr><td><p><span class="fas fa-lock fa-lg" title="Password"></span> '
                        '<input type="password" size="40" name="password" '
                        'pattern=".{4,}" placeholder="Password" required>'
                        '</p></td></tr><tr><td align="center"><p>'
                        '<button type="submit" value="Logon">'
                        '<span class="fas fa-sign-in-alt fa-lg"></span>&nbsp;Logon</button>'
                        '</p></td></tr></table></form>'))
        content.append(self.insert_return_link(returnurl))
        content.append('</td></tr></table>')
        content.append(self.insert_links('index', username, returnurl))
        content.append('</div>')
        content.append(self.footer())
        return "".join(content)
        
    def insert_return_link(self, returnurl):
        content = []
        if returnurl and returnurl not in ['/kanban', '/members/logoff']:
            content.append(('<p class="returnurl">You will be returned to '
                            f'<a href="{returnurl}">{returnurl}</a> '
                            'following a successful logon!</p>'))
        
        return "".join(content)

    @cherrypy.expose
    def validateusername(self, username):
        warning = ""
        if self.members_collection.count({'username': username}):
            warning = '<p class="warning">Username already in use!</p>'

        if not re.search(r'^[\w_\-\.]+@\S+\.\w+$',username):
            warning = '<p class="warning">Not a valid email address!</p>'

        return warning

    def read_version_ini_file(self):
        """Reads the version.ini file and extracts versioning information"""
        # TODO - We can switch to using configParser in Kanbanara version 2
        major = 0
        minor = 0
        revision = 0
        build = 0
        date = ""
        
        with open(os.path.join(self.CURRENT_DIR, '..', 'version.ini'), 'r') as ip:
            settings = ip.read()
            
            major_pattern = re.compile(r'(?i)major\s?=\s?(\d+)')
            results = major_pattern.findall(settings)
            if results:
                major = int(results[0])        
            
            minor_pattern = re.compile(r'(?i)minor\s?=\s?(\d+)')
            results = minor_pattern.findall(settings)
            if results:
                minor = int(results[0])

            revision_pattern = re.compile(r'(?i)revision\s?=\s?(\d+)')
            results = revision_pattern.findall(settings)
            if results:
                revision = int(results[0])

            build_pattern = re.compile(r'(?i)build\s?=\s?(\d+)')
            results = build_pattern.findall(settings)
            if results:
                build = int(results[0])
                                
            date_pattern = re.compile(r'(?i)date\s?=\s?(\d{4}\-\d{2}\-\d{2})')
            results = date_pattern.findall(settings)
            if results:
                date = results[0]

        return major, minor, revision, build, date

    @cherrypy.expose
    def register(self, returnurl="", firstname="", nickname="", lastname="", username="",
                 password="", password_confirm=""):
        warning = ""
        lowercase_username = username.lower()
        if all([firstname, lastname, username, password, password_confirm]):
            if password == password_confirm:
                if not self.members_collection.count({'username': lowercase_username}):
                    member_document = {'firstname': firstname,
                                       'lastname':  lastname,
                                       'username':  lowercase_username,
                                       'password':  password
                                      }
                    if nickname:
                        member_document['nickname'] = nickname
                                      
                    self.members_collection.insert_one(member_document)
                    
                    subject = "Kanbanara - Registration"
                    message = (f'Hi {firstname},\n\n'
                               'Thank you for registering with Kanbanara.\n\n'
                               'Your access details are:\n'
                               f'    Username: {lowercase_username}\n'
                               f'    Password: {password}\n\n'
                               'The Kanbanara Team')
                    self.send_email(lowercase_username, subject, message)                   
                    
                    message = ('Thank you for registering with Kanbanara. '
                               f'A confirmation email sent to {lowercase_username} '
                               'should arrive in due course.')
                    raise cherrypy.HTTPRedirect(f'/authenticate/confirmation?message={message}&returnurl={returnurl}', 302)
                else:
                    warning = ('<div class="warning">'
                               '<p class="warning">Username has already been registered!</p>'
                               '<form action="/authenticate/index" method="post">'
                               f'<input type="hidden" name="returnurl" value="{returnurl}">'
                               f'<input type="hidden" name="username" value="{lowercase_username}">'
                               '<button type="submit" value="Logon">'
                               '<span class="fas fa-sign-in-alt fa-lg">'
                               '</span>&nbsp;Logon</button></form></div>')

            else:
                warning = '<div class="warning"><p class="warning">Passwords do not match!</p></div>'

        content = []
        content.append(self.header(returnurl, title='Registration'))
        content.append(('<div id="maincontent">'
                        '<h3 class="subheading">Registration</h3>'
                        '<p>Should you need to register, you may do so here by entering<br>'
                        'your email address, first name, last name and a password</p>'))

        socket_hostname = socket.gethostname()
        if socket_hostname == 'www.kanbanara.com' or os.path.exists(os.path.join(self.CURRENT_DIR, '..', 'website')):
            content.append(('<p>By registering, you will automatically become<br>'
                            'a member of the <a href="/foundation">Kanbanara Software Foundation</a> and<br>'
                            'agree to be bound by its terms and conditions</p>'))

        content.append(warning)
        content.append(('<form action="/authenticate/register" method="post">'
                        f'<input type="hidden" name="returnurl" value="{returnurl}">'
                        '<table border="0">'
                        '<tr><td align="right">First Name</td><td>'
                        '<input type="text" size="40" name="firstname" '
                        'pattern="[a-zA-Z-]+"'))
        if firstname:
            content.append(f' value="{firstname}"')

        content.append((' placeholder="First Name" required>'
                        '</td><td></td></tr>'
                        '<tr><td align="right">Nickname</td><td>'
                        '<input type="text" size="40" name="nickname" '
                        'pattern="[a-zA-Z-]+"'))
        if nickname:
            content.append(f' value="{nickname}"')

        content.append((' placeholder="Nickname">'
                        '</td><td></td></tr>'
                        '<tr><td align="right">Last Name</td><td>'
                        '<input type="text" size="40" name="lastname" '
                        'pattern="[a-zA-Z-]+"'))
        if lastname:
            content.append(f' value="{lastname}"')

        content.append((' placeholder="Last Name" required></td><td></td></tr>'
                        '<tr><td align="right">Username (Email Address)</td><td>'
                        '<input id="username" type="email" size="40" name="username" '
                        'pattern="[a-z0-9._%+-]+@[a-z0-9.-]+\.[a-z]{2,}$"'))
        if username:
            content.append(f' value="{lowercase_username}"')

        content.append((' placeholder="Email Address" required>'
                        '</td><td>'
                        '<div id="validateusername">'
                        '<!-- This section is automatically populated by JQuery and Ajax -->'
                        '</div>'
                        '</td></tr>'
                        '<tr><td align="right">Password</td><td>'
                        '<input type="password" size="40" name="password" '
                        'pattern=".{4,}" placeholder="Password" required></td><td></td></tr>'
                        '<tr><td align="right">Password Confirmation</td><td>'
                        '<input type="password" size="40" name="password_confirm" '
                        'pattern=".{4,}" placeholder="Password" required>'
                        '</td><td></td></tr><tr><td align="center" colspan="3">'
                        '<button type="submit" value="Register">'
                        '<span class="fas fa-registered fa-lg"></span>&nbsp;Register</button>'
                        '</td></tr></table></form>'))
        content.append(self.insert_return_link(returnurl))
        content.append(self.insert_links('register', lowercase_username, returnurl))
        content.append('</div>')
        content.append(self.footer())
        return "".join(content)

    def send_email(self, recipient, subject, message):
        msg = MIMEText(message)
        msg['Subject'] = subject
        msg['From'] = self.kanbanara_settings['email_address']
        msg['To'] = recipient
        SERVER = self.kanbanara_settings['smtp_server_host']
        PORT = ""
        if self.kanbanara_settings.get('smtp_server_port', 0):
            PORT = int(self.kanbanara_settings['smtp_server_port'])
        
        USERNAME = self.kanbanara_settings['smtp_server_username']
        PASSWORD = self.kanbanara_settings['smtp_server_password']
        try:
            if PORT:
                server = smtplib.SMTP(host=SERVER, port=PORT)
            else:
                server = smtplib.SMTP(SERVER)
                
            server.set_debuglevel(True) # show communication with the server
            # identify ourselves to smtp client
            server.ehlo()
            if PORT in [465, 587]:
                # secure our email with TLS encryption
                try:
                    server.starttls()
                except:
                    True
                
                # re-identify ourselves as an encrypted connection
                server.ehlo()

            #if USERNAME and PASSWORD:
            #    server.login(USERNAME, PASSWORD)
                
            server.send_message(msg)
            server.quit()
                
        except socket.gaierror as e:
            print(f'Given host name is invalid. [{e}]')
        except smtplib.SMTPRecipientsRefused as e:
            print(f'{server} has rejected one or more recipients. [{e}]')
        except smtplib.SMTPServerDisconnected as e:
            print(f'{server} disconnected. [{e}]')
        except smtplib.SMTPConnectError as e:
            print(f'Unable to connect to {server}. [{e}]')
        except socket.timeout as e:
            print(f'Timedout connecting to {server}. [{e}]')
        
    @cherrypy.expose
    def reset_password(self, returnurl="", username="", reset_password="No"):
        '''Allows a registered user's password to be reset to a randomly-generated 8-letter one'''
        content = []
        content.append(self.header(returnurl, title='Reset Password'))
        content.append(('<div id="maincontent">'
                        '<h3 class="subheading">Reset Password</h3>'))
        if username and reset_password == "Yes":
            if self.members_collection.count({'username': username}):
                new_password = "".join([chr(random.randint(97, 122)) for i in range(8)])
                subject = "Kanbanara - Reset Password"               
                message = (f'Hi,\n\n'
                           'Someone, hopefully yourself, has requested that your Kanbanara password be reset.\n\n'
                           f"As requested, your password has been reset to '{new_password}'.\n\n"
                           "Feel free to use the 'Change Password' option to change it to something more memorable.\n\n"
                           'The Kanbanara Team')
                self.send_email(username, subject, message)
                #self.members_collection.find_one_and_update(
                #        {'username': username},
                #        {'$set': {'password': new_password}})
                message = ('Your password has been reset. '
                           'A confirmation email containing your new password '
                           f'has been sent to {username}.')
                raise cherrypy.HTTPRedirect(f'/authenticate/confirmation?message={message}&returnurl={returnurl}', 302)
            else:
                content.append('<p class="warning">Username does not exist!</p>')

        content.append(('<p>If you\'ve forgotten your password,<br>'
                        'you may reset it by sending us your username</p>'
                        '<form action="/authenticate/reset_password" method="post">'
                        f'<input type="hidden" name="returnurl" value="{returnurl}">'
                        '<input type="hidden" name="reset_password" value="Yes">'
                        '<table border="0"><tr><td align="right">Username</td><td>'
                        '<input type="email" size="40" name="username" '
                        'pattern="[a-z0-9._%+-]+@[a-z0-9.-]+\.[a-z]{2,}$"'))

        if username:
            content.append(f' value="{username}"')

        content.append((' placeholder="Email Address" required>'
                        '</td></tr><tr><td align="center" colspan="2">'
                        '<button type="submit" value="Reset Password">'
                        '<span class="fas fa-unlock-alt fa-lg"></span>&nbsp;Reset Password</button>'
                        '</td></tr></table></form>'))
        content.append(self.insert_return_link(returnurl))
        content.append(self.insert_links('reset_password', username, returnurl))
        content.append('</div>')
        content.append(self.footer())
        return "".join(content)

    @cherrypy.expose
    def logoff(self):
        cherrypy.session['username']=None
        cherrypy.session.delete()
        cherrypy.lib.sessions.expire()
        content = []
        content.append(self.header('', title='Logoff'))
        content.append(('<div id="maincontent">'
                        '<h3 class="subheading">Logoff</h3>'
                        '<p>You are now logged off!</p>'))        
        content.append(self.insert_links('logoff', '', ''))
        content.append('</div>')
        content.append(self.footer())
        return "".join(content)

    @cherrypy.expose
    def members(self, admin_username="", admin_password=""):
        content = []
        content.append(self.header('', title='Members'))
        content.append(('<div id="maincontent">'
                        '<h3 class="subheading">Members</h3>'))
        if (admin_username == self.kanbanara_settings['admin_username'] and
                admin_password == self.kanbanara_settings['admin_password']):
            content.append(('<table class="members"><tr>'
                            '<th>Last&nbsp;Name</th><th>First&nbsp;Name</th>'
                            '<th>Username&nbsp;(Email&nbsp;Address)</th><th>Password</th>'
                            '<th>Logon</th><th>Change Password</th><th>Reset Password</th>'
                            '<th>Delete Member</th></tr>'))
            for document in self.members_collection.find({}):
                username  = document['username']
                password  = document['password']
                firstname = document['firstname']
                lastname  = document['lastname']
                content.append((f'<tr><td>{lastname}</td><td>{firstname}</td>'
                                f'<td>{username}</td><td>{password}</td>'
                                '<td><form action="/authenticate" method="post">'
                                f'<input type="hidden" name="username" value="{username}">'
                                f'<input type="hidden" name="password" value="{password}">'
                                '<button type="submit" value="Logon">'
                                '<span class="fas fa-sign-in-alt fa-lg"></span>&nbsp;Logon</button>'
                                '</form></td>'
                                '<td><form action="/authenticate/change_password" method="post">'
                                f'<input type="hidden" name="username" value="{username}">'
                                f'<input type="hidden" name="existing_password" value="{password}">'
                                '<button type="submit" value="Change Password">'
                                '<span class="fas fa-unlock-alt fa-lg"></span>&nbsp;Change Password'
                                '</button></form></td>'
                                '<td><form action="/authenticate/reset_password" method="post">'
                                f'<input type="hidden" name="username" value="{username}">'
                                '<button type="submit" value="Reset Password">'
                                '<span class="fas fa-unlock-alt fa-lg"></span>&nbsp;Reset Password'
                                '</button></form></td>'
                                '<td><form action="/authenticate/delete_member" method="post">'
                                f'<input type="hidden" name="username" value="{username}">'
                                f'<input type="hidden" name="password" value="{password}">'
                                '<button type="submit" value="Delete Member">'
                                '<span class="fas fa-user fa-lg"></span>&nbsp;Delete Member</button>'
                                '</form></td></tr>'))

            content.append('</table>')
        else:
            content.append(('<form action="/authenticate/members" method="post">'
                            '<table border="0"><tr>'
                            '<td align="right">Administrator Username</td><td>'
                            '<input type="password" size="40" name="admin_username" '
                            'pattern=".{4,}" placeholder="Username" required></td></tr>'
                            '<tr><td align="right">Administrator Password</td><td>'
                            '<input type="password" size="40" name="admin_password" '
                            'pattern=".{4,}" placeholder="Password" required></td></tr>'
                            '<tr><td align="center" colspan="2">'
                            '<button type="submit" value="Logon">'
                            '<span class="fas fa-sign-in-alt fa-lg"></span>&nbsp;Logon</button>'
                            '</td></tr></table></form>'))

        content.append('</div>')
        content.append(self.footer())
        return "".join(content)

    def read_administrator_ini_file(self):
        """Reads the administrator.ini file and extracts username and password information"""
        # TODO - We can switch to using configParser in Kanbanara version 2
        with open(os.path.join(self.CURRENT_DIR, '..', 'administrator.ini'), 'r') as handle:
            data = handle.read()
            for (setting, regex) in [
                    ('admin_username', r'(?i)username\s?=\s?(\S+)'),
                    ('admin_password', r'(?i)password\s?=\s?(\S+)'),
                    ('smtp_server_host', r'(?i)smtp_server_host\s?=\s?(\S+)'),
                    ('smtp_server_password', r'(?i)smtp_server_password\s?=\s?(\S+)'),
                    ('smtp_server_port', r'(?i)smtp_server_port\s?=\s?(\d+)'),
                    ('smtp_server_username', r'(?i)smtp_server_username\s?=\s?(\S+)'),
                    ('email_address', r'(?i)email_address\s?=\s?(\S+)')]:
                pattern = re.compile(regex)
                results = pattern.findall(data)
                if results:
                    self.kanbanara_settings[setting] = results[0]
                else:
                    self.kanbanara_settings[setting] = ""

    def read_mongodb_ini_file(self):
        """Reads the common mongodb.ini file and extracts host and port information"""
        self.mongodb_host = 'localhost'
        self.mongodb_port = 27017
        # TODO - We can switch to using configParser in Kanbanara version 2
        with open(os.path.join(self.CURRENT_DIR, '..', 'mongodb.ini'), 'r') as handle:
            settings = handle.read()
            mongodb_host_pattern = re.compile(r'(?i)host\s?=\s?(\S+)')
            results = mongodb_host_pattern.findall(settings)
            if results != []:
                self.mongodb_host = results[0]

            mongodb_port_pattern = re.compile('(?i)port\s?=\s?(\d+)')
            results = mongodb_port_pattern.findall(settings)
            if results != []:
                self.mongodb_port = int(results[0])

CURRENT_DIR = os.path.dirname(os.path.abspath(__file__))
conf = {}
conf['/'] = {'tools.staticdir.root': CURRENT_DIR,
             'tools.sessions.on':    True}
for directory in ['css', 'images', 'scripts']:
    if os.path.exists(os.path.join(CURRENT_DIR, directory)):
        conf['/'+directory] = {'tools.staticdir.on':  True,
                               'tools.staticdir.dir': directory}

cherrypy.tree.mount(Authenticate(), '/authenticate', config=conf)
