/***************************************************************************
*   Copyright (C) 2004 by Kita Developers                                 *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "boardview.h"

// include files for Qt
#include <qtoolbutton.h>
#include <qtextbrowser.h>
#include <qtextcodec.h>
#include <qregexp.h>
#include <qfile.h>
#include <qclipboard.h>
#include <qapplication.h>
#include <qdatetime.h>
#include <qlabel.h>
#include <qmessagebox.h>
#include <qcombobox.h>
#include <qheader.h>

// kdelibs/kio
#include <kio/slaveconfig.h>
#include <kio/netaccess.h>

// include files for KDE
#include <kwin.h>
#include <kfilterdev.h>
#include <klistview.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kpopupmenu.h>
#include <krun.h>
#include <kdebug.h>
#include <kconfig.h>
#include <kstandarddirs.h>

#include "libkita/thread.h"
#include "libkita/kita_misc.h"
#include "libkita/kitaconfig.h"
#include "libkita/favoritethreads.h"
#include "libkita/datmanager.h"
#include "libkita/signalcollection.h"
#include "libkita/threadindex.h"
#include "libkita/boardmanager.h"
#include "libkita/threadinfo.h"

#include "threadproperty.h"
#include "threadlistviewitem.h"

KitaBoardView::KitaBoardView( QWidget* parent, const char* name )
        : Kita::ThreadListView( parent, name )
{
    init();
    closeButton->setEnabled( true );

    connect( subjectList, SIGNAL( contextMenuRequested( QListViewItem*, const QPoint&, int ) ),
             SLOT( slotContextMenuRequested( QListViewItem*, const QPoint&, int ) ) );
    connect( subjectList, SIGNAL( returnPressed( QListViewItem* ) ),
             SLOT( loadThread( QListViewItem* ) ) );
    connect( ReloadButton, SIGNAL( clicked() ),
             SLOT( reloadSubject() ) );

    Kita::SignalCollection* signalCollection = Kita::SignalCollection::getInstance();
    connect( signalCollection, SIGNAL( updateSubjectTab( const KURL& ) ),
             SLOT( slotUpdateSubject( const KURL& ) ) );
    connect( this, SIGNAL( closeThreadTab( const KURL& ) ),
             signalCollection, SIGNAL( closeThreadTab( const KURL& ) ) );
    connect( this, SIGNAL( loadBoardCompleted( const KURL& ) ),
             signalCollection, SIGNAL( setMainURLLine( const KURL& ) ) );
    connect( this, SIGNAL( createNewThread( const KURL& , const QString&, const QString& ) ),
             signalCollection, SIGNAL( createNewThread( const KURL& , const QString&, const QString& ) ) );

    connect( closeButton, SIGNAL( clicked() ),
             SLOT( slotCloseButton() ) );

    QHeader* header = subjectList->header();
    connect( header, SIGNAL( sizeChange( int, int, int ) ),
             SLOT( slotSizeChange( int, int, int ) ) );

    header->installEventFilter( this );
    loadLayout();
    loadHeaderOnOff();
}

KitaBoardView::~KitaBoardView()
{}

/* public */
void KitaBoardView::init()
{
    m_unreadNum = 0;
    m_readNum = 0;
    m_newNum = 0;
    m_showOldLogs = FALSE;
    m_enableSizeChange = FALSE;
}


void KitaBoardView::reloadSubject()
{
    if ( ! m_boardURL.isEmpty() ) {
        loadBoard( m_boardURL );
    }
}


/* private slot */
void KitaBoardView::slotCloseButton()
{
    emit closeCurrentTab();
}


const KURL KitaBoardView::boardURL()
{
    return m_boardURL;
}

void KitaBoardView::toggleShowOldLogs()
{
    m_showOldLogs = !m_showOldLogs;
    loadBoard( m_boardURL, FALSE );
}

void KitaBoardView::loadThread( QListViewItem* item )
{
    if ( ! item ) return ;

    KURL datURL = item->text( Col_DatURL );

    if ( KitaConfig::alwaysUseTab() ) {
        emit openURLRequestExt( datURL.prettyURL(), KParts::URLArgs(), "kita_open_2chthread", 1 );
    } else {
        emit openURLRequestExt( datURL.prettyURL(), KParts::URLArgs(), "kita_open_2chthread", 0 );
    }
}
/*
ͥ
ts_readed   : ɤ1000ĶƤ
ts_normal   : ̤1ʾФäƤ
ts_new      : ̤1Τ
ts_read   : ɤ1000ʲΤ
ts_hasunread: ̤ɤ
*/

enum {
    ts_old,
    ts_readed,
    ts_normal,
    ts_new,
    ts_read,
    ts_hasunread
};

void KitaBoardView::loadBoard( const KURL& url, bool online )
{
    setActiveWindow();
    topLevelWidget() ->raise();
    m_enableSizeChange = FALSE;

    // reset member variables.
    // FIXME: FavoriteListView::update()˥ԡ
    {
        m_hitList.clear();
        m_nextHitIndex = 0;
        m_prevquery = "";
    }

    m_boardURL = url;
    m_unreadNum = 0;
    m_readNum = 0;
    m_newNum = 0;

    /* get list of pointers of Thread classes */
    QPtrList<Kita::Thread> oldLogList;
    QPtrList<Kita::Thread> threadList;
    Kita::BoardManager::getThreadList( m_boardURL, m_showOldLogs, online, threadList, oldLogList );

    // reset list
    subjectList->clear();

    QDateTime current = QDateTime::currentDateTime();
    unsigned int countNew = threadList.count();
    unsigned int countOld = oldLogList.count();
    for ( unsigned i = 0; i < countNew + countOld; i++ ) {

        Kita::Thread* thread = i < countNew ? threadList.at( i ) : oldLogList.at( i - countNew );
        Kita::ThreadListViewItem* item = new Kita::ThreadListViewItem( subjectList );
        int id = ( i < countNew ? i + 1 : 0 );
        int order = i + 1;
        updateListViewItem( item, thread, current, id, order );
    }

    if ( HideButton->isOn() ) {
        HideButton->toggle();
    }
    emit loadBoardCompleted( m_boardURL );

    switch ( KitaConfig::sortOrder() ) {
    case KitaConfig::Order_Mark:
        subjectList->setSorting( Col_Mark );
        break;
    case KitaConfig::Order_ID:
        subjectList->setSorting( Col_ID );
        break;
    default:
        // do nothing
        break;
    }

    subjectList->setSelected( subjectList->firstChild(), TRUE );
    subjectList->setFocus();
    UpdateKindLabel();

    /* restore column size */
    loadLayout();
    loadHeaderOnOff();
    m_enableSizeChange = TRUE;
}

/* public slot */ /* virtual */
void KitaBoardView::setFocus()
{
    subjectList->setFocus();
}

/* public slot */
void KitaBoardView::slotFocusSearchCombo()
{
    if ( ! SearchCombo->hasFocus() ) {
        SearchCombo->setFocus();
    } else {
        setFocus();
    }
}

/* public slot */
void KitaBoardView::slotCreateNewThread()
{
    emit createNewThread( m_boardURL, QString::null, QString::null );
}


void KitaBoardView::UpdateKindLabel()
{
    QString fmtstr;
    fmtstr += QString( "<font color=#C50000>%1</font>" ).arg( m_unreadNum );
    fmtstr += QString( "/<font color=#00C200>%1</font>" ).arg( m_readNum );
    fmtstr += QString( "/<font color=#5AAAFF>%1</font>" ).arg( m_newNum );
    KindLabel->setText( fmtstr );
}

void KitaBoardView::setFont( const QFont& font )
{
    subjectList->setFont( font );
}

void KitaBoardView::slotUpdateSubject( const KURL& url )
{
    QDateTime current = QDateTime::currentDateTime();

    KURL datURL = Kita::getDatURL( url );
    for ( QListViewItem * item = subjectList->firstChild(); item; item = item->nextSibling() ) {
        if ( item->text( Col_DatURL ) == datURL.prettyURL() ) {

            switch ( item->text( Col_MarkOrder ).toInt() ) {
            case ts_readed :
            case ts_read : m_readNum--; break;
            case ts_new : m_newNum--; break;
            case ts_hasunread : m_unreadNum--; break;
            }

            Kita::Thread* thread = Kita::Thread::getByURLNew( datURL );
            if ( thread == NULL ) return ;
            int id = item->text( Col_ID ).toInt();
            int order = item->text( Col_IDOrder ).toInt();
            updateListViewItem( item, thread, current, id, order );

            UpdateKindLabel();

            return ;
        }
    }
}



/* if id == 0, this thread is old thread. */
/* private */
void KitaBoardView::updateListViewItem( QListViewItem* item, Kita::Thread* thread, const QDateTime& current, int id, int order )
{
    KURL datURL = thread->datURL();

    QDateTime since;
    since.setTime_t( Kita::datToSince( datURL ) );
    QString threadName = thread->threadName();
    int resNum = thread->resNum();
    int readNum = thread->readNum();
    int viewPos = thread->viewPos();
    double speed = resNum / ( since.secsTo( current ) / ( 60.0 * 60.0 * 24.0 ) );

    if ( id ) item->setText( Col_ID, QString().setNum( id ) );
    item->setText( Col_IDOrder, QString().setNum( order ) );
    item->setText( Col_Subject, threadName );
    item->setText( Col_ResNum, QString( "%1" ).arg( resNum, 4 ) );
    item->setText( Col_ReadNum, ( viewPos > 0 ) ? QString( "%1" ).arg( viewPos, 4 ) : QString( "" ) );
    item->setText( Col_Unread, ( viewPos > 0 && resNum > viewPos ) ? QString( "%1" ).arg( resNum - viewPos, 4 ) : QString( "" ) );
    item->setText( Col_Since, since.toString( "yy/MM/dd hh:mm" ) );
    item->setText( Col_DatURL, datURL.prettyURL() );
    item->setText( Col_Speed, QString( " %1 " ).arg( speed, 0, 'f', 2 ) );

    /* set mark order */

    if ( !id ) { /* old thread */

        item->setText( Col_MarkOrder, QString::number( ts_old ) );
    } else if ( readNum > 0 && ( resNum > readNum || resNum > viewPos ) ) {  /* There are new responses. */

        item->setPixmap( Col_Mark, SmallIcon( "unread" ) );
        m_unreadNum++;
        item->setText( Col_MarkOrder, QString::number( ts_hasunread ) );

    } else if ( readNum > 0 ) { /* Cache exists */

        item->setPixmap( Col_Mark, SmallIcon( "read" ) );
        m_readNum++;
        item->setText( Col_MarkOrder, QString::number( ( viewPos > 1000 ) ? ts_readed : ts_read ) );

    } else if ( since.secsTo( current ) < 3600 * KitaConfig::MarkTime() ) { /* new thread */

        item->setPixmap( Col_Mark, SmallIcon( "newthread" ) );
        m_newNum++;
        item->setText( Col_MarkOrder, QString::number( ts_new ) );

    } else { /* normal */

        item->setText( Col_MarkOrder, QString::number( ts_normal ) );
        item->setPixmap( Col_Mark, NULL );
    }

    // no effect: m_unreadNum, m_readNum, m_newNum, markOrder
    if ( thread->isOpened() && resNum == readNum ) { /* opened */
        item->setPixmap( Col_Mark, SmallIcon( "open" ) );
    }
}

void KitaBoardView::slotContextMenuRequested( QListViewItem* item, const QPoint& point, int )
{
    enum {
        Menu_OpenWithBrowser,
        Menu_OpenWithNewTab,
        Menu_CopyURL,
        Menu_CopyTitleAndURL,
        Menu_Favorites,
        Menu_DeleteLog,
        Menu_OpenByBackground,
        Menu_Property
    };

    if ( item == 0 ) {
        return ;
    }

    QString datURL = item->text( Col_DatURL );
    QString threadURL = Kita::DatManager::threadURL( datURL );
    bool isFavorites = FavoriteThreads::getInstance() ->contains( datURL );

    KPopupMenu popup( 0 );
    popup.insertItem( i18n( "Open with Web Browser" ), Menu_OpenWithBrowser );
    popup.insertItem( i18n( "Open with new tab" ), Menu_OpenWithNewTab );
    popup.insertItem( i18n( "Copy URL" ), Menu_CopyURL );
    popup.insertItem( i18n( "Copy title and URL" ), Menu_CopyTitleAndURL );
    if ( isFavorites ) {
        popup.insertItem( i18n( "Remove from Favorites" ), Menu_Favorites );
    } else {
        popup.insertItem( i18n( "Add to Favorites" ), Menu_Favorites );
    }

    if ( Kita::DatManager::getReadNum( datURL ) ) {
        popup.insertSeparator();
        popup.insertItem( i18n( "Delete Log" ), Menu_DeleteLog );
    }
    popup.insertItem( i18n( "Open by background" ), Menu_OpenByBackground );

    popup.insertSeparator();
    popup.insertItem( i18n( "Property" ), Menu_Property );

    QClipboard* clipboard = QApplication::clipboard();
    QString cliptxt;

    switch ( popup.exec( point ) ) {
    case Menu_OpenWithBrowser:
        KRun::runURL( threadURL, "text/html" );
        break;
    case Menu_OpenWithNewTab:
        emit openURLRequestExt( datURL, KParts::URLArgs(), "kita_open_2chthread", 1 );
        break;
    case Menu_CopyURL:
        clipboard->setText( threadURL );
        break;
    case Menu_CopyTitleAndURL:
        cliptxt = Kita::DatManager::threadName( datURL ) + "\n" + threadURL;
        clipboard->setText( cliptxt , QClipboard::Clipboard );
        clipboard->setText( cliptxt , QClipboard::Selection );
        break;
    case Menu_Favorites:
        if ( isFavorites ) {
            emit bookmarked( datURL, false );
        } else {
            emit bookmarked( datURL, true );
        }
        break;
    case Menu_DeleteLog:
        deleteLog( threadURL );
        break;
    case Menu_OpenByBackground:
        emit openURLRequestExt( datURL, KParts::URLArgs(), "kita_open_2chthread", 2 );
        break;
    case Menu_Property: {
            // FIXME: memory leak
            Kita::ThreadProperty* propertyWidget = new Kita::ThreadProperty( 0 );
            propertyWidget->threadURLLabel->setText( threadURL );
            propertyWidget->datURLLabel->setText( datURL );
            propertyWidget->threadNameLabel->setText( Kita::DatManager::threadName( datURL ) );
            propertyWidget->cachePathLabel->setText( Kita::DatManager::getCachePath( datURL ) );
            propertyWidget->indexPathLabel->setText( Kita::DatManager::getCacheIndexPath( datURL ) );

            propertyWidget->idx_threadNameWithIndexLabel->setText( Kita::ThreadIndex::getSubject( datURL ) );
            propertyWidget->idx_resNumLabel->setText( QString( "%1" ).arg( Kita::ThreadIndex::getResNum( datURL ) ) );
            propertyWidget->idx_readNumLabel->setText( QString( "%1" ).arg( Kita::ThreadIndex::getReadNum( datURL ) ) );
            propertyWidget->idx_viewPosLabel->setText( QString( "%1" ).arg( Kita::ThreadIndex::getViewPos( datURL ) ) );

            propertyWidget->cache_resNumLabel->setText( QString( "%1" ).arg( KitaThreadInfo::resNum( datURL ) ) );
            propertyWidget->cache_readNumLabel->setText( QString( "%1" ).arg( KitaThreadInfo::readNum( datURL ) ) );
            propertyWidget->show();
        }
        break;
    default:
        break;
    }
}

void KitaBoardView::deleteLog( const KURL& url )
{
    if ( QMessageBox::warning( this,
                               "Kita",
                               i18n( "Do you want to delete Log ?" ),
                               QMessageBox::Ok, QMessageBox::Cancel | QMessageBox::Default )
            == QMessageBox::Ok ) {
        if ( Kita::DatManager::deleteCache( url ) ) {
            emit closeThreadTab( url );
            slotUpdateSubject( url );
        }
    }
}

void KitaBoardView::slotSizeChange( int section, int oldSize, int newSize )
{
    if ( ! m_enableSizeChange ) return ;
    if ( autoResize() ) return ;
    // FIXME: cache KConfig
    QString configPath = locateLocal( "appdata", "subjectview.conf" );
    KConfig config( configPath );
    subjectList->saveLayout( &config, "Layout" );
}

void KitaBoardView::loadLayout()
{
    if ( autoResize() ) return ;
    QString configPath = locateLocal( "appdata", "subjectview.conf" );
    KConfig config( configPath );
    subjectList->restoreLayout( &config, "Layout" );
}

bool KitaBoardView::eventFilter( QObject* watched, QEvent* e )
{
    if ( e->type() == QEvent::MouseButtonPress ) {
        QMouseEvent * mouseEvent = static_cast<QMouseEvent *>( e );
        if ( mouseEvent->button() == RightButton ) {
            KPopupMenu popup;
            popup.setCheckable( true );

            for ( int i = Col_Begin; i <= Col_End; i++ ) {
                if ( i != Col_Subject && i != Col_MarkOrder && i != Col_IDOrder ) {
                    popup.insertItem( i18n( s_colAttr[ i ].itemName ), i );
                    popup.setItemChecked( i, subjectList->columnWidth( i ) != 0 );
                }
            }
            // TODO: don't use magic number.
            popup.insertItem( "Auto Resize", 999 );
            popup.setItemChecked( 999, autoResize() );

            int id = popup.exec( mouseEvent->globalPos() );
            if ( id == -1 ) {
                return true;
            }

            if ( id == 999 ) {
                setAutoResize( ! popup.isItemChecked( 999 ) );
            } else if ( popup.isItemChecked( id ) ) {
                hideColumn( id );
            } else {
                showColumn( id );
            }
            saveHeaderOnOff();
            return true;
        } else {
            return false;
        }
    } else {
        return subjectList->header() ->eventFilter( watched, e );
    }
}

void KitaBoardView::saveHeaderOnOff()
{
    QString configPath = locateLocal( "appdata", "subjectview.conf" );
    KConfig config( configPath );

    config.setGroup( "Column" );
    for ( int i = Col_Begin; i <= Col_End; i++ ) {
        if ( subjectList->columnWidth( i ) != 0 ) {
            config.writeEntry( s_colAttr[ i ].keyName, true );
        } else {
            config.writeEntry( s_colAttr[ i ].keyName, false );
        }
    }
}

void KitaBoardView::loadHeaderOnOff()
{
    QString configPath = locateLocal( "appdata", "subjectview.conf" );
    KConfig config( configPath );

    config.setGroup( "Column" );
    for ( int i = Col_Begin; i <= Col_End; i++ ) {
        bool isShown = config.readBoolEntry( s_colAttr[ i ].keyName, s_colAttr[ i ].showDefault );
        qDebug("%s: isShown %d", s_colAttr[i].keyName.latin1(), isShown );
        if ( isShown ) {
            showColumn( i );
        } else {
            hideColumn( i );
        }
    }
}

bool KitaBoardView::autoResize()
{
    QString configPath = locateLocal( "appdata", "subjectview.conf" );
    KConfig config( configPath );
    config.setGroup( "Column" );

    return config.readBoolEntry( "AutoResize", true );
}

void KitaBoardView::setAutoResize( bool flag )
{
    QString configPath = locateLocal( "appdata", "subjectview.conf" );
    KConfig config( configPath );
    config.setGroup( "Column" );

    config.writeEntry( "AutoResize", flag );
}
