/***************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto                                  *
*   ikemo@wakaba.jp                                                       *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "threadlistview.h"

#include <kiconloader.h>
#include <klistview.h>
#include <klocale.h>
#include <kurl.h>

#include <qtoolbutton.h>
#include <qcombobox.h>
#include <qregexp.h>
#include <qheader.h>

#include "libkita/thread.h"
#include "libkita/kitaconfig.h"
#include "libkita/signalcollection.h"

#include "threadlistviewitem.h"

using namespace Kita;

struct Col_Attr ThreadListView::s_colAttr[] =
    {
        // labelName, itemName, keyName, showDefault
        { I18N_NOOP( " " ), I18N_NOOP( "Mark" ), "Col_Mark", true },
        { I18N_NOOP( "No." ), I18N_NOOP( "ID" ), "Col_ID", true},
        { I18N_NOOP( " " ), I18N_NOOP( "Icon" ), "Col_Icon", true},
        { I18N_NOOP( "Title" ), I18N_NOOP( "Subject" ), "Col_Subject", true},
        { I18N_NOOP( "ResNum" ), I18N_NOOP( "ResNum" ), "Col_ResNum", true},
        { I18N_NOOP( "ReadNum" ), I18N_NOOP( "ReadNum" ), "Col_ReadNum", true},
        { I18N_NOOP( "ViewPos" ), I18N_NOOP( "ViewPos" ), "Col_ViewPos", false},
        { I18N_NOOP( "Unread" ), I18N_NOOP( "Unread" ), "Col_Unread", true},
        { I18N_NOOP( "Since" ), I18N_NOOP( "Since" ), "Col_Since", true},
        { I18N_NOOP( "Thread's speed" ), I18N_NOOP( "Speed" ), "Col_Speed", true},
        { I18N_NOOP( "Board" ), I18N_NOOP( "Board" ), "Col_Board", false},
        { I18N_NOOP( "Dat URL" ), I18N_NOOP( "DatURL" ), "Col_DatURL", false},
        { I18N_NOOP( "Mark Order" ), I18N_NOOP( "MarkOrder" ), "Col_MarkOrder", false},
        { I18N_NOOP( "ID Order" ), I18N_NOOP( "IDOrder" ), "Col_IDOrder", false}
    };

ThreadListView::ThreadListView( QWidget* parent, const char* name )
        : ThreadListViewBase( parent, name )
{
    SearchButton->setPixmap( SmallIcon( "find" ) );
    HideButton->setPixmap( SmallIcon( "filter" ) );
    ReloadButton->setPixmap( SmallIcon( "reload" ) );
    closeButton->setPixmap( SmallIcon( "fileclose" ) );

    QHeader* header = subjectList->header();
    for ( int i = Col_Begin; i <= Col_End; i++ ) {
        subjectList->addColumn( i18n( s_colAttr[ i ].labelName ) );

        if ( s_colAttr[ i ].showDefault != true ) {
            hideColumn( i );
        }
    }
    header->setStretchEnabled( true, Col_Subject );

    connect( SearchButton, SIGNAL( clicked() ),
             SLOT( slotSearchButton() ) );
    connect( SearchCombo, SIGNAL( activated( int ) ),
             SLOT( slotSearchButton() ) );
    connect( HideButton, SIGNAL( toggled( bool ) ),
             SLOT( slotHideButton( bool ) ) );
    connect( subjectList, SIGNAL( mouseButtonClicked( int, QListViewItem*, const QPoint&, int ) ),
             SLOT( slotMouseButtonClicked( int, QListViewItem* ) ) );
    connect( this, SIGNAL( bookmarked( const QString&, bool ) ),
             Kita::SignalCollection::getInstance(), SIGNAL( bookmarked( const QString&, bool ) ) );
    connect( this, SIGNAL( openURLRequestExt(
                               const KURL&, const KParts::URLArgs&, QString, int,
                               const KURL& ) ),
             Kita::SignalCollection::getInstance(), SIGNAL( openURLRequestExt(
                         const KURL& , const KParts::URLArgs&, QString, int,
                         const KURL& ) ) );
}

ThreadListView::~ThreadListView()
{}

void ThreadListView::slotSearchButton()
{
    insertSearchCombo();
    QStringList list = parseSearchQuery( SearchCombo->currentText() );
    searchNext( list );
}

void ThreadListView::insertSearchCombo()
{
    for ( int count = 0; count < SearchCombo->count(); ++count ) {
        if ( SearchCombo->text( count ) == SearchCombo->currentText() ) {
            return ;
        }
    }
    SearchCombo->insertItem( SearchCombo->currentText() );
}

QStringList ThreadListView::parseSearchQuery( const QString &input )
{
    QStringList tmp = QStringList::split( ' ', input );
    QStringList ret_list;
    QRegExp truncSpace( "\\s*$" );
    QStringList::iterator it = tmp.begin();

    for ( ; it != tmp.end(); ++it ) {
        ret_list += ( *it ).replace( truncSpace, "" );
    }
    return ret_list;
}

void ThreadListView::searchNext( const QStringList &query )
{
    if ( query.isEmpty() ) return ;

    if ( query != m_prevquery ) {
        searchAll( query );
        slotHideButton( HideButton->isOn() );
        m_nextHitIndex = 0; //A next jump-search target reset to '0'.
        return ;
    }

    if ( m_nextHitIndex >= m_hitList.size() ) {
        return ;
    }

    KListViewItem* item = m_hitList[ m_nextHitIndex ];
    subjectList->ensureItemVisible( item );
    subjectList->setSelected( item, true );
    m_nextHitIndex++;
    if ( m_nextHitIndex >= m_hitList.size() ) m_nextHitIndex = 0;
}

void ThreadListView::searchAll( const QStringList &query )
{
    m_hitList.clear();
    m_prevquery = query;

    QListViewItemIterator listIt( subjectList );
    while ( listIt.current() != 0 ) {
        KListViewItem * item = static_cast<KListViewItem*>( listIt.current() );
        item->setPixmap( Col_Icon, 0 );

        QStringList::const_iterator queryIt = query.begin();
        for ( ; queryIt != query.end(); ++queryIt ) {
            if ( item->text( Col_Subject ).contains( *queryIt, false ) ) {
                item->setPixmap( Col_Icon, SmallIcon( "find" ) );
                m_hitList.append( item );
                break;
            }
        }
        ++listIt;
    }
}

void ThreadListView::slotHideButton( bool on )
{
    if ( m_hitList.empty() ) return ;

    QListViewItemIterator listIt( subjectList );
    while ( listIt.current() != 0 ) {
        KListViewItem * item = static_cast<KListViewItem *>( listIt.current() );
        if ( on && ! item->pixmap( Col_Icon ) ) {
            item->setVisible( false );
        } else {
            item->setVisible( true );
        }
        ++listIt;
    }
}

void ThreadListView::slotMouseButtonClicked( int button, QListViewItem* item )
{
    if ( ! item ) return ;

    KURL datURL = item->text( Col_DatURL );

    switch ( button ) {
    case MidButton:
        emit openURLRequestExt( datURL.prettyURL(), KParts::URLArgs(), "kita_open_2chthread", 2 );
        break;
    case LeftButton:
        if ( KitaConfig::alwaysUseTab() ) {
            emit openURLRequestExt( datURL.prettyURL(), KParts::URLArgs(), "kita_open_2chthread", 1 );
        } else {
            emit openURLRequestExt( datURL.prettyURL(), KParts::URLArgs(), "kita_open_2chthread", 0 );
        }
        break;
    }
}

void ThreadListView::hideColumn( int col )
{
    QHeader* header = subjectList->header();
    subjectList->setColumnWidthMode( col, QListView::Manual );
    header->setResizeEnabled( false, col );
    subjectList->setColumnWidth( col, 0 );
}

void ThreadListView::showColumn( int col )
{
    QHeader* header = subjectList->header();
    subjectList->setColumnWidthMode( col, QListView::Maximum );
    header->setResizeEnabled( true, col );
#if QT_VERSION >= 0x030200
    subjectList->adjustColumn( col );
#else
    subjectList->setColumnWidth( col, 30 );
#endif
}

#include "threadlistview.moc"
