/***************************************************************************
*   Copyright (C) 2004 by Kita Developers                                 *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

/* Kita Image Tab */

#include "imagetabwidget.h"
#include "imageview.h"

#include "libkita/imgmanager.h"
#include "libkita/signalcollection.h"
#include "libkita/datmanager.h"
#include "libkita/boardmanager.h"
#include "libkita/kita_misc.h"

#include <kapplication.h>
#include <klocale.h>
#include <kstdaccel.h>
#include <kpopupmenu.h>
#include <kaction.h>
#include <kiconloader.h>

#include <qiconset.h>
#include <qclipboard.h>
#include <qmessagebox.h>


/*--------------------------------------------------------------*/


KitaImgTabWidget::KitaImgTabWidget( QWidget* parent, const char* name, WFlags fl )
        : KitaTabWidgetBase( parent, name, fl )
{
    setXMLFile( "imagetabwidgetui.rc" );
    setTabBar( new KitaImgTabBar( this ) );
    setupActions();

    /* When loading is done,  ImgManager emits finishImgLoad SIGNAL.     */
    /* See also ImgManager::slotResult.                                  */
    connect( Kita::ImgManager::getInstance(), SIGNAL( receiveImgData( const KURL&, unsigned int, unsigned int ) ),
             SLOT( slotReceiveImgData( const KURL&, unsigned int, unsigned int ) ) );

    connect( Kita::ImgManager::getInstance(), SIGNAL( finishImgLoad( const KURL& ) ),
             SLOT( slotUpdateView( const KURL& ) ) );

    connect( Kita::ImgManager::getInstance(), SIGNAL( cacheDeleted( const KURL& ) ),
             SLOT( slotCloseImgView( const KURL& ) ) );

    Kita::SignalCollection* signalCollection = Kita::SignalCollection::getInstance();
    connect( signalCollection, SIGNAL( redrawImage( const KURL& ) ),
             SLOT( slotRedrawImage( const KURL& ) ) );
}


KitaImgTabWidget::~KitaImgTabWidget() {}


/* public slot */
void KitaImgTabWidget::slotOpenURL( const KURL& url, const KURL& datURL )
{
    KitaImgView * view = findImgView( url );
    if ( view ) {
        setCurrentPage( indexOf( view ) );

        if ( Kita::ImgManager::code( url ) == 200 ) return ;
        /* else reload */
    } else view = createImgView( url, datURL );

    /* create and show new image viewer */
    if ( view ) {

        if ( Kita::ImgManager::isLoadingNow( url ) ) return ;

        /* cache exists. */
        if ( Kita::ImgManager::code( url ) == 200 ) {
            slotUpdateView( url );
            return ;
        }

        /* start loading the image.                    */
        /* When done, slotUpdateView() is called back. */
        if ( !Kita::ImgManager::load( url, datURL ) ) {
            static_cast< KitaDockWidgetBase* >( parentWidget() ) ->slotShowDock();
            QMessageBox::warning( this,
                                  "Kita",
                                  i18n( "can't start downloading. try again later." ) );
            setTabIconSet( view, QIconSet( SmallIcon( "unread" ), QIconSet::Small ) );
            setTabToolTip( view, "error" );
        }
    }
}


/* public slot */
void KitaImgTabWidget::slotCloseImgView( const KURL& url )
{
    KitaImgView * view = findImgView( url );
    if ( view ) slotCloseTab ( indexOf( view ) );
}


/* public slot */
void KitaImgTabWidget::slotReceiveImgData( const KURL& url, unsigned int size , unsigned int totalsize )
{
    KitaImgView * view = findImgView( url );

    if ( view ) {
        QString str = QString( "Loading...%1k/%2k" ).arg( size / 1024 ).arg( totalsize / 1024 );
        setTabToolTip( view, str );
        view->showMessage( str );
    }
}


/* public slot */
void KitaImgTabWidget::slotUpdateView( const KURL& url )
{
    KitaImgView * view = findImgView( url );

    if ( view ) {

        int code = Kita::ImgManager::code( url );

        if ( code == 200 ) {

            QString str = QString::null;
            KURL datURL = view->datURL();
            if ( !datURL.isEmpty() ) {
                QString ref;
                Kita::getDatURL( datURL, ref );
                QString boardName = Kita::BoardManager::boardName( datURL );
                if ( boardName != QString::null ) str = "[" + boardName + "] ";
                str += Kita::DatManager::threadName( datURL );
                if ( str != QString::null ) str += "\n";
                str += Kita::getThreadURL( datURL ) + ref;
            }
            if ( str != QString::null ) str += "\n";
            str += url.prettyURL();

            setTabIconSet( view, QIconSet( Kita::ImgManager::icon( url ), QIconSet::Small ) );
            setTabToolTip( view, str );
            view->slotFitToWin();
        } else { /* error */

            setTabIconSet( view, QIconSet( SmallIcon( "unread" ), QIconSet::Small ) );
            QString str = "error " + QString().setNum( code );
            setTabToolTip( view, str );
            view->showMessage( str );
        }
    }
}


/* public slot */
void KitaImgTabWidget::slotRedrawImage( const KURL& url )
{
    KitaImgView * view = findImgView( url );
    if ( view ) view->slotShowImage();
}


/* protected */ /* virtual */
void KitaImgTabWidget::deleteWidget( QWidget* w )
{
    /* stop loading */
    KitaImgView * view = isImgView( w );
    if ( view ) {
        Kita::ImgManager::stop( view->url() );
    }

    KitaTabWidgetBase::deleteWidget( w );

    if ( count() == 0 ) {
        static_cast< KitaDockWidgetBase* >( parentWidget() ) ->slotHideDock();
    }
}


/* private */
KitaImgView* KitaImgTabWidget::createImgView( const KURL& url, const KURL& datURL )
{
    KitaImgView * view = new KitaImgView( url, datURL, actionCollection(), this, "Viewer", 0L );
    if ( view ) {
        addTab( view, "" );
        setTabIconSet( view, QIconSet( SmallIcon( "read" ), QIconSet::Small ) );
        setTabToolTip( view, "Loading..." );
        showPage( view );
        view->showMessage( "Loading..." );
    }

    return view;
}


/* private */
KitaImgView* KitaImgTabWidget::findImgView( const KURL& url )
{
    int max = count();
    if ( max == 0 ) return NULL;
    int i = 0;

    while ( i < max ) {
        KitaImgView * view = isImgView( page ( i ) );
        if ( view ) {
            if ( view->url() == url ) return view;
        }
        i++;
    }

    return NULL;
}


/* private */
KitaImgView* KitaImgTabWidget::isImgView( QWidget* w )
{
    KitaImgView * view = NULL;
    if ( w ) {
        if ( w->isA( "KitaImgView" ) ) view = static_cast< KitaImgView* >( w );
    }

    return view;
}


/*------------------------------------*/
/* Image View actions */


/* private */
void KitaImgTabWidget::setupActions()
{
    new KAction( i18n( "Stop" ),
                 "stop",
                 Key_Escape,
                 this,
                 SLOT( slotStopLoading() ),
                 actionCollection(),
                 "imgviewer_stop" );

    new KAction( i18n( "Cancel mosaic" ),
                 Key_X,
                 this,
                 SLOT( slotCancelMosaic() ),
                 actionCollection(),
                 "imgviewer_cancelmosaic" );

    new KAction( i18n( "Cancel All mosaics" ),
                 0,
                 this,
                 SLOT( slotCancelAllMosaic() ),
                 actionCollection(),
                 "imgviewer_cancelallmosaic" );

    new KAction( i18n( "Fit Image to Window" ),
                 0,
                 this,
                 SLOT( slotShowFitToWin() ),
                 actionCollection(),
                 "imgviewer_fittowin" );

    new KAction( i18n( "Original Size" ),
                 0,
                 this,
                 SLOT( slotOriginalSize() ),
                 actionCollection(),
                 "imgviewer_original" );

    new KAction( i18n( "Zoom In" ),
                 0,
                 this,
                 SLOT( slotZoomIn() ),
                 actionCollection(),
                 "imgviewer_zoomin" );

    new KAction( i18n( "Zoom Out" ),
                 0,
                 this,
                 SLOT( slotZoomOut() ),
                 actionCollection(),
                 "imgviewer_zoomout" );

    new KAction( i18n( "copy URL" ),
                 0,
                 this,
                 SLOT( slotCopyURL() ),
                 actionCollection(),
                 "imgviewer_copyurl" );

    new KAction( i18n( "Open with Web Browser" ),
                 0,
                 this,
                 SLOT( slotOpenBrowser() ),
                 actionCollection(),
                 "imgviewer_openbrowser" );

    new KAction( i18n( "Delete" ),
                 Key_Delete,
                 this,
                 SLOT( slotDelete() ),
                 actionCollection(),
                 "imgviewer_delete" );

    new KAction( i18n( "Save" ),
                 KStdAccel::save(),
                 this,
                 SLOT( slotSave() ),
                 actionCollection(),
                 "imgviewer_save" );

    new KAction( i18n( "Close all error tabs" ),
                 0,
                 this,
                 SLOT( slotCloseErrorTab() ),
                 actionCollection(),
                 "imgviewer_closeerrortab" );

    new KAction( i18n( "Reload" ),
                 KStdAccel::reload(),
                 this,
                 SLOT( slotReload() ),
                 actionCollection(),
                 "imgviewer_reload" );

    new KAction( i18n( "Open Thread" ),
                 0,
                 this,
                 SLOT( slotOpenThread() ),
                 actionCollection(),
                 "imgviewer_openthread" );

    m_sizeaction = new KSelectAction( i18n( "Size" ),
                                      0,
                                      this,
                                      SLOT( slotSelectSize() ),
                                      actionCollection(),
                                      "imgviewer_selectsize" );

    QStringList selectlist;
    selectlist << "20%" << "25%" << "50%" << "75%" << "100%" << "125%"
    << "150%" << "175%" << "200%" << "300%" << "400%" << "500%" ;
    m_sizeaction->setItems( selectlist );
}


/* public slot */
void KitaImgTabWidget::slotStopLoading( int idx )
{
    QWidget * w = currentPage();
    if ( idx != -1 ) w = page( idx );

    KitaImgView* view = isImgView( w );
    if ( view ) {
        Kita::ImgManager::stop( view->url() );
    }
}


/* public slot */
void KitaImgTabWidget::slotShowFitToWin( int idx )
{
    QWidget * w = currentPage();
    if ( idx != -1 ) w = page( idx );

    KitaImgView* view = isImgView( w );
    if ( view ) view->slotFitToWin();
}


/* public slot */
void KitaImgTabWidget::slotOriginalSize( int idx )
{
    QWidget * w = currentPage();
    if ( idx != -1 ) w = page( idx );

    KitaImgView* view = isImgView( w );
    if ( view ) view->slotOriginalSize();
}


/* public slot */
void KitaImgTabWidget::slotZoomIn( int idx )
{
    QWidget * w = currentPage();
    if ( idx != -1 ) w = page( idx );

    KitaImgView* view = isImgView( currentPage() );
    if ( view ) view->slotZoomIn();
}


/* public slot */
void KitaImgTabWidget::slotZoomOut( int idx )
{
    QWidget * w = currentPage();
    if ( idx != -1 ) w = page( idx );

    KitaImgView* view = isImgView( currentPage() );
    if ( view ) view->slotZoomOut();
}


/* public slot */
void KitaImgTabWidget::slotCancelMosaic( int idx )
{
    QWidget * w = currentPage();
    if ( idx != -1 ) w = page( idx );

    KitaImgView* view = isImgView( w );
    if ( view
            && Kita::ImgManager::mosaic( view->url() )
       ) {

        KURL url = view->url();
        Kita::ImgManager::setMosaic( url, FALSE );
        view->slotShowImage();
    }
}



/* public slot */
void KitaImgTabWidget::slotCancelAllMosaic()
{
    int max = count();
    if ( max == 0 ) return ;

    int i = 0;

    while ( i < max ) {

        KitaImgView * view = isImgView( page( i ) );
        if ( view
                && Kita::ImgManager::mosaic( view->url() )
           ) {

            KURL url = view->url();
            Kita::ImgManager::setMosaic( url, FALSE );
            view->slotShowImage();
        }

        i++;
    }
}


/* public slot */
void KitaImgTabWidget::slotCopyURL( int idx )
{
    QWidget * w = currentPage();
    if ( idx != -1 ) w = page( idx );

    QClipboard* clipboard = QApplication::clipboard();

    KitaImgView* view = isImgView( w );
    if ( view ) {
        QString cliptxt = view->url().prettyURL();

        clipboard->setText( cliptxt , QClipboard::Clipboard );
        clipboard->setText( cliptxt , QClipboard::Selection );
    }
}



/* public slot */
void KitaImgTabWidget::slotOpenBrowser( int idx )
{
    QWidget * w = currentPage();
    if ( idx != -1 ) w = page( idx );

    KitaImgView* view = isImgView( w );
    if ( view ) {
        int code = Kita::ImgManager::code( view->url() );
        if ( code != 200 ) emit openURLRequestExt( view->url(), KParts::URLArgs(), "text/html" );
        else {
            emit openURLRequestExt( Kita::ImgManager::getPath( view->url() ), KParts::URLArgs(), "text/html" );
        }
    }
}


/* public slot */
void KitaImgTabWidget::slotDelete( int idx )
{
    QWidget * w = currentPage();
    if ( idx != -1 ) w = page( idx );

    KitaImgView* view = isImgView( w );
    if ( view ) {
        Kita::ImgManager::deleteCache( view->url(), this );

        /* slotCloseImgView() will be called later via cacheDeleted SIGNAL. */
    }
}


/* public slot */
void KitaImgTabWidget::slotSave( int idx )
{
    QWidget * w = currentPage();
    if ( idx != -1 ) w = page( idx );

    KitaImgView* view = isImgView( w );
    if ( view ) {
        Kita::ImgManager::copyCache( view->url(), this );
    }
}



/* public slot */
void KitaImgTabWidget::slotCloseErrorTab()
{
    int max = count();
    if ( max == 0 ) return ;

    int i = 0, i2 = 0;

    while ( i < max ) {

        KitaImgView * view = isImgView( page( i ) );
        if ( view ) {
            int code = Kita::ImgManager::code( view->url() );

            if ( code != 200 ) slotCloseTab( i2 );
            else i2++;
        } else i2++;

        i++;
    }
}


/* public slot */
void KitaImgTabWidget::slotReload( int idx )
{
    QWidget * w = currentPage();
    if ( idx != -1 ) w = page( idx );

    KitaImgView* view = isImgView( w );
    if ( view ) {
        slotOpenURL( view->url(), view->datURL() );
    }
}


/* public slot */
void KitaImgTabWidget::slotOpenThread( int idx )
{
    QWidget * w = currentPage();
    if ( idx != -1 ) w = page( idx );

    KitaImgView* view = isImgView( w );
    if ( view ) {
        KURL datURL = view->datURL();

        emit openURLRequestExt( datURL );
    }
}


/* public slot */
void KitaImgTabWidget::slotSelectSize()
{
    int size = m_sizeaction->currentText().remove( "%" ).toInt();

    QWidget * w = currentPage();
    KitaImgView* view = isImgView( w );
    if ( view ) view->slotSetSize( size );
}


/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/



KitaImgTabBar::KitaImgTabBar( QWidget* parent, const char* name ) : KitaTabBarBase( parent, name ) {}

KitaImgTabBar::~KitaImgTabBar() {}


/* private */ /* virtual */
void KitaImgTabBar::showPopupMenu( int idx, QPoint global )
{
    enum{
        MENU_CLOSE,
        MENU_CLOSEOTHER,
        MENU_CLOSELEFT,
        MENU_CLOSERIGHT,
        MENU_CANCELMOS,
        MENU_FITTOWIN,
        MENU_ORIGINAL,
        MENU_STOP,
        MENU_OPENBROWSER,
        MENU_OPENTHREAD,
        MENU_COPYURL,
        MENU_SAVE,
        MENU_RELOAD,
        MENU_DELETE
    };

    KitaImgTabWidget* tabwidget = static_cast<KitaImgTabWidget*>( parentWidget() );
    KActionCollection * collection = tabwidget->actionCollection();

    KPopupMenu* popup = new KPopupMenu( this );
    popup->clear();

    popup->insertItem( i18n( "Close this tab" ) , MENU_CLOSE );
    collection->action( "tab_prevtab" ) ->plug( popup );
    collection->action( "tab_nexttab" ) ->plug( popup );
    popup->insertSeparator();

    popup->insertItem( i18n( "Close Other Tabs" ) , MENU_CLOSEOTHER );
    popup->insertItem( i18n( "Close right tabs" ) , MENU_CLOSERIGHT );
    popup->insertItem( i18n( "Close left tabs" ) , MENU_CLOSELEFT );
    collection->action( "tab_closealltab" ) ->plug( popup );

    collection->action( "imgviewer_closeerrortab" ) ->plug( popup );

    popup->insertSeparator();

    popup->insertItem( i18n( "Cancel mosaic" ) , MENU_CANCELMOS );
    collection->action( "imgviewer_cancelallmosaic" ) ->plug( popup );
    popup->insertItem( i18n( "Fit Image to Window" ) , MENU_FITTOWIN );
    popup->insertItem( i18n( "Original Size" ) , MENU_ORIGINAL );

    popup->insertSeparator();
    popup->insertItem( i18n( "Open with Web Browser" ) , MENU_OPENBROWSER );
    popup->insertItem( i18n( "Open Thread" ) , MENU_OPENTHREAD );
    popup->insertItem( i18n( "copy URL" ) , MENU_COPYURL );
    popup->insertItem( i18n( "Save" ) , MENU_SAVE );
    popup->insertItem( i18n( "Reload" ) , MENU_RELOAD );
    popup->insertItem( i18n( "Stop" ) , MENU_STOP );
    popup->insertItem( i18n( "Delete" ) , MENU_DELETE );

    popup->insertSeparator();
    collection->action( "tab_configkeys" ) ->plug( popup );

    int ret = popup->exec( global );
    delete popup;

    switch ( ret ) {
    case MENU_CLOSE: tabwidget->slotCloseTab( idx ); break;
    case MENU_CLOSEOTHER: tabwidget->slotCloseOtherTab( idx ); break;
    case MENU_CLOSERIGHT: tabwidget->slotCloseRightTab( idx ); break;
    case MENU_CLOSELEFT: tabwidget->slotCloseLeftTab( idx ); break;
    case MENU_CANCELMOS: tabwidget->slotCancelMosaic( idx ); break;
    case MENU_FITTOWIN: tabwidget->slotShowFitToWin( idx ); break;
    case MENU_ORIGINAL: tabwidget->slotOriginalSize( idx ); break;
    case MENU_OPENBROWSER: tabwidget->slotOpenBrowser( idx ); break;
    case MENU_OPENTHREAD: tabwidget->slotOpenThread( idx ); break;
    case MENU_COPYURL: tabwidget->slotCopyURL( idx ); break;
    case MENU_SAVE: tabwidget->slotSave( idx ); break;
    case MENU_RELOAD: tabwidget->slotReload( idx ); break;
    case MENU_DELETE: tabwidget->slotDelete( idx ); break;
    case MENU_STOP: tabwidget->slotStopLoading( idx ); break;
    }

}




/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/



KitaImgDock::KitaImgDock( KDockManager* dockManager,
                          const char* name,
                          const QPixmap &pixmap,
                          QWidget* parent,
                          const QString& strCaption,
                          const QString& strTabPageLabel,
                          WFlags f )
    : KitaDockWidgetBase( dockManager, name, pixmap, parent, strCaption, strTabPageLabel, f ) {}


KitaImgDock::~KitaImgDock() {}


/* setup & create KitaImgTabWidget */ /* public */
KitaImgTabWidget* KitaImgDock::setup()
{
    /* create KitaNaviTabWidget */
    m_imgTab = new KitaImgTabWidget( this, "Image Viewer" );
    setWidget( m_imgTab );

    Kita::SignalCollection* signalCollection = Kita::SignalCollection::getInstance();
    connect( signalCollection, SIGNAL( switchToImgview() ),
             SLOT( slotShowDock() ) );

    return m_imgTab;
}


/* public slot */
void KitaImgDock::slotOpenURL( const KURL& url, const KURL& datURL, bool activate )
{
    if ( activate ) slotShowDock();

    if ( m_imgTab ) m_imgTab->slotOpenURL( url, datURL );
}
