/***************************************************************************
 *   Copyright (C) 2003 by Hideki Ikemoto                                  *
 *   ikemo@users.sourceforge.jp                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include "thread.h"

#include <config.h>

#include <kurl.h>
#include <kio/slaveconfig.h>
#include <kio/jobclasses.h>
#include <kio/scheduler.h>
#include <kdebug.h>

#include <qwidget.h>
#include <qapplication.h>
#include <qregexp.h>

#include "board.h"
#include "kita_misc.h"
#include "qcp932codec.h"
#include "threadinfo.h"

using namespace Kita;

ThreadFactory* ThreadFactory::instance = 0;

Thread::Thread()
{
}

Thread::Thread( const QString& datURL, const QString& threadName )
  : m_datURL( datURL ), m_threadName( threadName )
{
}

Thread::Thread( const Thread& obj )
{
  m_datURL = obj.m_datURL;
  m_threadName = obj.m_threadName;
  setResNum( obj.resNum() );
}

Thread::~Thread()
{
}

Thread& Thread::operator=( const Thread& obj )
{
  m_datURL = obj.m_datURL;
  m_threadName = obj.m_threadName;
  setResNum( obj.resNum() );

  return *this;
}

const QString& Thread::datURL() const
{
  return m_datURL;
}

const QString Thread::url() const
{
    return datToThread( m_datURL );
}

const QString Thread::datID() const
{
  return KURL( m_datURL ).filename().section( ".", 0, 0 );
}

int Thread::resNum() const
{
    return KitaThreadInfo::resNum( m_datURL );
}

const QString& Thread::name() const
{
  return m_threadName;
}

void Thread::setResNum( int resNum )
{
    KitaThreadInfo::setResNum( m_datURL, resNum );
}

const QString& Thread::boardName() const
{
    Board* board = Board::getByURL( boardURL() );
    return board->name();
}

const QString Thread::boardURL() const
{
  return datToBoard( m_datURL );
}

const QString Thread::boardID() const
{
  return KURL( datToBoard( m_datURL ) ).fileName();
}

const QString Thread::toXmlFragment() const
{
  QString ret;

  ret += "<thread xmlns=\"http://kita.sourceforge.jp/ns/thread\">\n";
  ret += QString("<daturl>%1</daturl>\n").arg( m_datURL );
  ret += QString("<name>%1</name>\n").arg( m_threadName );
  ret += QString("<resnum>%1</resnum>\n").arg( resNum() );

  // FIXME: Board饹饳ԡ
  ret += "<board xmlns=\"http://kita.sourceforge.jp/ns/board\">\n";
  ret += QString("<url>%1</url>\n").arg( boardURL() );
  ret += QString("<name>%1</name>\n").arg( boardName() );
  ret += "</board>\n";

  ret += "</thread>\n";

  return ret;
}

Thread Thread::fromXml( const QString& xml )
{
  Thread thread;

  ThreadXmlParser parser;
  QXmlSimpleReader reader;
  QXmlInputSource source;
  source.setData( xml );
  reader.setContentHandler( &parser );
  reader.parse( &source );

  if( parser.isValid() ) {
    thread = parser.getThread();
  } else {
    kdError() << "inValid" << endl;
  }

  return thread;
}

bool ThreadXmlParser::startElement( const QString& namespaceURI, const QString& localName, const QString& qName, const QXmlAttributes& atts )
{
  if( m_inBoard ) {
    m_boardParser->startElement( namespaceURI, localName, qName, atts );
  } else if( m_inThread ) {
    if( localName == "daturl" ) {
    } else if( localName == "name" ) {
    } else if( localName == "resnum" ) {
    } else if( localName == "board" ) {
      m_inBoard = true;
      m_boardParser = new BoardXmlParser();
      m_boardParser->startElement( namespaceURI, localName, qName, atts );
    } else {
      // error
      return false;
    }
  } else {
    if( localName == "thread" ) {
      m_inThread = true;
    } else {
      // error
      return false;
    }
  }

  return true;
}

// TODO: <resnum>FavoritesǤϻȤʤȤˤʤäΤǺɬפꡣ
bool ThreadXmlParser::endElement( const QString& namespaceURI, const QString& localName, const QString& qName )
{
  if( localName == "board" ) {
    m_inBoard = false;
    m_boardParser->endElement( namespaceURI, localName, qName );

    delete m_boardParser;
    m_boardParser = 0;

    // create board;
  } else if( m_inBoard ) {
    m_boardParser->endElement( namespaceURI, localName, qName );
  } else if( localName == "thread" ) {
    m_inThread = false;
    m_thread = Kita::Thread( m_datUrlStr, m_threadName );
    KitaThreadInfo::setResNum( m_datUrlStr, m_resNum );
    m_isValid = true;
    // create thread;
  } else if( localName == "daturl" ) {
    m_datUrlStr = m_characters;
  } else if( localName == "name" ) {
    m_threadName = m_characters;
  } else if( localName == "resnum" ) {
    m_resNum = m_characters.toInt();
  } else {
    // error
    return false;
  }
  return true;
}

bool ThreadXmlParser::characters( const QString& ch )
{
  if( m_inBoard ) {
    m_boardParser->characters( ch );
  } else {
    m_characters = ch;
  }
  return true;
}

QString ThreadAccess::get()
{
  if ( KURL( m_thread.datURL() ).protocol() != "k2ch" ) {
    KIO::SlaveConfig::self()->setConfigData("http",
      KURL( m_thread.datURL() ).host(),
      "UserAgent",
      QString( "Monazilla/1.00 (Kita/%1)" ).arg( VERSION ) );
  }

  KIO::TransferJob* job = KIO::get(m_thread.datURL(), true, true);
  m_currentJob = job;

  connect(job, SIGNAL(data(KIO::Job*, const QByteArray&)),
          SLOT(slotReceiveThreadData(KIO::Job*, const QByteArray&)));
  connect(job, SIGNAL(result(KIO::Job*)), SLOT(slotThreadResult(KIO::Job*)));
  connect(job, SIGNAL(redirection(KIO::Job *, const KURL&) ), SLOT(slotRedirection(KIO::Job *, const KURL&) ) );

  // use 'HTTP-Headers' metadata.
  job->addMetaData("PropagateHttpHeader", "true");

  enter_loop();

  QCp932Codec codec;
  return codec.toUnicode( m_threadData );
}

// from netaccess.cpp
void qt_enter_modal( QWidget* widget );
void qt_leave_modal( QWidget* widget );

void ThreadAccess::enter_loop()
{
  QWidget dummy( 0, 0, WType_Dialog | WShowModal );
  dummy.setFocusPolicy( QWidget::NoFocus );
  qt_enter_modal(&dummy);
  qApp->enter_loop();
  qt_leave_modal(&dummy);
}

void ThreadAccess::slotRedirection(KIO::Job *, const KURL & newURL)
{
  kdDebug() << "Redirected to newURL: " << newURL.url() << endl;
  Kita::Board::setName( datToBoard( newURL.url() ), m_thread.boardName() );
  m_thread = Kita::Thread( newURL.url() );
  emit redirection( newURL.url() );
}

void ThreadAccess::slotReceiveThreadData( KIO::Job*, const QByteArray& data )
{
  QCString cstr(data.data(), data.size()+1);
  m_threadData.append(cstr);
}

void ThreadAccess::slotThreadResult(KIO::Job* job)
{
  m_currentJob = 0;
  if(job->error()) {
    job->showErrorDialog();
  } else {
    m_header = job->queryMetaData("HTTP-Headers");
  }
  qApp->exit_loop();
}

void ThreadAccess::killJob()
{
  if( m_currentJob ) m_currentJob->kill();
}

int ThreadAccess::serverTime()
{
    // parse HTTP headers
    QStringList headerList = QStringList::split( "\n", m_header );
    QRegExp regexp( "Date: (...), (..) (...) (....) (..:..:..) .*" );
    QString dateStr = headerList.grep(regexp)[0];
    if( regexp.search(dateStr) == -1 ) {
        // invalid date format
        return QDateTime::currentDateTime().toTime_t();
    } else {
        // I hate this format ;p
        QString usLocalDateStr = regexp.cap(1) + " " + regexp.cap(3) + " " +
                                 regexp.cap(2) + " " + regexp.cap(5) + " " +
                                 regexp.cap(4);

        // 1970/01/01 00:00:00 GMT
        QDateTime zeroTime(QDate(1970, 1, 1), QTime(0, 0));
        return zeroTime.secsTo(QDateTime::fromString(usLocalDateStr));
    }
}

void ThreadFactory::insertThread( Thread& thread )
{
  if( ! instance ) {
    instance = new ThreadFactory();
  }

  Thread* new_thread = new Thread( thread.datURL(), thread.name() );
  KitaThreadInfo::setResNum( thread.datURL(), thread.resNum() );
  instance->m_dict.insert( thread.datURL(), new_thread );
}

Thread* ThreadFactory::getThread( QString& datURL )
{
  if( ! instance ) {
    instance = new ThreadFactory();
  }

  return instance->m_dict[ datURL ];
}

#include "thread.moc"
