/****************************************************************************
 * KONOHA COPYRIGHT, LICENSE NOTICE, AND DISCRIMER
 *
 * Copyright (c) 2005-2008, Kimio Kuramitsu <kimio at ynu.ac.jp>
 *           (c) 2008-      Konoha Software Foundation
 * All rights reserved.
 *
 * You may choose one of the following two licenses when you use konoha.
 * See www.konohaware.org/license.html for further information.
 *
 * (1) GNU General Public License 2.0      (with    KONOHA_UNDER_GPL2)
 * (2) Konoha Software Foundation License 1.0
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ****************************************************************************/

/* ************************************************************************ */

#include"commons.h"

/* ************************************************************************ */

#ifdef __cplusplus
extern "C" {
#endif

/* ======================================================================== */
/* [structs] */

void
knh_OutputStream_struct_init(Ctx *ctx, knh_OutputStream_struct *b, int init, Object *cs)
{
	b->fd = -1;
	b->driver = knh_System_getDefaultIODriver();
	KNH_INITv(b->ba, new_Bytes(ctx, 8));
	KNH_INITv(b->bconv, KNH_NULL);
	KNH_INITv(b->enc, TS_ENCODING);
	KNH_INITv(b->urn, TS_DEVNULL);
	b->size = 0;
	b->line = 0;
	KNH_INITv(b->NEWLINE, TS_LF);
	KNH_INITv(b->TAB, TS_TAB);
	b->indent = 0;
	b->flag = 0;
}

/* ------------------------------------------------------------------------ */

#define _knh_OutputStream_struct_copy      NULL

/* ------------------------------------------------------------------------ */

#define _knh_OutputStream_struct_compare   NULL

/* ------------------------------------------------------------------------ */

void
knh_OutputStream_struct_traverse(Ctx *ctx, knh_OutputStream_struct *b, f_traverse gc)
{
	if(IS_SWEEP(gc) && b->fd != -1) {
		b->driver.fclose(ctx, b->fd);
		b->fd = -1;
	}
	gc(ctx, UP(b->ba));
	gc(ctx, UP(b->enc));
	gc(ctx, UP(b->bconv));
	gc(ctx, UP(b->urn));
	gc(ctx, UP(b->NEWLINE));
	gc(ctx, UP(b->TAB));
}

/* ======================================================================== */
/* [methods] */

Object *knh_OutputStream_open(Ctx *ctx, OutputStream *o, String *urn, String *mode)
{
	knh_bytes_t fname = knh_String_tobytes(urn);
	knh_index_t loc = knh_bytes_index(fname, ':');

	if(loc == -1 || (loc == 1 && isalpha(fname.buf[0]))) {  /* 'C:/' */
		DP(o)->driver = *(knh_System_getIODriver(ctx, STEXT("file")));
	}
	else {
		DP(o)->driver = *(knh_System_getIODriver(ctx, knh_bytes_first(fname, loc)));
	}
	if(IS_NULL(mode)) {
		DP(o)->fd = DP(o)->driver.fopen(ctx, UP(o), fname, "w");
	}
	else {
		DP(o)->fd = DP(o)->driver.fopen(ctx, UP(o), fname, knh_String_tochar(mode));
	}

	if(DP(o)->fd != -1) {
		KNH_SETv(ctx, DP(o)->ba, new_Bytes(ctx, DP(o)->driver.bufsiz));
		knh_OutputStream_setBOL(o,1);
	}
	else {
		char buff[FILENAME_BUFSIZ];
		knh_snprintf(buff, sizeof(buff), "IO!!: cannot open %s", (char*)fname.buf);
		DP(o)->driver = knh_System_getDefaultIODriver();
		return (Object*)new_Nue__s(ctx, buff);
	}
	return (Object*)o;
}

/* ======================================================================== */
/* [methods] */

void knh_OutputStream_putc(Ctx *ctx, OutputStream *o, int ch)
{
	Bytes *ba = DP(o)->ba;
	KNH_ASSERT(IS_Bytes(ba));
	knh_Bytes_putc(ctx, ba, ch);
	if(!knh_OutputStream_isStoringBuffer(o) && ba->size > DP(o)->driver.bufsiz) {
		DP(o)->driver.fwrite(ctx, DP(o)->fd, (char*)(ba)->buf, (ba)->size);
		knh_Bytes_clear(ba);
	}
	DP(o)->size++;
}

/* ------------------------------------------------------------------------ */

void knh_OutputStream_write(Ctx *ctx, OutputStream *o, knh_bytes_t buf)
{
	Bytes *ba = DP(o)->ba;
	KNH_ASSERT(IS_Bytes(ba));
	knh_Bytes_write(ctx, ba, buf);
	if(!knh_OutputStream_isStoringBuffer(o) && ba->size > DP(o)->driver.bufsiz) {
		DP(o)->driver.fwrite(ctx, DP(o)->fd, (char*)(ba)->buf, (ba)->size);
		knh_Bytes_clear(ba);
	}
	DP(o)->size += buf.len;
}

/* ------------------------------------------------------------------------ */
/* @method void OutputStream.flush() */

void knh_OutputStream_flush(Ctx *ctx, OutputStream *o)
{
	if(!knh_OutputStream_isStoringBuffer(o)) {
		Bytes *ba = DP(o)->ba;
		DP(o)->driver.fwrite(ctx, DP(o)->fd, (char*)(ba)->buf, (ba)->size);
		knh_Bytes_clear(ba);
	}
}

/* ------------------------------------------------------------------------ */
/* @method void OutputStream.clear() */

void knh_OutputStream_clear(Ctx *ctx, OutputStream *o)
{
	if(knh_OutputStream_isStoringBuffer(o)) {
		knh_Bytes_clear(DP(o)->ba);
	}
}

/* ------------------------------------------------------------------------ */
/* @method void OutputStream.close() */

void knh_OutputStream_close(Ctx *ctx, OutputStream *o)
{
	f_io_close f = DP(o)->driver.fclose;
	DP(o)->driver = knh_System_getDefaultIODriver();
	f(ctx, DP(o)->fd);
	DP(o)->fd = -1;
}

/* ------------------------------------------------------------------------ */

int knh_OutputStream_isClosed(OutputStream *o)
{
	return (DP(o)->fd == -1);
}

/* ------------------------------------------------------------------------ */

void knh_OutputStream_indent_inc(Ctx *ctx, OutputStream *o)
{
	DP(o)->indent++;
}

/* ------------------------------------------------------------------------ */

void knh_OutputStream_indent_dec(Ctx *ctx, OutputStream *o)
{
	DP(o)->indent--;
}

/* ------------------------------------------------------------------------ */

void knh_OutputStream_write_indent(Ctx *ctx, OutputStream *o)
{
	int i;
	for(i = 0; i < DP(o)->indent; i++) {
		knh_OutputStream_write(ctx, o, knh_String_tobytes(DP(o)->TAB));
	}
}

/* ------------------------------------------------------------------------ */

void knh_OutputStream_print_(Ctx *ctx, OutputStream *o, knh_bytes_t str, knh_bool_t isnl)
{
	if(str.len > 0) {
		if(knh_OutputStream_isBOL(o)) {
			knh_write_BOL(ctx, o);
		}
		if(IS_NULL(DP(o)->bconv)) {
			knh_OutputStream_write(ctx, o, str);
		}
		else {
			Bytes *ba = knh_Context_openBConvBuf(ctx);
			knh_BytesConv_conv(ctx, DP(o)->bconv, str, ba);
			knh_OutputStream_write(ctx, o, knh_Bytes_tobytes(ba));
			knh_Context_closeBConvBuf(ctx, ba);
		}
	}
	if(isnl) {
		knh_write_EOL(ctx, o);
	}
}


/* ======================================================================== */

OutputStream *new_OutputStream__stdio(Ctx *ctx, FILE *fp)
{
	OutputStream* o = (OutputStream*)new_Object_malloc(ctx, FLAG_OutputStream, CLASS_OutputStream, sizeof(knh_OutputStream_struct));
	knh_OutputStream_struct_init(ctx, DP(o), 0, NULL);
	knh_OutputStream_setBOL(o, 1);

	KNH_ASSERT(fp == stdout || fp == stderr);

	DP(o)->driver = knh_System_getStdIODriver();
	DP(o)->fd = (knh_io_t)fp;

	if(fp == stdout) {
		KNH_SETv(ctx, DP(o)->urn, TS_DEVSTDOUT);
		knh_OutputStream_setAutoFlush(o, 1);
	}
	else if(fp == stderr) {
		KNH_SETv(ctx, DP(o)->urn, TS_DEVSTDERR);
		knh_OutputStream_setAutoFlush(o, 1);
	}
	return o;
}

/* ------------------------------------------------------------------------ */

KNHAPI(OutputStream*) new_FileOutputStream(Ctx *ctx, knh_bytes_t file, char *mode)
{
	OutputStream* o = (OutputStream*)new_Object_malloc(ctx, FLAG_OutputStream, CLASS_OutputStream, sizeof(knh_OutputStream_struct));
	knh_OutputStream_struct_init(ctx, DP(o), 0, NULL);
	knh_OutputStream_setBOL(o, 1);

	DP(o)->driver = *(knh_System_getIODriver(ctx, STEXT("file")));
	DP(o)->fd = DP(o)->driver.fopen(ctx, UP(o), file, mode);
	if(DP(o)->fd != -1) {
		KNH_SETv(ctx, DP(o)->ba, new_Bytes(ctx, DP(o)->driver.bufsiz));
		KNH_SETv(ctx, DP(o)->urn, new_String(ctx, file, NULL));
	}
	else {
		DP(o)->driver = knh_System_getDefaultIODriver();
	}
	return o;
}

/* ------------------------------------------------------------------------ */

KNHAPI(OutputStream*) new_BytesOutputStream(Ctx *ctx, Bytes *ba)
{
	OutputStream* o = (OutputStream*)new_Object_malloc(ctx, FLAG_OutputStream, CLASS_OutputStream, sizeof(knh_OutputStream_struct));
	knh_OutputStream_struct_init(ctx, DP(o), 0, NULL);
	knh_OutputStream_setBOL(o, 1);
	KNH_ASSERT(IS_Bytes(ba));
	KNH_SETv(ctx, DP(o)->ba, ba);
	knh_OutputStream_setStoringBuffer(o, 1);
	return o;
}

/* ------------------------------------------------------------------------ */

#ifdef __cplusplus
}
#endif
