/****************************************************************************
 * KONOHA COPYRIGHT, LICENSE NOTICE, AND DISCRIMER
 *
 * Copyright (c) 2006-2010, Kimio Kuramitsu <kimio at ynu.ac.jp>
 *           (c) 2008-      Konoha Team konohaken@googlegroups.com
 * All rights reserved.
 * You may choose one of the following two licenses when you use konoha.
 * See www.konohaware.org/license.html for further information.
 *
 * (1) GNU Lesser General Public License 3.0 (with K_UNDER_LGPL)
 * (2) Konoha Software Foundation License 1.0
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ****************************************************************************/

#ifdef __cplusplus
extern "C" {
#endif

#ifdef K_USING_SSA

#define USE_cwb_open    1
#define USE_cwb_size    1
#define USE_STEXT       1

#include "commons.h"

/* ------------------------------------------------------------------------ */
/* [SSA Options] */

//#define SSA_CONV_MINIMAL /* minimal SSA */
#define SSA_CONV_SEMI /* semi-pruned SSA */

#define SSA_CONVOUT_NAIVE /* naive (Sreedhar I) */
//#define SSA_CONVOUT_INTERFERENCE /* Interference Graph Update (Sreedhar III) */

/* ------------------------------------------------------------------------ */
/* [SSA Optimizations] */

//#define SSA_OPT_AGGRESSIVE_DCE /* Aggressive Dead Code Elimination */
#define SSA_OPT_CP /* Constnat Propagation */
//#define SSA_OPT_SCCP /* Sparse Conditional Constant Propagation */
//#define SSA_OPT_CSE /* Common Subexpression Elimination */

/* ------------------------------------------------------------------------ */

#if defined(SSA_CONVOUT_NAIVE)
#define eliminatePhi(ctx, ssa) eliminatePhi_naive(ctx, ssa)
#elif defined(SSA_CONVOUT_INTERFERENCE)
#define eliminatePhi(ctx, ssa) eliminatePhi_interference(ctx, ssa)
#endif

#define USE_MAX         512

#define Array__(data, n)   ((knh_Array_t*)knh_Array_n(data, n))
#define Array(T, data, n)  ((knh_##T##_t*)iArray_n(data, n))
#define PRED(bb)           Array__(ssa->pred, ID(bb))
#define DEFSITES(n)        Array__(defsites, n)
#define APHI(n)            Array__(Aphi, n)
#define STACK(n)           Array__(rs->stack, iArray_n(rs->origidx, n))
#define AORIG(n)           Array__(Aorig, n)
#define DEF(n)             Array(opline, ssa->def, n)
#define RDF(bb)            Array__(ssa->rdf, ID(bb))
#define DF(bb)             Array__(ssa->df, ID(bb))
#define USE(n)             Array__(ssa->use, n)
#define IFGRAPH(n)         Array__(ssa->interference, n)
#define PCC(n)             Array__(pcc, n)
#define UNMAP(n)           Array__(unresolvedNeighborMap, n)
#define LIVEIN(bb)         Array__(ssa->liveIn, ID(bb))
#define LIVEOUT(bb)        Array__(ssa->liveOut, ID(bb))
#define ID(bb)             (DP(bb)->id)
#define TOP(a)             iArray_n(a, knh_Array_size(a) - 1)
#define PUSH(a, n)         iArray_add(ctx, a, n)
#define LIVE(s, n)         iArray_n(live, n)
#define iArray_n(a, i)     ((a)->ilist[i])
#define iArray_pop(ctx, a) knh_Array_clear(ctx, a, knh_Array_size(a) - 1)
#define isOBJ(n)           ((n) % 2 == 0)
#define SWAP(a, b, size)\
	do {\
		register size_t __size = (size);\
		register char *__a = (char*)(a);\
		register char *__b = (char*)(b);\
		do {\
			char __tmp = *__a;\
			*__a++ = *__b;\
			*__b++ = __tmp;\
		} while (--__size > 0);\
	} while (0)

#define REUSE(op, on, nn)\
	/*DBG_P("replace USE %s %d to %d", knh_opcode_tochar(op->opcode), on, nn);*/\
	DBG_ASSERT(isOBJ(on) == isOBJ(nn));\
	on = nn
#define REDEF(op, on, nn)\
	/*DBG_P("replace DEF %s %d to %d", knh_opcode_tochar(op->opcode), on, nn);*/\
	DBG_ASSERT(isOBJ(on) == isOBJ(nn));\
	on = nn

#define INSERTED_ARG    ((size_t)-1)
#define MOD_LEFT        (0x10)
#define MOD_RIGHT       (0x01)
#define MOD_BOTH        (0x11)
#define op_isArg(op)    ((op)->count == INSERTED_ARG)

#if defined(K_USING_THREADEDCODE)
#define TADDR1   NULL, INSERTED_ARG/*counter*/
#else
#define TADDR1   INSERTED_ARG/*counter*/
#endif/*K_USING_THREADEDCODE*/

#define knh_BasicBlock_add1(ctx, bb, T, ...) { \
		klr_##T##_t op_ = {TADDR, OPCODE_##T, ASMLINE, ## __VA_ARGS__}; \
		knh_BasicBlock_add_(ctx, bb, 0, (knh_opline_t*)(&op_)); \
	}

#define INSERT_ARG(ctx, n, line, T, ...) { \
		klr_##T##_t op_ = {TADDR1, OPCODE_##T, ASMLINE, ## __VA_ARGS__}; \
		insertOpline(ctx, n, (knh_opline_t*)(&op_), line); \
		line += 1; \
	}

#define INSERT_OP(ctx, n, line, T, ...) { \
		klr_##T##_t op_ = {TADDR, OPCODE_##T, ASMLINE, ## __VA_ARGS__}; \
		insertOpline(ctx, n, (knh_opline_t*)(&op_), line); \
		line += 1; \
	}

#ifdef K_USING_DEBUG
#define RENAME_ASSERT(c)\
	if (!(c)) {\
		printArray_Int(rs->origidx);\
		printArray_Array(rs->stack);\
		DBG_ABORT();\
	}
#else
#define RENAME_ASSERT(c)
#endif

#define UNUSED(v) (void)(v)

#define op_eq(op, OP)  ((op)->opcode == OPCODE_##OP)
#define op_neq(op, OP) (!op_eq(op, OP))
#define isArraySET(op) (op_eq(op, OSETIDX) || op_eq(op, OSETIDXn) || \
	op_eq(op, NSETIDX) || op_eq(op, NSETIDXn) || \
	op_eq(op, BSETIDX) || op_eq(op, BSETIDXn))
#define isArrayGET(op) (op_eq(op, OGETIDX) || op_eq(op, OGETIDXn) || \
	op_eq(op, NGETIDX) || op_eq(op, NGETIDXn) || \
	op_eq(op, BGETIDX) || op_eq(op, BGETIDXn))
#define isDEF(op)      (knh_opcode_usedef(op->opcode, 0) && op_neq(op, TRYEND) && op_neq(op, THROW)\
	&& op_neq(op, LOADMTD))
#define isUSE(op, n)   knh_opcode_usedef(op->opcode, n)
#define isVDEF(op, n)  ((n == 0 && ((op->opcode >= OPCODE_XMOV && op->opcode <= OPCODE_XBMOV) ||\
	op_eq(op, LOADMTD) || op_eq(op, TRYEND) || op_eq(op, THROW))) || ((n == 1) && isArraySET(op)))
#define isVUSE(op, n)  (((n == 1) && ((op->opcode >= OPCODE_OMOVx && op->opcode <= OPCODE_bMOVx) || \
	op_eq(op, XMOVx) || isArrayGET(op))) || knh_opcode_sfpidx(op->opcode, n))
#define isCALL(op)     (op_eq(op, SCALL) || op_eq(op, VCALL) || \
	op_eq(op, VCALL_) || op_eq(op, CALL) || op_eq(op, FASTCALL0))
#define isCAST(op)     (op_eq(op, TCAST) || op_eq(op, SCAST) || op_eq(op, TR))

#define Array_each_cast(a, i, v, T)\
	Array_each_next(a, i, v, Array(T, a, i))

#define Array_each(a, i, v)\
	Array_each_next(a, i, v, iArray_n(a, i))

#define Array_each_next(a, i, v, NEXT)\
	for (i = 0, v = (NEXT);\
			i < knh_Array_size(a);\
			i++, v = (NEXT))

#define bblock_each(bb, i, op)\
	for (i = 0, op = DP(bb)->opbuf;\
			i < DP(bb)->size;\
			i++, op = DP(bb)->opbuf + i)

#define IDOM(bb)          iArray_n(idom, ID(bb))
#define DFN(bb)           ((cfg)->dfn(cfg, bb))
#define VERTEX(n)         Array(BasicBlock, ssa->vertex, n)
#define PARENT(data, bb)  ((data)->parent(data, bb))
#define ANCESTOR(bb)      Array(BasicBlock, ancestor, ID(bb))
#define SEMI(bb)          Array(BasicBlock, semi, ID(bb))
#define SAMEDOM(bb)       Array(BasicBlock, samedom, ID(bb))
#define BEST(bb)          Array(BasicBlock, best, ID(bb))
#define BUCKET(bb)        Array__(bucket, ID(bb))
#define LINK(bb1, bb2)\
	iArray_n(ancestor, ID(bb2)) = (knh_int_t)bb1;\
	iArray_n(best, ID(bb2)) = (knh_int_t)bb2

#define itr_bb(itr) ((itr)->bb)
#define itr_foreach(bb, itr)\
	for (bb = itr->next(itr); itr->pos < itr->end; itr->pos++, bb = itr->next(itr))

#define DBG_A(arg, f)\
	DBG_P("%s", arg);\
	DBG_(f)

typedef struct ssa_data_t {
	knh_Array_t *df, *pred, *idom, *vertex;
	knh_Array_t *liveIn, *liveOut, *interference;
	knh_Array_t *def, *use, *rdf;
	size_t stacktop;
	size_t rindex;
} ssa_data_t;

struct bbitr_t;
typedef knh_BasicBlock_t *(*fitrnext) (struct bbitr_t *itr);
typedef struct bbitr_t* (*fitrinit) (struct bbitr_t *, fitrnext f, knh_Array_t *, knh_BasicBlock_t *);

typedef struct bbitr_t {
	fitrnext next;
	knh_BasicBlock_t *bb;
	knh_Array_t *pred;
	int pos;
	int end;
} bbitr_t;

typedef struct cfg_data_t {
	knh_BasicBlock_t* (*parent)(struct cfg_data_t *cfg, knh_BasicBlock_t *bb);
	fitrinit init;
	fitrnext next;
	knh_int_t (*dfn)(struct cfg_data_t *cfg, knh_BasicBlock_t *bb);
	knh_Array_t *vertex;
	knh_Array_t *dfnum;
	knh_Array_t *pred;
} cfg_data_t;

typedef struct renamestack {
	knh_Array_t *stack;
	knh_Array_t *origidx;
} renamestack;

/* ------------------------------------------------------------------------ */
/* [API] */

/**
 * Add an int value to int array.
 * @param ctx Context
 * @param a   Target Array<int>
 * @param v   Int value
 */
static void iArray_add(Ctx *ctx, knh_Array_t *a, knh_int_t v)
{
	BEGIN_LOCAL(ctx, lsfp, 1);
	lsfp[0].ivalue = v;
	a->api->add(ctx, a, lsfp);
	END_LOCAL(ctx, lsfp);
}

/**
 * Pop the stack.
 * @param ctx   Context
 * @param a     Targe Array<int>
 * @return      A top number of the stack.
 */
static knh_int_t iArray_pop_ret(Ctx *ctx, knh_Array_t *a)
{
	knh_int_t v = TOP(a);
	iArray_pop(ctx, a);
	return v;
}

/**
 * Check an array contains an int object.
 * @param ctx Cotnext
 * @param a   A list of int object
 * @param n   Target int object
 * @return    1 if a contains n
 *            or 0 if a does not contain n.
 */
static knh_bool_t iArray_isContain(knh_Array_t *a, knh_int_t n)
{
	size_t i;
	for (i = 0; i < knh_Array_size(a); i++) {
		if (iArray_n(a, i) == n)
			return 1;
	}
	return 0;
}

static void iArray_insert(Ctx *ctx, knh_Array_t *a, knh_int_t n)
{
	if (!iArray_isContain(a, n))
		iArray_add(ctx, a, n);
}

static void iArray_copy(Ctx *ctx, knh_Array_t *to, knh_Array_t *from)
{
	DBG_ASSERT(knh_Array_size(to) == 0);
	size_t esize = knh_Array_size(from) * sizeof(knh_int_t);
	knh_cwb_t cwbbuf, *cwb = knh_cwb_open(ctx, &cwbbuf);
	knh_Bytes_write(ctx, cwb->ba, new_bytes2((char *)from->ilist, esize));
	knh_memcpy(to->ilist, knh_cwb_tochar(ctx, cwb), knh_cwb_size(cwb));
	knh_Array_size(to) = knh_Array_size(from);
	knh_cwb_close(cwb);
}

static void iArray_n_remove(Ctx *ctx, knh_Array_t *a, size_t i)
{
	SWAP(&iArray_n(a, i), &TOP(a), sizeof(knh_int_t));
	iArray_pop(ctx, a);
}

#ifdef SSA_CONVOUT_INTERFERENCE

static void iArray_remove(Ctx *ctx, knh_Array_t *a, knh_int_t n)
{
	size_t i;
	for (i = 0; i < knh_Array_size(a); i++) {
		if (iArray_n(a, i) == n) {
			iArray_n_remove(ctx, a, i);
			return;
		}
	}
	DBG_ABORT();
}

#endif /* SSA_CONVOUT_INTERFERENCE */

static knh_bool_t phi_isMod(knh_opline_t *op, int n)
{
	DBG_ASSERT(op_eq(op, PHI) && n >= 0 && n <= 2);
	if (n == 1)
		return (op->data[3] & MOD_LEFT) == MOD_LEFT;
	else if (n == 2)
		return (op->data[3] & MOD_RIGHT) == MOD_RIGHT;
	else
		return 0;
}

/* ------------------------------------------------------------------------ */
/* [debug] */

#ifdef K_USING_DEBUG

/*
static void printUseDef()
{
	int i, j;
	for (i = 0; i < (int)OPCODE_MAX; i++) {
		fprintf(stderr, "%s(%d) ", knh_opcode_tochar(i), (int)knh_opcode_size(i));
		for (j = 0; j < (int)knh_opcode_size(i); j++) {
			fprintf(stderr, "%d ", (int)knh_opcode_usedef(i, j));
		}
		fprintf(stderr, "\n");
	}
}
*/

/**
 * Print BasicBlock.
 * @param ctx Context
 * @param bb  BasicBlock
 */
static void printBB(Ctx *ctx, knh_BasicBlock_t *bb)
{
	size_t i;
	knh_opline_t *op;
	fprintf(stderr, " [%02d] incoming=%02d, visited=%d", DP(bb)->id, DP(bb)->incoming, knh_BasicBlock_isVisited(bb));
	fprintf(stderr, ", next=%02d", (bb->nextNC) ? (int)ID(bb->nextNC) : (-1));
	fprintf(stderr, ", jump=%02d\n", (bb->jumpNC) ? (int)ID(bb->jumpNC) : (-1));
	//fprintf(stderr, ", code=%p\n", DP(bb)->code);
	//fprintf(stderr, "| size=%02d, capa=%02d |\n",
	//DP(bb)->size, DP(bb)->capacity);
	bblock_each(bb, i, op) {
		if (KNH_STDERR != NULL)
			knh_opcode_dump(ctx, op, KNH_STDERR, op);
		else
			DBG_P("STDERR=NULL");
	}
}

/**
 * Print immediately dominator(idom) and others.
 * @param ctx      Context
 * @param vertex   Linear list of BasicBlocks(~= listNC)
 * @param samedom  It is used if a block has same dominator
 *                 as other block.
 * @param ancestor Ancestor block in Control Flow Graph
 * @param semi     Semidominator of the block
 * @param idom     Immediately dominator of the block
 */
//static void printIdom(knh_Array_t *vertex, knh_Array_t *samedom, knh_Array_t *ancestor, knh_Array_t *semi, knh_Array_t *idom)
//{
//	size_t i;
//	for (i = 0; i < knh_Array_size(vertex); i++) {
//		fprintf(stderr, "[%02lu] samedom=%02d", i, (int)iArray_n(samedom, i));
//		fprintf(stderr, ", ancestor=%02d", (int)iArray_n(ancestor, i));
//		fprintf(stderr, ", semi=%02d", (int)iArray_n(semi, i));
//		fprintf(stderr, ", idom=%02d\n", (int)iArray_n(idom, i));
//	}
//}

static void printTree(Ctx *ctx, knh_Array_t *vertex)
{
	size_t i;
	for (i = 0; i < knh_Array_size(vertex); i++) {
		printBB(ctx, Array(BasicBlock, vertex, i));
	}
}

static void printArray_Array(knh_Array_t *aa)
{
	size_t i, j;
	knh_Array_t *ai;
	Array_each_next(aa, i, ai, Array__(aa, i)) {
		fprintf(stderr, "[%02lu] {", i);
		for (j = 0; j < knh_Array_size(ai); j++) {
			fprintf(stderr, "%02ld, ", (long)iArray_n(ai, j));
		}
		fprintf(stderr, "}\n");
	}
}

static void printArray_Int(knh_Array_t *ai)
{
	size_t i;
	fprintf(stderr, "{");
	for (i = 0; i < knh_Array_size(ai); i++) {
		fprintf(stderr, "%02ld, ", (long)iArray_n(ai, i));
		if (i % 10 == 9)
			fprintf(stderr, "\n");
	}
	fprintf(stderr, "}\n");
}

static void printArray_ArrayOpline(knh_Array_t *aa)
{
	size_t i, j;
	knh_Array_t *ai;
	Array_each_next(aa, i, ai, Array__(aa, i)) {
		if (knh_Array_size(ai) > 0) {
			fprintf(stderr, "{%02lu:", i);
			knh_int_t v;
			Array_each(ai, j, v) {
				const char *s;
				if (v != 0)
					s = knh_opcode_tochar(((knh_opline_t*)v)->opcode);
				else
					s = "-1";
				fprintf(stderr, "%s, ", s);
			}
			fprintf(stderr, "}\n");
		}
	}
}

static void printArray_ArrayBB(knh_Array_t *aa)
{
	size_t i, j;
	knh_Array_t *ai;
	Array_each_next(aa, i, ai, Array__(aa, i)) {
		fprintf(stderr, "[%02lu] {", i);
		knh_BasicBlock_t *bb;
		Array_each_cast(ai, j, bb, BasicBlock) {
			fprintf(stderr, "%02d, ", DP(bb)->id);
		}
		fprintf(stderr, "}\n");
	}
}

static void printArray_Opline(knh_Array_t *ai)
{
	size_t i, count = 0;
	knh_int_t v;
	fprintf(stderr, "{");
	Array_each(ai, i, v) {
		if (v != 0) {
			fprintf(stderr, "%d:%s, ", (int)i, knh_opcode_tochar(((knh_opline_t*)v)->opcode));
			count++;
			if (count % 10 == 9)
				fprintf(stderr, "\n");
		}
	}
	fprintf(stderr, "}\n");
}

static void printArray_BB(knh_Array_t *a)
{
	size_t i;
	knh_BasicBlock_t *bb;
	fprintf(stderr, "{");
	Array_each_cast(a, i, bb, BasicBlock) {
		if (bb == 0) {
			fprintf(stderr, "-1, ");
			continue;
		}
		fprintf(stderr, "%02d, ", DP(bb)->id);
		if (i % 10 == 9)
			fprintf(stderr, "\n");
	}
	fprintf(stderr, "}\n");
}

static knh_bool_t checkVarStack(renamestack *rs)
{
	size_t i;
	knh_int_t v;
	Array_each(rs->origidx, i, v) {
		if (v == -1)
			continue;
		if (!iArray_isContain(STACK(i), v))
			return 0;
	}
	return 1;
}

static size_t countVars(knh_Array_t *stack) {
	size_t i, count = 0;
	for (i = 0; i < knh_Array_size(stack); i++) {
		count += knh_Array_size(Array__(stack, i));
	}
	return count;
}

static void printLiveness(Ctx *ctx, ssa_data_t *ssa, knh_Array_t *live)
{
	size_t i, j;
	int index = 0;
	knh_BasicBlock_t *n;
	knh_opline_t *s;
	Array_each_cast(ssa->vertex, i, n, BasicBlock) {
		bblock_each(n, j, s) {
			if (LIVE(s, index) == 1) {
				fprintf(stderr, "[%02d] LIVE ", ID(n));
				knh_opcode_dump(ctx, s, ctx->err, s);
//				DBG_P("[%02d] LIVE %s", ID(n), knh_opcode_tochar(s->opcode));
			} else {
//				DBG_P("[%02d] DEAD %s", ID(n), knh_opcode_tochar(s->opcode));
				fprintf(stderr, "[%02d] DEAD ", ID(n));
				knh_opcode_dump(ctx, s, ctx->err, s);
			}
			index++;
		}
	}
}

#endif /* K_USING_DEBUG */

/* ------------------------------------------------------------------------ */
/* [convert to SSA form] */

static void addPostBody(Ctx *ctx, knh_BasicBlock_t *bb, knh_BasicBlock_t *bbN)
{
	knh_BasicBlock_t *bbNEW = new_BasicBlockLABEL(ctx);
	knh_BasicBlock_t *bbS;
	int i;
	for (i = knh_Array_size(bb->listNC) - 1; i >= 0; i--) {
		bbS = Array(BasicBlock, bb->listNC, i);
		if (bbS->jumpNC == bbN && knh_BasicBlock_isVisited(bbS) && bbS != bb) {
			bbS->jumpNC = bbNEW;
			if (bb->jumpNC == bbN)
				break;
		}
	}
	if (bb->jumpNC == bbN)
		bb->jumpNC = bbNEW;
	bbNEW->jumpNC = bbN;
	DP(bbNEW)->incoming = 2;
	knh_BasicBlock_setVisited(bbNEW, 1);
}

/**
 * Do depth-first search(DFS) and number each block with
 * depth-first number(dfnum).
 * @param ctx    Context
 * @param p      Parent block
 * @param n      Target block
 * @param vertex Linear list of BasicBlocks(~= listNC)
 * @param pred   Predecessor of the block
 */
static void depthFirstSearch(Ctx *ctx, ssa_data_t *ssa, knh_BasicBlock_t *bbp, knh_BasicBlock_t *bbn)
{
	knh_BasicBlock_t *bbN, *bbJ;
	knh_BasicBlock_setVisited(bbn, 0);
	DP(bbn)->id = knh_Array_size(ssa->vertex);
	iArray_add(ctx, ssa->vertex, (knh_int_t)bbn);
	bbN = bbn->nextNC;
	bbJ = bbn->jumpNC;
	if (bbp != NULL)
		iArray_add(ctx, PRED(bbn), (knh_int_t)bbp);
	if (bbN != NULL) {
		if (knh_BasicBlock_isVisited(bbN))
			depthFirstSearch(ctx, ssa, bbn, bbN);
		else
			iArray_add(ctx, PRED(bbN), (knh_int_t)bbn);
	}
	if (bbJ != NULL) {
		if (knh_BasicBlock_isVisited(bbJ))
			depthFirstSearch(ctx, ssa, bbn, bbJ);
		else
			iArray_add(ctx, PRED(bbJ), (knh_int_t)bbn);
	}
}

/**
 * Get ancestor block with lowest Semidominator.
 * @param ctx      Context
 * @param vertex   Linear list of BasicBlocks(~= listNC)
 * @param v        Target block
 * @param ancestor Ancestor block in Control Flow Graph
 * @param best     A node with lowest dfnum from ancestor to v
 * @param semi     Semidominator of the block
 * @return         The ancestor block with lowest semidominator
 */
static knh_BasicBlock_t* getAncestorWLS(Ctx *ctx, knh_BasicBlock_t *bbv, knh_Array_t *ancestor, knh_Array_t *best, knh_Array_t *semi, cfg_data_t *cfg)
{
	knh_BasicBlock_t *bba = ANCESTOR(bbv);
//	DBG_P("v=%02d, a=%02d", ID(v), ID(a));
	knh_BasicBlock_t *bbb = NULL;
	if (iArray_n(ancestor, ID(bba)) != 0) {
		bbb = getAncestorWLS(ctx, bba, ancestor, best, semi, cfg);
//		DBG_P("ancestor[%02d]=%02d", ID(v), ID(a));
		iArray_n(ancestor, ID(bbv)) = iArray_n(ancestor, ID(bba));
		if (DFN(SEMI(bbb)) <
				DFN(SEMI(BEST(bbv)))) {
//			DBG_P("best[%02d]=%02d", DFN(cfg, v), DFN(cfg, b));
			iArray_n(best, ID(bbv)) = (knh_int_t)bbb;
		}
	}
//	DBG_P("[%d]->ancestor=[%d]", DP(v)->id, DP(VERTEX_(cfg, BEST(v)))->id);
	return BEST(bbv);
}

static knh_BasicBlock_t *getReverseParent(cfg_data_t *cfg, knh_BasicBlock_t *bb)
{
	knh_BasicBlock_t *bbN = bb->nextNC;
	knh_BasicBlock_t *bbJ = bb->jumpNC;
	if (bbN != NULL) {
		if (bbJ != NULL) {
			if (DFN(bbN) > DFN(bbJ))
				return bbJ;
			else
				return bbN;
		} else {
			return bbN;
		}
	}
	if (bbJ != NULL)
		return bbJ;
	return NULL;
}

static knh_BasicBlock_t *getParent(cfg_data_t *cfg, knh_BasicBlock_t *bb)
{
	if (knh_Array_size(Array__(cfg->pred, ID(bb))) > 0)
		return Array(BasicBlock, Array__(cfg->pred, ID(bb)), 0);
	else
		return NULL;
}

static bbitr_t *bbP_init(bbitr_t *itr, fitrnext next, knh_Array_t *pred, knh_BasicBlock_t *bb)
{
	itr->next = next;
	itr->bb = bb;
	itr->pos = 0;
	itr->end = knh_Array_size(Array__(pred, ID(bb)));
	itr->pred = pred;
	return itr;
}

static knh_BasicBlock_t *bbP_next(bbitr_t *itr)
{
	if ((knh_int_t)knh_Array_size(itr->pred) > itr->pos)
		return Array(BasicBlock, Array__(itr->pred, ID(itr->bb)), itr->pos);
	else
		return NULL;
}

/**
 * Set immediately dominator(IDOM) to each BasicBlock.
 * @param ctx    Context
 * @param vertex Linear list of BasicBlocks(~= listNC)
 * @param idom   Immediately dominator of the block
 * @param pred   Predecessor of the block
 */
static void setIdom(Ctx *ctx, knh_Array_t *idom, cfg_data_t *cfg)
{
	size_t i, j;
	size_t size = knh_Array_size(cfg->vertex);

	BEGIN_LOCAL(ctx, lsfp, 5);
	LOCAL_NEW(ctx, lsfp, 0, knh_Array_t*, bucket, new_Array(ctx, CLASS_Array, size));
	LOCAL_NEW(ctx, lsfp, 1, knh_Array_t*, semi, new_Array(ctx, CLASS_Int, size));
	LOCAL_NEW(ctx, lsfp, 2, knh_Array_t*, ancestor, new_Array(ctx, CLASS_Int, size));
	LOCAL_NEW(ctx, lsfp, 3, knh_Array_t*, samedom, new_Array(ctx, CLASS_Int, size));
	LOCAL_NEW(ctx, lsfp, 4, knh_Array_t*, best, new_Array(ctx, CLASS_Int, size));
	knh_BasicBlock_t *bbn, *bbp, *bbs, *bb_s, *bbv, *bby;

	for (i = 0; i < size; i++) {
		knh_Array_add(ctx, bucket, new_Array(ctx, CLASS_Int, 0));
		iArray_add(ctx, semi, 0);
		iArray_add(ctx, ancestor, 0);
		iArray_add(ctx, samedom, 0);
		iArray_add(ctx, best, 0);
	}

	for (i = size - 1; i > 0; i--) {
		bbn = Array(BasicBlock, cfg->vertex, i);
		bbp = PARENT(cfg, bbn);
//		DBG_P("bbn=%d,bbp=%d", ID(bbn), ID(bbp));
		bbs = bbp;
		bbitr_t itrbuf, *itr = cfg->init(&itrbuf, cfg->next, cfg->pred, bbn);
		itr_foreach(bbv, itr) {
			if (DFN(bbv) <= DFN(bbn)) {
				bb_s = bbv;
			} else {
				bb_s = SEMI(getAncestorWLS(ctx, bbv, ancestor, best, semi, cfg));
			}
			if (DFN(bb_s) < DFN(bbs)) {
//				DBG_P("bbs[%d] = bb_s[%d]", DFN(cfg, bbs), DFN(cfg, bb_s));
				bbs = bb_s;
			}
		}
//		DBG_P("semi[%02d]=%02d", DFN(cfg, bbn), DFN(cfg, bbs));
		iArray_n(semi, ID(bbn)) = (knh_int_t)bbs;
//		DBG_P("bucket[%02d]<<%02d", DFN(cfg, bbs), DFN(cfg, bbn));
		iArray_insert(ctx, BUCKET(bbs), (knh_int_t)bbn);
		LINK(bbp, bbn);
		for (j = 0; j < knh_Array_size(BUCKET(bbp)); j++) {
			bbv = Array(BasicBlock, BUCKET(bbp), j);
			bby = getAncestorWLS(ctx, bbv, ancestor, best, semi, cfg);
			if (SEMI(bby) == SEMI(bbv)) {
				iArray_n(idom, ID(bbv)) = (knh_int_t)bbp;
			} else {
				iArray_n(samedom, ID(bbv)) = (knh_int_t)bby;
			}
		}
		knh_Array_clear(ctx, BUCKET(bbp), 0);
	}
	for (i = 1; i < size; i++) {
		bbn = Array(BasicBlock, cfg->vertex, i);
		if (iArray_n(samedom, ID(bbn)) != 0)
			iArray_n(idom, ID(bbn)) = iArray_n(idom, ID(SAMEDOM(bbn)));
	}
#ifdef K_USING_DEBUG
	//printIdom(ssa->vertex, samedom, ancestor, semi, ssa->idom);
#endif
	END_LOCAL(ctx, lsfp);
}

/**
 * Check a block is dominated by another block.
 * @param ctx  Context
 * @param n    Target block
 * @param w    A block that might be dominate the target block
 * @param idom Immediately dominator of the block
 * @return     1 if n is dominated by w
 *             or 0 if n is not dominated by w
 */
static knh_bool_t isDominated(knh_Array_t *idom, knh_BasicBlock_t *n, knh_BasicBlock_t *w)
{
	knh_int_t tmp;
	for (tmp = iArray_n(idom, ID(w)); tmp != 0; tmp = iArray_n(idom, ID((knh_BasicBlock_t*)tmp))) {
		if (tmp == (knh_int_t)n)
			return 1;
	}
	return 0;
}

/**
 * Compute Dominance Frontier(DF).
 * @param ctx    Context
 * @param vertex Linear list of BasicBlocks(~= listNC)
 * @param n      Target block
 * @param df     Dominance Frontier of the block
 * @param idom   Immediately dominator of the block
 */

static knh_BasicBlock_t *bbnext(bbitr_t *itr)
{
	if (itr->pos == 0) {
		if (itr_bb(itr)->nextNC == NULL)
			return itr_bb(itr)->jumpNC;
		return itr_bb(itr)->nextNC;
	} else if (itr->pos == 1) {
		return itr_bb(itr)->jumpNC;
	} else {
		return NULL;
	}
}

static bbitr_t *bbinit(bbitr_t *itr, fitrnext next, knh_Array_t *pred, knh_BasicBlock_t *bb)
{
	itr->next = next;
	itr->bb = bb;
	itr->pos = itr->end = 0;
	itr->pred = pred;
	if (bb->nextNC != NULL) itr->end++;
	if (bb->jumpNC != NULL) itr->end++;
	return itr;
}

static knh_int_t getDFN(cfg_data_t *cfg, knh_BasicBlock_t *bb)
{
	return (knh_int_t)ID(bb);
	UNUSED(cfg);
}

static knh_int_t getReverseDFN(cfg_data_t *cfg, knh_BasicBlock_t *bb)
{
	return iArray_n(cfg->dfnum, ID(bb));
}

static void setDF(Ctx *ctx, knh_Array_t *df, knh_Array_t *idom, knh_BasicBlock_t *bbn, cfg_data_t *cfg)
{
	size_t i, j;
	knh_BasicBlock_t *bbw, *bbc;
	BEGIN_LOCAL(ctx, lsfp, 1);
	LOCAL_NEW(ctx, lsfp, 0, knh_Array_t*, S, new_Array(ctx, CLASS_Int, 0));
	knh_BasicBlock_t *bby;
	bbitr_t itrbuf, *itr = cfg->init(&itrbuf, cfg->next, cfg->pred, bbn);
	itr_foreach(bby, itr) {
//		DBG_P("[%02d]->succ[%02d]=[%02d]", ID(bbn), itr->pos, ID(bby));
		if (IDOM(bby) != (knh_int_t)bbn)
			iArray_insert(ctx, S, (knh_int_t)bby);
	}
	for (i = 0; i < knh_Array_size(cfg->vertex); i++) {
		if (IDOM(Array(BasicBlock, cfg->vertex, i)) == (knh_int_t)bbn) {
			bbc = Array(BasicBlock, cfg->vertex, i);
			setDF(ctx, df, idom, bbc, cfg);
			for (j = 0; j < knh_Array_size(Array__(df, ID(bbc))); j++) {
				bbw = Array(BasicBlock, Array__(df, ID(bbc)), j);
				if (!isDominated(idom, bbn, bbw))
					iArray_insert(ctx, S, (knh_int_t)bbw);
			}
		}
	}
	iArray_copy(ctx, Array__(df, ID(bbn)), S);
	END_LOCAL(ctx, lsfp);
}

/**
 * Insert phi function.
 * @param ctx  Context
 * @param n    Target block
 * @param idx  An index of register
 * @param argc A number of Predecessor
 */
static void insertOpline(Ctx *ctx, knh_BasicBlock_t *n, knh_opline_t *op, knh_ushort_t line)
{
	knh_ushort_t size = DP(n)->size - line;
	if (size > 0) {
		knh_opline_t *buf;
		size_t bsize = size * sizeof(knh_opline_t);
		knh_cwb_t cwbbuf, *cwb = knh_cwb_open(ctx, &cwbbuf);
		knh_BasicBlock_add_(ctx, n, 0, op);
		buf = DP(n)->opbuf + line;
		knh_Bytes_write(ctx, cwb->ba, new_bytes2((char *)op, sizeof(knh_opline_t)));
		knh_Bytes_write(ctx, cwb->ba, new_bytes2((char *)buf, bsize));
		knh_memcpy(buf, knh_cwb_tochar(ctx, cwb), knh_cwb_size(cwb));
		knh_cwb_close(cwb);
	} else {
		knh_BasicBlock_add_(ctx, n, 0, op);
	}
}

/**
 * Set a list of variable that defined each block n.
 * @param ctx   Context
 * @param Aorig A list of variable that defined each block
 * @param n     Target block
 */
static void setAorig(Ctx *ctx, knh_Array_t *Aorig, knh_BasicBlock_t *n)
{
#define INSERT(T, v0, ...) {\
		DBG_P("Aorig[%02d] add r%d %s", ID(n), v0, knh_opcode_tochar(OPCODE_##T));\
		INSERT_OP(ctx, n, i, T, v0, ## __VA_ARGS__);\
		iArray_add(ctx, Aorig, v0);\
		op = DP(n)->opbuf + i;\
	}

#define REPLACE(T, v0, v1)\
	op->opcode = OPCODE_##T;\
	op->data[0] = v0;\
	op->data[1] = v1

	knh_ushort_t i;
	knh_opline_t *op;
	bblock_each(n, i, op) {
		if (isDEF(op)) {
			switch (op->opcode) {
			case OPCODE_iINC:
				REPLACE(iADDn, op->data[0], 1);
				break;
			case OPCODE_iDEC:
				REPLACE(iSUBn, op->data[1], 1);
				break;
			case OPCODE_ONMOV:
				INSERT(OMOV, op->data[0], op->data[1]);
				REPLACE(NMOV, op->data[2], op->data[3]);
				break;
			case OPCODE_OOMOV:
				INSERT(OMOV, op->data[0], op->data[1]);
				REPLACE(OMOV, op->data[2], op->data[3]);
				break;
			case OPCODE_NNMOV:
				INSERT(NMOV, op->data[0], op->data[1]);
				REPLACE(NMOV, op->data[2], op->data[3]);
				break;
			case OPCODE_NSET2:
				INSERT(NSET, op->data[0] + 0 * K_NEXTIDX, op->data[1]);
				REPLACE(NSET, op->data[0] + 1 * K_NEXTIDX, op->data[2]);
				break;
			case OPCODE_NSET3:
				INSERT(NSET, op->data[0] + 0 * K_NEXTIDX, op->data[1]);
				INSERT(NSET, op->data[0] + 1 * K_NEXTIDX, op->data[2]);
				REPLACE(NSET, op->data[0] + 2 * K_NEXTIDX, op->data[3]);
				break;
			case OPCODE_NSET4:
				INSERT(NSET, op->data[0] + 0 * K_NEXTIDX, op->data[1]);
				INSERT(NSET, op->data[0] + 1 * K_NEXTIDX, op->data[2]);
				INSERT(NSET, op->data[0] + 2 * K_NEXTIDX, op->data[3]);
				REPLACE(NSET, op->data[0] + 3 * K_NEXTIDX, op->data[4]);
				break;
			case OPCODE_OSET2:
				INSERT(OSET, op->data[0], op->p[1]);
				knh_Object_RCdec((knh_Object_t*)op->p[1]);
				REPLACE(OSET, op->data[0] + 1 * K_NEXTIDX, op->data[2]);
				break;
			case OPCODE_OSET3:
				INSERT(OSET, op->data[0] + 0 * K_NEXTIDX, op->p[1]);
				INSERT(OSET, op->data[0] + 1 * K_NEXTIDX, op->p[2]);
				knh_Object_RCdec((knh_Object_t*)op->p[1]);
				knh_Object_RCdec((knh_Object_t*)op->p[2]);
				REPLACE(OSET, op->data[0] + 2 * K_NEXTIDX, op->data[3]);
				break;
			case OPCODE_OSET4:
				INSERT(OSET, op->data[0] + 0 * K_NEXTIDX, op->p[1]);
				INSERT(OSET, op->data[0] + 1 * K_NEXTIDX, op->p[2]);
				INSERT(OSET, op->data[0] + 2 * K_NEXTIDX, op->p[3]);
				knh_Object_RCdec((knh_Object_t*)op->p[1]);
				knh_Object_RCdec((knh_Object_t*)op->p[2]);
				knh_Object_RCdec((knh_Object_t*)op->p[3]);
				REPLACE(OSET, op->data[0] + 3 * K_NEXTIDX, op->data[4]);
				break;
			}
			if ((int)op->data[0] >= 0) {
//				DBG_P("Aorig[%02d] add r%d %s", ID(n), op->data[0], knh_opcode_tochar(op->opcode));
				iArray_add(ctx, Aorig, op->data[0]);
			}
		}
	}
}

#ifdef SSA_CONV_SEMI

static void setNonlocals(Ctx *ctx, knh_BasicBlock_t *bb, knh_Array_t *nonlocals)
{
	size_t i, j;
	knh_opline_t *op;
	BEGIN_LOCAL(ctx, lsfp, 1);
	LOCAL_NEW(ctx, lsfp, 0, knh_Array_t*, killed, new_Array(ctx, CLASS_Int, 0));
	bblock_each(bb, i, op) {
		if (isCALL(op)) {
			knh_intptr_t min, max;
			if (op_eq(op, FASTCALL0)) {
				min = op->data[1];
				max = op->data[1] + 2 * (knh_Method_psize((knh_Method_t*)op->p[4]) + 1);
			} else {
				min = op->data[1];
				max = op->data[2];
			}
			for (j = (size_t)min; j < (size_t)max; j++) {
				if (!iArray_isContain(killed, j)) {
//					DBG_P("[%02d] set nonlocals %d", DFN(bb), j);
					iArray_insert(ctx, nonlocals, j);
				}
			}
			if (op->data[0] >= 0) {
//				DBG_P("[%02d] set killed %d", DFN(bb), op->data[0]);
				iArray_insert(ctx, killed, op->data[0]);
			}
		} else if (op_eq(op, P)) {
			if (op->data[4] > 0) {
				knh_opline_t *opP = op - 1;
				if (!iArray_isContain(killed, opP->data[0])) {
//					DBG_P("[%02d] set nonlocals %d", DFN(bb), opP->data[0]);
					iArray_insert(ctx, nonlocals, opP->data[0]);
				}
			}
		} else if (isCAST(op)) {
			if (op->data[1] >= 0 && !iArray_isContain(killed, op->data[1])) {
//				DBG_P("[%02d] set nonlocals %d", DFN(bb), op->data[1]);
				iArray_insert(ctx, nonlocals, op->data[1]);
			}
			if (op->data[0] >= 0) {
//				DBG_P("[%02d] set killed %d", DFN(bb), op->data[0]);
				iArray_insert(ctx, killed, op->data[0]);
			}
//		} else if (op_eq(op, LOADMTD)) {
//			if (op->data[0] >= 0) {
////				DBG_P("[%02d] set nonlocals %d", DFN(bb), op->data[0]);
//				iArray_insert(ctx, nonlocals, op->data[0]);
//			}
		} else {
			for (j = 1; j < knh_opcode_size(op->opcode); j++) {
				if (isUSE(op, j) && !iArray_isContain(killed, op->data[j])) {
//					DBG_P("[%02d] set nonlocals %d", DFN(bb), op->data[j]);
					iArray_insert(ctx, nonlocals, op->data[j]);
				}
				else if (isVUSE(op, j) && !iArray_isContain(killed, op->data[j])) {
//					DBG_P("[%02d] set nonlocals %d", DFN(bb), op->data[j]);
					iArray_insert(ctx, nonlocals, op->data[j]);
				}
			}
			if (isVDEF(op, 0)) {
//				DBG_P("[%02d] set nonlocals %d", DFN(bb), op->data[0]);
				iArray_insert(ctx, nonlocals, op->data[0]);
			}
			if (op->data[0] >= 0 && isDEF(op)) {
//				DBG_P("[%02d] set killed %d", DFN(bb), op->data[0]);
				iArray_insert(ctx, killed, op->data[0]);
			}
		}
	}
	END_LOCAL(ctx, lsfp);
}

#endif /* SSA_CONV_SEMI */

/**
 * Put phi function to BasicBlock.
 * @param ctx    Context
 * @param vertex Linear list of BasicBlocks(~= listNC)
 * @param df     Dominance Frontier of the block
 * @param max    Max index of register
 */
static int putPhi(Ctx *ctx, ssa_data_t *ssa)
{
	knh_int_t a, max, min;
	size_t i, j;
	knh_BasicBlock_t *bbn, *bby;
#if defined(SSA_CONV_MINIMAL)
	BEGIN_LOCAL(ctx, lsfp, 4);
#elif defined(SSA_CONV_SEMI)
	BEGIN_LOCAL(ctx, lsfp, 5);
#endif
	LOCAL_NEW(ctx, lsfp, 0, knh_Array_t*, Aorig, new_Array(ctx, CLASS_Array, knh_Array_size(ssa->vertex)));
	LOCAL_NEW(ctx, lsfp, 1, knh_Array_t*, Aphi, new_Array(ctx, CLASS_Array, 0));
	LOCAL_NEW(ctx, lsfp, 2, knh_Array_t*, defsites, new_Array(ctx, CLASS_Array, 0));
	LOCAL_NEW(ctx, lsfp, 3, knh_Array_t*, W, new_Array(ctx, CLASS_Int, 0));
#if defined(SSA_CONV_SEMI)
	LOCAL_NEW(ctx, lsfp, 4, knh_Array_t*, nonlocals, new_Array(ctx, CLASS_Int, 0));
#endif
	Array_each_cast(ssa->vertex, i, bbn, BasicBlock) {
		knh_Array_add(ctx, Aorig, new_Array(ctx, CLASS_Int, 0));
		if (DP(bbn)->size > 0) {
			setAorig(ctx, AORIG(i), bbn);
			Array_each(AORIG(i), j, a) {
				while ((knh_int_t)knh_Array_size(defsites) <= a) {
					knh_Array_add(ctx, defsites, new_Array(ctx, CLASS_Int, 0));
				}
				iArray_insert(ctx, DEFSITES(a), ID(bbn));
			}
		}
#ifdef SSA_CONV_SEMI
		setNonlocals(ctx, bbn, nonlocals);
#endif
	}
//#if defined(SSA_CONV_SEMI)
//	DBG_A("nonlocals", printArray_Int(nonlocals));
//#endif
	max = (knh_int_t)knh_Array_size(defsites);
	// set ssa->stacktop
	min = max;
	knh_Array_t *ao;
	Array_each_next(Aorig, i, ao, Array__(Aorig, i)) {
		if (knh_Array_size(ao) > 0 && iArray_n(ao, 0) < min) {
			min = iArray_n(ao, 0);
		}
	}
	if (!isOBJ(min))
		min -= 1;
	DBG_P("set stacktop %d", min);
	ssa->stacktop = min;
	for (a = 0; a < max; a++) {
		knh_Array_add(ctx, Aphi, new_Array(ctx, CLASS_Int, 0));
		if (knh_Array_size(DEFSITES(a)) == 0)
			continue;
		iArray_copy(ctx, W, DEFSITES(a));
		while (knh_Array_size(W) > 0) {
			bbn = VERTEX(iArray_pop_ret(ctx, W));
			Array_each_cast(DF(bbn), i, bby, BasicBlock) {
#if defined(SSA_CONV_MINIMAL)
				if (!iArray_isContain(APHI(a), ID(bby)) && (DP(bby)->size == 0 || op_neq(DP(bby)->opbuf, RET))) {
#elif defined(SSA_CONV_SEMI)
				if (!iArray_isContain(APHI(a), ID(bby)) && (DP(bby)->size == 0 || op_neq(DP(bby)->opbuf, RET)) && iArray_isContain(nonlocals, a)) {
#endif
						DBG_ASSERT(DP(bby)->incoming == 2);
						DBG_P("[%02d] insert PHI of r%d", ID(bby), a);
						klr_PHI_t phi = {TADDR, OPCODE_PHI, 0, a, a, a, 0};
						insertOpline(ctx, bby, (knh_opline_t*)(&phi), 0);
						iArray_insert(ctx, APHI(a), ID(bby));
						if (!iArray_isContain(AORIG(ID(bby)), a))
							iArray_insert(ctx, W, ID(bby));
				} else {
					DBG_P("[%02d] Don't insert phi of r%d", ID(bby), a);
				}
			}
		}
	}
	END_LOCAL(ctx, lsfp);
	return max;
}

/**
 * Get Predecessor number
 * @param pred Predecessor of the block
 * @param p    Parent block
 * @param n    Target block
 * @return     An index of p in pred.
 */
static int getPredNum(ssa_data_t *ssa, knh_BasicBlock_t *bbp, knh_BasicBlock_t *bbn)
{
	size_t i;
	for (i = 0; i < knh_Array_size(PRED(bbn)); i++) {
		if (bbp == Array(BasicBlock, PRED(bbn), i))
			return i;
	}
	DBG_ABORT();
	return -1;
}

static void copyArgs(Ctx *ctx, renamestack *rs, knh_BasicBlock_t *n, knh_opline_t *s, int *idx, knh_ushort_t *pos)
{
	knh_intptr_t i, shift, min, max, newName;
	if (op_eq(s, FASTCALL0)) {
		DBG_P("fastcall psize=%d", knh_Method_psize((knh_Method_t*)s->p[4]));
		min = s->data[1];
		max = s->data[1] + 2 * (knh_Method_psize((knh_Method_t*)s->p[4]) + 1);
		shift = *idx - (s->data[1] - 2);
	} else {
		min = s->data[1];
		max = s->data[2];
		shift = *idx - (s->data[1] - 6);
	}
	DBG_P("shift=%d", shift);
	for (i = min; i < max; i++) {
		if (iArray_n(rs->origidx, i) != -1 && ((knh_Array_size(STACK(i)) > 1 && (knh_intptr_t)knh_Array_size(rs->stack) > i) || (op_eq(s, FASTCALL0) && i == min))) {
			newName = i + shift;
			if (isOBJ(i)) {
				while ((knh_intptr_t)knh_Array_size(rs->origidx) < newName) {
					iArray_add(ctx, rs->origidx, -1);
				}
				iArray_add(ctx, rs->origidx, i);
				iArray_add(ctx, rs->origidx, -1);
				DBG_P("insert OMOV r%d r%d", newName, TOP(STACK(i)));
				INSERT_ARG(ctx, n, *pos, OMOV, newName, TOP(STACK(i)));
//				DBG_P("op=%s", knh_opcode_tochar(newop->opcode));
				PUSH(STACK(i), newName);
			} else {
				while ((knh_intptr_t)knh_Array_size(rs->origidx) <= newName) {
					iArray_add(ctx, rs->origidx, -1);
				}
				DBG_P("insert NMOV r%d r%d", newName, TOP(STACK(i)));
				INSERT_ARG(ctx, n, *pos, NMOV, newName, TOP(STACK(i)));
			}
			s = DP(n)->opbuf + *pos;
		} else {
			DBG_P("Don't insert MOV %s %d", knh_opcode_tochar(s->opcode), i);
		}
	}
	if (op_eq(s, CALL)) {
		newName = min - 1 + shift;
		knh_intptr_t oldName = iArray_n(STACK(min), knh_Array_size(STACK(min)) - 2) - 1;
		DBG_P("insert NMOV r%d r%d", newName, oldName);
		INSERT_ARG(ctx, n, *pos, NMOV, newName, oldName);
		s = DP(n)->opbuf + *pos;
	}
	*idx = knh_Array_size(rs->origidx);
	if (op_eq(s, FASTCALL0)) {
		DBG_P("shift USE %s %d to %d", knh_opcode_tochar(s->opcode), s->data[1], s->data[1] + shift);
		s->data[1] += shift;
//		iArray_add(ctx, USE(s->data[1]), (knh_int_t)s);
	} else {
		for (i = 1; i <= 2; i++) {
			DBG_P("shift USE %s %d to %d", knh_opcode_tochar(s->opcode), s->data[i], s->data[i] + shift);
			s->data[i] += shift;
//			iArray_add(ctx, USE(s->data[i]), (knh_int_t)s);
		}
	}
	if (s->data[0] >= 0) {
		if (op_eq(s, FASTCALL0)) {
			if (isOBJ(s->data[0]))
				shift = s->data[1] - 2 - s->data[0];
			else
				shift = s->data[1] - 1 - s->data[0];
		}
		newName = s->data[0] + shift;
		if ((knh_intptr_t)knh_Array_size(rs->origidx) <= newName) {
			if (isOBJ(s->data[0])) {
				iArray_add(ctx, rs->origidx, s->data[0]);
				iArray_add(ctx, rs->origidx, -1);
			} else {
				iArray_add(ctx, rs->origidx, -1);
				iArray_add(ctx, rs->origidx, s->data[0]);
			}
			*idx += 2;
			DBG_ASSERT(*idx == (int)knh_Array_size(rs->origidx));
		} else {
			iArray_n(rs->origidx, newName) = s->data[0];
		}
		PUSH(STACK(s->data[0]), newName);
		REDEF(s, s->data[0], newName);
	}
}

static void replaceUse(renamestack *rs, knh_opline_t *s)
{
	size_t i;
	knh_opline_t *op;
	if (op_eq(s, P)) {
		if (s->data[4] != 0) {
			op = s - 1;
			i = isOBJ(op->data[0]) ? op->data[0] / 2 : (op->data[0] - 1) / 2;
			// REUSE
			DBG_P("replace USE %s %d to %d", knh_opcode_tochar(op->opcode), s->data[4], i);\
			s->data[4] = i;
		} else {
			DBG_P("Don't replace USE %s %d", knh_opcode_tochar(s->opcode), s->data[4]);
		}
	} else if (/*op_eq(s, LOADMTD) ||*/op_eq(s, THROW) || op_eq(s, CHKSTACK)) {
//		DBG_P("replace VUSE %s %d to %d", knh_opcode_tochar(s->opcode), s->data[0], TOP(STACK(s->data[0])));
		REUSE(s, s->data[0], TOP(STACK(s->data[0])));
	} else {
		for (i = 1; i < knh_opcode_size(s->opcode); i++) {
			if (isUSE(s, i)) {
				REUSE(s, s->data[i], TOP(STACK(s->data[i])));
			} else if (isVUSE(s, i)) {
//				DBG_P("replace VUSE %s %d to %d", knh_opcode_tochar(s->opcode), s->data[i], TOP(STACK(s->data[i])));
				REUSE(s, s->data[i], TOP(STACK(s->data[i])));
			}
		}
		if (isVDEF(s, 0)) {
//			DBG_P("replace VDEF %s %d to %d", knh_opcode_tochar(s->opcode), s->data[0], TOP(STACK(s->data[0])));
			REUSE(s, s->data[0], TOP(STACK(s->data[0])));
		}
	}
}

static void replaceDef(Ctx *ctx, renamestack *rs, knh_opline_t *s, int *idx)
{
	if (s->data[0] < 0) {
//		DBG_P("Don't replace DEF %s %d", knh_opcode_tochar(s->opcode), s->data[0]);
		return;
	}
	int newName;
	if (isOBJ(s->data[0])) {
		iArray_add(ctx, rs->origidx, s->data[0]);
		iArray_add(ctx, rs->origidx, -1);
		newName = *idx;
	} else {
		iArray_add(ctx, rs->origidx, -1);
		iArray_add(ctx, rs->origidx, s->data[0]);
		newName = *idx + 1;
	}
	//DBG_P("s->data[0]=%d, esp=%d", s->data[0], ctx->esp->data);
	PUSH(STACK(s->data[0]), newName);
	REDEF(s, s->data[0], newName);
	*idx += 2;
}

static void replaceRix(knh_opline_t *s)
{
	if (op_eq(s, NEXT)) {
		knh_intptr_t newrix = isOBJ(s->data[1] - s->data[2]) ? (s->data[1] - s->data[2]) / 2 : (s->data[1] - s->data[2] - 1) / 2;
//		DBG_P("replace RIX %s %d to %d", knh_opcode_tochar(s->opcode), s->data[3], newrix);
		s->data[3] = newrix;
	} else {
		knh_intptr_t newrix = isOBJ(s->data[0] - s->data[1]) ? (s->data[0] - s->data[1]) / 2 : (s->data[0] - s->data[1] - 1) / 2;
//		DBG_P("replace RIX %s %d to %d", knh_opcode_tochar(s->opcode), s->data[2], newrix);
		s->data[2] = newrix;
	}
}

/**
 * Rename variable to make SSA form.
 * @param ctx    Cotnext
 * @param vertex Linear list of BasicBlocks(~= listNC)
 * @param n      Target block
 * @param stack  A stack that records rename in each variable
 * @param idx    An index of register
 * @param pred   Predecessor of the block
 * @param idom   Immediately dominator of the block
 */
static void renameVars(Ctx *ctx, ssa_data_t *ssa, renamestack *rs, knh_BasicBlock_t *n, int *idx)
{
#ifdef K_USING_DEBUG
	size_t count = countVars(rs->stack);
#endif
	knh_ushort_t i, j;
	knh_opline_t *s;
	knh_BasicBlock_t *x ,*y;
	bblock_each(n, i, s) {
		if (isCALL(s)) {
			copyArgs(ctx, rs, n, s, idx, &i);
			s = DP(n)->opbuf + i;
			if (op_eq(s, FASTCALL0))
				replaceRix(s);
		} else if (isCAST(s)) {
			knh_intptr_t newName;
			if (s->data[1] > 0 && (int)knh_Array_size(rs->stack) > s->data[1]) {
				RENAME_ASSERT(isOBJ(s->data[1]));
				newName = *idx;
				iArray_add(ctx, rs->origidx, s->data[1]);
				iArray_add(ctx, rs->origidx, -1);
//				DBG_P("insert OMOV r%d r%d", newName, TOP(STACK(s->data[1])));
				INSERT_ARG(ctx, n, i, OMOV, newName, TOP(STACK(s->data[1])));
				s = DP(n)->opbuf + i;
				PUSH(STACK(s->data[1]), newName);
				if (iArray_n(rs->origidx, s->data[1] + 1) != -1 && TOP(STACK(s->data[1] + 1)) != s->data[1] + 1) {
					newName = *idx + 1;
//					DBG_P("insert NMOV %d to %d", TOP(STACK(s->data[1] + 1)), newName);
					INSERT_ARG(ctx, n, i, NMOV, newName, TOP(STACK(s->data[1] + 1)));
					s = DP(n)->opbuf + i;
				}
				REUSE(s, s->data[1], TOP(STACK(s->data[1])));
				*idx += 2;
			} else {
//				DBG_P("Don't replace USE %s %d", knh_opcode_tochar(s->opcode), s->data[1]);
			}
			if (s->data[0] >= 0) {
				if (isOBJ(s->data[0])) {
					newName = *idx;
					iArray_add(ctx, rs->origidx, s->data[0]);
					if ((int)knh_Array_size(rs->stack) > s->data[0] + 1) {
						iArray_add(ctx, rs->origidx, s->data[0] + 1);
						PUSH(STACK(s->data[0] + 1), newName + 1);
					} else {
						iArray_add(ctx, rs->origidx, -1);
					}
				} else {
					iArray_add(ctx, rs->origidx, -1);
					iArray_add(ctx, rs->origidx, s->data[0]);
					newName = *idx + 1;
				}
				PUSH(STACK(s->data[0]), newName);
				REDEF(s, s->data[0], newName);
				*idx += 2;
//			} else {
//				DBG_P("Don't replace DEF %s %d", knh_opcode_tochar(s->opcode), s->data[0]);
			}
			s = DP(n)->opbuf + i;
			replaceRix(s);
		} else if (op_eq(s, NEXT)) {
			if (s->data[2] > 0 && (int)knh_Array_size(rs->stack) > s->data[2]) {
				REUSE(s, s->data[2], TOP(STACK(s->data[2])));
//			} else {
//				DBG_P("Don't replace USE %s %d", knh_opcode_tochar(s->opcode), s->data[2]);
			}
			knh_intptr_t newName;
			if (isOBJ(s->data[1])) {
				iArray_add(ctx, rs->origidx, s->data[1]);
				iArray_add(ctx, rs->origidx, -1);
				newName = *idx;
			} else {
				iArray_add(ctx, rs->origidx, -1);
				iArray_add(ctx, rs->origidx, s->data[1]);
				newName = *idx + 1;
			}
			PUSH(STACK(s->data[1]), newName);
			REDEF(s, s->data[1], newName);
			*idx += 2;
			replaceRix(s);
		} else {
			if (op_neq(s, PHI)) {
				replaceUse(rs, s);
			}
			if (op_neq(s, LOADMTD) && op_neq(s, THROW) && op_neq(s, CHKSTACK) && (isDEF(s) || op_eq(s, PHI)))
				replaceDef(ctx, rs, s, idx);
		}
	}
	bbitr_t itrbuf, *itr = bbinit(&itrbuf, bbnext, NULL, n);
	itr_foreach(y, itr) {
		j = getPredNum(ssa, n, y) + 1;
		bblock_each(y, i, s) {
			if (op_eq(s, PHI)) {
				REUSE(s, s->data[j], TOP(STACK(s->data[j])));
				//DBG_P("phi source r%d is live at end of [%02d]", s->data[j], DFN(n));
				//iArray_add(ctx, LIVEOUT(DFN(n)), (int)s->data[j]);
			} else {
				break;
			}
		}
	}
	Array_each_cast(ssa->vertex, i, x, BasicBlock) {
		if (iArray_n(ssa->idom, ID(x)) == (knh_int_t)n)
			renameVars(ctx, ssa, rs, x, idx);
	}
	for (i = DP(n)->size - 1; (int)i >= 0; i--) {
		s = DP(n)->opbuf + i;
		if (isDEF(s) || op_eq(s, PHI)) {
			if ((int)s->data[0] > 0 && knh_Array_size(rs->origidx) > (size_t)s->data[0] && (int)iArray_n(rs->origidx, s->data[0]) != -1) {
//				DBG_P("pop %02d", s->data[0]);
				if (isCAST(s) && (int)knh_Array_size(rs->origidx) > s->data[0] + 1 && iArray_n(rs->origidx, s->data[0] + 1) != -1) {
					RENAME_ASSERT(knh_Array_size(STACK(s->data[0] + 1)) > 1);
					iArray_pop(ctx, STACK(s->data[0] + 1));
					iArray_n(rs->origidx, s->data[0] + 1) = -1;
//				} else {
//					DBG_P("debug");
				}
				if (op_neq(s, THROW) && op_neq(s, CHKSTACK)) {
					RENAME_ASSERT(knh_Array_size(STACK(s->data[0])) > 1);
					iArray_pop(ctx, STACK(s->data[0]));
					iArray_n(rs->origidx, s->data[0]) = -1;
				}
//			} else {
//				DBG_P("don't pop %d", s->data[0]);
			}
			if (isCALL(s)) {
				knh_intptr_t min = s->data[1];
				knh_intptr_t max;
				if (op_eq(s, FASTCALL0)) {
					max = s->data[1] + 2 * (knh_Method_psize((knh_Method_t*)s->p[4]) + 1);
				} else {
					max = s->data[2];
				}
				s = DP(n)->opbuf + i - 1;
				while ((int)i > 0 && isDEF(s) && s->data[0] >= min && s->data[0] < max) {
					if (isOBJ(s->data[0]) && iArray_n(rs->origidx, s->data[0]) != -1) {
//						DBG_P("pop %02d", s->data[0]);
						RENAME_ASSERT(knh_Array_size(STACK(s->data[0])) > 1);
						iArray_pop(ctx, STACK(s->data[0]));
						iArray_n(rs->origidx, s->data[0]) = -1;
//					} else {
//						DBG_P("skip pop args %d", s->data[0]);
					}
					i--;
					s = DP(n)->opbuf + i - 1;
				}
			} else if (isCAST(s)) {
				knh_intptr_t cfg = s->data[1] + 1;
				s = DP(n)->opbuf + i - 1;
				if ((int)i > 0 && isDEF(s) && s->data[0] == cfg && !isCALL(s)) {
//					DBG_P("skip pop args %d", s->data[0]);
					i--;
				}
			}
		} else if (op_eq(s, NEXT)) {
			RENAME_ASSERT(knh_Array_size(STACK(s->data[1])) > 1);
			iArray_pop(ctx, STACK(s->data[1]));
			iArray_n(rs->origidx, s->data[1]) = -1;
		}
	}
//	DBG_P("end %02d", DFN(n));
//	DBG_P("idx=%02d", *idx);
	RENAME_ASSERT(count == countVars(rs->stack) && checkVarStack(rs));
}

static void splitEdge(Ctx *ctx, knh_BasicBlock_t *bb, knh_BasicBlock_t *bbN)
{
	knh_BasicBlock_t *bbNEW = new_BasicBlockLABEL(ctx);
	if (bb->nextNC == bbN) {
		bbNEW->nextNC = bb->nextNC;
		bb->nextNC = bbNEW;
	} else {
		bbNEW->jumpNC = bb->jumpNC;
		bb->jumpNC = bbNEW;
	}
	DP(bbNEW)->incoming = 1;
	knh_BasicBlock_setVisited(bbNEW, 1);
}

static void adjustBlocks(Ctx *ctx, knh_BasicBlock_t *bbp, knh_BasicBlock_t *bbn, size_t *size)
{
	knh_BasicBlock_t *bbN, *bbJ;
	knh_BasicBlock_setVisited(bbn, 1);
	*size += 1;
	bbN = bbn->nextNC;
	bbJ = bbn->jumpNC;
	if (bbp != NULL)
		DP(bbn)->incoming = 1;
	if (bbN != NULL) {
		if (!knh_BasicBlock_isVisited(bbN)) {
			adjustBlocks(ctx, bbn, bbN, size);
		} else if (DP(bbN)->incoming < 2) {
			DP(bbN)->incoming += 1;
			if (bbJ != NULL) {
				splitEdge(ctx, bbn, bbN);
				*size += 1;
			}
		} else if (DP(bbN)->size > 0 && op_neq(DP(bbN)->opbuf, RET)) {
			//DBG_P("add empty block between [%02d] and NEXT[%02d]", DFN(n), DFN(bbN));
			addPostBody(ctx, bbn, bbN);
			DBG_A("printBB", printBB(ctx, bbN));
			*size += 1;
		} else {
			DBG_P("jump RET");
			DP(bbN)->incoming += 1;
		}
	}
	if (bbJ != NULL) {
		if (!knh_BasicBlock_isVisited(bbJ)) {
			adjustBlocks(ctx, bbn, bbJ, size);
		} else if (DP(bbJ)->incoming < 2) {
			DP(bbJ)->incoming += 1;
			if (bbN != NULL) {
				splitEdge(ctx, bbn, bbJ);
				*size += 1;
			}
		} else if (DP(bbJ)->size > 0 && op_neq(DP(bbJ)->opbuf, RET)) {
			//DBG_P("add empty block between [%02d] and JUMP[%02d]", DFN(n), DFN(bbJ));
			addPostBody(ctx, bbn, bbJ);
			DBG_A("printBB", printBB(ctx, bbJ));
			*size += 1;
		} else {
			DBG_P("jump RET");
			DP(bbJ)->incoming += 1;
		}
	}
}

/**
 * Convert to SSA form.
 * @param ctx    Context
 * @param bb     BasicBlock
 * @param df     Dominance Frontier of the block
 * @param pred   Predecessor of the block
 * @param idom   Immediately dominator of the block
 * @param vertex Linear list of BasicBlocks(~= listNC)
 * @param stack  A stack that records rename in each variable
 * @param max    Max index of register
 */
static void convertSSA(Ctx *ctx, ssa_data_t *ssa, knh_BasicBlock_t *bb)
{
	int i;
	int idx = 0;
	size_t size = 0;
	adjustBlocks(ctx, NULL, bb, &size);
	for (i = 0; i < (int)size; i++) {
		knh_Array_add(ctx, ssa->df, new_Array(ctx, CLASS_Int, 0));
		knh_Array_add(ctx, ssa->pred, new_Array(ctx, CLASS_Int, 0));
		iArray_add(ctx, ssa->idom, 0);
	}
	depthFirstSearch(ctx, ssa, NULL, bb);
	DBG_A("printTree", printTree(ctx, ssa->vertex));
	cfg_data_t cfg = {
		getParent,
		bbP_init,
		bbP_next,
		getDFN,
		ssa->vertex,
		NULL,
		ssa->pred
	};
	setIdom(ctx, ssa->idom, &cfg);
	DBG_A("idom", printArray_BB(ssa->idom));
	cfg.init = bbinit;
	cfg.next = bbnext;
	setDF(ctx, ssa->df, ssa->idom, bb, &cfg);
//	DBG_A("df", printArray_ArrayBB(ssa->df));
	idx = putPhi(ctx, ssa);
	BEGIN_LOCAL(ctx, lsfp, 2);
	LOCAL_NEW(ctx, lsfp, 0, knh_Array_t*, stack, new_Array(ctx, CLASS_Array, idx));
	LOCAL_NEW(ctx, lsfp, 1, knh_Array_t*, origidx, new_Array(ctx, CLASS_Int, idx));
	for (i = 0; i < idx; i++) {
		knh_Array_add(ctx, stack, new_Array(ctx, CLASS_Int, 1));
		iArray_add(ctx, Array__(stack, i), i);
		iArray_add(ctx, origidx, i);
	}
	if ((int)knh_Array_size(stack) > 0) {
		if (!isOBJ(idx)) {
			iArray_add(ctx, origidx, -1);
			idx += 1;
		}
		DBG_P("idx=%d", idx);
		renamestack rs = {stack, origidx};
		int codesize = 0;
		for (i = 0; i < (int)knh_Array_size(ssa->vertex); i++) {
			codesize += DP(VERTEX(i))->size;
		}
		DBG_P("codesize=%d", codesize);
		renameVars(ctx, ssa, &rs, bb, &idx);
//#if defined(K_USING_DEBUG)
//		printArray_Opline(ssa->def);
//		printArray_ArrayOpline(ssa->use);
//#endif
	} else {
		DBG_P("Don't rename!");
	}
	//setUnvisited(vertex);
	ssa->rindex = knh_Array_size(origidx);
	END_LOCAL(ctx, lsfp);
}

/* ------------------------------------------------------------------------ */
/* [convert out of SSA form] */

static knh_BasicBlock_t *getContainingBB(ssa_data_t *ssa, knh_opline_t *op)
{
	size_t i;
	knh_BasicBlock_t *bbn;
	Array_each_cast(ssa->vertex, i, bbn, BasicBlock) {
		if (op >= DP(bbn)->opbuf && op < DP(bbn)->opbuf + DP(bbn)->size)
			return bbn;
	}
	DBG_ABORT();
	return NULL;
}

/**
 * Eliminate phi resource interferences based on
 * data-flow and interference graph updates.
 */

#ifdef SSA_CONVOUT_NAIVE

/**
 * Convert out of SSA form.
 * @param ctx Context
 * @param bb  BasicBlock
 */
static void eliminatePhi_naive(Ctx *ctx, ssa_data_t *ssa)
{
	size_t i, j;
	knh_ushort_t k;
	knh_BasicBlock_t *bbn, *bbp;
	knh_opline_t *op;
	int tmpName;
	Array_each_cast(ssa->vertex, i, bbn, BasicBlock) {
		bblock_each(bbn, j, op) {
			if (op_eq(op, PHI)) {
				if (isOBJ(op->data[0])) {
					tmpName = ssa->rindex;
				} else {
					tmpName = ssa->rindex + 1;
				}
				ssa->rindex += 2;
				Array_each_cast(PRED(bbn), k, bbp, BasicBlock) {
					if (isOBJ(op->data[0])) {
						if (phi_isMod(op, k + 1)) {
							DBG_P("insert OSET r%d", tmpName);
							knh_BasicBlock_add1(ctx, bbp, OSET, tmpName, op->p[k + 1]);
							knh_Object_RCdec((knh_Object_t*)op->p[k + 1]);
						} else {
							DBG_P("insert OMOV r%d r%d", tmpName, op->data[k + 1]);
							knh_BasicBlock_add1(ctx, bbp, OMOV, tmpName, op->data[k + 1]);
						}
					} else {
						if (phi_isMod(op, k + 1)) {
							DBG_P("insert NSET r%d %d", tmpName, op->data[k + 1]);
							knh_BasicBlock_add1(ctx, bbp, NSET, tmpName, op->data[k + 1]);
						} else {
							DBG_P("insert NMOV r%d r%d", tmpName, op->data[k + 1]);
							knh_BasicBlock_add1(ctx, bbp, NMOV, tmpName, op->data[k + 1]);
						}
					}
					if (bbp->nextNC != NULL && bbp->jumpNC != NULL) {
						knh_opline_t *opLAST = DP(bbp)->opbuf + DP(bbp)->size - 1;
						SWAP(opLAST, opLAST - 1, sizeof(knh_opline_t));
					}
				}
				op->data[1] = tmpName;
				if (isOBJ(op->data[0])) {
					DBG_P("replace PHI to OMOV r%d r%d", op->data[0], tmpName);
					op->opcode = OPCODE_OMOV;
				} else {
					DBG_P("replace PHI to NMOV r%d r%d", op->data[0], tmpName);
					op->opcode = OPCODE_NMOV;
				}
			} else if (op_neq(op, NOP)) {
				break;
			}
		}
	}
}

#endif /* SSA_CONVOUT_NAIVE */
#ifdef SSA_CONVOUT_INTERFERENCE

static knh_bool_t isInterfering(ssa_data_t *ssa, knh_intptr_t x, knh_intptr_t y)
{
	size_t i, j;
	knh_bool_t isx, isy;
	knh_intptr_t data;
	for (i = 0; i < knh_Array_size(ssa->liveIn); i++) {
		isx = 0;
		isy = 0;
		Array_each(Array__(ssa->liveIn, i), j, data) {
			if (data == x) {
				isx = 1;
				if (isy)
					return 1;
			} else if (data == y) {
				isy = 1;
				if (isx)
					return 1;
			}
		}
		isx = 0;
		isy = 0;
		Array_each(Array__(ssa->liveOut, i), j, data) {
			if (data == x) {
				isx = 1;
				if (isy)
					return 1;
			} else if (data == y) {
				isy = 1;
				if (isx)
					return 1;
			}
		}
	}
	return 0;
}

static knh_bool_t isSameArray(knh_Array_t *aa, knh_Array_t *ab)
{
	size_t i, j;
	knh_int_t va, vb;
	if (knh_Array_size(aa) != knh_Array_size(ab))
		return 0;
	Array_each(aa, i, va) {
		Array_each(ab, j, vb) {
			if (va == vb)
				goto L_CONT;
		}
		return 0;
L_CONT:;
	}
	return 1;
}

static knh_bool_t hasSamePart(knh_Array_t *aa, knh_Array_t *ab)
{
	size_t i, j;
	knh_int_t va, vb;
	Array_each(aa, i, va) {
		Array_each(ab, j, vb) {
			if (va == vb)
				return 1;
		}
	}
	return 0;
}

static void setCandidateResource(Ctx *ctx, ssa_data_t *ssa, knh_BasicBlock_t *bb, knh_opline_t *op, knh_Array_t *pcc, knh_Array_t *candidateResourceSet, knh_Array_t *unresolvedNeighborMap)
{
	size_t i, j, k, l;
	knh_intptr_t xi, xj, yi, yj;
	knh_Array_t *live_i, *live_j;
	for (i = 0; i < 2; i++) {
		for (j = i + 1; j < 3; j++) {
			if (phi_isMod(op, i) || phi_isMod(op, j))
				continue;
			xi = op->data[i];
			xj = op->data[j];
			live_i = (i == 0 ? LIVEIN(bb) : LIVEOUT(bb));
			live_j = LIVEOUT(bb);
			Array_each(PCC(xi), k, yi) {
				Array_each(PCC(xj), l, yj) {
					if (isInterfering(ssa, yi, yj)) {
						DBG_P("yi[%d] and yj[%d] is interfering", yi, yj);
						if (isSameArray(PCC(xi), PCC(xj))) {
							DBG_P("pcc(%d) and pcc(%d) are same array", xi, xj);
							continue;
						}
						// case 1
						if (hasSamePart(PCC(xi), live_j) && !hasSamePart(PCC(xj), live_i)) {
							iArray_add(ctx, candidateResourceSet, xi);
						}
						// case 2
						else if (!hasSamePart(PCC(xi), live_j) && hasSamePart(PCC(xj), live_i)) {
							iArray_add(ctx, candidateResourceSet, xj);
						}
						// case 3
						else if (hasSamePart(PCC(xi), live_j) && hasSamePart(PCC(xj), live_i)) {
							iArray_add(ctx, candidateResourceSet, xi);
							iArray_add(ctx, candidateResourceSet, xj);
						}
						// case 4
						else /* if (!hasSamePart(PCC(xi), live_j) && !hasSamePart(PCC(xj), live_i)) */ {
							iArray_add(ctx, UNMAP(xi), xj);
							iArray_add(ctx, UNMAP(xj), xi);
						}
					} else {
						DBG_P("yi[%d] and yj[%d] is not interfering", yi, yj);
					}
				}
			}
		}
	}
}

static void resolveNeighborMap(Ctx *ctx, knh_Array_t *candidateResourceSet, knh_Array_t *unresolvedNeighborMap)
{
	size_t i, j;
	knh_Array_t *a;
	knh_intptr_t v, xi;
	Array_each_next(unresolvedNeighborMap, i, a, Array__(unresolvedNeighborMap, i)) {
		if (knh_Array_size(a) > 0) {
			v = i;
			if (!iArray_isContain(candidateResourceSet, v)) {
				Array_each(a, j, xi) {
					if (!iArray_isContain(candidateResourceSet, xi)) {
						DBG_P("add CRSET r%d", xi);
						iArray_add(ctx, candidateResourceSet, xi);
					}
				}
			}
		}
	}
}

static void liveOutAtStatement(Ctx *ctx, ssa_data_t *ssa, knh_Array_t *aM, knh_BasicBlock_t *bbn, knh_opline_t *ops, knh_intptr_t v);
static void liveOutAtBlock(Ctx *ctx, ssa_data_t *ssa, knh_Array_t *aM, knh_BasicBlock_t *bbn, knh_intptr_t v);

static void liveInAtStatement(Ctx *ctx, ssa_data_t *ssa, knh_Array_t *aM, knh_BasicBlock_t *bbn, knh_opline_t *ops, knh_intptr_t v)
{
	// v is liveIn at ops
	size_t i;
	knh_BasicBlock_t *bbp;
	knh_opline_t *opstart;
	bblock_each(bbn, i, opstart) {
		if (op_neq(opstart, NOP))
			break;
	}
	if (ops == opstart) {
		DBG_P("r%d is liveIn at [%02d]", v, ID(bbn));
		iArray_add(ctx, LIVEIN(bbn), v);
		Array_each_cast(PRED(bbn), i, bbp, BasicBlock) {
			liveOutAtBlock(ctx, ssa, aM, bbp, v);
		}
	} else {
		knh_opline_t *ops_ = ops - 1;
		liveOutAtStatement(ctx, ssa, aM, bbn, ops_, v);
	}
}

static void interferenceGraph_add(Ctx *ctx, ssa_data_t *ssa, knh_intptr_t v, knh_intptr_t w)
{
	iArray_insert(ctx, IFGRAPH(v), w);
	iArray_insert(ctx, IFGRAPH(w), v);
}

static void liveOutAtStatement(Ctx *ctx, ssa_data_t *ssa, knh_Array_t *aM, knh_BasicBlock_t *bbn, knh_opline_t *ops, knh_intptr_t v)
{
	// v is liveOut at ops
	knh_intptr_t w;
	if (isDEF(ops)) {
		w = ops->data[0];
		if (w != v)
			interferenceGraph_add(ctx, ssa, v, w);
	} else {
		w = -1;
	}
	if (w != v) {
		liveInAtStatement(ctx, ssa, aM, bbn, ops, v);
	}
}

static void liveOutAtBlock(Ctx *ctx, ssa_data_t *ssa, knh_Array_t *aM, knh_BasicBlock_t *bbn, knh_intptr_t v)
{
	DBG_P("r%d is liveOut at [%02d]", v, ID(bbn));
	iArray_insert(ctx, LIVEOUT(bbn), v);
	if (!iArray_isContain(aM, v)) {
		iArray_insert(ctx, aM, v);
		knh_opline_t *ops = DP(bbn)->opbuf + DP(bbn)->size - 1;
		liveOutAtStatement(ctx, ssa, aM, bbn, ops, v);
	}
}

static void analyzeLiveness(Ctx *ctx, ssa_data_t *ssa)
{
	size_t i, j, k;
	knh_intptr_t v;
	knh_opline_t *op, *ops;
	BEGIN_LOCAL(ctx, lsfp, 1);
	LOCAL_NEW(ctx, lsfp, 0, knh_Array_t*, aM, new_Array(ctx, CLASS_Int, 0));
	// initialize
	for (i = 0; i < knh_Array_size(ssa->vertex); i++) {
		knh_Array_clear(ctx, LIVEIN(VERTEX(i)), 0);
		knh_Array_clear(ctx, LIVEOUT(VERTEX(i)), 0);
	}
	Array_each_cast(ssa->def, i, op, opline) {
		if (op == NULL || op_eq(op, NOP))
			continue;
		v = op->data[0];
		// added start
		if (op_eq(op, PHI)) {
			DBG_P("r%d is liveIn at [%02d]", v, ID(getContainingBB(ssa, op)));
			iArray_add(ctx, LIVEIN(getContainingBB(ssa, op)), v);
		}
		// added end
		knh_Array_clear(ctx, aM, 0);
		Array_each_cast(USE(v), j, ops, opline) {
			if (op_eq(ops, PHI)) {
				for (k = 1; k < 3; k++) {
					if (!phi_isMod(ops, k)) {
						knh_BasicBlock_t *bbp = Array(BasicBlock, PRED(getContainingBB(ssa, ops)), k - 1);
						liveOutAtBlock(ctx, ssa, aM, bbp, v);
					}
				}
			} else if (op_neq(ops, NOP)) {
				liveInAtStatement(ctx, ssa, aM, getContainingBB(ssa, ops), ops, v);
			}
		}
	}
	END_LOCAL(ctx, lsfp);
}

static void updateUseDef(Ctx *ctx, ssa_data_t *ssa, knh_BasicBlock_t *bb, knh_opline_t *buf, int line)
{
	DBG_P("line=%d", line);
	DBG_P("updateUseDef");
	DBG_A("before", printArray_Opline(ssa->def));
	DBG_A("before", printArray_ArrayOpline(ssa->use));
	size_t i, j;
	int offset;
	knh_opline_t *opdef, *opuse;
	knh_ushort_t bbsize = DP(bb)->size;
	knh_opline_t *newbuf = DP(bb)->opbuf;
	Array_each_cast(ssa->def, i, opdef, opline) {
		if (opdef == NULL)
			continue;
		if (buf <= opdef && buf + bbsize >= opdef) {
			DBG_P("DEF(%d)=%s", i, knh_opcode_tochar(DEF(i)->opcode));
			offset = opdef - buf;
			DBG_P("offset=%d", offset);
			if (offset > line) {
				iArray_n(ssa->def, i) = (knh_int_t)(newbuf + offset);
			} else if (DEF(i) != opdef) {
				iArray_n(ssa->def, i) = (knh_int_t)(newbuf + offset + 1);
			}
		} else if (op_eq(opdef, NOP)) {
			iArray_n(ssa->def, i) = 0;
			knh_Array_clear(ctx, USE(i), 0);
		}
		Array_each_cast(USE(i), j, opuse, opline) {
			DBG_ASSERT(opuse != NULL);
			if (buf <= opuse && buf + bbsize >= opuse) {
				DBG_P("USE(%d)=%s", j, knh_opcode_tochar(Array(opline, USE(i), j)->opcode));
				offset = opuse - buf;
				DBG_P("offset=%d", offset);
				if (offset > line) {
					iArray_n(USE(i), j) = (knh_int_t)(newbuf + offset);
				} else if (iArray_n(USE(i), j) != (knh_int_t)opuse) {
					iArray_n(USE(i), j) = (knh_int_t)(newbuf + offset + 1);
				}
			} else if (op_eq(opuse, NOP)) {
				iArray_n_remove(ctx, USE(i), j);
			}
		}
	}
	DBG_A("after", printArray_Opline(ssa->def));
	DBG_A("after", printArray_ArrayOpline(ssa->use));
}

static knh_int_t insertCopy(Ctx *ctx, ssa_data_t *ssa, knh_Array_t *candidateResourceSet, knh_Array_t *unresolvedNeighborMap, knh_Array_t *pcc, knh_BasicBlock_t *bb, knh_opline_t *op, size_t count)
{
	size_t i, j, k, l;
	int line = 0;
	knh_intptr_t xi, xnew_i;
	knh_BasicBlock_t *bbk, *bbj;
	knh_bool_t remove;
	knh_opline_t *opj, *op_inserted;
	knh_int_t ret = 0;
	knh_opline_t *opbuf;
	if (knh_Array_size(candidateResourceSet) > 0) {
		if (isOBJ(iArray_n(candidateResourceSet, 0))) {
			xnew_i = ssa->rindex;
		} else {
			xnew_i = ssa->rindex + 1;
		}
		ssa->rindex += 2;
		iArray_add(ctx, ssa->def, 0);
		iArray_add(ctx, ssa->def, 0);
		knh_Array_add(ctx, ssa->interference, new_Array(ctx, CLASS_Int, 0));
		knh_Array_add(ctx, ssa->interference, new_Array(ctx, CLASS_Int, 0));
		knh_Array_add(ctx, ssa->use, new_Array(ctx, CLASS_Int, 0));
		knh_Array_add(ctx, ssa->use, new_Array(ctx, CLASS_Int, 0));
	}
	Array_each(candidateResourceSet, i, xi) {
		for (j = 1; j < 3; j++) {
			if (xi == op->data[j]) {
				bbk = Array(BasicBlock, PRED(bb), j - 1);
//				DBG_P("bb[%02d], bbk[%02d]", ID(bb), ID(bbk));
				opbuf = DP(bbk)->opbuf;
				if (isOBJ(xi)) {
					DBG_P("[%02d] insert OMOV r%d r%d", ID(bbk), xnew_i, xi);
					knh_BasicBlock_add1(ctx, bbk, OMOV, xnew_i, xi);
				} else {
					DBG_P("[%02d] insert NMOV r%d r%d", ID(bbk), xnew_i, xi);
					knh_BasicBlock_add1(ctx, bbk, NMOV, xnew_i, xi);
				}
				if (bbk == bb) {
					ret += 1;
					op = DP(bb)->opbuf + count + ret;
				}
				line = DP(bb)->size - 1;
				if (bbk->nextNC != NULL && bbk->jumpNC != NULL) {
					knh_opline_t *opLAST = DP(bbk)->opbuf + DP(bbk)->size - 1;
					SWAP(opLAST, opLAST - 1, sizeof(knh_opline_t));
					line -= 1;
				}
				updateUseDef(ctx, ssa, bbk, opbuf, line);
				DBG_P("op_inserted=%s", knh_opcode_tochar(op_inserted->opcode));
				iArray_n(ssa->def, xnew_i) = (knh_int_t)op_inserted;
				iArray_add(ctx, USE(xnew_i), (knh_int_t)op_inserted);
				DBG_P("replace PHI r%d to r%d", op->data[j], xnew_i);
				op->data[j] = xnew_i;
				while ((knh_intptr_t)knh_Array_size(pcc) <= xnew_i) {
					knh_Array_add(ctx, pcc, new_Array(ctx, CLASS_Int, 0));
				}
				iArray_add(ctx, PCC(xnew_i), xnew_i);
				iArray_add(ctx, LIVEOUT(bbk), xnew_i);
				remove = 1;
				bbitr_t itrbuf, *itr = bbinit(&itrbuf, bbnext, NULL, bbk);
				itr_foreach(bbj, itr) {
					if (iArray_isContain(LIVEIN(bbj), xi)) {
						remove = 0;
					}
					bblock_each(bbj, k, opj) {
						if (op_eq(opj, PHI)) {
							for (l = 1; l < 3; l++) {
								if (opj->data[l] == xi && Array(BasicBlock, PRED(bbj), l - 1) == bbk) {
									remove = 0;
								}
							}
						} else {
							break;
						}
					}
				}
				if (remove == 1) {
					iArray_remove(ctx, LIVEOUT(bbk), xi);
				}
			}
		}
		if (xi == op->data[0]) {
			bblock_each(bb, j, opj) {
				if (op_neq(opj, PHI)) {
					line = j;
					break;
				}
			}
			opbuf = DP(bb)->opbuf;
			if (isOBJ(xi)) {
				DBG_P("[%02d] insert OMOV r%d r%d", ID(bb), xi, xnew_i);
				INSERT_OP(ctx, bb, line, OMOV, xi, xnew_i);
			} else {
				DBG_P("[%02d] insert NMOV r%d r%d", ID(bb), xi, xnew_i);
				INSERT_OP(ctx, bb, line, NMOV, xi, xnew_i);
			}
			updateUseDef(ctx, ssa, bb, opbuf, line);
			ret += 1;
			op = DP(bb)->opbuf + count + ret;
			DBG_P("replace PHI r%d to r%d", op->data[0], xnew_i);
			knh_intptr_t oldDest = op->data[0];
			op->data[0] = xnew_i;
			while ((knh_intptr_t)knh_Array_size(pcc) <= xnew_i) {
				knh_Array_add(ctx, pcc, new_Array(ctx, CLASS_Int, 0));
			}
			while ((knh_intptr_t)knh_Array_size(unresolvedNeighborMap) <= xnew_i) {
				knh_Array_add(ctx, unresolvedNeighborMap, new_Array(ctx, CLASS_Int, 0));
			}
			iArray_copy(ctx, UNMAP(xnew_i), UNMAP(oldDest));
			knh_Array_clear(ctx, UNMAP(oldDest), 0);
			iArray_remove(ctx, LIVEIN(bb), xi);
			iArray_add(ctx, LIVEOUT(bb), xnew_i);
		}
		// update interferenceGraph
		analyzeLiveness(ctx, ssa);
	}
	return ret;
}

static void mergepcc(Ctx *ctx, knh_Array_t *pcc, knh_opline_t *op, knh_Array_t *currentPhiCongruenceClass)
{
	DBG_ASSERT(op_eq(op, PHI));
	size_t i, j, k;
	knh_intptr_t xi, yj, val;
	for (i = 0; i < 3; i++) {
		if (!phi_isMod(op, i)) {
			xi = op->data[i];
//			DBG_P("add pcc(%d) to currentPhiCongruenceClass", xi);
			Array_each(PCC(xi), j, val) {
				iArray_insert(ctx, currentPhiCongruenceClass, val);
			}
			Array_each(PCC(xi), j, yj) {
//				DBG_P("add pcc(%d) to currentPhiCongruenceClass", yj);
				Array_each(PCC(yj), k, val) {
					iArray_insert(ctx, currentPhiCongruenceClass, val);
				}
			}
		}
	}
	Array_each(currentPhiCongruenceClass, i, xi) {
//		DBG_P("merge pcc(%d)", xi);
		knh_Array_clear(ctx, PCC(xi), 0);
		KNH_SETv(ctx, knh_Array_n(pcc, xi), currentPhiCongruenceClass);
	}
}

static void coalescePhi(Ctx *ctx, ssa_data_t *ssa, knh_Array_t *pcc)
{
	size_t i, j, k, l;
	knh_BasicBlock_t *bb;
	knh_opline_t *op;
	knh_intptr_t x, y, val;
	Array_each_cast(ssa->vertex, i, bb, BasicBlock) {
		bblock_each(bb, j, op) {
			if ((op_eq(op, OMOV) || op_eq(op, NMOV))) {
				if (op->data[0] >= (knh_intptr_t)knh_Array_size(pcc)) {
					DBG_P("don't remove copy");
					continue;
				}
				x = op->data[0];
				y = op->data[1];
				// case 1
				if (knh_Array_size(PCC(x)) == 0 && knh_Array_size(PCC(y)) == 0) {
					DBG_P("remove MOV r%d r%d", x, y);
					op->opcode = OPCODE_NOP;
				}
				// case 2
				else if (knh_Array_size(PCC(x)) == 0 && knh_Array_size(PCC(y)) > 0) {
					Array_each(PCC(y), k, val) {
						if (val != y) {
							if (isInterfering(ssa, x, val)) {
								DBG_P("can't remove MOV r%d r%d", x, y);
							} else {
								DBG_P("remove MOV r%d r%d", x, y);
								op->opcode = OPCODE_NOP;
							}
						}
					}
				}
				else if (knh_Array_size(PCC(x)) > 0 && knh_Array_size(PCC(y)) == 0) {
					Array_each(PCC(x), k, val) {
						if (val != x) {
							if (isInterfering(ssa, val, y)) {
								DBG_P("can't remove MOV r%d r%d", x, y);
							} else {
								DBG_P("remove MOV r%d r%d", x, y);
								op->opcode = OPCODE_NOP;
							}
						}
					}
				}
				// case 3
				else /* if (knh_Array_size(PCC(x)) > 0 && knh_Array_size(PCC(y)) > 0) */{
					Array_each(PCC(y), k, val) {
						if (val != y) {
							if (isInterfering(ssa, x, val)) {
								DBG_P("can't remove MOV r%d r%d", x, y);
							} else {
								Array_each(PCC(x), l, val) {
									if (val != x) {
										if (isInterfering(ssa, val, y)) {
											DBG_P("can't remove MOV r%d r%d", x, y);
										} else {
											DBG_P("remove MOV r%d r%d", x, y);
											op->opcode = OPCODE_NOP;
										}
									}
								}
							}
						}
					}
				}
				if (op_eq(op, NOP)) {
					DBG_P("merge pcc(%d) and pcc(%d)", x, y);
					while (knh_Array_size(PCC(y)) > 0) {
						iArray_insert(ctx, PCC(x), iArray_pop_ret(ctx, PCC(y)));
					}
					KNH_SETv(ctx, knh_Array_n(pcc, y), knh_Array_n(pcc, x));
				}
			}
		}
	}
}

static void eliminatePhi_interference(Ctx *ctx, ssa_data_t *ssa)
{
	size_t i, j, k;
	knh_BasicBlock_t *bb;
	knh_opline_t *op;
	knh_intptr_t x;
	BEGIN_LOCAL(ctx, lsfp, 4);
	LOCAL_NEW(ctx, lsfp, 0, knh_Array_t*, pcc, new_Array(ctx, CLASS_Array, ssa->rindex));
	LOCAL_NEW(ctx, lsfp, 1, knh_Array_t*, candidateResourceSet, new_Array(ctx, CLASS_Int, 0));
	LOCAL_NEW(ctx, lsfp, 2, knh_Array_t*, unresolvedNeighborMap, new_Array(ctx, CLASS_Array, ssa->rindex));
	LOCAL_NEW(ctx, lsfp, 3, knh_Array_t*, currentPhiCongruenceClass, new_Array(ctx, CLASS_Array, 0));
	for (i = 0; i < knh_Array_size(ssa->vertex); i++) {
		knh_Array_add(ctx, ssa->liveIn, new_Array(ctx, CLASS_Int, 0));
		knh_Array_add(ctx, ssa->liveOut, new_Array(ctx, CLASS_Int, 0));
	}
	analyzeLiveness(ctx, ssa);
	// step 1
	Array_each_cast(ssa->vertex, i, bb, BasicBlock) {
		bblock_each(bb, j, op) {
			if (op_eq(op, PHI)) {
				for (k = 1; k < 3; k++) {
					if (!phi_isMod(op, k)) {
						while (op->data[k] >= (int)knh_Array_size(pcc)) {
							knh_Array_add(ctx, pcc, new_Array(ctx, CLASS_Int, 0));
							knh_Array_add(ctx, unresolvedNeighborMap, new_Array(ctx, CLASS_Int, 0));
						}
						iArray_add(ctx, PCC(op->data[k]), op->data[k]);
					}
				}
			} else {
				break;
			}
		}
	}
	DBG_A("pcc", printArray_Array(pcc));
	DBG_A("interferenceGraph", printArray_Array(ssa->interference));
	DBG_A("liveIn", printArray_Array(ssa->liveIn));
	DBG_A("liveOut", printArray_Array(ssa->liveOut));
	DBG_A("cfg", printTree(ctx, ssa->vertex));
	// step 2
	Array_each_cast(ssa->vertex, i, bb, BasicBlock) {
		bblock_each(bb, j, op) {
			if (op_eq(op, PHI)) {
				// step 3
				for (k = 0; k < 3; k++) {
					if (!phi_isMod(op, k)) {
						knh_Array_clear(ctx, UNMAP(op->data[k]), 0);
					}
				}
				// step 4
				setCandidateResource(ctx, ssa, bb, op, pcc, candidateResourceSet, unresolvedNeighborMap);
				// step 5
				resolveNeighborMap(ctx, candidateResourceSet, unresolvedNeighborMap);
				// step 6
				op = DP(bb)->opbuf + j + insertCopy(ctx, ssa, candidateResourceSet, unresolvedNeighborMap, pcc, bb, op, j);
				// step 7
				knh_Array_add(ctx, currentPhiCongruenceClass, new_Array(ctx, CLASS_Int, 0));
				mergepcc(ctx, pcc, op, Array__(currentPhiCongruenceClass, knh_Array_size(currentPhiCongruenceClass) - 1));
			} else {
				break;
			}
		}
	}
	DBG_A("currentPhiCongruenceClass", printArray_Array(currentPhiCongruenceClass));
	Array_each_cast(ssa->vertex, i, bb, BasicBlock) {
		bblock_each(bb, j, op) {
			if (op_eq(op, PHI)) {
				for (k = 0; k < 3; k++) {
					if (!phi_isMod(op, k)) {
						x = op->data[k];
						if (knh_Array_size(PCC(x)) == 1) {
							DBG_P("remove pcc(%d)", x);
							iArray_pop(ctx, PCC(x));
						}
					}
				}
			} else {
				break;
			}
		}
	}
	coalescePhi(ctx, ssa, pcc);
	DBG_A("pcc", printArray_Array(pcc));
	DBG_A("interferenceGraph", printArray_Array(ssa->interference));
	DBG_A("liveIn", printArray_Array(ssa->liveIn));
	DBG_A("liveOut", printArray_Array(ssa->liveOut));
	END_LOCAL(ctx, lsfp);
}

#endif /* SSA_CONVOUT_INTERFERENCE */

/* ------------------------------------------------------------------------ */
/**
 * Optimize the BasicBlocks.
 * @param ctx Context
 * @param bbs BasicBlocks
 */

static knh_BasicBlock_t* getExitNode(ssa_data_t *ssa)
{
	size_t i;
	knh_BasicBlock_t *n;
	Array_each_cast(ssa->vertex, i, n, BasicBlock) {
		if (DP(n)->size > 0 && op_eq(DP(n)->opbuf, RET))
			return n;
	}
	DBG_ABORT();
	return NULL;
}

static void reverseDFS(Ctx *ctx, ssa_data_t *ssa, knh_BasicBlock_t *bbn, knh_Array_t *dfnum, knh_Array_t *rvertex, knh_int_t *ridx)
{
	DBG_ASSERT((int)ID(bbn) >= 0);
	if (iArray_n(dfnum, ID(bbn)) == -1) {
		iArray_n(dfnum, ID(bbn)) = *ridx;
		iArray_add(ctx, rvertex, (knh_int_t)bbn);
		*ridx += 1;
		size_t i;
		knh_BasicBlock_t *bbw;
		Array_each_cast(PRED(bbn), i, bbw, BasicBlock) {
			reverseDFS(ctx, ssa, bbw, dfnum, rvertex, ridx);
		}
	}
}

static void setValueUseDef(Ctx *ctx, ssa_data_t *ssa)
{
	size_t i, j, k;
	knh_opline_t *op;
	knh_BasicBlock_t *n;
	for (i = 0; i < ssa->rindex; i++) {
		iArray_add(ctx, ssa->def, 0);
		knh_Array_add(ctx, ssa->use, new_Array(ctx, CLASS_Int, 0));
		knh_Array_add(ctx, ssa->interference, new_Array(ctx, CLASS_Int, 0));
	}
	Array_each_cast(ssa->vertex, i, n, BasicBlock) {
		bblock_each(n, j, op) {
//			DBG_P("%s count %d", knh_opcode_tochar(op->opcode), (int)op->count);
			if (isCALL(op) && (int)op->data[0] >= 0) {
				DBG_ASSERT(iArray_n(ssa->def, op->data[0]) == 0);
				iArray_n(ssa->def, op->data[0]) = (knh_int_t)op;
			} else if (op_eq(op, NEXT)) {
				iArray_add(ctx, USE(op->data[2]), (knh_int_t)op);
				DBG_ASSERT(iArray_n(ssa->def, op->data[1]) == 0);
				iArray_n(ssa->def, op->data[1]) = (knh_int_t)op;
			} else if (op_isArg(op)) {
				k = 0;
				do {
					DBG_ASSERT(op_eq(op, NSET) || op_eq(op, NMOV) || op_eq(op, OSET) || op_eq(op, OMOV) || op_eq(op, PHI));
					if (isUSE(op, 1))
						iArray_add(ctx, USE(op->data[1]), (knh_int_t)op);
					DBG_ASSERT(iArray_n(ssa->def, op->data[0]) == 0);
					iArray_n(ssa->def, op->data[0]) = (knh_int_t)op;
					DBG_P("inserted %s", knh_opcode_tochar(op->opcode));
					k += 1;
					op = DP(n)->opbuf + ++j;
				} while (op_isArg(op));
				while (k > 0) {
					knh_opline_t *opP = op - k;
					iArray_add(ctx, USE(opP->data[0]), (knh_int_t)op);
					k -= 1;
				}
				if ((int)op->data[0] >= 0) {
					DBG_ASSERT(iArray_n(ssa->def, op->data[0]) == 0);
					iArray_n(ssa->def, op->data[0]) = (knh_int_t)op;
				}
			} else {
				if (op_eq(op, P) && op->data[4] != 0) {
					knh_opline_t *opP = op - 1;
					iArray_add(ctx, USE(opP->data[0]), (knh_int_t)op);
				} else if (isCAST(op) && (int)op->data[0] >= 0) {
					DBG_ASSERT(iArray_n(ssa->def, op->data[0]) == 0);
					iArray_n(ssa->def, op->data[0]) = (knh_int_t)op;
				} else if (op_eq(op, LOADMTD) || op_eq(op, THROW) || op_eq(op, CHKSTACK)) {
					iArray_add(ctx, USE(op->data[0]), (knh_int_t)op);
				} else {
					for (k = 1; k < knh_opcode_size(op->opcode); k++) {
						if ((int)op->data[k] >= 0) {
							if (isUSE(op, k)) {
								//			DBG_P("adduse %s %d", knh_opcode_tochar(op->opcode), (int)op->data[k]);
								iArray_add(ctx, USE(op->data[k]), (knh_int_t)op);
							} else if (isVUSE(op, k)) {
								iArray_add(ctx, USE(op->data[k]), (knh_int_t)op);
							}
						}
					}
					if ((int)op->data[0] >= 0) {
						if (isVDEF(op, 0)) {
							if (iArray_n(ssa->def, op->data[0]) == 0) {
								iArray_n(ssa->def, op->data[0]) = (knh_int_t)op;
							} else {
								iArray_add(ctx, USE(op->data[0]), (knh_int_t)op);
							}
						}
						if (isDEF(op)) {
							DBG_ASSERT(iArray_n(ssa->def, op->data[0]) == 0);
							iArray_n(ssa->def, op->data[0]) = (knh_int_t)op;
						}
					}
				}
			}
		}
	}
}

static knh_bool_t isLiveBB(knh_BasicBlock_t *bb, knh_Array_t *live)
{
	knh_ushort_t i;
	knh_opline_t *op;
	bblock_each(bb, i, op) {
		if (LIVE(op, op->count) == 1)
			return 1;
	}
	return 0;
}

/* ------------------------------------------------------------------------ */
/* [SSA optimize] */

#if defined(SSA_OPT_CP) || defined(SSA_OPT_AGGRESSIVE_DCE)

static void removeBlock(knh_BasicBlock_t *bb, int index)
{
	size_t i;
	knh_opline_t *op;
	knh_BasicBlock_t *bbC;
	bbitr_t itrbuf, *itr = bbinit(&itrbuf, bbnext, NULL, bb);
	itr_foreach(bbC, itr) {
		bblock_each(bbC, i, op) {
			if (op_eq(op, PHI)) {
				if (phi_isMod(op, index + 1)) {
					if (!isOBJ(op->data[0])) {
						DBG_P("REPLACE PHI to NSET", knh_opcode_tochar(op->opcode));
						op->opcode = OPCODE_NSET;
					} else {
						DBG_P("REPLACE PHI to OSET", knh_opcode_tochar(op->opcode));
						op->opcode = OPCODE_OSET;
					}
				} else {
					if (!isOBJ(op->data[0])) {
						DBG_P("REPLACE PHI to NMOV", knh_opcode_tochar(op->opcode));
						op->opcode = OPCODE_NMOV;
					} else {
						DBG_P("REPLACE PHI to OMOV", knh_opcode_tochar(op->opcode));
						op->opcode = OPCODE_OMOV;
					}
				}
				if (index == 1)
					op->data[1] = op->data[2];
			} else {
				return;
			}
		}
	}
}

#endif /* SSA_OPT_CP || SSA_OPT_AGGRESSIVE_DCE */

typedef void (*fssaopt) (Ctx *ctx, ssa_data_t *ssa);

#ifdef SSA_OPT_AGGRESSIVE_DCE
#include "./opt/deadCodeElim.c"
#endif
#if defined(SSA_OPT_CP)
#include "./opt/constProp.c"
#elif defined(SSA_OPT_SCCP)
#include "./opt/sparseConstProp.c"
#endif
#ifdef SSA_OPT_CSE
#include "./opt/commonSubExpElim.c"
#endif

/**
 * Optimize the KLRcode.
 * @param ctx Context
 * @param bb  BasicBlock
 */
void knh_BasicBlock_optimize(Ctx *ctx, knh_BasicBlock_t *bb)
{
	size_t i, size = knh_Array_size(bb->listNC);
	BEGIN_LOCAL(ctx, lsfp, 10);
	LOCAL_NEW(ctx, lsfp, 0, knh_Array_t*, df, new_Array(ctx, CLASS_Array, size));
	LOCAL_NEW(ctx, lsfp, 1, knh_Array_t*, pred, new_Array(ctx, CLASS_Array, size));
	LOCAL_NEW(ctx, lsfp, 2, knh_Array_t*, idom, new_Array(ctx, CLASS_Int, size));
	LOCAL_NEW(ctx, lsfp, 3, knh_Array_t*, vertex, new_Array(ctx, CLASS_Int, size));
	LOCAL_NEW(ctx, lsfp, 4, knh_Array_t*, liveIn, new_Array(ctx, CLASS_Array, size));
	LOCAL_NEW(ctx, lsfp, 5, knh_Array_t*, liveOut, new_Array(ctx, CLASS_Array, size));
	LOCAL_NEW(ctx, lsfp, 6, knh_Array_t*, interference, new_Array(ctx, CLASS_Array, 0));
	LOCAL_NEW(ctx, lsfp, 7, knh_Array_t*, def, new_Array(ctx, CLASS_Int, 0));
	LOCAL_NEW(ctx, lsfp, 8, knh_Array_t*, use, new_Array(ctx, CLASS_Array, 0));
	LOCAL_NEW(ctx, lsfp, 9, knh_Array_t*, rdf, new_Array(ctx, CLASS_Int, 0));

	ssa_data_t ssa = {
		df, pred, idom, vertex, liveIn, liveOut,
		interference, def, use, rdf, 0, 0}
	;
	convertSSA(ctx, &ssa, bb);
	setValueUseDef(ctx, &ssa);
	fssaopt opts[] = {
#if defined(SSA_OPT_CP)
		constantPropagation,
#elif defined(SSA_OPT_SCCP)
		sparseConditionalConstantPropagation,
#endif
#ifdef SSA_OPT_CSE
		commonSubExpressionElimination,
#endif
#ifdef SSA_OPT_AGGRESSIVE_DCE
		setRDF,
		aggressiveDeadCodeElimination,
#endif
	};
	if (ssa.rindex > 0) {
		size = sizeof(opts) / sizeof(fssaopt);
		DBG_A("printtree", printTree(ctx, ssa.vertex));
		for (i = 0; i < size; i++) {
			opts[i](ctx, &ssa);
			DBG_A("def", printArray_Opline(ssa.def));
			DBG_A("use", printArray_ArrayOpline(ssa.use));
		}
		eliminatePhi(ctx, &ssa);
	}
	END_LOCAL(ctx, lsfp);
}

#endif /* K_USING_SSA */

#ifdef __cplusplus
}
#endif


