/***************************************************************************
                          docTables.cpp  -  description
                             -------------------
    begin                :   2 22 2003
    copyright            : (C) 2003 by K.Kobayashi
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
//using namespace std;
//#include <iostream>

// include files for Qt
#include <qdatetime.h>
#include <qlistview.h>
#include <qfile.h>
#include <qregexp.h>
#include <qfileinfo.h>

// include files for KDE

// application specific includes
#include "option.h"
#include "docTables.h"

extern Option*    g_Option;
extern QDateTime* g_CurTime;

/////////////////////////////////////////////////////////////////////////////
// VarTbl - Variable Table

//
// Constructor - fron 'cron' Variable Data
VarTbl::VarTbl( const QString& s, const QString& c )
  : Cmnt( c )
{

  // ex : "NAME = VARIABLE"
  Name = s.section( QRegExp("\\s*=\\s*"), 0, 0 );
  Val = s.section( QRegExp("\\s*=\\s*"), 1, 1 );

}

//
// Constructor - ( name, variable, comment )
VarTbl::VarTbl( const QString& n, const QString& v, const QString& c )
  : Name(n), Val(v), Cmnt( c )
{
}

/////////////////////////////////////////////////////////////////////////////
// UserTbl - User Table

//
// Constructor - ( DataType, User, Comment, Variables )
UserTbl::UserTbl( DATATYPE t, const QString& u, const QString& h,
                QPtrList<VarTbl>* v )
  : Flag(0), dType(t), User(u), Head(h), Vars(v), itemCnt(0), exeCnt(0)
{
}

//
// Destructor
UserTbl::~UserTbl()
{
//cout << "UserTbl::~UserTbl()" << endl;
  Vars->clear();
  delete Vars;
}

/////////////////////////////////////////////////////////////////////////////
// CronTbl - 'cron' Table

//
// Constructor - ( User, crontab Command Line, Comment, User Table )
CronTbl::CronTbl( const QString& u, const QString& s, const QString& c,
           UserTbl* e )
  : Flag(0), User(u), Cmnt( c ), userTbl( e ), exeCnt(0)
{

  int n;
  QRegExp sep( "\\s+" );
  // Time
  if ( s[0] == '@' ) n=0;
  else               n=4;
  Time = s.section( sep, 0, n );
#ifdef CRON_DILLON
    Cmnd = s.section( sep, n+1 );
#else
  if ( User == "" ){
    // System 'crontab'
    //  ex: '0 0 * * * root echo "Midnight"'
    cronType = CRON_ETC;
    User = s.section( sep, n+1, n+1 );
    Cmnd = s.section( sep, n+2 );
  }else{
    // User 'crontab'
    //  ex: '0 0 * * * echo "Midnight"'
    cronType = CRON_USER;
    Cmnd = s.section( sep, n+1 );
  }
#endif
  userTbl->itemCnt++;
}

CronTbl::~CronTbl()
{
//cout << "CronTbl::~CronTbl()" << endl;
}

/////////////////////////////////////////////////////////////////////////////
// AnacTbl - 'anacron' Table

//
// Constructor - ( anacrontab Command Line, User Table )
AnacTbl::AnacTbl( const QString& s, const QString& c,
              UserTbl* e, bool newFlag )
  : Flag(0), Cmnt( c ), userTbl( e ), exeCnt(0)
{
  QRegExp sep( "\\s+" );

  // Period ( Day )
  Prod = ( s.section( sep, 0, 0 ) ).toInt();
  // Delay ( Minute )
  Dlay = ( s.section( sep, 1, 1 ) ).toInt();
  // Identifier
  Idnt = s.section( sep, 2, 2 );
  // Command
  Cmnd = s.section( sep, 3 );

  // Get Last Execute Time
  QString f( g_Option->anacPathSpool + "/" + Idnt );
  QFileInfo info( f );
  if ( ! newFlag && info.exists() ){
    // Executed
    Mtim = info.lastModified();
    Mtim.setTime( QTime( Mtim.time().hour(), Mtim.time().minute() ) );
  }else{
    // Not Executed
    int p = Prod;
    if ( p == 0 ) p = 2;
    Mtim = g_CurTime->addDays( -p );
    Mtim.setTime( g_Option->ponTime.addSecs(Dlay*60) );
  }
  userTbl->itemCnt++;

}

/////////////////////////////////////////////////////////////////////////////
// AtTbl - 'at' Table

//
// Constructor
AtTbl::AtTbl( const QString& s )
  : Flag(0), exeCnt(0)
{
  QRegExp sep( "\\s+" );

  // Job No
  Jno = ( s.section( sep, 0, 0 ) ).toInt();

  // Execute Time
  Time = QDateTime::fromString( s.section( sep, 1, 2 ), Qt::ISODate );

  // Queue No
  const QString qn(s.section( sep, 3, 3 ));
  Qno = qn[0];

  // User
  User = s.section( sep, 4, 4 );
}
#ifndef CRON_DILLON

/////////////////////////////////////////////////////////////////////////////
// LogTbl - Log Table

//
// Constructor
LogTbl::LogTbl( const QString &str, DATATYPE dty, const QString &prog, const QDate& mt )
{

  QRegExp reg( "\\s+" );
  // Get ymd
  int y = mt.year();
  int m = cnvMonth( str.section( reg, 0, 0 ) );
  int d = (str.section( reg, 1, 1 )).toInt();
  if ( QDate(y,m,d) > mt ) y = mt.year() - 1;
  // Get hms
  QString s = str.section( reg, 2, 2 );
  int h = (s.section( ':', 0, 0 )).toInt();
  int mm = (s.section( ':', 1, 1 )).toInt();
  int ss = (s.section( ':', 2, 2 )).toInt();
  Time = QDateTime( QDate(y,m,d), QTime(h,mm,ss) );

  // Host Name
  Host = str.section( reg, 3, 3 );
  Prog = prog;
  
  // Get Log Type
  s = str.section( reg, 4 );
  dType = dty;
  if ( dType == CRON_LOG ){
    User = QString(s.section( QRegExp( "[\\(\\)]"), 1, 1 ) );
    LogStr = s.section( ") ", 1 );
  }
  else if ( dType == ANACRON_LOG ) {
    User = "-";
    LogStr = s.section( "]: ", 1, 1 );
  }
  else {        // if ( dType == AT_LOG ){
    LogStr = s.section( ": ", 1, 1 );
    User = "-";
  }

  if ( User == "*system*" ) User = "system";

  // Pid
  Pid = QString(s.section( QRegExp( "[\\[\\]]"), 1, 1 ) ).toInt();

  // Description
//  LogStr = s.section( QRegExp("\\]: "), 1, 1 );

}

// convert Month
int LogTbl::cnvMonth( const QString &s )
{
const char* MonthName[] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun",
                             "Jul", "Aug", "Sep", "Oct", "Nov", "Dec", NULL};
  int i;
  for( i=0; MonthName[i] != NULL; i++) if ( s == MonthName[i] ) break;
  if ( MonthName[i] != 0 ) i++;
  else i=0;

  return i;
}

//
// Destructor
LogTbl::~LogTbl()
{
//cout << "LogTbl::~LogTbl()" << endl;  
}
#endif

/////////////////////////////////////////////////////////////////////////////
// BitLL - 'cron' Data Bit

//
// from 'cron' Time String
//  ex: "*" , "1,2,4" , . . .
void BitLL::fromString( const QString& str, bool shift )
{

  if ( str == "*" ){
    fill( true );
    return;
  }

  // All Bit Off
  fill( false );

  QString s1;
  int p1, p2;

  for( int i=0; ( s1 = str.section( ',', i, i ) ) != "" ; i++ ){
    p1 = s1.find( '-' );
    if ( p1 == 0 ){     // skip first char '-'
      s1 = s1.mid(1);
      p1 = -1;
    }
    p2 = s1.find( '/' );
    int d = 1;
    if ( p2 != -1 ) d = ( s1.mid( p2+1 ) ).toInt();
    int f, t;
    if ( p1 == -1 ) {
      if ( s1.find( "*/" ) == -1){
        f = t = s1.toInt();
      }else{
        t = 59;
        if ( shift ) f = 1;
        else f = 0;
      }
    }else{
      f = ( s1.section( '-', 0, 0 ) ).toInt();
      t = ( s1.section( '-', 1, 1 ) ).toInt();
    }
    if ( f > t ) f = t;
    for( int j=f; j<=t; j+=d ) bSet( j );
  }
  if ( shift ) {
    if ( bTest(0) ){
      bitLL = bitLL >> 1LL;
      bSet(0);
    }else{
      bitLL = bitLL >> 1LL;
    }
  }

}

//
// Convert data to String
void BitLL::addString( QString* str, int base, int intv, int num, bool shift )
{
  if ( *str != "" ) *str = *str + ",";

  if ( shift ) base++;
  if ( num == 1 ){
    *str = *str + QString().sprintf( "%d", base );
  }else if ( num == 2 ){
    *str = *str + QString().sprintf( "%d,%d", base, base+intv+1 );
  }else{
    if ( intv == 0 ){
      *str = *str + QString().sprintf( "%d-%d", base, base+num-1 );
    }else{
      *str = *str + QString().sprintf( "%d-%d/%d", base, base+(intv+1)*(num-1), intv+1 );
    }
  }
}

//
// Convert to 'cron' TIme
QString BitLL::toString( bool shift )
{
  QString s;
  if ( isFill( true ) || isFill( false ) ){
    s = "*";
  }else{
    s = "";
    int bp = -1;    // Base Number
    int n = 0;      // Data Number
    int v = 0;      // Interval
    int c = 0;      // Space Count
    for( int i=0; i<60; i++){
      if ( ! bNext( i ) )break;
      if ( bTest(i) ){
        // Bit ON
        if ( bp == -1 ) {   // reset
          bp = i;
          v = 0;
          c = 0;
          n = 1;
        }else{
          if ( v == c ){    // Space = Interval
            n++;
          }else{            // Space != Interval
            // Set String
            if ( n < 3 ){
              n--;
              addString( &s, bp, v, n, shift );
              bp = i - c - 1;
              v = c;
              n = 2;
            }else{
              addString( &s, bp, v, n, shift );
              bp = i;
              v = 0;
              c = 0;
              n = 1;
            }
          }
        }
        c = 0;
      }else{
        // BIT OFF
        if ( bp != -1 ){
          c++;
          if ( n == 1 ){
            v = c;
          }else{
            if ( c > v ){     // Spece > Interval
              // Set String
              addString( &s, bp, v, n, shift );
              bp = -1;
              v = 0;
              c = 0;
              n = 0;
            }
          }
        }
      }
    }
    if ( bp != -1 ){
      addString( &s, bp, v, n, shift );
    }
  }
  return s;
}


/////////////////////////////////////////////////////////////////////////////
// CronBit - 'cron' Data Bit

//
// Conver Week
QString CronBit::convWeek( const QString& str, int flg )
{
  const char *WeekNames[] =
  { "sun", "mon", "tue", "wed", "thu", "fri", "sat", "sun", NULL };

  QString s = str.lower();
  if ( flg == 0 && s.contains( ',' ) == 0 && s.contains( '-' ) == 0 ){  // num -> literal
    QRegExp rx( "\\d+" );
    int d;
    int p=0;
    int l;
    while( true ){
      p = rx.search( s, p );
      if ( p == -1 ) break;
      l = rx.matchedLength();
      d = ( s.mid( p, l ) ).toInt();
      if ( l > 7 || ( p != 0 && s[p-1] == '/' ) ){
        p = p + l;
      }else{
        QString w = WeekNames[d];
        w[0] = w[0].upper();
        s.replace( p, l, w );
        p = p + 3;
      }
    }
  }else{            // literal -> num
    for( int i=0; i<7; i++ ){
      s = s.replace( QRegExp( WeekNames[i] ), QString().setNum(i) );
    }
  }
  return s;
}

// Conver Month
QString CronBit::convMonth( const QString& str, int flg )
{
  const char *MonthNames[] =
  { "jan", "feb", "mar", "apr", "may", "jun",
    "jul", "aug", "sep", "oct", "nov", "dec", NULL };

  QString s = str.lower();
  if ( flg == 0  && s.contains( ',' ) == 0 && s.contains( '-' ) == 0  ){  // num -> literal
    QRegExp rx( "\\d+" );
    int d;
    int p=0;
    int l;
    while( true ){
      p = rx.search( s, p );
      if ( p == -1 ) break;
      l = rx.matchedLength();
      d = ( s.mid( p, l ) ).toInt();
      if ( d < 1 || d > 12 || ( p != 0 && s[p-1] == '/' ) ){
        p = p + l;
      }else{
        QString m = MonthNames[d-1];
        m[0] = m[0].upper();
        s.replace( p, l, m );
        p = p + 3;
      }
    }
  }else{            // literal -> num
    for( int i=0; i<12; i++ ){
      s = s.replace( QRegExp( MonthNames[i] ), QString().setNum(i+1) );
    }
  }
  return s;
}

//
// Initialize from 'cron' String
void CronBit::fromString( const QString& str )
{

  if ( str[0] == '@' ){
    QString s = (str.lower()).mid(1);
    Min.fill( false );
    Hour.fill( true );
    Week.fill( false );
    Day.fill( true );
    Month.fill( true );
    if ( s == "hourly" ){
      Min.bSet( 0 );
    } else if ( s == "daily" || s == "midnight" ){
      Min.bSet( 0 );
      Hour.fill( false );
      Hour.bSet( 0 );
    } else if ( s == "weekly" ){
      Min.bSet( 0 );
      Hour.fill( false );
      Hour.bSet( 0 );
      Week.bSet( 0 );
      Day.fill( false );
    } else if ( s == "monthly" ){
      Min.bSet( 0 );
      Hour.fill( false );
      Hour.bSet( 0 );
      Day.fill( false );
      Day.bSet( 0 );
    } else if ( s == "yearly" || s == "annually" ){
      Min.bSet( 0 );
      Hour.fill( false );
      Hour.bSet( 0 );
      Day.fill( false );
      Day.bSet( 0 );
      Month.fill( false );
      Month.bSet( 0 );
    } else if ( s == "reboot" ){
      if ( g_Option->ponWeekDay ){
        Week.fromString("1-5", false );
        Day.fill( false );
      }
      Hour.fill( false );
      Hour.bSet( g_Option->ponTime.hour() );
      Min.bSet( g_Option->ponTime.minute() );
    }
  }else{
    QStringList sl = QStringList::split( QRegExp( "\\s+" ), str );

    Min.fromString( sl[0] );
    Hour.fromString( sl[1] );
    Week.fromString( convWeek( sl[4], 1 ) );
    if ( Week.isFill( true ) ) Week.fill( false );
    Day.fromString( sl[2], true );
    if ( Day.isFill( true ) && ! Week.isFill( false ) ) Day.fill( false );
    Month.fromString( convMonth( sl[3], 1 ), true );
    if ( Week.bTest( 7 ) ) Week.bSet( 0 );
  }
  return;

}

//
// To 'cron' Time String
QString CronBit::toString()
{
  QString s="";
#ifndef CRON_DILLON
  if ( Min.is0() ){
    if ( Hour.is0() ){
      if ( Day.is0() && Week.isFill(false) ){
        if (  Month.is0() ) s = "@yearly";
        else if ( Month.isFill(true) ) s = "@monthly";
      }else if ( Month.isFill(true) ){
        if ( Day.isFill(true) && Week.is0() ) s = "@weekly";
        else if ( Day.isFill(true) && Week.isFill(false) ) s = "@daily";
      }
    }else if ( Hour.isFill(true) && Day.isFill(true) && Week.isFill(false) && Month.isFill(true) ){
      s = "@hourly";
    }
  }
#endif
  if ( s == "" ){
    QString w = Week.toString();
    QString m = Month.toString(true);
    if ( g_Option->cronLiteral ){
      w = convWeek( w, 0 );
      m = convMonth( m, 0 );
    }
    s = Min.toString() + " " + Hour.toString() + " " +
        Day.toString(true) + " " +  m + " " + w;
  }
  return s;
}

//
// Increment Time
void CronBit::UpTime ( UPTIME u, QDateTime *t )
{
  QDate dt;
  QTime tm;
  switch( u ){
    case UP_MIN :
      tm.setHMS( t->time().hour(), t->time().minute(), 0 );
      t->setTime( tm );
      *t = t->addSecs( 60 );
      break;
    case UP_HOUR :
      tm.setHMS( t->time().hour(), 0, 0 );
      t->setTime( tm );
      *t = t->addSecs( 60 * 60 );
      break;
    case UP_DAY :
      tm.setHMS( 0, 0, 0 );
      t->setTime( tm );
      *t = t->addDays(1);
      break;
    case UP_MON :
      tm.setHMS( 0, 0, 0 );
      dt.setYMD( t->date().year(), t->date().month(), 1 );
      t->setTime( tm );
      t->setDate( dt );
      *t = t->addMonths(1);
      break;
    default :
      break;
    }
}

//
// Get Next Time
QDateTime CronBit::GetNextTime( const QDateTime& ckt )
{

  QDateTime tm = ckt;
  UpTime( UP_MIN, &tm );
  while( true ){
      if ( ! Month.bTest( tm.date().month()-1 ) ){
        UpTime( UP_MON, &tm );
        continue;
      }
      if ( ! Day.bTest( tm.date().day()-1 ) ){
        int wk = tm.date().dayOfWeek();
        if ( wk == 7 ) wk = 0;
        if ( ! Week.bTest( wk ) ){
          UpTime( UP_DAY, &tm );
          continue;
        }
      }
      if ( ! Hour.bTest( tm.time().hour() ) ){
        if ( Hour.bNext( tm.time().hour() ) ){
          UpTime( UP_HOUR, &tm );
        }else{
          UpTime( UP_DAY, &tm );
        }
        continue;
      }
      if ( Min.bTest( tm.time().minute() ) ) {
        break;
      }else{
        if ( Min.bNext( tm.time().minute() ) ){
          UpTime( UP_MIN, &tm );
        }else{
          UpTime( UP_HOUR, &tm );
        }
      }
  }
  return tm;
}


/////////////////////////////////////////////////////////////////////////////
// CronTime - 'cron' Time

//
// Constructor - cron
CronTime::CronTime( CronTbl *cr)
      : dType(CRON), TblPtr((void*)cr)
{
  cronBit.fromString( cr->Time );
}

//
// Constructor - anacron
CronTime::CronTime( AnacTbl *an)
      : dType(ANACRON), TblPtr((void*)an)
{}

//
// Constructor - at
CronTime::CronTime( AtTbl *at)
      : dType(AT), TblPtr((void*)at)
{}

//
// Get Next Time
QDateTime CronTime::GetNextTime( const QDateTime& ckt )
{
  QDateTime ret;
  if ( dType == CRON ){
    ret = cronBit.GetNextTime( ckt );
  }else if ( dType == ANACRON ){
    AnacTbl* an = anacTbl();
    ret = an->Mtim;
    ret.setTime( g_Option->ponTime.addSecs( an->Dlay*60 ) );
    int p = an->Prod;
    if ( p == 0 ) p = 1;
    while( TRUE ){
      ret = ret.addDays( p );
      if ( g_Option->ponWeekDay ){
        int w = ret.date().dayOfWeek();
        if ( w == 6 ) ret=ret.addDays( 2 );
        else if ( w == 7 ) ret=ret.addDays( 1 );
      }
      if ( ret > ckt ) break;
    }
  }else if ( dType == AT ){
    ret = atTbl()->Time;
  }

  return ret;
}

