/*******************************************************************************
 * Copyright (c) 2012 cnfree.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  cnfree  - initial API and implementation
 *******************************************************************************/

package org.sf.feeling.decompiler.editor;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.jdt.core.IBuffer;
import org.eclipse.jdt.core.IClassFile;
import org.eclipse.jdt.internal.core.BufferManager;
import org.eclipse.jdt.internal.core.ClassFile;
import org.eclipse.jdt.internal.ui.javaeditor.ClassFileEditor;
import org.eclipse.jdt.internal.ui.javaeditor.IClassFileEditorInput;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.ui.IEditorInput;
import org.sf.feeling.decompiler.JavaDecompilerPlugin;
import org.sf.feeling.decompiler.util.UIUtil;

public class JavaDecompilerClassFileEditor extends ClassFileEditor
{

	public static final String ID = "net.sf.feeling.decompiler.ClassFileEditor"; //$NON-NLS-1$
	public static final String MARK = "/*** Eclipse Class Decompiler plugin, copyright (c) 2012 Chao Chen (cnfree2000@hotmail.com) ***/"; //$NON-NLS-1$

	public JavaDecompilerClassFileEditor( )
	{
		super( );
	}

	private boolean doOpenBuffer( IEditorInput input, boolean force )
	{
		IPreferenceStore prefs = JavaDecompilerPlugin.getDefault( )
				.getPreferenceStore( );
		String decompilerType = prefs.getString( JavaDecompilerPlugin.DECOMPILER_TYPE );
		return doOpenBuffer( input, decompilerType, force );
	}

	private boolean doOpenBuffer( IEditorInput input, String type, boolean force )
	{
		IPreferenceStore prefs = JavaDecompilerPlugin.getDefault( )
				.getPreferenceStore( );
		boolean reuseBuf = prefs.getBoolean( JavaDecompilerPlugin.REUSE_BUFFER );
		boolean always = prefs.getBoolean( JavaDecompilerPlugin.IGNORE_EXISTING );
		return doOpenBuffer( input, type, force, reuseBuf, always );
	}

	private boolean doOpenBuffer( IEditorInput input, String type,
			boolean force, boolean reuseBuf, boolean always )
	{
		if ( input instanceof IClassFileEditorInput )
		{
			try
			{
				boolean opened = false;
				IClassFile cf = ( (IClassFileEditorInput) input ).getClassFile( );

				String decompilerType = type;
				String origSrc = cf.getSource( );
				// have to check our mark since all line comments are stripped
				// in debug align mode
				if ( origSrc == null
						|| always
						&& !origSrc.startsWith( MARK )
						|| ( origSrc.startsWith( MARK ) && ( !reuseBuf || force ) ) )
				{
					DecompilerSourceMapper sourceMapper = SourceMapperFactory.getSourceMapper( decompilerType );
					char[] src = sourceMapper.findSource( cf.getType( ) );
					if ( src == null )
					{
						if ( DecompilerType.JAD.equals( decompilerType ) )
						{
							src = SourceMapperFactory.getSourceMapper( DecompilerType.JDCORE )
									.findSource( cf.getType( ) );
						}
						else if ( DecompilerType.JDCORE.equals( decompilerType )
								&& UIUtil.isWin32( ) )
						{
							src = SourceMapperFactory.getSourceMapper( DecompilerType.JAD )
									.findSource( cf.getType( ) );
						}
					}
					if ( src == null )
					{
						return false;
					}
					char[] markedSrc = src;
					IBuffer buffer = BufferManager.createBuffer( cf );
					buffer.setContents( markedSrc );
					getBufferManager( ).addBuffer( buffer );
					// buffer.addBufferChangedListener((IBufferChangedListener)cf);
					sourceMapper.mapSource( cf.getType( ), markedSrc, true );

					ClassFileSourceMap.updateSource( getBufferManager( ),
							(ClassFile) cf,
							markedSrc );

					opened = true;
				}
				return opened;
			}
			catch ( Exception e )
			{
				JavaDecompilerPlugin.logError( e, "" ); //$NON-NLS-1$
			}
		}
		return false;
	}

	/**
	 * Sets edditor input only if buffer was actually opened.
	 * 
	 * @param force
	 *            if <code>true</code> initialize no matter what
	 */
	public void doSetInput( boolean force )
	{
		IEditorInput input = getEditorInput( );
		if ( doOpenBuffer( input, force ) )
		{
			try
			{
				super.doSetInput( input );
			}
			catch ( Exception e )
			{
				JavaDecompilerPlugin.logError( e, "" ); //$NON-NLS-1$
			}
		}
	}

	public void doSetInput( String type, boolean force )
	{
		IEditorInput input = getEditorInput( );
		if ( doOpenBuffer( input, type, force ) )
		{
			try
			{
				super.doSetInput( input );
			}
			catch ( Exception e )
			{
				JavaDecompilerPlugin.logError( e, "" ); //$NON-NLS-1$
			}
		}
	}

	protected void doSetInput( IEditorInput input ) throws CoreException
	{
		doOpenBuffer( input, false );
		super.doSetInput( input );
	}

	protected JavaDecompilerBufferManager getBufferManager( )
	{
		JavaDecompilerBufferManager manager;
		BufferManager defManager = BufferManager.getDefaultBufferManager( );
		if ( defManager instanceof JavaDecompilerBufferManager )
			manager = (JavaDecompilerBufferManager) defManager;
		else
			manager = new JavaDecompilerBufferManager( defManager );
		return manager;
	}

}