//#if _MSC_VER >= 1400
#define _CRT_SECURE_NO_DEPRECATE
//#endif

#include <windows.h>
#include <io.h>
#include <stdio.h>

#define BEEP_DEBUG 0

#define VERSION_STRING "beep-1.2.2-win32/64"
// char *copyright = 
// "Copyright (C) JOJO, 2015.\n"
// "Copyright (C) Johnathan Nightingale, 2002.\n"
// "Use and Distribution subject to GPL.\n"
// "For information: http://www.gnu.org/copyleft/.";

#pragma warning(push)
#pragma warning(disable:4996)
static inline void stdbeep() { write(0, "\a", 1); }
#pragma warning(pop)

typedef void	(__stdcall *lpOut32)(short, short);
typedef short	(__stdcall *lpInp32)(short);
typedef BOOL	(__stdcall *lpIsInpOutDriverOpen)(void);

#ifndef CLOCK_TICK_RATE
#define CLOCK_TICK_RATE 1193180
#endif

/* Other Constants */
#define NO_END_DELAY    0
#define YES_END_DELAY   1

/* Meaningful Defaults */
#define DEFAULT_FREQ       440.0 /* Middle A */
#define DEFAULT_LENGTH     200   /* milliseconds */
#define DEFAULT_REPS       1
#define DEFAULT_DELAY      100   /* milliseconds */
#define DEFAULT_END_DELAY  NO_END_DELAY

typedef struct beep_parms_t {
	float freq;     /* tone frequency (Hz)      */
	int length;     /* tone length    (ms)      */
	int reps;       /* # of repetitions         */
	int delay;      /* delay between reps  (ms) */
	int end_delay;  /* do we delay after last rep? */
	struct beep_parms_t *next;  /* in case -n/--new is used. */
} beep_parms_t;

/* print usage and exit */
// http://wiki.linuxquestions.org/wiki/Beep
void usage_bail(const char *executable_name, int exitcode = 1) {
	printf("Usage:\n%s [-f freq] [-l length] [-r reps] [-d delay] [-D delay] [-n]\n", executable_name);

	printf("-f <frequency in Hz>\n"
	       "-l <tone length in ms>\n"
	       "-r <repetitions>\n"
	       "-d <delay in ms>\n"
	       "-D <delay in ms> (similar to -d, but delay after last repetition as well)\n"
	       "-h/--help\n"
	       "-v/-V/--version\n"
	       "-n new\n");

	exit(exitcode);
}

// .\beep.exe -f 2000 -l120 -n -f 1000 -l 120

int main(int argc, char* argv[])
{
	beep_parms_t *parms = (beep_parms_t *)malloc(sizeof(beep_parms_t));
	parms->freq       = 0;
	parms->length     = DEFAULT_LENGTH;
	parms->reps       = DEFAULT_REPS;
	parms->delay      = DEFAULT_DELAY;
	parms->end_delay  = DEFAULT_END_DELAY;
	parms->next       = NULL;

	// Parse the command line.
	beep_parms_t *result = parms;
	for (char **argp = argv; *++argp; ) {
		char *arg = *argp;
		if      (strcmp(arg, "--new") == 0) arg = "-n";
		else if (strcmp(arg, "--help") == 0) arg = "-h";
		else if (strcmp(arg, "--version") == 0) arg = "-v";
		if (arg[0] == '-' && arg[1] != '\0' && arg[2] == '\0') {
			int argval = -1;    /* handle parsed numbers for various arguments */
			float argfreq = -1; 
			char *optarg = strchr("flrdD", arg[1]) != NULL ? *++argp : NULL;
			if (optarg == NULL) optarg = "";
			switch (arg[1]) {
			case 'f': /* freq */
				if (!sscanf(optarg, "%f", &argfreq) || argfreq >= 20000 /* ack! */ || argfreq <= 0)
					usage_bail(argv[0]);
				else
					if (result->freq != 0)
						fprintf(stderr, "WARNING: multiple -f values given, only last ""one is used.\n");
				result->freq = argfreq;
				break;
			case 'l' : /* length */
				if (!sscanf(optarg, "%d", &argval) || argval < 0)
					usage_bail(argv[0]);
				else
					result->length = argval;
				break;
			case 'r' : /* repetitions */
				if (!sscanf(optarg, "%d", &argval) || argval < 0)
					usage_bail(argv[0]);
				else
					result->reps = argval;
				break;
			case 'd' : /* delay between reps - WITHOUT delay after last beep*/
				if (!sscanf(optarg, "%d", &argval) || (argval < 0))
					usage_bail(argv[0]);
				else {
					result->delay = argval;
					result->end_delay = NO_END_DELAY;
				}
				break;
			case 'D' : /* delay between reps - WITH delay after last beep */
				if (!sscanf(optarg, "%d", &argval) || (argval < 0))
					usage_bail(argv[0]);
				else {
					result->delay = argval;
					result->end_delay = YES_END_DELAY;
				}
				break;
			case 'v' :
			case 'V' : /* also --version */
				printf("%s\n", VERSION_STRING);
				exit(0);
				break;
			case 'n' : /* also --new - create another beep */
				if (result->freq == 0)
					result->freq = DEFAULT_FREQ;
				result->next = (beep_parms_t *)malloc(sizeof(beep_parms_t));
				result->next->freq       = 0;
				result->next->length     = DEFAULT_LENGTH;
				result->next->reps       = DEFAULT_REPS;
				result->next->delay      = DEFAULT_DELAY;
				result->next->end_delay  = DEFAULT_END_DELAY;
				result->next->next       = NULL;
				result = result->next; /* yes, I meant to do that. */
				break;
			case 'h' : /* notice that this is also --help */
				usage_bail(argv[0], 0);
				break;
			default:
				usage_bail(argv[0]);
				break;
			}
		}
		else {
			usage_bail(argv[0]);
		}
	}
	if (result->freq == 0)
		result->freq = DEFAULT_FREQ;

	// Dynamically load the DLL at runtime (not linked at compile time)
	HINSTANCE hInpOutDll;
	if ((hInpOutDll = LoadLibrary("inpout32.dll")) == NULL
	 && (hInpOutDll = LoadLibrary("inpoutx64.dll")) == NULL)
	{
		stdbeep();
#if BEEP_DEBUG
		fprintf(stderr, "Unable to load inpout32 DLL!\n");
#endif
		return -1;
	}

	const lpOut32 outp = (lpOut32)GetProcAddress(hInpOutDll, "Out32");
	const lpInp32 inp = (lpInp32)GetProcAddress(hInpOutDll, "Inp32");
	const lpIsInpOutDriverOpen gfpIsInpOutDriverOpen = (lpIsInpOutDriverOpen)GetProcAddress(hInpOutDll, "IsInpOutDriverOpen");

	if (gfpIsInpOutDriverOpen())
	{
		while (parms) {
			for (int i = 0; i < parms->reps; i++) {
				// start beep
				const int freq = (int)(CLOCK_TICK_RATE / parms->freq);
				outp(0x43, 0xB6);			// Intel 8254 PIT control register
				outp(0x42, freq);			// Intel 8254 PIT counter #2 (for beep)
				outp(0x42, freq >> 8);		// Intel 8254 PIT counter #2 (for beep)
			//	Sleep(10);
				outp(0x61, inp(0x61) | 0x03);	// beep ON
				// duration[ms]
				Sleep(parms->length);
				// stop beep
				outp(0x61, inp(0x61) & ~0x03);	// beep OFF
				// rest
				if (parms->end_delay || i + 1 < parms->reps)
					Sleep(parms->delay);
			}
			/* Junk each parms struct after playing it */
			beep_parms_t *prev = parms;
			parms = parms->next;
			free(prev);
		}
	}
	else
	{
		stdbeep();
#if BEEP_DEBUG
		fprintf(stderr, "Unable to open inpout32 Driver!\n");
#endif
	}

	// done
	FreeLibrary(hInpOutDll);
	return 0;
}
