﻿// This file is part of Notepad++ project
// Copyright (C)2003 Don HO <don.h@free.fr>
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// Note that the GPL places important restrictions on "derived works", yet
// it does not provide a detailed definition of that term.  To avoid
// misunderstandings, we consider an application to constitute a
// "derivative work" for the purpose of this license if it does any of the
// following:
// 1. Integrates source code from Notepad++.
// 2. Integrates/includes/aggregates Notepad++ into a proprietary executable
//    installer, such as those produced by InstallShield.
// 3. Links to a library or executes a program that does any of the above.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <memory>
#include <shlwapi.h>
#include "Notepad_plus_Window.h"
#include "EncodingMapper.h"
#include "ShortcutMapper.h"
#include "TaskListDlg.h"
#include "clipboardFormats.h"
#include "VerticalFileSwitcher.h"
#include "documentMap.h"
#include "functionListPanel.h"
#include "fileBrowser.h"
#include "Sorters.h"
#include "LongRunningOperation.h"
#include "md5.h"
#ifdef MOD_ENCODING_ANY
#include <MLang.h>
#endif

using namespace std;

void Notepad_plus::macroPlayback(Macro macro)
{
	LongRunningOperation op;

	_playingBackMacro = true;
	_pEditView->execute(SCI_BEGINUNDOACTION);

	for (Macro::iterator step = macro.begin(); step != macro.end(); ++step)
	{
		if (step->isPlayable())
			step->PlayBack(this->_pPublicInterface, _pEditView);
		else
			_findReplaceDlg.execSavedCommand(step->_message, step->_lParameter, step->_sParameter);
	}

	_pEditView->execute(SCI_ENDUNDOACTION);
	_playingBackMacro = false;
}

#ifdef MOD_CHANGE_ENCODING
static void /*Notepad_plus::*/changeEncoding(const HWND hWnd, ScintillaEditView* _pEditView, const UniMode unicodeMode, const int encoding)
{
	Buffer* buf = _pEditView->getCurrentBuffer();
#ifdef INTERNAL_ENCODING_UTF8N
	assert(buf->getUnicodeMode() != uni7Bit);
	assert(buf->getUnicodeMode() != uni8Bit);
	assert(unicodeMode != uni7Bit);
	assert(unicodeMode != uni8Bit);
#endif
	if (buf->getUnicodeMode() == unicodeMode && buf->getEncoding() == encoding)
		{ MessageBeep(0); return; }

	bool shoulBeDirty = !buf->isUntitled() && buf->docLength() > 0;
	if (shoulBeDirty) {
		static const WCHAR *title = L"文字コード変更";
		const WCHAR *caption;
		UINT style;
		if (buf->isDirty()) {
			caption = L"ファイルを読み込み直しますか?\r\rはい：読み込み直す\r現在の変更は失われます\r\rいいえ：このまま変換する";
			style = MB_YESNOCANCEL | MB_APPLMODAL | MB_ICONEXCLAMATION | MB_DEFBUTTON2;
		}
		else {
			caption = L"ファイルを読み込み直しますか?\r\rはい：読み込み直す\r\rいいえ：このまま変換する";
			if (buf->getUnicodeMode() == unicodeMode && encoding != -1)
				style = MB_YESNOCANCEL | MB_APPLMODAL | MB_ICONQUESTION | MB_DEFBUTTON1;
			else
				style = MB_YESNOCANCEL | MB_APPLMODAL | MB_ICONQUESTION | MB_DEFBUTTON2;
		}

		int answer = ::MessageBoxW(hWnd, caption, title, style);
		if (answer == IDYES) {
			_pEditView->saveCurrentPos();
			if (MainFileManager->reloadBuffer(buf, unicodeMode, encoding))
				shoulBeDirty = false;
			_pEditView->restoreCurrentPos();
		}
		else if (answer == IDNO)
			; // do nothing
		else /*if (answer == IDCANCEL)*/
			return;
	}
	buf->setUnicodeMode(unicodeMode);
	buf->setEncoding(encoding);
#ifdef FIX_L_ASCII_ENCODING
	buf->canEncodeAutoChange = false;
#endif
	if (shoulBeDirty) {
		buf->setDirty(true);
		buf->setModifiedStatus(true);
	}
}
#endif



void Notepad_plus::command(int id)
{
	switch (id)
	{
		case IDM_FILE_NEW:
		{
			fileNew();

			/*
			bool isFirstTime = not _pluginsAdminDlg.isCreated();
			_pluginsAdminDlg.setPluginsManager(&_pluginsManager);
			_pluginsAdminDlg.doDialog(_nativeLangSpeaker.isRTL());
			if (isFirstTime)
			{
				_nativeLangSpeaker.changeConfigLang(_pluginsAdminDlg.getHSelf());
				_pluginsAdminDlg.downloadPluginList();
				_pluginsAdminDlg.loadFomList();
			}
			*/
		}
		break;

		case IDM_FILE_OPEN:
		{
			fileOpen();
		}
		break;

		case IDM_FILE_OPEN_FOLDER:
		{
#ifdef MOD_FILE_OPEN_FOLDER
			const WCHAR *fullPathName = _pEditView->getCurrentBuffer()->getFullPathName();
			WCHAR param[MAX_PATH + 10];
			if (!::PathFileExistsW(fullPathName)) {	// "new 1"
				wcscpy_s(param, _countof(param), fullPathName);
				::PathRemoveFileSpecW(param);
			} else {
				wcscpy_s(param, _countof(param), L"/select,");
				wcscat_s(param, _countof(param), fullPathName);
			}
			::ShellExecuteW(_pPublicInterface->getHSelf(), L"open", L"explorer.exe", param, NULL, SW_SHOW);
#else
			Command cmd(TEXT("explorer /select,\"$(FULL_CURRENT_PATH)\""));
			cmd.run(_pPublicInterface->getHSelf());
#endif
		}
		break;

		case IDM_FILE_OPEN_CMD:
#ifdef MOD_FILE_OPEN_POWERSHELL
		case IDM_FILE_OPEN_POWERSHELL:
#endif
		{
#ifdef FIX_FILE_OPEN_CMD
			WCHAR docDir[MAX_PATH];
 #ifdef MOD_FILE_OPEN_POWERSHELL
			wcscpy_s(docDir, _countof(docDir), _pEditView->getCurrentBuffer()->getFullPathName());
			::PathRemoveFileSpecW(docDir);
			if (id == IDM_FILE_OPEN_POWERSHELL)
				RunPowerShell(_pPublicInterface->getHSelf(), docDir);
			else // IDM_FILE_OPEN_CMD
 #endif
				RunCommandPrompt(_pPublicInterface->getHSelf(), docDir);
#else
			Command cmd(TEXT("cmd /K cd /d $(CURRENT_DIRECTORY)"));
			cmd.run(_pPublicInterface->getHSelf());
#endif
		}
		break;
		
		case IDM_FILE_OPEN_DEFAULT_VIEWER:
		{
			// Opens file in its default viewer. 
            // Has the same effect as double–clicking this file in Windows Explorer.
            BufferID buf = _pEditView->getCurrentBufferID();
			HINSTANCE res = ::ShellExecute(NULL, TEXT("open"), buf->getFullPathName(), NULL, NULL, SW_SHOW);

			// As per MSDN (https://msdn.microsoft.com/en-us/library/windows/desktop/bb762153(v=vs.85).aspx)
			// If the function succeeds, it returns a value greater than 32.
			// If the function fails, it returns an error value that indicates the cause of the failure.
			int retResult = reinterpret_cast<int>(res);
			if (retResult <= 32)
			{
				generic_string errorMsg;
				errorMsg += GetLastErrorAsString(retResult);
				errorMsg += TEXT("An attempt was made to execute the below command.");
				errorMsg += TEXT("\n----------------------------------------------------------");
				errorMsg += TEXT("\nCommand: ");
				errorMsg += buf->getFullPathName();
				errorMsg += TEXT("\nError Code: ");
				errorMsg += intToString(retResult);
				errorMsg += TEXT("\n----------------------------------------------------------");
				
				::MessageBox(_pPublicInterface->getHSelf(), errorMsg.c_str(), TEXT("ShellExecute - ERROR"), MB_ICONINFORMATION | MB_APPLMODAL);
			}
		}
		break;

		case IDM_FILE_OPENFOLDERASWORSPACE:
		{
			generic_string folderPath = folderBrowser(_pPublicInterface->getHSelf(), TEXT("Select a folder to add in Folder as Workspace panel"));
			if (not folderPath.empty())
			{
				if (_pFileBrowser == nullptr) // first launch, check in params to open folders
				{
					vector<generic_string> dummy;
					launchFileBrowser(dummy);
					if (_pFileBrowser != nullptr)
					{
						checkMenuItem(IDM_VIEW_FILEBROWSER, true);
						_toolBar.setCheck(IDM_VIEW_FILEBROWSER, true);
						_pFileBrowser->setClosed(false);
					}
					else // problem
						return;
				}
				else
				{
					if (_pFileBrowser->isClosed())
					{
						_pFileBrowser->display();
						checkMenuItem(IDM_VIEW_FILEBROWSER, true);
						_toolBar.setCheck(IDM_VIEW_FILEBROWSER, true);
						_pFileBrowser->setClosed(false);
					}
				}
				_pFileBrowser->addRootFolder(folderPath);
			}
		}
		break;

		case IDM_FILE_RELOAD:
			fileReload();
			break;

		case IDM_FILESWITCHER_FILESCLOSE:
		case IDM_FILESWITCHER_FILESCLOSEOTHERS:
			if (_pFileSwitcherPanel)
			{
				vector<SwitcherFileInfo> files = _pFileSwitcherPanel->getSelectedFiles(id == IDM_FILESWITCHER_FILESCLOSEOTHERS);
				for (size_t i = 0, len = files.size(); i < len; ++i)
				{
					fileClose((BufferID)files[i]._bufID, files[i]._iView);
				}
				if (id == IDM_FILESWITCHER_FILESCLOSEOTHERS)
				{
					// Get current buffer and its view
					_pFileSwitcherPanel->activateItem(_pEditView->getCurrentBufferID(), currentView());
				}
			}
			break;

		case IDM_FILE_CLOSE:
			if (fileClose())
                checkDocState();
			break;

		case IDM_FILE_DELETE:
			if (fileDelete())
                checkDocState();
			break;

		case IDM_FILE_RENAME:
			fileRename();
			break;

		case IDM_FILE_CLOSEALL:
		{
			bool isSnapshotMode = NppParameters::getInstance()->getNppGUI().isSnapshotMode();
			fileCloseAll(isSnapshotMode, false);
            checkDocState();
			break;
		}

		case IDM_FILE_CLOSEALL_BUT_CURRENT :
			fileCloseAllButCurrent();
            checkDocState();
			break;

		case IDM_FILE_CLOSEALL_TOLEFT :
			fileCloseAllToLeft();
			checkDocState();
			break;

		case IDM_FILE_CLOSEALL_TORIGHT :
			fileCloseAllToRight();
			checkDocState();
			break;

		case IDM_FILE_SAVE :
			fileSave();
			break;

		case IDM_FILE_SAVEALL :
			fileSaveAll();
			break;

		case IDM_FILE_SAVEAS :
			fileSaveAs();
			break;

		case IDM_FILE_SAVECOPYAS :
			fileSaveAs(BUFFER_INVALID, true);
			break;

		case IDM_FILE_LOADSESSION:
			fileLoadSession();
			break;

		case IDM_FILE_SAVESESSION:
			fileSaveSession();
			break;

		case IDM_FILE_PRINTNOW :
			filePrint(false);
			break;

		case IDM_FILE_PRINT :
			filePrint(true);
			break;

		case IDM_FILE_EXIT:
			::SendMessage(_pPublicInterface->getHSelf(), WM_CLOSE, 0, 0);
			break;

		case IDM_EDIT_UNDO:
		{
			LongRunningOperation op;
			_pEditView->execute(WM_UNDO);
			checkClipboard();
			checkUndoState();
			break;
		}

		case IDM_EDIT_REDO:
		{
			LongRunningOperation op;
			_pEditView->execute(SCI_REDO);
			checkClipboard();
			checkUndoState();
			break;
		}

		case IDM_EDIT_CUT:
			_pEditView->execute(WM_CUT);
			checkClipboard();
			break;

		case IDM_EDIT_COPY:
			_pEditView->execute(WM_COPY);
			checkClipboard();
			break;

		case IDM_EDIT_COPY_BINARY:
		case IDM_EDIT_CUT_BINARY:
		{
			int textLen = static_cast<int32_t>(_pEditView->execute(SCI_GETSELTEXT, 0, 0)) - 1;
			if (!textLen)
				return;

#ifdef ORZ // memory leak IDM_EDIT_COPY_BINARY / IDM_EDIT_CUT_BINARY / IDM_EDIT_PASTE_BINARY
#else
			char *pBinText = new char[textLen + 1];
			_pEditView->getSelectedText(pBinText, textLen + 1);
#endif

			// Open the clipboard, and empty it.
			if (!OpenClipboard(NULL))
				return;
			EmptyClipboard();

			// Allocate a global memory object for the text.
			HGLOBAL hglbCopy = GlobalAlloc(GMEM_MOVEABLE, (textLen + 1) * sizeof(unsigned char));
			if (hglbCopy == NULL)
			{
				CloseClipboard();
				return;
			}

			// Lock the handle and copy the text to the buffer.
#ifdef ORZ // memory leak IDM_EDIT_COPY_BINARY / IDM_EDIT_CUT_BINARY / IDM_EDIT_PASTE_BINARY
			char *lpucharCopy = (char *)GlobalLock(hglbCopy);
			_pEditView->getSelectedText(lpucharCopy, textLen + 1);
#else
			unsigned char *lpucharCopy = (unsigned char *)GlobalLock(hglbCopy);
			memcpy(lpucharCopy, pBinText, textLen * sizeof(unsigned char));
			lpucharCopy[textLen] = 0;    // null character
#endif

			GlobalUnlock(hglbCopy);

			// Place the handle on the clipboard.
			SetClipboardData(CF_TEXT, hglbCopy);


			// Allocate a global memory object for the text length.
			HGLOBAL hglbLenCopy = GlobalAlloc(GMEM_MOVEABLE, sizeof(unsigned long));
			if (hglbLenCopy == NULL)
			{
				CloseClipboard();
				return;
			}

			// Lock the handle and copy the text to the buffer.
			unsigned long *lpLenCopy = (unsigned long *)GlobalLock(hglbLenCopy);
			*lpLenCopy = textLen;

			GlobalUnlock(hglbLenCopy);

			// Place the handle on the clipboard.
			UINT f = RegisterClipboardFormat(CF_NPPTEXTLEN);
			SetClipboardData(f, hglbLenCopy);

			CloseClipboard();

			if (id == IDM_EDIT_CUT_BINARY)
				_pEditView->execute(SCI_REPLACESEL, 0, reinterpret_cast<LPARAM>(""));
		}
		break;

		case IDM_EDIT_PASTE:
		{
			LongRunningOperation op;
			int eolMode = int(_pEditView->execute(SCI_GETEOLMODE));
			_pEditView->execute(SCI_PASTE);
			_pEditView->execute(SCI_CONVERTEOLS, eolMode);
		}
		break;

		case IDM_EDIT_PASTE_BINARY:
		{
			LongRunningOperation op;
			if (!IsClipboardFormatAvailable(CF_TEXT))
				return;

			if (!OpenClipboard(NULL))
				return;

#ifdef FIX_MACRORECORD
			const auto recordingMacro = _pEditView->execute(SCI_STOPRECORD);	// SCI_REPLACESEL
#endif
			HGLOBAL hglb = GetClipboardData(CF_TEXT);
			if (hglb != NULL)
			{
				char *lpchar = (char *)GlobalLock(hglb);
				if (lpchar != NULL)
				{
					UINT cf_nppTextLen = RegisterClipboardFormat(CF_NPPTEXTLEN);
					if (IsClipboardFormatAvailable(cf_nppTextLen))
					{
						HGLOBAL hglbLen = GetClipboardData(cf_nppTextLen);
						if (hglbLen != NULL)
						{
							unsigned long *lpLen = (unsigned long *)GlobalLock(hglbLen);
							if (lpLen != NULL)
							{
#ifdef MOD_SCI_REPLACESEL_LENGTH
								_pEditView->execute(SCI_REPLACESEL, ~(WPARAM)*lpLen, (LPARAM)lpchar);
#else
								_pEditView->execute(SCI_REPLACESEL, 0, reinterpret_cast<LPARAM>(""));
								_pEditView->execute(SCI_ADDTEXT, *lpLen, reinterpret_cast<LPARAM>(lpchar));
#endif

#ifdef ORZ // memory leak IDM_EDIT_COPY_BINARY / IDM_EDIT_CUT_BINARY / IDM_EDIT_PASTE_BINARY
								GlobalUnlock(hglbLen);
#else
								GlobalUnlock(hglb);
#endif
							}
						}
					}
					else
					{
						_pEditView->execute(SCI_REPLACESEL, 0, reinterpret_cast<LPARAM>(lpchar));
					}
					GlobalUnlock(hglb);
				}
			}
			CloseClipboard();
#ifdef FIX_MACRORECORD
			if (recordingMacro) _pEditView->execute(SCI_STARTRECORD);
#endif

		}
		break;

		case IDM_EDIT_OPENINFOLDER:
		case IDM_EDIT_OPENASFILE:
		{
			if (_pEditView->execute(SCI_GETSELECTIONS) != 1) // Multi-Selection || Column mode || no selection
				return;

			HWND hwnd = _pPublicInterface->getHSelf();
			TCHAR curentWord[CURRENTWORD_MAXLENGTH];
			::SendMessage(hwnd, NPPM_GETFILENAMEATCURSOR, CURRENTWORD_MAXLENGTH, reinterpret_cast<LPARAM>(curentWord));
			
			TCHAR cmd2Exec[CURRENTWORD_MAXLENGTH];
			if (id == IDM_EDIT_OPENINFOLDER)
			{
				lstrcpy(cmd2Exec, TEXT("explorer"));
			}
			else
			{
				::SendMessage(hwnd, NPPM_GETNPPFULLFILEPATH, CURRENTWORD_MAXLENGTH, reinterpret_cast<LPARAM>(cmd2Exec));
			}

			// Full file path
			if (::PathFileExists(curentWord))
			{
				generic_string fullFilePath = id == IDM_EDIT_OPENINFOLDER ? TEXT("/select,") : TEXT("");
				fullFilePath += TEXT("\"");
				fullFilePath += curentWord;
				fullFilePath += TEXT("\"");

				if (id == IDM_EDIT_OPENINFOLDER ||
					(id == IDM_EDIT_OPENASFILE && not ::PathIsDirectory(curentWord)))
					::ShellExecute(hwnd, TEXT("open"), cmd2Exec, fullFilePath.c_str(), TEXT("."), SW_SHOW);
			}
			else // Full file path - need concatenate with current full file path
			{
				TCHAR currentDir[CURRENTWORD_MAXLENGTH];
				::SendMessage(hwnd, NPPM_GETCURRENTDIRECTORY, CURRENTWORD_MAXLENGTH, reinterpret_cast<LPARAM>(currentDir));

				generic_string fullFilePath = id == IDM_EDIT_OPENINFOLDER ? TEXT("/select,") : TEXT("");
				fullFilePath += TEXT("\"");
				fullFilePath += currentDir;
				fullFilePath += TEXT("\\");
				fullFilePath += curentWord;

				if ((id == IDM_EDIT_OPENASFILE &&
					(not::PathFileExists(fullFilePath.c_str() + 1) || ::PathIsDirectory(fullFilePath.c_str() + 1))))
				{
					_nativeLangSpeaker.messageBox("FilePathNotFoundWarning",
						_pPublicInterface->getHSelf(),
						TEXT("The file you're trying to open doesn't exist."),
						TEXT("File Open"),
						MB_OK | MB_APPLMODAL);
					return;
				}
				fullFilePath += TEXT("\"");
				::ShellExecute(hwnd, TEXT("open"), cmd2Exec, fullFilePath.c_str(), TEXT("."), SW_SHOW);
			}
		}
		break;

		case IDM_EDIT_SEARCHONINTERNET:
		{
			if (_pEditView->execute(SCI_GETSELECTIONS) != 1) // Multi-Selection || Column mode || no selection
				return;

			const NppGUI & nppGui = (NppParameters::getInstance())->getNppGUI();
			generic_string url;
			if (nppGui._searchEngineChoice == nppGui.se_custom)
			{
				if (nppGui._searchEngineCustom.empty())
				{
					url = TEXT("https://www.google.com/search?q=$(CURRENT_WORD)");
				}
				else
				{
					url = nppGui._searchEngineCustom.c_str();
				}
			}
			else if (nppGui._searchEngineChoice == nppGui.se_duckDuckGo)
			{
				url = TEXT("https://duckduckgo.com/?q=$(CURRENT_WORD)");
			}
			else if (nppGui._searchEngineChoice == nppGui.se_google)
			{
				url = TEXT("https://www.google.com/search?q=$(CURRENT_WORD)");
			}
			else if (nppGui._searchEngineChoice == nppGui.se_bing)
			{
				url = TEXT("https://www.bing.com/search?q=$(CURRENT_WORD)");
			}
			else if (nppGui._searchEngineChoice == nppGui.se_yahoo)
			{
				url = TEXT("https://search.yahoo.com/search?q=$(CURRENT_WORD)");
			}

			Command cmd(url.c_str());
			cmd.run(_pPublicInterface->getHSelf());	
		}
		break;

		case IDM_EDIT_CHANGESEARCHENGINE:
		{
			command(IDM_SETTING_PREFERECE);
			_preference.showDialogByName(TEXT("SearchEngine"));
		}
		break;

		case IDM_EDIT_PASTE_AS_RTF:
		case IDM_EDIT_PASTE_AS_HTML:
		{
			LongRunningOperation op;
			UINT f = RegisterClipboardFormat(id==IDM_EDIT_PASTE_AS_HTML?CF_HTML:CF_RTF);

			if (!IsClipboardFormatAvailable(f))
				return;

			if (!OpenClipboard(NULL))
				return;

#ifdef FIX_MACRORECORD
			const auto recordingMacro = _pEditView->execute(SCI_STOPRECORD);	// SCI_REPLACESEL
#endif
			HGLOBAL hglb = GetClipboardData(f);
			if (hglb != NULL)
			{
				LPSTR lptstr = (LPSTR)GlobalLock(hglb);
				if (lptstr != NULL)
				{
					// Call the application-defined ReplaceSelection
					// function to insert the text and repaint the
					// window.
					_pEditView->execute(SCI_REPLACESEL, 0, reinterpret_cast<LPARAM>(lptstr));

					GlobalUnlock(hglb);
				}
			}
			CloseClipboard();
#ifdef FIX_MACRORECORD
			if (recordingMacro) _pEditView->execute(SCI_STARTRECORD);
#endif
		}
		break;

		case IDM_EDIT_BEGINENDSELECT:
		{
			::CheckMenuItem(_mainMenuHandle, IDM_EDIT_BEGINENDSELECT, MF_BYCOMMAND | (_pEditView->beginEndSelectedIsStarted() ? MF_UNCHECKED : MF_CHECKED));
			_pEditView->beginOrEndSelect();
		}
		break;

		case IDM_EDIT_SORTLINES_LEXICOGRAPHIC_ASCENDING:
		case IDM_EDIT_SORTLINES_LEXICOGRAPHIC_DESCENDING:
#ifdef MOD_SORTLINES_FILENAME
		case IDM_EDIT_SORTLINES_FILENAME_DESCENDING:
		case IDM_EDIT_SORTLINES_FILENAME_ASCENDING:
#endif
		case IDM_EDIT_SORTLINES_INTEGER_ASCENDING:
		case IDM_EDIT_SORTLINES_INTEGER_DESCENDING:
		case IDM_EDIT_SORTLINES_DECIMALCOMMA_ASCENDING:
		case IDM_EDIT_SORTLINES_DECIMALCOMMA_DESCENDING:
		case IDM_EDIT_SORTLINES_DECIMALDOT_ASCENDING:
		case IDM_EDIT_SORTLINES_DECIMALDOT_DESCENDING:
		{
			LongRunningOperation op;

			size_t fromLine = 0, toLine = 0;
			size_t fromColumn = 0, toColumn = 0;

			bool hasLineSelection = false;
			if (_pEditView->execute(SCI_GETSELECTIONS) > 1)
			{
				if (_pEditView->execute(SCI_SELECTIONISRECTANGLE))
				{
					ColumnModeInfos colInfos = _pEditView->getColumnModeSelectInfo();
					int leftPos = colInfos.begin()->_selLpos;
					int rightPos = colInfos.rbegin()->_selRpos;
					int startPos = min(leftPos, rightPos);
					int endPos = max(leftPos, rightPos);
					fromLine = _pEditView->execute(SCI_LINEFROMPOSITION, startPos);
					toLine = _pEditView->execute(SCI_LINEFROMPOSITION, endPos);
					fromColumn = _pEditView->execute(SCI_GETCOLUMN, leftPos);
					toColumn = _pEditView->execute(SCI_GETCOLUMN, rightPos);
				}
				else
				{
					return;
				}
			}
			else
			{
				auto selStart = _pEditView->execute(SCI_GETSELECTIONSTART);
				auto selEnd = _pEditView->execute(SCI_GETSELECTIONEND);
				hasLineSelection = selStart != selEnd;
				if (hasLineSelection)
				{
					pair<int, int> lineRange = _pEditView->getSelectionLinesRange();
					// One single line selection is not allowed.
					if (lineRange.first == lineRange.second)
					{
						return;
					}
					fromLine = lineRange.first;
					toLine = lineRange.second;
				}
				else
				{
					// No selection.
					fromLine = 0;
					toLine = _pEditView->execute(SCI_GETLINECOUNT) - 1;
				}
			}

			bool isDescending = id == IDM_EDIT_SORTLINES_LEXICOGRAPHIC_DESCENDING ||
#ifdef MOD_SORTLINES_FILENAME
								id == IDM_EDIT_SORTLINES_FILENAME_DESCENDING ||
#endif
								id == IDM_EDIT_SORTLINES_INTEGER_DESCENDING ||
								id == IDM_EDIT_SORTLINES_DECIMALCOMMA_DESCENDING ||
								id == IDM_EDIT_SORTLINES_DECIMALDOT_DESCENDING;

			_pEditView->execute(SCI_BEGINUNDOACTION);
			std::unique_ptr<ISorter> pSorter;
			if (id == IDM_EDIT_SORTLINES_LEXICOGRAPHIC_DESCENDING || id == IDM_EDIT_SORTLINES_LEXICOGRAPHIC_ASCENDING)
			{
				pSorter = std::unique_ptr<ISorter>(new LexicographicSorter(isDescending, fromColumn, toColumn));
			}
#ifdef MOD_SORTLINES_FILENAME
			else if (id == IDM_EDIT_SORTLINES_FILENAME_DESCENDING || id == IDM_EDIT_SORTLINES_FILENAME_ASCENDING)
			{
				pSorter = std::unique_ptr<ISorter>(new FilenameSorter(isDescending, fromColumn, toColumn));
			}
#endif
			else if (id == IDM_EDIT_SORTLINES_INTEGER_DESCENDING || id == IDM_EDIT_SORTLINES_INTEGER_ASCENDING)
			{
				pSorter = std::unique_ptr<ISorter>(new IntegerSorter(isDescending, fromColumn, toColumn));
			}
			else if (id == IDM_EDIT_SORTLINES_DECIMALCOMMA_DESCENDING || id == IDM_EDIT_SORTLINES_DECIMALCOMMA_ASCENDING)
			{
				pSorter = std::unique_ptr<ISorter>(new DecimalCommaSorter(isDescending, fromColumn, toColumn));
			}
			else
			{
				pSorter = std::unique_ptr<ISorter>(new DecimalDotSorter(isDescending, fromColumn, toColumn));
			}
			try
			{
				_pEditView->sortLines(fromLine, toLine, pSorter.get());
			}
#ifdef FIX_BADALLOC_EDIT_SORTLINES // 58E2304F MultiByteToWideChar
			catch (const std::bad_alloc & e)
			{
				::MessageBoxA(_pPublicInterface->getHSelf(), e.what(), "Sorting Error", MB_OK | MB_ICONWARNING | MB_APPLMODAL);
			}
#endif
			catch (size_t& failedLineIndex)
			{
				generic_string lineNo = std::to_wstring(1 + fromLine + failedLineIndex);
				_nativeLangSpeaker.messageBox("SortingError",
					_pPublicInterface->getHSelf(),
					TEXT("Unable to perform numeric sort due to line $STR_REPLACE$."),
					TEXT("Sorting Error"),
					MB_OK | MB_ICONINFORMATION | MB_APPLMODAL,
					0,
					lineNo.c_str()); // We don't use intInfo since it would require casting size_t -> int.
			}
			_pEditView->execute(SCI_ENDUNDOACTION);

			if (hasLineSelection) // there was 1 selection, so we restore it
			{
				auto posStart = _pEditView->execute(SCI_POSITIONFROMLINE, fromLine);
				auto posEnd = _pEditView->execute(SCI_GETLINEENDPOSITION, toLine);
				_pEditView->execute(SCI_SETSELECTIONSTART, posStart);
				_pEditView->execute(SCI_SETSELECTIONEND, posEnd);
			}
		}
		break;

		case IDM_EDIT_BLANKLINEABOVECURRENT:
		{
			_pEditView->insertNewLineAboveCurrentLine();
		}
		break;

		case IDM_EDIT_BLANKLINEBELOWCURRENT:
		{
			_pEditView->insertNewLineBelowCurrentLine();
		}
		break;

		case IDM_EDIT_CHAR_PANEL:
		{
			launchAnsiCharPanel();
		}
		break;

		case IDM_EDIT_CLIPBOARDHISTORY_PANEL:
		{
			launchClipboardHistoryPanel();
		}
		break;

		case IDM_VIEW_FILESWITCHER_PANEL:
		{
#ifdef MOD_VIEW_FILESWITCHER_PANEL // 5963FB8F
			const bool isDocSwitcherShown = _pFileSwitcherPanel != nullptr && _pFileSwitcherPanel->isVisible(); // NPPM_ISDOCSWITCHERSHOWN
			const bool toShow = !isDocSwitcherShown; // toggle
			if (toShow)
				launchFileSwitcherPanel();
			else
				_pFileSwitcherPanel->display(false);
			checkMenuItem(IDM_VIEW_FILESWITCHER_PANEL, toShow);
 // future	_toolBar.setCheck(IDM_VIEW_FILESWITCHER_PANEL, toShow);
#else
			launchFileSwitcherPanel();
#endif
		}
		break;

		case IDM_VIEW_PROJECT_PANEL_1:
		{
			launchProjectPanel(id, &_pProjectPanel_1, 0);
		}
		break;
		case IDM_VIEW_PROJECT_PANEL_2:
		{
			launchProjectPanel(id, &_pProjectPanel_2, 1);
		}
		break;
		case IDM_VIEW_PROJECT_PANEL_3:
		{
			launchProjectPanel(id, &_pProjectPanel_3, 2);
		}
		break;

		case IDM_VIEW_FILEBROWSER:
		{
			if (_pFileBrowser == nullptr) // first launch, check in params to open folders
			{
				NppParameters *pNppParam = NppParameters::getInstance();
				launchFileBrowser(pNppParam->getFileBrowserRoots());
				if (_pFileBrowser != nullptr)
				{
					checkMenuItem(IDM_VIEW_FILEBROWSER, true);
					_toolBar.setCheck(IDM_VIEW_FILEBROWSER, true);
					_pFileBrowser->setClosed(false);
				}
			}
			else
			{
				if (not _pFileBrowser->isClosed())
				{
					_pFileBrowser->display(false);
					_pFileBrowser->setClosed(true);
					checkMenuItem(IDM_VIEW_FILEBROWSER, false);
					_toolBar.setCheck(IDM_VIEW_FILEBROWSER, false);
				}
				else
				{
					vector<generic_string> dummy;
					launchFileBrowser(dummy);
					checkMenuItem(IDM_VIEW_FILEBROWSER, true);
					_toolBar.setCheck(IDM_VIEW_FILEBROWSER, true);
					_pFileBrowser->setClosed(false);
				}
			}
		}
		break;

		case IDM_VIEW_DOC_MAP:
		{
			if (_pDocMap && (not _pDocMap->isClosed()))
			{
				_pDocMap->display(false);
				_pDocMap->vzDlgDisplay(false);
				_pDocMap->setClosed(true);
				checkMenuItem(IDM_VIEW_DOC_MAP, false);
				_toolBar.setCheck(IDM_VIEW_DOC_MAP, false);
			}
			else
			{
				launchDocMap();
				if (_pDocMap)
				{
					checkMenuItem(IDM_VIEW_DOC_MAP, true);
					_toolBar.setCheck(IDM_VIEW_DOC_MAP, true);
					_pDocMap->setClosed(false);
				}
			}
		}
		break;

		case IDM_VIEW_FUNC_LIST:
		{
			if (_pFuncList && (not _pFuncList->isClosed()))
			{
				_pFuncList->display(false);
				_pFuncList->setClosed(true);
				checkMenuItem(IDM_VIEW_FUNC_LIST, false);
				_toolBar.setCheck(IDM_VIEW_FUNC_LIST, false);
			}
			else
			{
				checkMenuItem(IDM_VIEW_FUNC_LIST, true);
				_toolBar.setCheck(IDM_VIEW_FUNC_LIST, true);
				launchFunctionList();
				_pFuncList->setClosed(false);
			}
		}
		break;

		case IDM_VIEW_TAB1:
		case IDM_VIEW_TAB2:
		case IDM_VIEW_TAB3:
		case IDM_VIEW_TAB4:
		case IDM_VIEW_TAB5:
		case IDM_VIEW_TAB6:
		case IDM_VIEW_TAB7:
		case IDM_VIEW_TAB8:
		case IDM_VIEW_TAB9:
		{
			const int index = id - IDM_VIEW_TAB1;
			BufferID buf = _pDocTab->getBufferByIndex(index);
			if(buf == BUFFER_INVALID)
			{
				// No buffer at chosen index, select the very last buffer instead.
				const int last_index = _pDocTab->getItemCount() - 1;
				if(last_index > 0)
					switchToFile(_pDocTab->getBufferByIndex(last_index));
			}
			else
				switchToFile(buf);
		}
		break;

#ifdef MOD_VIEW_TAB_RECENT
		case IDM_VIEW_TAB_RECENT:
		{
			const size_t current_index = /*int32_t*/ _pDocTab->getCurrentTabIndex();
			long max = -1;
			BufferID last_buf = nullptr;
			for (size_t i = 0, len = _pDocTab->nbItem(); i < len; ++i) {
				if (i == current_index) continue;
				BufferID b = _pDocTab->getBufferByIndex(i);
				if (max < b->getRecentTag()) { max = b->getRecentTag(); last_buf = b; }
			}
			if (last_buf != nullptr)
				switchToFile(last_buf);
		}
		break;
#endif
		case IDM_VIEW_TAB_NEXT:
		{
			const int current_index = _pDocTab->getCurrentTabIndex();
			const int last_index = _pDocTab->getItemCount() - 1;
			if(current_index < last_index)
				switchToFile(_pDocTab->getBufferByIndex(current_index + 1));
			else
				switchToFile(_pDocTab->getBufferByIndex(0)); // Loop around.
		}
		break;

		case IDM_VIEW_TAB_PREV:
		{
			const int current_index = _pDocTab->getCurrentTabIndex();
			if(current_index > 0)
				switchToFile(_pDocTab->getBufferByIndex(current_index - 1));
			else
			{
				const int last_index = _pDocTab->getItemCount() - 1;
				switchToFile(_pDocTab->getBufferByIndex(last_index)); // Loop around.
			}
		}
		break;

		case IDM_VIEW_TAB_MOVEFORWARD:
		case IDM_VIEW_TAB_MOVEBACKWARD:
		{
			const int currentTabIndex = _pDocTab->getCurrentTabIndex();
			const int lastTabIndex = _pDocTab->getItemCount() - 1;
			int newTabIndex = currentTabIndex;

			if (id == IDM_VIEW_TAB_MOVEFORWARD)
			{
				if (currentTabIndex >= lastTabIndex)
					return;
				++newTabIndex;
			}
			else
			{
				if (currentTabIndex < 1)
					return;
				--newTabIndex;
			}

			TCITEM tciMove, tciShift;
			tciMove.mask = tciShift.mask = TCIF_IMAGE | TCIF_TEXT | TCIF_PARAM;

			const int strSizeMax = 256;
			TCHAR strMove[strSizeMax];
			TCHAR strShift[strSizeMax];

			tciMove.pszText = strMove;
			tciMove.cchTextMax = strSizeMax;

			tciShift.pszText = strShift;
			tciShift.cchTextMax = strSizeMax;

			::SendMessage(_pDocTab->getHSelf(), TCM_GETITEM, currentTabIndex, reinterpret_cast<LPARAM>(&tciMove));

			::SendMessage(_pDocTab->getHSelf(), TCM_GETITEM, newTabIndex, reinterpret_cast<LPARAM>(&tciShift));
			::SendMessage(_pDocTab->getHSelf(), TCM_SETITEM, currentTabIndex, reinterpret_cast<LPARAM>(&tciShift));

			::SendMessage(_pDocTab->getHSelf(), TCM_SETITEM, newTabIndex, reinterpret_cast<LPARAM>(&tciMove));

			::SendMessage(_pDocTab->getHSelf(), TCM_SETCURSEL, newTabIndex, 0);

			// Notify plugins that the document order has changed
			::SendMessage(_pDocTab->getHParent(), NPPM_INTERNAL_DOCORDERCHANGED, 0, newTabIndex);
		}
		break;

		case IDM_EDIT_DELETE:
			_pEditView->execute(WM_CLEAR);
			break;

		case IDM_MACRO_STARTRECORDINGMACRO:
		case IDM_MACRO_STOPRECORDINGMACRO:
		case IDC_EDIT_TOGGLEMACRORECORDING:
		{
			if (_recordingMacro)
			{
				// STOP !!!
				_mainEditView.execute(SCI_STOPRECORD);
				_subEditView.execute(SCI_STOPRECORD);

				_mainEditView.execute(SCI_SETCURSOR, static_cast<WPARAM>(SC_CURSORNORMAL));
				_subEditView.execute(SCI_SETCURSOR, static_cast<WPARAM>(SC_CURSORNORMAL));

				_recordingMacro = false;
				_runMacroDlg.initMacroList();
			}
			else
			{
				_mainEditView.execute(SCI_SETCURSOR, 9);
				_subEditView.execute(SCI_SETCURSOR, 9);
				_macro.clear();

				// START !!!
				_mainEditView.execute(SCI_STARTRECORD);
				_subEditView.execute(SCI_STARTRECORD);
				_recordingMacro = true;
			}
			checkMacroState();
			break;
		}

		case IDM_MACRO_PLAYBACKRECORDEDMACRO:
			if (!_recordingMacro) // if we're not currently recording, then playback the recorded keystrokes
			{
				macroPlayback(_macro);
			}
			break;

		case IDM_MACRO_RUNMULTIMACRODLG :
		{
			if (!_recordingMacro) // if we're not currently recording, then playback the recorded keystrokes
			{
				bool isFirstTime = !_runMacroDlg.isCreated();
				_runMacroDlg.doDialog(_nativeLangSpeaker.isRTL());

				if (isFirstTime)
				{
					_nativeLangSpeaker.changeDlgLang(_runMacroDlg.getHSelf(), "MultiMacro");
				}
				break;

			}
		}
		break;

		case IDM_MACRO_SAVECURRENTMACRO :
		{
			if (addCurrentMacro())
				_runMacroDlg.initMacroList();
			break;
		}
		case IDM_EDIT_FULLPATHTOCLIP :
		case IDM_EDIT_CURRENTDIRTOCLIP :
		case IDM_EDIT_FILENAMETOCLIP :
		{
			Buffer * buf = _pEditView->getCurrentBuffer();
			if (id == IDM_EDIT_FULLPATHTOCLIP)
			{
				str2Cliboard(buf->getFullPathName());
			}
			else if (id == IDM_EDIT_CURRENTDIRTOCLIP)
			{
				generic_string dir(buf->getFullPathName());
				PathRemoveFileSpec(dir);
				str2Cliboard(dir);
			}
			else if (id == IDM_EDIT_FILENAMETOCLIP)
			{
				str2Cliboard(buf->getFileName());
			}
		}
		break;

		case IDM_SEARCH_FIND :
		case IDM_SEARCH_REPLACE :
		case IDM_SEARCH_MARK :
		{
			const int strSize = FINDREPLACE_MAXLENGTH;
			TCHAR str[strSize];

			bool isFirstTime = !_findReplaceDlg.isCreated();

			DIALOG_TYPE dlgID = FIND_DLG;
			if (id == IDM_SEARCH_REPLACE)
				dlgID = REPLACE_DLG;
			else if (id == IDM_SEARCH_MARK)
				dlgID = MARK_DLG;
			_findReplaceDlg.doDialog(dlgID, _nativeLangSpeaker.isRTL());

#ifdef MOD_FIND_START_POS
			const Sci_Position pos = (Sci_Position) _pEditView->execute(SCI_GETCURRENTPOS, 0, 0);
			// getGenericSelectedText() --> SCI_SETSELECTIONEND
#endif
			_pEditView->getGenericSelectedText(str, strSize);
#ifdef MOD_FIND_START_POS
			if (pos != _pEditView->execute(SCI_GETCURRENTPOS, 0, 0))
				_pEditView->execute(SCI_SETSELECTION, pos, pos);	// restore pos
#endif
			_findReplaceDlg.setSearchText(str);
			setFindReplaceFolderFilter(NULL, NULL);

			if (isFirstTime)
				_nativeLangSpeaker.changeFindReplaceDlgLang(_findReplaceDlg);
#ifdef FIX_FIND_ACCESS_KEY
			if (isFirstTime)
				_findReplaceDlg.doDialog(dlgID, _nativeLangSpeaker.isRTL());
#endif
			break;
		}

		case IDM_SEARCH_FINDINFILES:
		{
			::SendMessage(_pPublicInterface->getHSelf(), NPPM_LAUNCHFINDINFILESDLG, 0, 0);
			break;
		}

		case IDM_SEARCH_FINDINCREMENT :
		{
			const int strSize = FINDREPLACE_MAXLENGTH;
			TCHAR str[strSize];

			_pEditView->getGenericSelectedText(str, strSize, false);
			if (0 != str[0])         // the selected text is not empty, then use it
				_incrementFindDlg.setSearchText(str, _pEditView->getCurrentBuffer()->getUnicodeMode() != uni8Bit);

			_incrementFindDlg.display();
		}
		break;

		case IDM_SEARCH_FINDNEXT :
		case IDM_SEARCH_FINDPREV :
		{
			if (!_findReplaceDlg.isCreated())
				return;

			FindOption op = _findReplaceDlg.getCurrentOptions();
			op._whichDirection = (id == IDM_SEARCH_FINDNEXT?DIR_DOWN:DIR_UP);
			generic_string s = _findReplaceDlg.getText2search();
			FindStatus status = FSNoMessage;
			_findReplaceDlg.processFindNext(s.c_str(), &op, &status);
			if (status == FSEndReached)
				_findReplaceDlg.setStatusbarMessage(TEXT("Find: Found the 1st occurrence from the top. The end of document has been reached."), FSEndReached);
			else if (status == FSTopReached)
				_findReplaceDlg.setStatusbarMessage(TEXT("Find: Found the 1st occurrence from the bottom. The begin of document has been reached."), FSTopReached);
			break;
		}
		break;

        case IDM_SEARCH_SETANDFINDNEXT :
		case IDM_SEARCH_SETANDFINDPREV :
        {
            bool isFirstTime = !_findReplaceDlg.isCreated();
			if (isFirstTime)
				_findReplaceDlg.doDialog(FIND_DLG, _nativeLangSpeaker.isRTL(), false);

			const int strSize = FINDREPLACE_MAXLENGTH;
			TCHAR str[strSize];
			_pEditView->getGenericSelectedText(str, strSize);
			_findReplaceDlg.setSearchText(str);
			_findReplaceDlg._env->_str2Search = str;
			setFindReplaceFolderFilter(NULL, NULL);
			if (isFirstTime)
				_nativeLangSpeaker.changeFindReplaceDlgLang(_findReplaceDlg);
#ifdef FIX_FIND_ACCESS_KEY
			if (isFirstTime)
				_findReplaceDlg.doDialog(FIND_DLG, _nativeLangSpeaker.isRTL(), false);
#endif

			FindOption op = _findReplaceDlg.getCurrentOptions();
			op._whichDirection = (id == IDM_SEARCH_SETANDFINDNEXT?DIR_DOWN:DIR_UP);

			FindStatus status = FSNoMessage;
			_findReplaceDlg.processFindNext(str, &op, &status);
			if (status == FSEndReached)
				_findReplaceDlg.setStatusbarMessage(TEXT("Find: Found the 1st occurrence from the top. The end of document has been reached."), FSEndReached);
			else if (status == FSTopReached)
				_findReplaceDlg.setStatusbarMessage(TEXT("Find: Found the 1st occurrence from the bottom. The begin of document has been reached."), FSTopReached);
			break;
        }

		case IDM_SEARCH_GOTONEXTFOUND:
		{
			_findReplaceDlg.gotoNextFoundResult();
			break;
		}
		case IDM_SEARCH_GOTOPREVFOUND:
		{
			_findReplaceDlg.gotoNextFoundResult(-1);
			break;
		}
		case IDM_FOCUS_ON_FOUND_RESULTS:
		{
			if (GetFocus() == _findReplaceDlg.getHFindResults())
				// focus already on found results, switch to current edit view
				switchEditViewTo(currentView());
			else
				_findReplaceDlg.focusOnFinder();
			break;
		}

		case IDM_SEARCH_VOLATILE_FINDNEXT :
		case IDM_SEARCH_VOLATILE_FINDPREV :
		{
			TCHAR text2Find[MAX_PATH];
			_pEditView->getGenericSelectedText(text2Find, MAX_PATH);

			FindOption op;
			op._isWholeWord = false;
			op._whichDirection = (id == IDM_SEARCH_VOLATILE_FINDNEXT?DIR_DOWN:DIR_UP);

			FindStatus status = FSNoMessage;
			_findReplaceDlg.processFindNext(text2Find, &op, &status);
			if (status == FSEndReached)
				_findReplaceDlg.setStatusbarMessage(TEXT("Find: Found the 1st occurrence from the top. The end of document has been reached."), FSEndReached);
			else if (status == FSTopReached)
				_findReplaceDlg.setStatusbarMessage(TEXT("Find: Found the 1st occurrence from the bottom. The begin of document has been reached."), FSTopReached);

			break;
		}

		case IDM_SEARCH_MARKALLEXT1 :
		case IDM_SEARCH_MARKALLEXT2 :
		case IDM_SEARCH_MARKALLEXT3 :
		case IDM_SEARCH_MARKALLEXT4 :
		case IDM_SEARCH_MARKALLEXT5 :
		{
			int styleID;
			if (id == IDM_SEARCH_MARKALLEXT1)
				styleID = SCE_UNIVERSAL_FOUND_STYLE_EXT1;
			else if (id == IDM_SEARCH_MARKALLEXT2)
				styleID = SCE_UNIVERSAL_FOUND_STYLE_EXT2;
			else if (id == IDM_SEARCH_MARKALLEXT3)
				styleID = SCE_UNIVERSAL_FOUND_STYLE_EXT3;
			else if (id == IDM_SEARCH_MARKALLEXT4)
				styleID = SCE_UNIVERSAL_FOUND_STYLE_EXT4;
			else // (id == IDM_SEARCH_MARKALLEXT5)
				styleID = SCE_UNIVERSAL_FOUND_STYLE_EXT5;

			const int strSize = FINDREPLACE_MAXLENGTH;
			TCHAR text2Find[strSize];
			TCHAR text2Find2[strSize];

			_pEditView->getGenericSelectedText(text2Find, strSize, false);
			_pEditView->getGenericWordOnCaretPos(text2Find2, strSize);

            if (text2Find[0] == '\0')
            {
                _findReplaceDlg.markAll(text2Find2, styleID, true);
            }
			else
			{
				_findReplaceDlg.markAll(text2Find, styleID, lstrlen(text2Find) == lstrlen(text2Find2));
			}
			break;
		}
		case IDM_SEARCH_UNMARKALLEXT1 :
		case IDM_SEARCH_UNMARKALLEXT2 :
		case IDM_SEARCH_UNMARKALLEXT3 :
		case IDM_SEARCH_UNMARKALLEXT4 :
		case IDM_SEARCH_UNMARKALLEXT5 :
		{
			int styleID;
			if (id == IDM_SEARCH_UNMARKALLEXT1)
				styleID = SCE_UNIVERSAL_FOUND_STYLE_EXT1;
			else if (id == IDM_SEARCH_UNMARKALLEXT2)
				styleID = SCE_UNIVERSAL_FOUND_STYLE_EXT2;
			else if (id == IDM_SEARCH_UNMARKALLEXT3)
				styleID = SCE_UNIVERSAL_FOUND_STYLE_EXT3;
			else if (id == IDM_SEARCH_UNMARKALLEXT4)
				styleID = SCE_UNIVERSAL_FOUND_STYLE_EXT4;
			else // (id == IDM_SEARCH_UNMARKALLEXT5)
				styleID = SCE_UNIVERSAL_FOUND_STYLE_EXT5;

			_pEditView->clearIndicator(styleID);
			break;
		}

		case IDM_SEARCH_GONEXTMARKER1 :
		case IDM_SEARCH_GONEXTMARKER2 :
		case IDM_SEARCH_GONEXTMARKER3 :
		case IDM_SEARCH_GONEXTMARKER4 :
		case IDM_SEARCH_GONEXTMARKER5 :
		case IDM_SEARCH_GONEXTMARKER_DEF :
		{
			int styleID;
			if (id == IDM_SEARCH_GONEXTMARKER1)
				styleID = SCE_UNIVERSAL_FOUND_STYLE_EXT1;
			else if (id == IDM_SEARCH_GONEXTMARKER2)
				styleID = SCE_UNIVERSAL_FOUND_STYLE_EXT2;
			else if (id == IDM_SEARCH_GONEXTMARKER3)
				styleID = SCE_UNIVERSAL_FOUND_STYLE_EXT3;
			else if (id == IDM_SEARCH_GONEXTMARKER4)
				styleID = SCE_UNIVERSAL_FOUND_STYLE_EXT4;
			else if (id == IDM_SEARCH_GONEXTMARKER5)
				styleID = SCE_UNIVERSAL_FOUND_STYLE_EXT5;
			else // (id == IDM_SEARCH_GONEXTMARKER_DEF)
				styleID = SCE_UNIVERSAL_FOUND_STYLE;

			goToNextIndicator(styleID);

			break;
		}

		case IDM_SEARCH_GOPREVMARKER1 :
		case IDM_SEARCH_GOPREVMARKER2 :
		case IDM_SEARCH_GOPREVMARKER3 :
		case IDM_SEARCH_GOPREVMARKER4 :
		case IDM_SEARCH_GOPREVMARKER5 :
		case IDM_SEARCH_GOPREVMARKER_DEF :
		{
			int styleID;
			if (id == IDM_SEARCH_GOPREVMARKER1)
				styleID = SCE_UNIVERSAL_FOUND_STYLE_EXT1;
			else if (id == IDM_SEARCH_GOPREVMARKER2)
				styleID = SCE_UNIVERSAL_FOUND_STYLE_EXT2;
			else if (id == IDM_SEARCH_GOPREVMARKER3)
				styleID = SCE_UNIVERSAL_FOUND_STYLE_EXT3;
			else if (id == IDM_SEARCH_GOPREVMARKER4)
				styleID = SCE_UNIVERSAL_FOUND_STYLE_EXT4;
			else if (id == IDM_SEARCH_GOPREVMARKER5)
				styleID = SCE_UNIVERSAL_FOUND_STYLE_EXT5;
			else // (id == IDM_SEARCH_GOPREVMARKER_DEF)
				styleID = SCE_UNIVERSAL_FOUND_STYLE;

			goToPreviousIndicator(styleID);

			break;
		}

		case IDM_SEARCH_CLEARALLMARKS :
		{
			_pEditView->clearIndicator(SCE_UNIVERSAL_FOUND_STYLE_EXT1);
			_pEditView->clearIndicator(SCE_UNIVERSAL_FOUND_STYLE_EXT2);
			_pEditView->clearIndicator(SCE_UNIVERSAL_FOUND_STYLE_EXT3);
			_pEditView->clearIndicator(SCE_UNIVERSAL_FOUND_STYLE_EXT4);
			_pEditView->clearIndicator(SCE_UNIVERSAL_FOUND_STYLE_EXT5);
			break;
		}

        case IDM_SEARCH_GOTOLINE :
		{
			bool isFirstTime = !_goToLineDlg.isCreated();
			_goToLineDlg.doDialog(_nativeLangSpeaker.isRTL());
			if (isFirstTime)
				_nativeLangSpeaker.changeDlgLang(_goToLineDlg.getHSelf(), "GoToLine");
			break;
		}
#ifdef MOD_INDIRECT_JUMP
		case IDM_SEARCH_INDIRECTJUMP:
		{
			const int cl = (int) _pEditView->execute(SCI_LINEFROMPOSITION, _pEditView->execute(SCI_GETCURRENTPOS));
			const Sci_Position startPos = (Sci_Position) _pEditView->execute(SCI_POSITIONFROMLINE, cl);
			const int lineLength = (int) _pEditView->execute(SCI_LINELENGTH, cl);
			const unsigned char * const lineStart = (const unsigned char*) _pEditView->execute(SCI_GETRANGEPOINTER, startPos, lineLength);	//to get characters directly from Scintilla buffer
			const unsigned char * const lineEnd = lineStart + lineLength;

			const unsigned char *bx = lineStart;
			while (bx < lineEnd && *bx <= 0x20) ++bx;	// Skip white space
			const unsigned char * const fileNameStart = bx;
			for ( ; bx < lineEnd; ++bx) {
				const unsigned char ch = *bx;
				if (ch == '*' || ch == '?' || ch == '"' || ch == '<' || ch == '>' || ch == '|' || ch < 0x20)
					break;
				if (ch == ':')
					if (!( bx - fileNameStart == 1 && isalpha(*fileNameStart) )) break;
			}
			const unsigned char * /*const*/ fileNameEnd = bx;
			for (const unsigned char *it = fileNameEnd; --it >= fileNameStart; )			// Find (linenumber)
				if (*it == '(' && it + 1 < fileNameEnd && isdigit(*(it + 1)))
					{ fileNameEnd = bx = it; break; }
			while (fileNameEnd - 1 >= fileNameStart && *(fileNameEnd - 1) == 0x20) --fileNameEnd;	// Trim space
			if (fileNameEnd <= fileNameStart)
				{ ::MessageBeep(0xFFFFFFFF); break; /*switch*/ }

			std::string fileNameA(fileNameStart, fileNameEnd);
			WcharMbcsConvertor *wmc = WcharMbcsConvertor::getInstance();
			std::wstring fileNameW(wmc->char2wchar(fileNameA.c_str(), SC_CP_UTF8));

			if (bx < lineEnd /* && *bx > 0x20 */) ++bx;	// Skip '(' or other separator
			while (bx < lineEnd && *bx <= 0x20) ++bx;	// Skip white space
			int lineNumber = bx < lineEnd ? atoi((const /*unsigned*/ char*)bx) : 0;

			WCHAR nppDir[MAX_PATH];
			::GetCurrentDirectory(_countof(nppDir), nppDir);	// PUSHD
			WCHAR docDir[MAX_PATH];
			::SendMessage(_pPublicInterface->getHSelf(), NPPM_GETCURRENTDIRECTORY, _countof(docDir), (LPARAM)docDir);
			SetCurrentDirectoryW(docDir);	// CHDIR document directory
			BufferID target = !PathFileExistsW(fileNameW.c_str()) || PathIsDirectoryW(fileNameW.c_str()) ? BUFFER_INVALID : doOpen(fileNameW.c_str());
			::SetCurrentDirectory(nppDir);						// POPD
			if (target == BUFFER_INVALID)
				{ ::MessageBeep(0xFFFFFFFF); break; /*switch*/ }
			if (!switchToFile(target))
				{ ::MessageBeep(0xFFFFFFFF); break; /*switch*/ }

			if (lineNumber > 0) {
				const auto recordingMacro = _pEditView->execute(SCI_STOPRECORD);	//SCI_GOTOLINE
				_pEditView->execute(SCI_ENSUREVISIBLEENFORCEPOLICY, lineNumber - 1);
				_pEditView->execute(SCI_GOTOLINE, lineNumber - 1);
				if (recordingMacro) _pEditView->execute(SCI_STARTRECORD);
			}
			break; // switch
		}
#endif

		case IDM_SEARCH_FINDCHARINRANGE :
		{
			bool isFirstTime = !_findCharsInRangeDlg.isCreated();
			_findCharsInRangeDlg.doDialog(_nativeLangSpeaker.isRTL());
			if (isFirstTime)
				_nativeLangSpeaker.changeDlgLang(_findCharsInRangeDlg.getHSelf(), "FindCharsInRange");
			break;
		}

        case IDM_EDIT_COLUMNMODETIP :
		{
			_nativeLangSpeaker.messageBox("ColumnModeTip",
					_pPublicInterface->getHSelf(),
					TEXT("Please use \"ALT+Mouse Selection\" or \"Alt+Shift+Arrow key\" to switch to column mode."),
					TEXT("Column Mode Tip"),
					MB_OK|MB_APPLMODAL);
			break;
		}

        case IDM_EDIT_COLUMNMODE :
		{
			bool isFirstTime = !_colEditorDlg.isCreated();
			_colEditorDlg.doDialog(_nativeLangSpeaker.isRTL());
			if (isFirstTime)
				_nativeLangSpeaker.changeDlgLang(_colEditorDlg.getHSelf(), "ColumnEditor");
			break;
		}

		case IDM_SEARCH_GOTOMATCHINGBRACE :
		case IDM_SEARCH_SELECTMATCHINGBRACES :
		{
#ifdef FIX_MACRORECORD
			const auto recordingMacro = _pEditView->execute(SCI_STOPRECORD);	// SCI_GOTOPOS
#endif
			int braceAtCaret = -1;
			int braceOpposite = -1;
			findMatchingBracePos(braceAtCaret, braceOpposite);

			if (braceOpposite != -1)
			{
				if(id == IDM_SEARCH_GOTOMATCHINGBRACE)
					_pEditView->execute(SCI_GOTOPOS, braceOpposite);
				else
					_pEditView->execute(SCI_SETSEL, min(braceAtCaret, braceOpposite), max(braceAtCaret, braceOpposite) + 1); // + 1 so we always include the ending brace in the selection.
			}
#ifdef FIX_MACRORECORD
			if (recordingMacro) _pEditView->execute(SCI_STARTRECORD);
#endif
			break;
		}
#ifdef MOD_SEARCH_GOTOBRACKET
		case IDM_SEARCH_GOTOLEFTCURLYBRACKET:
		case IDM_SEARCH_GOTORIGHTCURLYBRACKET:
		{
			// <WordsStyle name="OPERATOR" styleID="10"
			const int styBrace = 10;	// c++/java/perl: 10={}  1=/*{}*/  2=//{}  6="{}"  7='{' '}'
			unsigned char chSeek, chBrace;
			int direction;
			switch (id) {
			case IDM_SEARCH_GOTOLEFTCURLYBRACKET:  chSeek = '{'; chBrace = '}'; direction = -1; break;	// Goto left curly bracket
			case IDM_SEARCH_GOTORIGHTCURLYBRACKET: chSeek = '}'; chBrace = '{'; direction =  1; break;	// Goto right curly bracket
			}
			ScintillaCellBuffer cb(_pEditView);
			const int length = cb.Length();
			const Sci_Position endStyled = (Sci_Position)_pEditView->execute(SCI_GETENDSTYLED);
			Sci_Position position = (Sci_Position) _pEditView->execute(SCI_GETCURRENTPOS);
			int depth = 1;
			while ((position += direction) >= 0 && position < length) {
				unsigned char chAtPos = cb.UCharAt(position);
				/*unsigned char styAtPos = (unsigned char)_pEditView->execute(SCI_GETSTYLEAT, position);*/
				if (chAtPos == chBrace && (position > endStyled || _pEditView->execute(SCI_GETSTYLEAT, position) == styBrace))
					++depth;
				else if (chAtPos == chSeek && (position > endStyled || _pEditView->execute(SCI_GETSTYLEAT, position) == styBrace))
					if (--depth == 0) {
						const auto recordingMacro = _pEditView->execute(SCI_STOPRECORD);	// SCI_GOTOPOS
						auto firstVisibleDisplayLine = _pEditView->execute(SCI_GETFIRSTVISIBLELINE);
						_pEditView->execute(SCI_GOTOPOS, position);
						if (firstVisibleDisplayLine != _pEditView->execute(SCI_GETFIRSTVISIBLELINE))
							if (_pEditView->execute(SCI_LINESONSCREEN) >= 3)
								_pEditView->execute(SCI_LINESCROLL, 0, direction);
						if (recordingMacro) _pEditView->execute(SCI_STARTRECORD);
						break; // while
					}
			}
		}
		break;
#endif

        case IDM_SEARCH_TOGGLE_BOOKMARK :
	        bookmarkToggle(-1);
            break;

	    case IDM_SEARCH_NEXT_BOOKMARK:
		    bookmarkNext(true);
		    break;

	    case IDM_SEARCH_PREV_BOOKMARK:
		    bookmarkNext(false);
		    break;

	    case IDM_SEARCH_CLEAR_BOOKMARKS:
			bookmarkClearAll();
		    break;

        case IDM_LANG_USER_DLG :
        {
		    bool isUDDlgVisible = false;

		    UserDefineDialog *udd = _pEditView->getUserDefineDlg();

		    if (!udd->isCreated())
		    {
			    _pEditView->doUserDefineDlg(true, _nativeLangSpeaker.isRTL());
				_nativeLangSpeaker.changeUserDefineLang(udd);
				if (_isUDDocked)
					::SendMessage(udd->getHSelf(), WM_COMMAND, IDC_DOCK_BUTTON, 0);

		    }
			else
			{
				isUDDlgVisible = udd->isVisible();
				bool isUDDlgDocked = udd->isDocked();

				if ((isUDDlgDocked)&&(isUDDlgVisible))
				{
					::ShowWindow(_pMainSplitter->getHSelf(), SW_HIDE);

					if (bothActive())
						_pMainWindow = &_subSplitter;
					else
						_pMainWindow = _pDocTab;

					::SendMessage(_pPublicInterface->getHSelf(), WM_SIZE, 0, 0);

					udd->display(false);
					_mainWindowStatus &= ~WindowUserActive;
				}
				else if ((isUDDlgDocked)&&(!isUDDlgVisible))
				{
                    if (!_pMainSplitter)
                    {
                        _pMainSplitter = new SplitterContainer;
                        _pMainSplitter->init(_pPublicInterface->getHinst(), _pPublicInterface->getHSelf());

                        Window *pWindow;
                        if (bothActive())
                            pWindow = &_subSplitter;
                        else
                            pWindow = _pDocTab;

                        _pMainSplitter->create(pWindow, ScintillaEditView::getUserDefineDlg(), 8, SplitterMode::RIGHT_FIX, 45);
                    }

					_pMainWindow = _pMainSplitter;

					_pMainSplitter->setWin0((bothActive())?(Window *)&_subSplitter:(Window *)_pDocTab);

					::SendMessage(_pPublicInterface->getHSelf(), WM_SIZE, 0, 0);
					_pMainWindow->display();

					_mainWindowStatus |= WindowUserActive;
				}
				else if ((!isUDDlgDocked)&&(isUDDlgVisible))
				{
					udd->display(false);
				}
				else //((!isUDDlgDocked)&&(!isUDDlgVisible))
					udd->display();
			}
			checkMenuItem(IDM_LANG_USER_DLG, !isUDDlgVisible);
			_toolBar.setCheck(IDM_LANG_USER_DLG, !isUDDlgVisible);

            break;
        }

#ifdef MOD_EDIT_SELECTWORD
		case IDM_EDIT_SELECTWORD:
			_pEditView->selectWord();
			break;
#endif
		case IDM_EDIT_SELECTALL:
			_pEditView->execute(SCI_SELECTALL);
			checkClipboard();
			break;

		case IDM_EDIT_INS_TAB:
			_pEditView->execute(SCI_TAB);
			break;

		case IDM_EDIT_RMV_TAB:
			_pEditView->execute(SCI_BACKTAB);
			break;

		case IDM_EDIT_DUP_LINE:
			_pEditView->execute(SCI_LINEDUPLICATE);
			break;

		case IDM_EDIT_SPLIT_LINES:
			_pEditView->execute(SCI_TARGETFROMSELECTION);
			if (_pEditView->execute(SCI_GETEDGEMODE) == EDGE_NONE)
			{
				_pEditView->execute(SCI_LINESSPLIT);
			}
			else
			{
#ifdef FIX_EDIT_SPLIT_LINES
				_pEditView->execute(SCI_LINESSPLIT, _pEditView->execute(SCI_GETSPACEWIDTH) * _pEditView->execute(SCI_GETEDGECOLUMN));
#else
				auto textWidth = _pEditView->execute(SCI_TEXTWIDTH, STYLE_LINENUMBER, reinterpret_cast<LPARAM>("P"));
				auto edgeCol = _pEditView->execute(SCI_GETEDGECOLUMN);
				_pEditView->execute(SCI_LINESSPLIT, textWidth * edgeCol);
#endif
			}
			break;

		case IDM_EDIT_JOIN_LINES:
			_pEditView->execute(SCI_TARGETFROMSELECTION);
			_pEditView->execute(SCI_LINESJOIN);
			break;

		case IDM_EDIT_LINE_UP:
			_pEditView->currentLinesUp();
			break;

		case IDM_EDIT_LINE_DOWN:
			_pEditView->currentLinesDown();
			break;

		case IDM_EDIT_REMOVEEMPTYLINES:
			_pEditView->execute(SCI_BEGINUNDOACTION);
			removeEmptyLine(false);
			_pEditView->execute(SCI_ENDUNDOACTION);
			break;

		case IDM_EDIT_REMOVEEMPTYLINESWITHBLANK:
			_pEditView->execute(SCI_BEGINUNDOACTION);
			removeEmptyLine(true);
			_pEditView->execute(SCI_ENDUNDOACTION);
			break;

		case IDM_EDIT_UPPERCASE:
            _pEditView->convertSelectedTextToUpperCase();
			break;

		case IDM_EDIT_LOWERCASE:
            _pEditView->convertSelectedTextToLowerCase();
			break;

		case IDM_EDIT_PROPERCASE_FORCE:
			_pEditView->convertSelectedTextToNewerCase(TITLECASE_FORCE);
			break;

		case IDM_EDIT_PROPERCASE_BLEND:
			_pEditView->convertSelectedTextToNewerCase(TITLECASE_BLEND);
			break;

		case IDM_EDIT_SENTENCECASE_FORCE:
			_pEditView->convertSelectedTextToNewerCase(SENTENCECASE_FORCE);
			break;

		case IDM_EDIT_SENTENCECASE_BLEND:
			_pEditView->convertSelectedTextToNewerCase(SENTENCECASE_BLEND);
			break;

		case IDM_EDIT_INVERTCASE:
			_pEditView->convertSelectedTextToNewerCase(INVERTCASE);
			break;

		case IDM_EDIT_RANDOMCASE:
			_pEditView->convertSelectedTextToNewerCase(RANDOMCASE);
			break;

#ifdef MOD_EDIT_CONVERT_KANJI
		case IDM_EDIT_FULLWIDTH:
			_pEditView->convertSelectedTextToNewerCase(TC_FULLWIDTH);
			break;
		case IDM_EDIT_HALFWIDTH:
			_pEditView->convertSelectedTextToNewerCase(TC_HALFWIDTH);
			break;
		case IDM_EDIT_KATAKANA:
			_pEditView->convertSelectedTextToNewerCase(TC_KATAKANA);
			break;
		case IDM_EDIT_HIRAGANA:
			_pEditView->convertSelectedTextToNewerCase(TC_HIRAGANA);
			break;
#endif

		case IDM_EDIT_BLOCK_COMMENT:
			doBlockComment(cm_toggle);
 			break;

		case IDM_EDIT_BLOCK_COMMENT_SET:
			doBlockComment(cm_comment);
			break;

		case IDM_EDIT_BLOCK_UNCOMMENT:
			doBlockComment(cm_uncomment);
			break;

		case IDM_EDIT_STREAM_COMMENT:
			doStreamComment();
			break;

		case IDM_EDIT_STREAM_UNCOMMENT:
			undoStreamComment();
			break;

		case IDM_EDIT_TRIMTRAILING:
			_pEditView->execute(SCI_BEGINUNDOACTION);
			doTrim(lineTail);
			_pEditView->execute(SCI_ENDUNDOACTION);
			break;

		case IDM_EDIT_TRIMLINEHEAD:
			_pEditView->execute(SCI_BEGINUNDOACTION);
			doTrim(lineHeader);
			_pEditView->execute(SCI_ENDUNDOACTION);
			break;

		case IDM_EDIT_TRIM_BOTH:
			_pEditView->execute(SCI_BEGINUNDOACTION);
			doTrim(lineTail);
			doTrim(lineHeader);
			_pEditView->execute(SCI_ENDUNDOACTION);
			break;

		case IDM_EDIT_EOL2WS:
			_pEditView->execute(SCI_BEGINUNDOACTION);
			_pEditView->execute(SCI_SETTARGETRANGE, 0, _pEditView->getCurrentDocLen());
			_pEditView->execute(SCI_LINESJOIN);
			_pEditView->execute(SCI_ENDUNDOACTION);
			break;

		case IDM_EDIT_TRIMALL:
			_pEditView->execute(SCI_BEGINUNDOACTION);
			doTrim(lineTail);
			doTrim(lineHeader);
			_pEditView->execute(SCI_SETTARGETRANGE, 0, _pEditView->getCurrentDocLen());
			_pEditView->execute(SCI_LINESJOIN);
			_pEditView->execute(SCI_ENDUNDOACTION);
			break;

		case IDM_EDIT_TAB2SW:
			wsTabConvert(tab2Space);
			break;
#ifdef MOD_EDIT_TAB2SW_LEADING
		case IDM_EDIT_TAB2SW_LEADING:
			wsTabConvert(tab2SpaceLeading);
			break;
#endif

		case IDM_EDIT_SW2TAB_LEADING:
			wsTabConvert(space2TabLeading);
			break;

		case IDM_EDIT_SW2TAB_ALL:
			wsTabConvert(space2TabAll);
			break;

		case IDM_EDIT_SETREADONLY:
		{
			Buffer * buf = _pEditView->getCurrentBuffer();
			buf->setUserReadOnly(!buf->getUserReadOnly());
		}
		break;

		case IDM_EDIT_CLEARREADONLY:
		{
			Buffer * buf = _pEditView->getCurrentBuffer();

			DWORD dwFileAttribs = ::GetFileAttributes(buf->getFullPathName());
#ifdef ORZ // IDM_EDIT_CLEARREADONLY
			dwFileAttribs &= ~FILE_ATTRIBUTE_READONLY;
#else
			dwFileAttribs ^= FILE_ATTRIBUTE_READONLY;
#endif

			::SetFileAttributes(buf->getFullPathName(), dwFileAttribs);
#ifdef FIX_SYMBOLIC_LINK
			buf->checkFileState();
			if (buf->getFileReadOnly()) MessageBeep(0);
#else
			buf->setFileReadOnly(false);
#endif
		}
		break;

		case IDM_SEARCH_CUTMARKEDLINES :
			cutMarkedLines();
			break;

		case IDM_SEARCH_COPYMARKEDLINES :
			copyMarkedLines();
			break;

		case IDM_SEARCH_PASTEMARKEDLINES :
			pasteToMarkedLines();
			break;

		case IDM_SEARCH_DELETEMARKEDLINES :
			deleteMarkedLines(true);
			break;

		case IDM_SEARCH_DELETEUNMARKEDLINES :
			deleteMarkedLines(false);
			break;

		case IDM_SEARCH_INVERSEMARKS :
			inverseMarks();
			break;

		case IDM_VIEW_FULLSCREENTOGGLE :
			fullScreenToggle();
			break;

	    case IDM_VIEW_ALWAYSONTOP:
		{
			int check = (::GetMenuState(_mainMenuHandle, id, MF_BYCOMMAND) == MF_CHECKED)?MF_UNCHECKED:MF_CHECKED;
			::CheckMenuItem(_mainMenuHandle, id, MF_BYCOMMAND | check);
			SetWindowPos(_pPublicInterface->getHSelf(), check == MF_CHECKED?HWND_TOPMOST:HWND_NOTOPMOST, 0, 0, 0, 0, SWP_NOMOVE|SWP_NOSIZE);
		}
		break;


		case IDM_VIEW_FOLD_CURRENT :
		case IDM_VIEW_UNFOLD_CURRENT :
			_pEditView->foldCurrentPos((id==IDM_VIEW_FOLD_CURRENT)?fold_collapse:fold_uncollapse);
			break;

		case IDM_VIEW_TOGGLE_FOLDALL:
		case IDM_VIEW_TOGGLE_UNFOLDALL:
		{
			_isFolding = true; // So we can ignore events while folding is taking place
			bool doCollapse = (id==IDM_VIEW_TOGGLE_FOLDALL)?fold_collapse:fold_uncollapse;
 			_pEditView->foldAll(doCollapse);
			if (_pDocMap)
			{
				_pDocMap->foldAll(doCollapse);
			}
			_isFolding = false;
		}
		break;

		case IDM_VIEW_FOLD_1:
		case IDM_VIEW_FOLD_2:
		case IDM_VIEW_FOLD_3:
		case IDM_VIEW_FOLD_4:
		case IDM_VIEW_FOLD_5:
		case IDM_VIEW_FOLD_6:
		case IDM_VIEW_FOLD_7:
		case IDM_VIEW_FOLD_8:
			_isFolding = true; // So we can ignore events while folding is taking place
 			_pEditView->collapse(id - IDM_VIEW_FOLD - 1, fold_collapse);
			_isFolding = false;
			break;

		case IDM_VIEW_UNFOLD_1:
		case IDM_VIEW_UNFOLD_2:
		case IDM_VIEW_UNFOLD_3:
		case IDM_VIEW_UNFOLD_4:
		case IDM_VIEW_UNFOLD_5:
		case IDM_VIEW_UNFOLD_6:
		case IDM_VIEW_UNFOLD_7:
		case IDM_VIEW_UNFOLD_8:
			_isFolding = true; // So we can ignore events while folding is taking place
 			_pEditView->collapse(id - IDM_VIEW_UNFOLD - 1, fold_uncollapse);
			_isFolding = false;
			break;


		case IDM_VIEW_TOOLBAR_REDUCE:
		{
            toolBarStatusType state = _toolBar.getState();

            if (state != TB_SMALL)
            {
			    _toolBar.reduce();
			    changeToolBarIcons();
            }
		}
		break;

		case IDM_VIEW_TOOLBAR_ENLARGE:
		{
            toolBarStatusType state = _toolBar.getState();

            if (state != TB_LARGE)
            {
			    _toolBar.enlarge();
			    changeToolBarIcons();
            }
		}
		break;

		case IDM_VIEW_TOOLBAR_STANDARD:
		{
			toolBarStatusType state = _toolBar.getState();

            if (state != TB_STANDARD)
            {
				_toolBar.setToUglyIcons();
			}
		}
		break;

		case IDM_VIEW_REDUCETABBAR :
		{
			_toReduceTabBar = !_toReduceTabBar;

			//Resize the  icon
			int iconDpiDynamicalSize = NppParameters::getInstance()->_dpiManager.scaleY(_toReduceTabBar?12:18);

			//Resize the tab height
			int tabDpiDynamicalWidth = NppParameters::getInstance()->_dpiManager.scaleX(45);
			int tabDpiDynamicalHeight = NppParameters::getInstance()->_dpiManager.scaleY(_toReduceTabBar?22:25);
			TabCtrl_SetItemSize(_mainDocTab.getHSelf(), tabDpiDynamicalWidth, tabDpiDynamicalHeight);
			TabCtrl_SetItemSize(_subDocTab.getHSelf(), tabDpiDynamicalWidth, tabDpiDynamicalHeight);
			_docTabIconList.setIconSize(iconDpiDynamicalSize);

			//change the font
			int stockedFont = _toReduceTabBar?DEFAULT_GUI_FONT:SYSTEM_FONT;
			HFONT hf = (HFONT)::GetStockObject(stockedFont);

			if (hf)
			{
				::SendMessage(_mainDocTab.getHSelf(), WM_SETFONT, reinterpret_cast<WPARAM>(hf), MAKELPARAM(TRUE, 0));
				::SendMessage(_subDocTab.getHSelf(), WM_SETFONT, reinterpret_cast<WPARAM>(hf), MAKELPARAM(TRUE, 0));
			}

			::SendMessage(_pPublicInterface->getHSelf(), WM_SIZE, 0, 0);
			break;
		}

		case IDM_VIEW_REFRESHTABAR :
		{
			::SendMessage(_pPublicInterface->getHSelf(), WM_SIZE, 0, 0);
			break;
		}
        case IDM_VIEW_LOCKTABBAR:
		{
			bool isDrag = TabBarPlus::doDragNDropOrNot();
            TabBarPlus::doDragNDrop(!isDrag);
            break;
		}


		case IDM_VIEW_DRAWTABBAR_INACIVETAB:
		{
			TabBarPlus::setDrawInactiveTab(!TabBarPlus::drawInactiveTab());
			break;
		}
		case IDM_VIEW_DRAWTABBAR_TOPBAR:
		{
			TabBarPlus::setDrawTopBar(!TabBarPlus::drawTopBar());
			break;
		}

		case IDM_VIEW_DRAWTABBAR_CLOSEBOTTUN :
		{
			TabBarPlus::setDrawTabCloseButton(!TabBarPlus::drawTabCloseButton());

			// This part is just for updating (redraw) the tabs
			int tabDpiDynamicalHeight = NppParameters::getInstance()->_dpiManager.scaleY(TabBarPlus::drawTabCloseButton() ? 22 : 22);
			int tabDpiDynamicalWidth = NppParameters::getInstance()->_dpiManager.scaleX(TabBarPlus::drawTabCloseButton() ? 60 : 45);
			TabCtrl_SetItemSize(_mainDocTab.getHSelf(), tabDpiDynamicalWidth, tabDpiDynamicalHeight);
			TabCtrl_SetItemSize(_subDocTab.getHSelf(), tabDpiDynamicalWidth, tabDpiDynamicalHeight);

			::SendMessage(_pPublicInterface->getHSelf(), WM_SIZE, 0, 0);
			break;
		}

		case IDM_VIEW_DRAWTABBAR_DBCLK2CLOSE :
		{
			TabBarPlus::setDbClk2Close(!TabBarPlus::isDbClk2Close());
			break;
		}

		case IDM_VIEW_DRAWTABBAR_VERTICAL :
		{
			TabBarPlus::setVertical(!TabBarPlus::isVertical());
			::SendMessage(_pPublicInterface->getHSelf(), WM_SIZE, 0, 0);
			break;
		}

		case IDM_VIEW_DRAWTABBAR_MULTILINE :
		{
			TabBarPlus::setMultiLine(!TabBarPlus::isMultiLine());
			::SendMessage(_pPublicInterface->getHSelf(), WM_SIZE, 0, 0);
			break;
		}

		case IDM_VIEW_POSTIT :
		{
			postItToggle();
		}
		break;

#ifdef MOD_WHITESPACE_VISIBILITY
		case IDM_VIEW_TAB:
		case IDM_VIEW_ASPACE:
		case IDM_VIEW_USPACE:
		case IDM_VIEW_EOF:
		case IDM_VIEW_EOB:
		case IDM_VIEW_CRLF:
		{
			bool isChecked = !(::GetMenuState(_mainMenuHandle, id, MF_BYCOMMAND) == MF_CHECKED);
			                  ::CheckMenuItem(_mainMenuHandle, id, MF_BYCOMMAND | (isChecked ? MF_CHECKED : MF_UNCHECKED));
			unsigned int mask = 0;
			switch (id) {
				case IDM_VIEW_TAB:    mask = SCWS_TAB;    break;
				case IDM_VIEW_ASPACE: mask = SCWS_ASPACE; break;
				case IDM_VIEW_USPACE: mask = SCWS_USPACE; break;
				case IDM_VIEW_EOF:    mask = SCWS_EOF;    break;
				case IDM_VIEW_EOB:    mask = SCWS_EOB;    break;
				case IDM_VIEW_CRLF:   mask = SCWS_CRLF;   break;
			}
			ScintillaViewParams & svp1 = const_cast<ScintillaViewParams &>(NppParameters::getInstance()->getSVP());
			if (isChecked) svp1.whiteSpaceShow |=  mask;
			else           svp1.whiteSpaceShow &= ~mask;

			if (isChecked && svp1.viewAllCharacters == 0) {
				svp1.viewAllCharacters = ~0u;
				checkMenuItem(IDM_VIEW_ALL_CHARACTERS, true);
				_toolBar.setCheck(IDM_VIEW_ALL_CHARACTERS, true);
			}

			auto viewWhitespace = svp1.whiteSpaceShow & svp1.viewAllCharacters;
			_mainEditView.showWSAndTab(viewWhitespace);	// SCI_SETVIEWWS
			_subEditView.showWSAndTab(viewWhitespace);	// SCI_SETVIEWWS
			break;
		}
#else
		case IDM_VIEW_TAB_SPACE:
		{
			bool isChecked = !(::GetMenuState(_mainMenuHandle, IDM_VIEW_TAB_SPACE, MF_BYCOMMAND) == MF_CHECKED);
			::CheckMenuItem(_mainMenuHandle, IDM_VIEW_EOL, MF_BYCOMMAND | MF_UNCHECKED);
			::CheckMenuItem(_mainMenuHandle, IDM_VIEW_ALL_CHARACTERS, MF_BYCOMMAND | MF_UNCHECKED);
			::CheckMenuItem(_mainMenuHandle, IDM_VIEW_TAB_SPACE, MF_BYCOMMAND | (isChecked?MF_CHECKED:MF_UNCHECKED));
			_toolBar.setCheck(IDM_VIEW_ALL_CHARACTERS, false);
			_mainEditView.showEOL(false);
			_mainEditView.showWSAndTab(isChecked);
			_subEditView.showEOL(false);
			_subEditView.showWSAndTab(isChecked);

            ScintillaViewParams & svp1 = (ScintillaViewParams &)(NppParameters::getInstance())->getSVP();
            svp1._whiteSpaceShow = isChecked;
            svp1._eolShow = false;
			break;
		}
		case IDM_VIEW_EOL:
		{
			bool isChecked = !(::GetMenuState(_mainMenuHandle, IDM_VIEW_EOL, MF_BYCOMMAND) == MF_CHECKED);
			::CheckMenuItem(_mainMenuHandle, IDM_VIEW_TAB_SPACE, MF_BYCOMMAND | MF_UNCHECKED);
			::CheckMenuItem(_mainMenuHandle, IDM_VIEW_EOL, MF_BYCOMMAND | (isChecked?MF_CHECKED:MF_UNCHECKED));
			::CheckMenuItem(_mainMenuHandle, IDM_VIEW_ALL_CHARACTERS, MF_BYCOMMAND | MF_UNCHECKED);
			_toolBar.setCheck(IDM_VIEW_ALL_CHARACTERS, false);
			_mainEditView.showEOL(isChecked);
			_subEditView.showEOL(isChecked);
			_mainEditView.showWSAndTab(false);
			_subEditView.showWSAndTab(false);

            ScintillaViewParams & svp1 = (ScintillaViewParams &)(NppParameters::getInstance())->getSVP();
            svp1._whiteSpaceShow = false;
            svp1._eolShow = isChecked;
			break;
		}
#endif // MOD_WHITESPACE_VISIBILITY
		case IDM_VIEW_ALL_CHARACTERS:
		{
#ifdef MOD_WHITESPACE_VISIBILITY
			bool isChecked = !(::GetMenuState(_mainMenuHandle, id, MF_BYCOMMAND) == MF_CHECKED);
			ScintillaViewParams & svp1 = const_cast<ScintillaViewParams &>(NppParameters::getInstance()->getSVP());
			svp1.viewAllCharacters = isChecked ? ~0 : 0;

			::CheckMenuItem(_mainMenuHandle, id, MF_BYCOMMAND | (isChecked ? MF_CHECKED : MF_UNCHECKED));
			_toolBar.setCheck(IDM_VIEW_ALL_CHARACTERS, isChecked);
			auto viewWhitespace = svp1.whiteSpaceShow & svp1.viewAllCharacters;
			_mainEditView.showWSAndTab(viewWhitespace);	// SCI_SETVIEWWS
			_subEditView.showWSAndTab(viewWhitespace);	// SCI_SETVIEWWS
#else
			bool isChecked = !(::GetMenuState(_mainMenuHandle, id, MF_BYCOMMAND) == MF_CHECKED);
			::CheckMenuItem(_mainMenuHandle, IDM_VIEW_EOL, MF_BYCOMMAND | MF_UNCHECKED);
			::CheckMenuItem(_mainMenuHandle, IDM_VIEW_TAB_SPACE, MF_BYCOMMAND | MF_UNCHECKED);
			::CheckMenuItem(_mainMenuHandle, IDM_VIEW_ALL_CHARACTERS, MF_BYCOMMAND | (isChecked?MF_CHECKED:MF_UNCHECKED));
			_mainEditView.showInvisibleChars(isChecked);
			_subEditView.showInvisibleChars(isChecked);
			_toolBar.setCheck(IDM_VIEW_ALL_CHARACTERS, isChecked);

            ScintillaViewParams & svp1 = (ScintillaViewParams &)(NppParameters::getInstance())->getSVP();
            svp1._whiteSpaceShow = isChecked;
            svp1._eolShow = isChecked;
#endif
			break;
		}

		case IDM_VIEW_INDENT_GUIDE:
		{
#ifdef ORZ // IDM_VIEW_INDENT_GUIDE
			const bool isChecked = !_pEditView->isShownIndentGuide();
			_mainEditView.showIndentGuideLine(isChecked);
			_subEditView.showIndentGuideLine(isChecked);
			_toolBar.setCheck(IDM_VIEW_INDENT_GUIDE, isChecked);
			checkMenuItem(IDM_VIEW_INDENT_GUIDE, isChecked);

			ScintillaViewParams & svp1 = const_cast<ScintillaViewParams &>(NppParameters::getInstance()->getSVP());
            svp1._indentGuideLineShow = isChecked;
#else
			_mainEditView.showIndentGuideLine(!_pEditView->isShownIndentGuide());
			_subEditView.showIndentGuideLine(!_pEditView->isShownIndentGuide());
            _toolBar.setCheck(IDM_VIEW_INDENT_GUIDE, _pEditView->isShownIndentGuide());
			checkMenuItem(IDM_VIEW_INDENT_GUIDE, _pEditView->isShownIndentGuide());

            ScintillaViewParams & svp1 = (ScintillaViewParams &)(NppParameters::getInstance())->getSVP();
            svp1._indentGuideLineShow = _pEditView->isShownIndentGuide();
#endif
			break;
		}

#ifdef MOD_EACH_VIEW_TABSIZE
		case IDM_VIEW_TABSIZE_2:
		case IDM_VIEW_TABSIZE_4:
		case IDM_VIEW_TABSIZE_8:
		{
			const int tabSize =
				  id == IDM_VIEW_TABSIZE_2 ? 2
				: id == IDM_VIEW_TABSIZE_4 ? 4
				: 8;
			BufferID const bufID = _pEditView->getCurrentBufferID();

			bufID->tabSize = tabSize;
			bufID->setLastXChosen();

			ScintillaEditView *views[] = { &_mainEditView, &_subEditView };
			for (int i = _countof(views); --i >=0; ) {
				ScintillaEditView *v = views[i];
				if (v->getCurrentBufferID() == bufID) {
					v->execute(SCI_SETTABWIDTH, tabSize);
					v->setWrapRestoreNeeded(true);	// Call SCI_CHOOSECARETX later.
					v->saveCurrentPos();
				}
			}

			int tabWidth = tabSize;
		 /*	checkMenuItem(IDM_VIEW_TABSIZE_2, tabWidth == 2);	*/
			checkMenuItem(IDM_VIEW_TABSIZE_4, tabWidth == 4);
			checkMenuItem(IDM_VIEW_TABSIZE_8, tabWidth == 8);

			if (_pDocMap)
			{
				_pDocMap->initWrapMap();
				_pDocMap->wrapMap();
			}
			break;
		}
#endif

#ifdef MOD_EDGE
		case IDM_VIEW_WRAP_NONE:
		case IDM_VIEW_WRAP_EDGE:
		case IDM_VIEW_WRAP_WINDOW:
		{
			int /*id,*/ di;
			bool isChecked;
			switch (id) {
			default:
			case IDM_VIEW_WRAP_NONE:
				id = IDM_VIEW_WRAP_EDGE;
				di = IDM_VIEW_WRAP_WINDOW;
				isChecked = false;
				break;
			case IDM_VIEW_WRAP_EDGE:
			/*	id = IDM_VIEW_WRAP_EDGE;	*/
				di = IDM_VIEW_WRAP_WINDOW;
				isChecked = !(::GetMenuState(_mainMenuHandle, id, MF_BYCOMMAND) == MF_CHECKED);	// toggle
				break;
			case IDM_VIEW_WRAP_WINDOW:
			/*	id = IDM_VIEW_WRAP_WINDOW;	*/
				di = IDM_VIEW_WRAP_EDGE;
				isChecked = !(::GetMenuState(_mainMenuHandle, id, MF_BYCOMMAND) == MF_CHECKED);	// toggle
				break;
			}

			if (isChecked) {
				int edgeState = (int) _pEditView->execute(SCI_GETEDGEMODE);
				switch (id) {
					case IDM_VIEW_WRAP_EDGE:   edgeState |=  EDGE_WRAP; break;
					case IDM_VIEW_WRAP_WINDOW: edgeState &= ~EDGE_WRAP; break;
				}
				_pEditView->execute(SCI_SETEDGEMODE, edgeState);
			}
			_pEditView->wrap(isChecked);	// execute SCI_SETWRAPMODE
			::CheckMenuItem(_mainMenuHandle, id, isChecked ? MF_BYCOMMAND | MF_CHECKED : MF_BYCOMMAND | MF_UNCHECKED);
			::CheckMenuItem(_mainMenuHandle, di, MF_BYCOMMAND | MF_UNCHECKED);
			_toolBar.setCheck(id, isChecked);
			_toolBar.setCheck(di, false);

			//--FLS: ViewMoveAtWrappingDisableFix: Disable wrapping messes up visible lines. Therefore save view position before in IDM_VIEW_WRAP and restore after SCN_PAINTED, as Scintilla-Doc. says
			_pEditView->setWrapRestoreNeeded(true);	// Call SCI_CHOOSECARETX later.
			_pEditView->saveCurrentPos();

			if (_pDocMap)
			{
				_pDocMap->initWrapMap();
				_pDocMap->wrapMap();
			}
			break;
		}
#else // MOD_EDGE
		case IDM_VIEW_WRAP:
		{
			bool isWraped = !_pEditView->isWrap();
			// ViewMoveAtWrappingDisableFix: Disable wrapping messes up visible lines. Therefore save view position before in IDM_VIEW_WRAP and restore after SCN_PAINTED, as Scintilla-Doc. says
			if (!isWraped)
			{
				_mainEditView.saveCurrentPos();
				_mainEditView.setWrapRestoreNeeded(true);
				_subEditView.saveCurrentPos();
				_subEditView.setWrapRestoreNeeded(true);
			}
			_mainEditView.wrap(isWraped);
			_subEditView.wrap(isWraped);
			_toolBar.setCheck(IDM_VIEW_WRAP, isWraped);
			checkMenuItem(IDM_VIEW_WRAP, isWraped);

			ScintillaViewParams & svp1 = (ScintillaViewParams &)(NppParameters::getInstance())->getSVP();
			svp1._doWrap = isWraped;

			if (_pDocMap)
			{
				_pDocMap->initWrapMap();
				_pDocMap->wrapMap();
			}
			break;
		}
#endif // MOD_EDGE
		case IDM_VIEW_WRAP_SYMBOL:
		{
			_mainEditView.showWrapSymbol(!_pEditView->isWrapSymbolVisible());
			_subEditView.showWrapSymbol(!_pEditView->isWrapSymbolVisible());
			checkMenuItem(IDM_VIEW_WRAP_SYMBOL, _pEditView->isWrapSymbolVisible());

            ScintillaViewParams & svp1 = (ScintillaViewParams &)(NppParameters::getInstance())->getSVP();
            svp1._wrapSymbolShow = _pEditView->isWrapSymbolVisible();
			break;
		}

		case IDM_VIEW_HIDELINES:
		{
			_pEditView->hideLines();
			break;
		}

		case IDM_VIEW_ZOOMIN:
		{
			_pEditView->execute(SCI_ZOOMIN);
			break;
		}
		case IDM_VIEW_ZOOMOUT:
			_pEditView->execute(SCI_ZOOMOUT);
			break;

		case IDM_VIEW_ZOOMRESTORE:
			//Zoom factor of 0 points means default view
			_pEditView->execute(SCI_SETZOOM, 0);//_zoomOriginalValue);
			break;

		case IDM_VIEW_SYNSCROLLV:
		{
            bool isSynScollV = !_syncInfo._isSynScollV;

			checkMenuItem(IDM_VIEW_SYNSCROLLV, isSynScollV);
			_toolBar.setCheck(IDM_VIEW_SYNSCROLLV, isSynScollV);

            _syncInfo._isSynScollV = isSynScollV;
			if (_syncInfo._isSynScollV)
			{
				int mainCurrentLine = static_cast<int32_t>(_mainEditView.execute(SCI_GETFIRSTVISIBLELINE));
				int subCurrentLine = static_cast<int32_t>(_subEditView.execute(SCI_GETFIRSTVISIBLELINE));
				_syncInfo._line = mainCurrentLine - subCurrentLine;
			}

		}
		break;

		case IDM_VIEW_SYNSCROLLH:
		{
            bool isSynScollH = !_syncInfo._isSynScollH;
			checkMenuItem(IDM_VIEW_SYNSCROLLH, isSynScollH);
			_toolBar.setCheck(IDM_VIEW_SYNSCROLLH, isSynScollH);

            _syncInfo._isSynScollH = isSynScollH;
			if (_syncInfo._isSynScollH)
			{
				int mxoffset = static_cast<int32_t>(_mainEditView.execute(SCI_GETXOFFSET));
				int pixel = static_cast<int32_t>(_mainEditView.execute(SCI_TEXTWIDTH, STYLE_DEFAULT, reinterpret_cast<LPARAM>("P")));
				int mainColumn = mxoffset/pixel;

				int sxoffset = static_cast<int32_t>(_subEditView.execute(SCI_GETXOFFSET));
				pixel = int(_subEditView.execute(SCI_TEXTWIDTH, STYLE_DEFAULT, reinterpret_cast<LPARAM>("P")));
				int subColumn = sxoffset/pixel;
				_syncInfo._column = mainColumn - subColumn;
			}
		}
		break;

		case IDM_VIEW_SUMMARY:
		{
			generic_string characterNumber = TEXT("");

			Buffer * curBuf = _pEditView->getCurrentBuffer();
			int fileLen = curBuf->getFileLength();

			if (fileLen != -1)
			{
				TCHAR *filePathLabel = TEXT("Full file path: ");
				TCHAR *fileCreateTimeLabel = TEXT("Created: ");
				TCHAR *fileModifyTimeLabel = TEXT("Modified: ");
				TCHAR *fileLenLabel = TEXT("File length (in byte): ");

				characterNumber += filePathLabel;
				characterNumber += curBuf->getFullPathName();
				characterNumber += TEXT("\r");

				characterNumber += fileCreateTimeLabel;
				characterNumber += curBuf->getFileTime(Buffer::ft_created);
				characterNumber += TEXT("\r");

				characterNumber += fileModifyTimeLabel;
				characterNumber += curBuf->getFileTime(Buffer::ft_modified);
				characterNumber += TEXT("\r");

				characterNumber += fileLenLabel;
				characterNumber += commafyInt(fileLen).c_str();
				characterNumber += TEXT("\r");
#if 9999 //+[JOJO]
				const DWORD attribute = GetFileAttributes(curBuf->getFullPathName());
				if (attribute != -1) {
					std::wstring as;
					characterNumber += L"Attribute: ";
					characterNumber += FileAttributesToString(attribute, 0);
					characterNumber += L' ';
					characterNumber += FileAttributesToDescription(attribute, 0);
					characterNumber += L'\r';
				}
				if (curBuf->getUserReadOnly()) characterNumber += L"編集禁止・";
				if (curBuf->isDirty()) characterNumber += L"変更あり・";
				if (characterNumber.back() == L'・') { characterNumber.pop_back(); characterNumber += L'\r'; }
#endif
#if 0000 //+[JOJO]
//				HANDLE hFile = ::CreateFileW(curBuf->getFullPathName(),0, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
//				if (hFile != INVALID_HANDLE_VALUE) {
//					BY_HANDLE_FILE_INFORMATION fileInformation;
//					auto res = GetFileInformationByHandle(hFile, &fileInformation);
//					::CloseHandle(hFile);
//					if (res != 0)
//						swsprintf(characterNumber, L"GetFileInformationByHandle: dwVolumeSerialNumber=>%u nFileIndexHigh=>%u nFileIndexLow=>%u\r", fileInformation.dwVolumeSerialNumber, fileInformation.nFileIndexHigh, fileInformation.nFileIndexLow);
//				}
#endif
#if 0000 //+[JOJO]
//				WIN32_FIND_DATA stat;
//				HANDLE handle = FindFirstFile(curBuf->getFullPathName(), &stat);
//				if (handle != INVALID_HANDLE_VALUE) {
//					FindClose(handle);
//					swsprintf(characterNumber, L"*dwFileAttributes: %08X%s\r", stat.dwFileAttributes, stat.dwFileAttributes & FILE_ATTRIBUTE_REPARSE_POINT ? L" リパースポイント" : L"");
//					swsprintf(characterNumber, L"*dwReserved0: %08X%s\r", stat.dwReserved0, stat.dwReserved0 == IO_REPARSE_TAG_MOUNT_POINT ? L" IO_REPARSE_TAG_MOUNT_POINT" : L"");
//					swsprintf(characterNumber, L"*dwReserved1: %08X\r", stat.dwReserved1);
//				}
#endif
				characterNumber += TEXT("\r");
			}

#ifdef INTERNAL_ENCODING_UTF8N
			auto nbChar = getCurrentDocCharCount(uniCookie);
#else
			UniMode um = _pEditView->getCurrentBuffer()->getUnicodeMode();
			auto nbChar = getCurrentDocCharCount(um);
#endif
#ifdef ORZ // wordCount() too heavy
#else
			int nbWord = wordCount();
#endif
			size_t nbLine = _pEditView->execute(SCI_GETLINECOUNT);
			int nbByte = (int) _pEditView->execute(SCI_GETLENGTH);
#ifdef ORZ // SCI_GETLINECOUNT
			if (nbByte - 1 >= 0) { unsigned char c = (unsigned char)_pEditView->execute(SCI_GETCHARAT, nbByte - 1); if (c == '\r' || c == '\n') --nbLine; }
#endif
#ifdef INTERNAL_ENCODING_UTF8N
			long nbSel = _pEditView->countUnicodeSelectedLength();
#else
			int nbSel = getSelectedCharNumber(um);
#endif
			int nbSelByte = getSelectedBytes();
			int nbRange = getSelectedAreas();

			swsprintf(characterNumber, L"Characters (without blanks): %s\r", commafyInt(nbChar).c_str());
#ifdef ORZ // wordCount() too heavy
#else
			swsprintf(characterNumber, L"Words: %s\r", commafyInt(nbWord).c_str());
#endif
			swsprintf(characterNumber, L"Lines: %s\r", commafyInt(nbLine).c_str());
			swsprintf(characterNumber, L"Current document length: %s\r", commafyInt(nbByte).c_str());
			swsprintf(characterNumber, L"%s selected characters (%s bytes) in %s ranges\r", commafyInt(nbSel).c_str(), commafyInt(nbSelByte).c_str(), commafyInt(nbRange).c_str());
#if 9999 //+[JOJO]
			characterNumber += L'\r';
			auto e = EncodeToSession(curBuf->getEncoding(), curBuf->getUnicodeMode());
			const int cp = std::abs(e);
			const bool hasBOM = e < 0;
			swsprintf(characterNumber, L"Encoding: cp%d", cp);
			if (curBuf->getEncoding() == -1) {
				if (hasBOM) characterNumber += L" + BOM";
			}
			else {
				int cmdID = EncodingMapper::getInstance()->getIndexFromEncoding(curBuf->getEncoding());
				if (cmdID != -1)
				{
					std::wstring str; getNameStrFromCmd(cmdID + IDM_FORMAT_ENCODE, str);
					characterNumber = characterNumber + L" " + str.c_str();
				}
			}
			characterNumber += L'\r';
			characterNumber += L'\r';
			characterNumber += L"Character: ";
			unsigned char utf8Chars[UTF8MaxBytes + 1] = {};
			const Sci_Position position = (Sci_Position) _pEditView->execute(SCI_GETCURRENTPOS);
			_pEditView->getText((char*)utf8Chars, position, position + min(4, nbByte - position));	// SCI_GETTEXTRANGE
			if (nbByte <= position)
				characterNumber += L"--";
			else if (utf8Chars[0] == 0x000D && utf8Chars[1] == 0x000A)
				characterNumber += L"U+000D U+000A";
			else {
				int xxBytes = Scintilla::UTF8BytesOfLead[utf8Chars[0]];
				WCHAR wideCharStr[2] = {};
				int wwLength = MultiByteToWideChar(65001, MB_ERR_INVALID_CHARS, (LPCSTR)utf8Chars, xxBytes, wideCharStr, 2);

				uint32_t u = wideCharStr[0], v = wideCharStr[1];
				if (wwLength == 0) {
					swsprintf(characterNumber, L"0x%02X", utf8Chars[0]);
				}
				else {
					if (0xD800 <= u && u < 0xDC00 && 0xDC00 <= v && v < 0xE000) {
						u = (u - 0xD800) * 0x0400 + (v - 0xDC00) + 0x10000;
						wwLength = 2;
					}
					else
						wwLength = 1;
					
					swsprintf(characterNumber, L"U+%04X", u);
					if (0x20 <= u && u < 0x7F || u >= 0xA0)
						swsprintf(characterNumber, L" [%.*s]", wwLength, wideCharStr);
					if (u > 0xFFFF) {
						characterNumber += L"\r\t""UTF-16: ";
						for (int i = 0; i < wwLength; ++i) swsprintf(characterNumber, L"%04X ", wideCharStr[i]);
					}
					if (u > 0x7F) {
						characterNumber += L"\r\t""UTF-8: ";
						for (int i = 0; i < xxBytes; ++i) swsprintf(characterNumber, L"%02X ", utf8Chars[i]);
					}
					if (curBuf->getEncoding() != -1) {
						unsigned char ansiCharStr[10];
						WCHAR verifyWStr[2] = {};
						int aaLength = WideCharToMultiByteN(curBuf->getEncoding(), wideCharStr, wwLength, (char*) ansiCharStr, _countof(ansiCharStr));
						int vvLength = MultiByteToWideCharN(curBuf->getEncoding(), (char *) ansiCharStr, aaLength, verifyWStr, _countof(verifyWStr));
						unsigned char a[1];
						int n = WideCharToMultiByteN(curBuf->getEncoding(), L"0", 1, (char*) a, _countof(a));
						const bool isEBCDIC = n == 1 && a[0] == 0xF0;
						characterNumber += isEBCDIC ? L"\r\t""EBCDIC: " : L"\r\t""ANSI: "; // TODO: characterNumber += curBuf->isEBCDIC() ? ...
						if (aaLength == 0)
							characterNumber += L"--";
						else if (wwLength != vvLength || memcmp(wideCharStr, verifyWStr, wwLength * sizeof(WCHAR)) != 0) {
							characterNumber += L"--";
							characterNumber += L"\r\t""Best fit char: ";
							swsprintf(characterNumber, L" [%.*s] ", vvLength, verifyWStr);
						}
						for (int i = 0; i < aaLength; ++i) swsprintf(characterNumber, L"%02X ", ansiCharStr[i]);
					}
				}
				swsprintf(characterNumber, L"\r\tEast Asian Width: %d",  UnicodeWidth(u));
			}
#endif
			::MessageBox(_pPublicInterface->getHSelf(), characterNumber.c_str(), TEXT("Summary"), MB_OK|MB_APPLMODAL);
		}
		break;

		case IDM_VIEW_MONITORING:
		{
			Buffer * curBuf = _pEditView->getCurrentBuffer();
			if (curBuf->isMonitoringOn())
			{
				curBuf->stopMonitoring();
				checkMenuItem(IDM_VIEW_MONITORING, false);
				_toolBar.setCheck(IDM_VIEW_MONITORING, false);
#ifdef MOD_USER_READONLY
				curBuf->setUserReadOnly(curBuf->getUserReadOnly());	// CALL curBuf->doNotify(BufferChangeReadonly);
#else
				curBuf->setUserReadOnly(false);
#endif
			}
			else
			{
				const TCHAR *longFileName = curBuf->getFullPathName();
				if (::PathFileExists(longFileName))
				{
					if (curBuf->isDirty())
					{
						::MessageBox(_pPublicInterface->getHSelf(), TEXT("The document is dirty. Please save the modification before monitoring it."), TEXT("Monitoring problem"), MB_OK);
					}
					else
					{
						curBuf->startMonitoring(); // monitoring firstly for making monitoring icon 
						curBuf->setUserReadOnly(true);
						
						MonitorInfo *monitorInfo = new MonitorInfo(curBuf, _pPublicInterface->getHSelf());
						HANDLE hThread = ::CreateThread(NULL, 0, monitorFileOnChange, (void *)monitorInfo, 0, NULL); // will be deallocated while quitting thread
						::CloseHandle(hThread);
						checkMenuItem(IDM_VIEW_MONITORING, true);
						_toolBar.setCheck(IDM_VIEW_MONITORING, true);
					}
				}
				else
				{
					::MessageBox(_pPublicInterface->getHSelf(), TEXT("The file should exist to be monitored."), TEXT("Monitoring problem"), MB_OK);
				}
			}

			break;
		}

		case IDM_EXECUTE:
		{
			bool isFirstTime = !_runDlg.isCreated();
			_runDlg.doDialog(_nativeLangSpeaker.isRTL());
			if (isFirstTime)
				_nativeLangSpeaker.changeDlgLang(_runDlg.getHSelf(), "Run");

			break;
		}

		case IDM_FORMAT_TODOS:
		case IDM_FORMAT_TOUNIX:
		case IDM_FORMAT_TOMAC:
		{
			EolType newFormat = (id == IDM_FORMAT_TODOS)
				? EolType::windows
				: (id == IDM_FORMAT_TOUNIX) ? EolType::unix : EolType::macos;

			Buffer* buf = _pEditView->getCurrentBuffer();

			if (not buf->isReadOnly())
			{
				LongRunningOperation op;
				buf->setEolFormat(newFormat);
				_pEditView->execute(SCI_CONVERTEOLS, static_cast<int>(buf->getEolFormat()));
			}

			break;
		}

#ifdef MOD_ENCODING_ANY
		case IDM_FORMAT_ANY:
		{
			ValueDlg codePageDlg;
			Buffer *buf = _pEditView->getCurrentBuffer();
			const int cp = std::abs( EncodeToSession(buf->getEncoding(), buf->getUnicodeMode()) );
			codePageDlg.init(_pPublicInterface->getHinst(), _pPublicInterface->getHSelf(), cp, TEXT("Code Page : "));
			codePageDlg.setNBNumber(5);
			const int a = codePageDlg.doDialog({ 0, 0 });
			if (a == cp || a == -1)
				break;
			const int en = SessionToBufEncode(a);
			const UniMode um = SessionToUniMode(a);
			if (en != -1) {	// ANSI
				DWORD dwMode = 0;  // dummy
				INT nWideCharCount = 0;  // dummy
				HRESULT result = ConvertINetMultiByteToUnicode(&dwMode, a, "A", NULL, NULL, &nWideCharCount);	// MLANG_DLL
				if (result != S_OK) {
					::MessageBoxA(_pPublicInterface->getHSelf(), "Does not support code page", "Change Any Encoding", MB_OK | MB_ICONERROR);
					break;
				}
			}
			changeEncoding(_pPublicInterface->getHSelf(), _pEditView, um, en);
			break;
		}
#endif
#ifdef INTERNAL_ENCODING_UTF8N
#else
		case IDM_FORMAT_ANSI :
#endif
		case IDM_FORMAT_UTF_8 :
		case IDM_FORMAT_UCS_2BE :
		case IDM_FORMAT_UCS_2LE :
		case IDM_FORMAT_AS_UTF_8 :
#ifdef MOD_CHANGE_ENCODING
		case IDM_FORMAT_UTF_16BE_N :
		case IDM_FORMAT_UTF_16LE_N :
#endif
		{
#ifdef INTERNAL_ENCODING_UTF8N
#else
			Buffer * buf = _pEditView->getCurrentBuffer();
#endif

			UniMode um;
#ifdef MOD_CHANGE_ENCODING
#else
			bool shoulBeDirty = true;
#endif
			switch (id)
			{
				case IDM_FORMAT_AS_UTF_8:
#ifdef MOD_CHANGE_ENCODING
#else
					shoulBeDirty = buf->getUnicodeMode() != uni8Bit;
#endif
					um = uniCookie;
					break;

				case IDM_FORMAT_UTF_8:
					um = uniUTF8;
					break;

				case IDM_FORMAT_UCS_2BE:
					um = uni16BE;
					break;

				case IDM_FORMAT_UCS_2LE:
					um = uni16LE;
					break;

#ifdef MOD_CHANGE_ENCODING
				case IDM_FORMAT_UTF_16BE_N:
					um = uni16BE_NoBOM;
					break;

				case IDM_FORMAT_UTF_16LE_N:
					um = uni16LE_NoBOM;
					break;
#endif

#ifdef INTERNAL_ENCODING_UTF8N
#else
				default : // IDM_FORMAT_ANSI
					shoulBeDirty = buf->getUnicodeMode() != uniCookie;
					um = uni8Bit;
#endif
			}

#ifdef MOD_CHANGE_ENCODING
			changeEncoding(_pPublicInterface->getHSelf(), _pEditView, um, -1);
#else
			if (buf->getEncoding() != -1)
			{
				if (buf->isDirty())
				{
					int answer = _nativeLangSpeaker.messageBox("SaveCurrentModifWarning",
						NULL,
						TEXT("You should save the current modification.\rAll the saved modifications can not be undone.\r\rContinue?"),
						TEXT("Save Current Modification"),
						MB_YESNO);

					if (answer == IDYES)
					{
						fileSave();
						_pEditView->execute(SCI_EMPTYUNDOBUFFER);
					}
					else
						return;
				}

				if (_pEditView->execute(SCI_CANUNDO) == TRUE)
				{
					generic_string msg, title;
					int answer = _nativeLangSpeaker.messageBox("LoseUndoAbilityWarning",
						NULL,
						TEXT("You should save the current modification.\rAll the saved modifications can not be undone.\r\rContinue?"),
						TEXT("Lose Undo Ability Waning"),
						MB_YESNO);
					if (answer == IDYES)
					{
						// Do nothing
					}
					else
						return;
				}

				buf->setEncoding(-1);

				if (um == uni8Bit)
					_pEditView->execute(SCI_SETCODEPAGE, CP_ACP);
				else
					buf->setUnicodeMode(um);
				fileReload();
			}
			else
			{
				if (buf->getUnicodeMode() != um)
				{
					buf->setUnicodeMode(um);
					if (shoulBeDirty)
						buf->setDirty(true);
				}
			}
#endif
			break;
		}

        case IDM_FORMAT_WIN_1250 :
        case IDM_FORMAT_WIN_1251 :
        case IDM_FORMAT_WIN_1252 :
        case IDM_FORMAT_WIN_1253 :
        case IDM_FORMAT_WIN_1254 :
        case IDM_FORMAT_WIN_1255 :
        case IDM_FORMAT_WIN_1256 :
        case IDM_FORMAT_WIN_1257 :
        case IDM_FORMAT_WIN_1258 :
        case IDM_FORMAT_ISO_8859_1  :
        case IDM_FORMAT_ISO_8859_2  :
        case IDM_FORMAT_ISO_8859_3  :
        case IDM_FORMAT_ISO_8859_4  :
        case IDM_FORMAT_ISO_8859_5  :
        case IDM_FORMAT_ISO_8859_6  :
        case IDM_FORMAT_ISO_8859_7  :
        case IDM_FORMAT_ISO_8859_8  :
        case IDM_FORMAT_ISO_8859_9  :
        case IDM_FORMAT_ISO_8859_10 :
        case IDM_FORMAT_ISO_8859_11 :
        case IDM_FORMAT_ISO_8859_13 :
        case IDM_FORMAT_ISO_8859_14 :
        case IDM_FORMAT_ISO_8859_15 :
        case IDM_FORMAT_ISO_8859_16 :
        case IDM_FORMAT_DOS_437 :
        case IDM_FORMAT_DOS_720 :
        case IDM_FORMAT_DOS_737 :
        case IDM_FORMAT_DOS_775 :
        case IDM_FORMAT_DOS_850 :
        case IDM_FORMAT_DOS_852 :
        case IDM_FORMAT_DOS_855 :
        case IDM_FORMAT_DOS_857 :
        case IDM_FORMAT_DOS_858 :
        case IDM_FORMAT_DOS_860 :
        case IDM_FORMAT_DOS_861 :
        case IDM_FORMAT_DOS_862 :
        case IDM_FORMAT_DOS_863 :
        case IDM_FORMAT_DOS_865 :
        case IDM_FORMAT_DOS_866 :
        case IDM_FORMAT_DOS_869 :
        case IDM_FORMAT_BIG5 :
        case IDM_FORMAT_GB2312 :
        case IDM_FORMAT_SHIFT_JIS :
        case IDM_FORMAT_KOREAN_WIN :
        case IDM_FORMAT_EUC_KR :
        case IDM_FORMAT_TIS_620 :
        case IDM_FORMAT_MAC_CYRILLIC :
        case IDM_FORMAT_KOI8U_CYRILLIC :
        case IDM_FORMAT_KOI8R_CYRILLIC :
#ifdef MLANG_DLL
        case IDM_FORMAT_ISO_2022_JP :
        case IDM_FORMAT_ISO_2022_JP_MS :
        case IDM_FORMAT_ISO_2022_JP_SIO :
        case IDM_FORMAT_EUC_JP :
        case IDM_FORMAT_JOHAB :
#endif
        {
			int index = id - IDM_FORMAT_ENCODE;

			EncodingMapper *em = EncodingMapper::getInstance();
			int encoding = em->getEncodingFromIndex(index);
			if (encoding == -1)
			{
				//printStr(TEXT("Encoding problem. Command is not added in encoding_table?"));
				return;
			}

#ifdef INTERNAL_ENCODING_UTF8N
#else
            Buffer* buf = _pEditView->getCurrentBuffer();
#endif
#ifdef MOD_CHANGE_ENCODING
			changeEncoding(_pPublicInterface->getHSelf(), _pEditView, uniCookie, encoding);
#else
            if (buf->isDirty())
            {
				generic_string warning, title;
				int answer = _nativeLangSpeaker.messageBox("SaveCurrentModifWarning",
					NULL,
					TEXT("You should save the current modification.\rAll the saved modifications can not be undone.\r\rContinue?"),
					TEXT("Save Current Modification"),
					MB_YESNO);

                if (answer == IDYES)
                {
                    fileSave();
					_pEditView->execute(SCI_EMPTYUNDOBUFFER);
                }
                else
                    return;
            }

            if (_pEditView->execute(SCI_CANUNDO) == TRUE)
            {
				generic_string msg, title;
				int answer = _nativeLangSpeaker.messageBox("LoseUndoAbilityWarning",
					NULL,
					TEXT("You should save the current modification.\rAll the saved modifications can not be undone.\r\rContinue?"),
					TEXT("Lose Undo Ability Waning"),
					MB_YESNO);

                if (answer != IDYES)
                    return;
            }

            if (not buf->isDirty())
            {
				buf->setEncoding(encoding);
				buf->setUnicodeMode(uniCookie);
				fileReload();
            }
#endif
			break;
		}


#ifdef MOD_CHANGE_ENCODING
#else
		case IDM_FORMAT_CONV2_ANSI:
		case IDM_FORMAT_CONV2_AS_UTF_8:
		case IDM_FORMAT_CONV2_UTF_8:
		case IDM_FORMAT_CONV2_UCS_2BE:
		case IDM_FORMAT_CONV2_UCS_2LE:
		{
			int idEncoding = -1;
			Buffer *buf = _pEditView->getCurrentBuffer();
            UniMode um = buf->getUnicodeMode();
            int encoding = buf->getEncoding();

			switch(id)
			{
				case IDM_FORMAT_CONV2_ANSI:
				{
                    if (encoding != -1)
                    {
                        // do nothing
                        return;
                    }
                    else
                    {
					    if (um == uni8Bit)
						    return;

                        // set scintilla to ANSI
					    idEncoding = IDM_FORMAT_ANSI;
                    }
					break;
				}
				case IDM_FORMAT_CONV2_AS_UTF_8:
				{
                    if (encoding != -1)
                    {
                        buf->setDirty(true);
                        buf->setUnicodeMode(uniCookie);
                        buf->setEncoding(-1);
                        return;
                    }

					idEncoding = IDM_FORMAT_AS_UTF_8;
					if (um == uniCookie)
						return;

					if (um != uni8Bit)
					{
						::SendMessage(_pPublicInterface->getHSelf(), WM_COMMAND, idEncoding, 0);
						_pEditView->execute(SCI_EMPTYUNDOBUFFER);
						return;
					}

					break;
				}
				case IDM_FORMAT_CONV2_UTF_8:
				{
                    if (encoding != -1)
                    {
                        buf->setDirty(true);
                        buf->setUnicodeMode(uniUTF8);
                        buf->setEncoding(-1);
                        return;
                    }

					idEncoding = IDM_FORMAT_UTF_8;
					if (um == uniUTF8)
						return;

					if (um != uni8Bit)
					{
						::SendMessage(_pPublicInterface->getHSelf(), WM_COMMAND, idEncoding, 0);
						_pEditView->execute(SCI_EMPTYUNDOBUFFER);
						return;
					}
					break;
				}

				case IDM_FORMAT_CONV2_UCS_2BE:
				{
                    if (encoding != -1)
                    {
                        buf->setDirty(true);
                        buf->setUnicodeMode(uni16BE);
                        buf->setEncoding(-1);
                        return;
                    }

					idEncoding = IDM_FORMAT_UCS_2BE;
					if (um == uni16BE)
						return;

					if (um != uni8Bit)
					{
						::SendMessage(_pPublicInterface->getHSelf(), WM_COMMAND, idEncoding, 0);
						_pEditView->execute(SCI_EMPTYUNDOBUFFER);
						return;
					}
					break;
				}

				case IDM_FORMAT_CONV2_UCS_2LE:
				{
                    if (encoding != -1)
                    {
                        buf->setDirty(true);
                        buf->setUnicodeMode(uni16LE);
                        buf->setEncoding(-1);
                        return;
                    }

					idEncoding = IDM_FORMAT_UCS_2LE;
					if (um == uni16LE)
						return;
					if (um != uni8Bit)
					{
						::SendMessage(_pPublicInterface->getHSelf(), WM_COMMAND, idEncoding, 0);
						_pEditView->execute(SCI_EMPTYUNDOBUFFER);
						return;
					}
					break;
				}
			}

			if (idEncoding != -1)
			{
				// Save the current clipboard content
				::OpenClipboard(_pPublicInterface->getHSelf());
				HANDLE clipboardData = ::GetClipboardData(CF_TEXT);
				int len = static_cast<int32_t>(::GlobalSize(clipboardData));
				LPVOID clipboardDataPtr = ::GlobalLock(clipboardData);

				HANDLE allocClipboardData = ::GlobalAlloc(GMEM_MOVEABLE, len);
				LPVOID clipboardData2 = ::GlobalLock(allocClipboardData);

				::memcpy(clipboardData2, clipboardDataPtr, len);
				::GlobalUnlock(clipboardData);
				::GlobalUnlock(allocClipboardData);
				::CloseClipboard();

				_pEditView->saveCurrentPos();

				// Cut all text
				int docLen = _pEditView->getCurrentDocLen();
				_pEditView->execute(SCI_COPYRANGE, 0, docLen);
				_pEditView->execute(SCI_CLEARALL);

				// Change to the proper buffer, save buffer status

				::SendMessage(_pPublicInterface->getHSelf(), WM_COMMAND, idEncoding, 0);

				// Paste the texte, restore buffer status
				_pEditView->execute(SCI_PASTE);
				_pEditView->restoreCurrentPos();

				// Restore the previous clipboard data
				::OpenClipboard(_pPublicInterface->getHSelf());
				::EmptyClipboard();
				::SetClipboardData(CF_TEXT, clipboardData2);
				::CloseClipboard();

				//Do not free anything, EmptyClipboard does that
				_pEditView->execute(SCI_EMPTYUNDOBUFFER);
			}
			break;
		}
#endif

		case IDM_SETTING_IMPORTPLUGIN :
        {
            // get plugin source path
            TCHAR *extFilterName = TEXT("Notepad++ plugin");
            TCHAR *extFilter = TEXT(".dll");
            TCHAR *destDir = TEXT("plugins");

            vector<generic_string> copiedFiles = addNppComponents(destDir, extFilterName, extFilter);

            // load plugin
            vector<generic_string> dll2Remove;
            for (size_t i = 0, len = copiedFiles.size() ; i < len ; ++i)
            {
                int index = _pluginsManager.loadPlugin(copiedFiles[i].c_str(), dll2Remove);
                if (_pluginsManager.getMenuHandle())
                    _pluginsManager.addInMenuFromPMIndex(index);
            }
            if (!_pluginsManager.getMenuHandle())
                _pluginsManager.setMenu(_mainMenuHandle, NULL);
            ::DrawMenuBar(_pPublicInterface->getHSelf());
            break;
        }

        case IDM_SETTING_IMPORTSTYLETHEMS :
        {
            // get plugin source path
            TCHAR *extFilterName = TEXT("Notepad++ style theme");
            TCHAR *extFilter = TEXT(".xml");
            TCHAR *destDir = TEXT("themes");

            // load styler
            NppParameters *pNppParams = NppParameters::getInstance();
            ThemeSwitcher & themeSwitcher = pNppParams->getThemeSwitcher();

            vector<generic_string> copiedFiles = addNppComponents(destDir, extFilterName, extFilter);
            for (size_t i = 0, len = copiedFiles.size(); i < len ; ++i)
            {
                generic_string themeName(themeSwitcher.getThemeFromXmlFileName(copiedFiles[i].c_str()));
		        if (!themeSwitcher.themeNameExists(themeName.c_str()))
		        {
			        themeSwitcher.addThemeFromXml(copiedFiles[i].c_str());
                    if (_configStyleDlg.isCreated())
                    {
                        _configStyleDlg.addLastThemeEntry();
                    }
		        }
            }
            break;
        }

		case IDM_SETTING_SHORTCUT_MAPPER :
		case IDM_SETTING_SHORTCUT_MAPPER_MACRO :
        case IDM_SETTING_SHORTCUT_MAPPER_RUN :
		{
            GridState st = id==IDM_SETTING_SHORTCUT_MAPPER_MACRO?STATE_MACRO:id==IDM_SETTING_SHORTCUT_MAPPER_RUN?STATE_USER:STATE_MENU;
			ShortcutMapper shortcutMapper;
            shortcutMapper.init(_pPublicInterface->getHinst(), _pPublicInterface->getHSelf(), st);
#ifdef FIX_LOCALIZATION // ShortcutMapper
			shortcutMapper.nativeLangSpeaker = &_nativeLangSpeaker;
#else
            _nativeLangSpeaker.changeShortcutmapperLang(&shortcutMapper);
#endif
			shortcutMapper.doDialog(_nativeLangSpeaker.isRTL());
			shortcutMapper.destroy();
			break;
		}
		case IDM_SETTING_PREFERECE :
		{
			bool isFirstTime = !_preference.isCreated();
			_preference.doDialog(_nativeLangSpeaker.isRTL());

			if (isFirstTime)
			{
				_nativeLangSpeaker.changePrefereceDlgLang(_preference);
			}
			break;
		}

        case IDM_SETTING_EDITCONTEXTMENU :
        {
			_nativeLangSpeaker.messageBox("ContextMenuXmlEditWarning",
				_pPublicInterface->getHSelf(),
				TEXT("Editing contextMenu.xml allows you to modify your Notepad++ popup context menu on edit zone.\rYou have to restart your Notepad++ to take effect after modifying contextMenu.xml."),
				TEXT("Editing contextMenu"),
				MB_OK|MB_APPLMODAL);

            NppParameters *pNppParams = NppParameters::getInstance();
            BufferID bufID = doOpen((pNppParams->getContextMenuPath()));
			switchToFile(bufID);
            break;
        }

        case IDM_VIEW_GOTO_ANOTHER_VIEW:
            docGotoAnotherEditView(TransferMove);
			checkSyncState();
            break;

        case IDM_VIEW_CLONE_TO_ANOTHER_VIEW:
            docGotoAnotherEditView(TransferClone);
			checkSyncState();
            break;

        case IDM_VIEW_GOTO_NEW_INSTANCE :
            docOpenInNewInstance(TransferMove);
            break;

        case IDM_VIEW_LOAD_IN_NEW_INSTANCE:
            docOpenInNewInstance(TransferClone);
            break;

		case IDM_VIEW_SWITCHTO_OTHER_VIEW:
		{
			int view_to_focus;
			HWND wnd = GetFocus();
			if (_pEditView->getHSelf() == wnd)
			{
				view_to_focus = otherView();
				if (!viewVisible(view_to_focus)) view_to_focus = _activeView;
			}
			else
			{
				view_to_focus = currentView();
			}
			switchEditViewTo(view_to_focus);
			break;
		}

		case IDM_TOOL_MD5_GENERATE:
		{
			bool isFirstTime = !_md5FromTextDlg.isCreated();
			_md5FromTextDlg.doDialog(_nativeLangSpeaker.isRTL());
			if (isFirstTime)
				_nativeLangSpeaker.changeDlgLang(_md5FromTextDlg.getHSelf(), "MD5FromTextDlg");
		}
		break;

		case IDM_TOOL_MD5_GENERATEFROMFILE:
		{
			bool isFirstTime = !_md5FromFilesDlg.isCreated();
			_md5FromFilesDlg.doDialog(_nativeLangSpeaker.isRTL());
			if (isFirstTime)
				_nativeLangSpeaker.changeDlgLang(_md5FromFilesDlg.getHSelf(), "MD5FromFilesDlg");
		}
		break;

		case IDM_TOOL_MD5_GENERATEINTOCLIPBOARD:
		{
			if (_pEditView->execute(SCI_GETSELECTIONS) == 1)
			{
				size_t selectionStart = _pEditView->execute(SCI_GETSELECTIONSTART);
				size_t selectionEnd = _pEditView->execute(SCI_GETSELECTIONEND);

				int32_t strLen = static_cast<int32_t>(selectionEnd - selectionStart);
				if (strLen)
				{
					int strSize = strLen + 1;
					char *selectedStr = new char[strSize];
					_pEditView->execute(SCI_GETSELTEXT, 0, reinterpret_cast<LPARAM>(selectedStr));

					MD5 md5;
					std::string md5ResultA = md5.digestString(selectedStr);
					std::wstring md5ResultW(md5ResultA.begin(), md5ResultA.end());
					str2Clipboard(md5ResultW, _pPublicInterface->getHSelf());
					
					delete [] selectedStr;
				}
			}
		}
		break;

		case IDM_DEBUGINFO:
		{
			_debugInfoDlg.doDialog();
			break;
		}

        case IDM_ABOUT:
		{
#ifdef EASTER_EGG
			bool doAboutDlg = false;
			const int maxSelLen = 32;
			auto textLen = _pEditView->execute(SCI_GETSELTEXT, 0, 0) - 1;
			if (!textLen)
				doAboutDlg = true;
			if (textLen > maxSelLen)
				doAboutDlg = true;

			if (!doAboutDlg)
			{
				char author[maxSelLen+1] = "";
				_pEditView->getSelectedText(author, maxSelLen + 1);
				int iQuote = getQuoteIndexFrom(author);

				if (iQuote == -1)
				{
					doAboutDlg = true;
				}
				else if (iQuote == -2)
				{
					generic_string noEasterEggsPath((NppParameters::getInstance())->getNppPath());
					noEasterEggsPath.append(TEXT("\\noEasterEggs.xml"));
					if (!::PathFileExists(noEasterEggsPath.c_str()))
						showAllQuotes();
					return;
				}
				if (iQuote != -1)
				{
					generic_string noEasterEggsPath((NppParameters::getInstance())->getNppPath());
					noEasterEggsPath.append(TEXT("\\noEasterEggs.xml"));
					if (!::PathFileExists(noEasterEggsPath.c_str()))
						showQuoteFromIndex(iQuote);
					return;
				}
			}

			if (doAboutDlg)
#endif
			{
				bool isFirstTime = !_aboutDlg.isCreated();
				_aboutDlg.doDialog();
				if (isFirstTime && _nativeLangSpeaker.getNativeLangA())
				{
					if (_nativeLangSpeaker.getLangEncoding() == NPP_CP_BIG5)
					{
						char *authorName = "侯今吾";
						HWND hItem = ::GetDlgItem(_aboutDlg.getHSelf(), IDC_AUTHOR_NAME);

						WcharMbcsConvertor *wmc = WcharMbcsConvertor::getInstance();
						const wchar_t *authorNameW = wmc->char2wchar(authorName, NPP_CP_BIG5);
						::SetWindowText(hItem, authorNameW);
					}
				}
			}
			break;
		}

		case IDM_HELP :
		{
			generic_string tmp((NppParameters::getInstance())->getNppPath());
			generic_string nppHelpPath = tmp.c_str();

			nppHelpPath += TEXT("\\user.manual\\documentation\\notepad-online-document.html");
			if (::PathFileExists(nppHelpPath.c_str()))
				::ShellExecute(NULL, TEXT("open"), nppHelpPath.c_str(), NULL, NULL, SW_SHOWNORMAL);
			else
			{
				generic_string msg = nppHelpPath;
				generic_string warning, title;
				if (!_nativeLangSpeaker.getMsgBoxLang("NppHelpAbsentWarning", title, warning))
				{
					title = TEXT("File does not exist");
					warning = TEXT("\rdoesn't exist. Please download it on Notepad++ site.");
				}
				msg += warning;
				::MessageBox(_pPublicInterface->getHSelf(), msg.c_str(), title.c_str(), MB_OK);
			}
		}
		break;

		case IDM_HOMESWEETHOME :
		{
			::ShellExecute(NULL, TEXT("open"), TEXT("https://notepad-plus-plus.org/"), NULL, NULL, SW_SHOWNORMAL);
			break;
		}
		case IDM_PROJECTPAGE :
		{
			::ShellExecute(NULL, TEXT("open"), TEXT("https://github.com/notepad-plus-plus/notepad-plus-plus/"), NULL, NULL, SW_SHOWNORMAL);
			break;
		}

		case IDM_ONLINEHELP:
		{
			::ShellExecute(NULL, TEXT("open"), TEXT("http://docs.notepad-plus-plus.org/"), NULL, NULL, SW_SHOWNORMAL);
			break;
		}

		case IDM_CMDLINEARGUMENTS:
		{
			::MessageBox(NULL, COMMAND_ARG_HELP, TEXT("Notepad++ Command Argument Help"), MB_OK);
			break;
		}

		case IDM_FORUM:
		{
			::ShellExecute(NULL, TEXT("open"), TEXT("https://notepad-plus-plus.org/community/"), NULL, NULL, SW_SHOWNORMAL);
			break;
		}

		case IDM_ONLINESUPPORT:
		{
			::ShellExecute(NULL, TEXT("open"), TEXT("https://gitter.im/notepad-plus-plus/notepad-plus-plus"), NULL, NULL, SW_SHOWNORMAL);
			break;
		}

		case IDM_PLUGINSHOME:
		{
			::ShellExecute(NULL, TEXT("open"), TEXT("http://docs.notepad-plus-plus.org/index.php/Plugin_Central"), NULL, NULL, SW_SHOWNORMAL);
			break;
		}

		case IDM_UPDATE_NPP :
		case IDM_CONFUPDATERPROXY :
		{
			// wingup doesn't work with the obsolete security layer (API) under xp since downloadings are secured with SSL on notepad_plus_plus.org
			winVer ver = NppParameters::getInstance()->getWinVersion();
			if (ver <= WV_XP)
			{
				long res = ::MessageBox(NULL, TEXT("Notepad++ updater is not compatible with XP due to the obsolete security layer under XP.\rDo you want to go to Notepad++ page to download the latest version?"), TEXT("Notepad++ Updater"), MB_YESNO);
				if (res == IDYES)
				{
					::ShellExecute(NULL, TEXT("open"), TEXT("https://notepad-plus-plus.org/download/"), NULL, NULL, SW_SHOWNORMAL);
				}
			}
			else
			{
				generic_string updaterDir = (NppParameters::getInstance())->getNppPath();
				PathAppend(updaterDir, TEXT("updater"));

				generic_string updaterFullPath = updaterDir;
				PathAppend(updaterFullPath, TEXT("gup.exe"));

				generic_string param;
				if (id == IDM_CONFUPDATERPROXY)
				{
					param = TEXT("-options");
				}
				else
				{
					param = TEXT("-verbose -v");
					param += VERSION_VALUE;

					if (NppParameters::getInstance()->isx64())
					{
						param += TEXT(" -px64");
					}
				}
				Process updater(updaterFullPath.c_str(), param.c_str(), updaterDir.c_str());

				updater.run();
			}
			break;
		}

		case IDM_EDIT_AUTOCOMPLETE :
			showAutoComp();
			break;

		case IDM_EDIT_AUTOCOMPLETE_CURRENTFILE :
			autoCompFromCurrentFile();
			break;

		case IDM_EDIT_AUTOCOMPLETE_PATH :
			showPathCompletion();
			break;

		case IDM_EDIT_FUNCCALLTIP :
			showFunctionComp();
			break;

        case IDM_LANGSTYLE_CONFIG_DLG :
		{
			bool isFirstTime = !_configStyleDlg.isCreated();
			_configStyleDlg.doDialog(_nativeLangSpeaker.isRTL());
			if (isFirstTime)
                _nativeLangSpeaker.changeConfigLang(_configStyleDlg.getHSelf());
			break;
		}

        case IDM_LANG_C	:
        case IDM_LANG_CPP :
        case IDM_LANG_JAVA :
        case IDM_LANG_CS :
        case IDM_LANG_HTML :
        case IDM_LANG_XML :
        case IDM_LANG_JS :
		case IDM_LANG_JSON :
        case IDM_LANG_PHP :
        case IDM_LANG_ASP :
        case IDM_LANG_CSS :
        case IDM_LANG_LUA :
        case IDM_LANG_PERL :
        case IDM_LANG_PYTHON :
        case IDM_LANG_PASCAL :
        case IDM_LANG_BATCH :
        case IDM_LANG_OBJC :
        case IDM_LANG_VB :
        case IDM_LANG_SQL :
        case IDM_LANG_ASCII :
        case IDM_LANG_TEXT :
        case IDM_LANG_RC :
        case IDM_LANG_MAKEFILE :
        case IDM_LANG_INI :
        case IDM_LANG_TEX :
        case IDM_LANG_FORTRAN :
		case IDM_LANG_FORTRAN_77 :
        case IDM_LANG_BASH :
        case IDM_LANG_FLASH :
		case IDM_LANG_NSIS :
		case IDM_LANG_TCL :
		case IDM_LANG_LISP :
		case IDM_LANG_SCHEME :
		case IDM_LANG_ASM :
		case IDM_LANG_DIFF :
		case IDM_LANG_PROPS :
		case IDM_LANG_PS:
		case IDM_LANG_RUBY:
		case IDM_LANG_SMALLTALK:
		case IDM_LANG_VHDL :
        case IDM_LANG_KIX :
        case IDM_LANG_CAML :
        case IDM_LANG_ADA :
        case IDM_LANG_VERILOG :
		case IDM_LANG_MATLAB :
		case IDM_LANG_HASKELL :
        case IDM_LANG_AU3 :
		case IDM_LANG_INNO :
		case IDM_LANG_CMAKE :
		case IDM_LANG_YAML :
        case IDM_LANG_COBOL :
        case IDM_LANG_D :
        case IDM_LANG_GUI4CLI :
        case IDM_LANG_POWERSHELL :
        case IDM_LANG_R :
        case IDM_LANG_JSP :
		case IDM_LANG_COFFEESCRIPT:
		case IDM_LANG_BAANC:
		case IDM_LANG_SREC:
		case IDM_LANG_IHEX:
		case IDM_LANG_TEHEX:
		case IDM_LANG_SWIFT:
        case IDM_LANG_ASN1 :
        case IDM_LANG_AVS :
        case IDM_LANG_BLITZBASIC :
        case IDM_LANG_PUREBASIC :
        case IDM_LANG_FREEBASIC :
        case IDM_LANG_CSOUND :
        case IDM_LANG_ERLANG :
        case IDM_LANG_ESCRIPT :
        case IDM_LANG_FORTH :
        case IDM_LANG_LATEX :
        case IDM_LANG_MMIXAL :
        case IDM_LANG_NIMROD :
        case IDM_LANG_NNCRONTAB :
        case IDM_LANG_OSCRIPT :
        case IDM_LANG_REBOL :
        case IDM_LANG_REGISTRY :
        case IDM_LANG_RUST :
        case IDM_LANG_SPICE :
        case IDM_LANG_TXT2TAGS :
        case IDM_LANG_VISUALPROLOG:
#ifdef MOD_JAPANESE_AA
        case IDM_LANG_JAPANESE_AA:
#endif
		case IDM_LANG_USER :
		{
            setLanguage(menuID2LangType(id));
			// Manually set language, don't change language even file extension changes.
			Buffer *buffer = _pEditView->getCurrentBuffer();
			buffer->langHasBeenSetFromMenu();

			if (_pDocMap)
			{
				_pDocMap->setSyntaxHiliting();
			}
		}
        break;

        case IDC_PREV_DOC :
        case IDC_NEXT_DOC :
        {
			size_t nbDoc = viewVisible(MAIN_VIEW) ? _mainDocTab.nbItem() : 0;
			nbDoc += viewVisible(SUB_VIEW)?_subDocTab.nbItem():0;

			bool doTaskList = ((NppParameters::getInstance())->getNppGUI())._doTaskList;
			if (nbDoc > 1)
			{
				bool direction = (id == IDC_NEXT_DOC)?dirDown:dirUp;
				if (!doTaskList)
				{
					activateNextDoc(direction);
				}
				else
				{
					if (TaskListDlg::_instanceCount == 0)
					{
						TaskListDlg tld;
						HIMAGELIST hImgLst = _docTabIconList.getHandle();
						tld.init(_pPublicInterface->getHinst(), _pPublicInterface->getHSelf(), hImgLst, direction);
						tld.doDialog();
					}
				}
			}
			_linkTriggered = true;
		}
        break;

		case IDM_OPEN_ALL_RECENT_FILE :
		{
			BufferID lastOne = BUFFER_INVALID;
			int size = _lastRecentFileList.getSize();
			for (int i = size - 1; i >= 0; i--)
			{
				BufferID test = doOpen(_lastRecentFileList.getIndex(i));
				if (test != BUFFER_INVALID)
					lastOne = test;
			}
			if (lastOne != BUFFER_INVALID)
			{
				switchToFile(lastOne);
			}
			break;
		}

		case IDM_CLEAN_RECENT_FILE_LIST :
			_lastRecentFileList.clear();
			break;

		case IDM_EDIT_RTL :
		case IDM_EDIT_LTR :
		{
			_pEditView->changeTextDirection(id == IDM_EDIT_RTL);

			// Wrap then !wrap to fix problem of mirror characters
			bool isWraped = _pEditView->isWrap();
			_pEditView->wrap(!isWraped);
			_pEditView->wrap(isWraped);
			if (_pDocMap)
			{
				_pDocMap->changeTextDirection(id == IDM_EDIT_RTL);
			}
		}
		break;

		case IDM_WINDOW_WINDOWS :
		{
			WindowsDlg _windowsDlg;
			_windowsDlg.init(_pPublicInterface->getHinst(), _pPublicInterface->getHSelf(), _pDocTab);

            const TiXmlNodeA *nativeLangA = _nativeLangSpeaker.getNativeLangA();
			TiXmlNodeA *dlgNode = NULL;
			if (nativeLangA)
			{
				dlgNode = nativeLangA->FirstChild("Dialog");
				if (dlgNode)
					dlgNode = _nativeLangSpeaker.searchDlgNode(dlgNode, "Window");
			}
			_windowsDlg.doDialog(dlgNode);
		}
		break;


		case IDM_SYSTRAYPOPUP_NEWDOC:
		{
			NppGUI & nppGUI = const_cast<NppGUI &>((NppParameters::getInstance())->getNppGUI());
			::ShowWindow(_pPublicInterface->getHSelf(), nppGUI._isMaximized?SW_MAXIMIZE:SW_SHOW);
			fileNew();
		}
		break;

		case IDM_SYSTRAYPOPUP_ACTIVATE :
		{
			NppGUI & nppGUI = const_cast<NppGUI &>((NppParameters::getInstance())->getNppGUI());
			::ShowWindow(_pPublicInterface->getHSelf(), nppGUI._isMaximized?SW_MAXIMIZE:SW_SHOW);

			// Send sizing info to make window fit (specially to show tool bar. Fixed issue #2600)
			::SendMessage(_pPublicInterface->getHSelf(), WM_SIZE, 0, 0);
		}
		break;

		case IDM_SYSTRAYPOPUP_NEW_AND_PASTE:
		{
			NppGUI & nppGUI = const_cast<NppGUI &>((NppParameters::getInstance())->getNppGUI());
			::ShowWindow(_pPublicInterface->getHSelf(), nppGUI._isMaximized?SW_MAXIMIZE:SW_SHOW);
			BufferID bufferID = _pEditView->getCurrentBufferID();
			Buffer * buf = MainFileManager->getBufferByID(bufferID);
			if (!buf->isUntitled() || buf->docLength() != 0)
			{
				fileNew();
			}
			command(IDM_EDIT_PASTE);
		}
		break;

		case IDM_SYSTRAYPOPUP_OPENFILE:
		{
			NppGUI & nppGUI = const_cast<NppGUI &>((NppParameters::getInstance())->getNppGUI());
			::ShowWindow(_pPublicInterface->getHSelf(), nppGUI._isMaximized?SW_MAXIMIZE:SW_SHOW);

			// Send sizing info to make window fit (specially to show tool bar. Fixed issue #2600)
			::SendMessage(_pPublicInterface->getHSelf(), WM_SIZE, 0, 0);
			fileOpen();
		}
		break;

		case IDM_SYSTRAYPOPUP_CLOSE:
		{
			_pPublicInterface->setIsPrelaunch(false);
			_pTrayIco->doTrayIcon(REMOVE);
			if (!::IsWindowVisible(_pPublicInterface->getHSelf()))
				::SendMessage(_pPublicInterface->getHSelf(), WM_CLOSE, 0,0);
		}
		break;

		case IDM_FILE_RESTORELASTCLOSEDFILE:
		{
			generic_string lastOpenedFullPath = _lastRecentFileList.getFirstItem();
			if (not lastOpenedFullPath.empty())
			{
				BufferID lastOpened = doOpen(lastOpenedFullPath);
				if (lastOpened != BUFFER_INVALID)
					switchToFile(lastOpened);
			}
		}
		break;

		case IDM_VIEW_LINENUMBER:
		case IDM_VIEW_SYMBOLMARGIN:
		case IDM_VIEW_DOCCHANGEMARGIN:
		{
			int margin;
			if (id == IDM_VIEW_LINENUMBER)
				margin = ScintillaEditView::_SC_MARGE_LINENUMBER;
			else //if (id == IDM_VIEW_SYMBOLMARGIN)
				margin = ScintillaEditView::_SC_MARGE_SYBOLE;

			if (_mainEditView.hasMarginShowed(margin))
			{
				_mainEditView.showMargin(margin, false);
				_subEditView.showMargin(margin, false);
			}
			else
			{
				_mainEditView.showMargin(margin);
				_subEditView.showMargin(margin);
			}
		}
		break;

		case IDM_VIEW_FOLDERMAGIN_SIMPLE:
		case IDM_VIEW_FOLDERMAGIN_ARROW:
		case IDM_VIEW_FOLDERMAGIN_CIRCLE:
		case IDM_VIEW_FOLDERMAGIN_BOX:
		case IDM_VIEW_FOLDERMAGIN:
		{
			folderStyle fStyle = (id == IDM_VIEW_FOLDERMAGIN_SIMPLE) ? FOLDER_STYLE_SIMPLE : \
				(id == IDM_VIEW_FOLDERMAGIN_ARROW) ? FOLDER_STYLE_ARROW : \
				(id == IDM_VIEW_FOLDERMAGIN_CIRCLE) ? FOLDER_STYLE_CIRCLE : \
				(id == IDM_VIEW_FOLDERMAGIN) ? FOLDER_STYLE_NONE : FOLDER_STYLE_BOX;

			_mainEditView.setMakerStyle(fStyle);
			_subEditView.setMakerStyle(fStyle);
		}
		break;

		case IDM_VIEW_CURLINE_HILITING:
		{
			COLORREF colour = (NppParameters::getInstance())->getCurLineHilitingColour();
			_mainEditView.setCurrentLineHiLiting(!_pEditView->isCurrentLineHiLiting(), colour);
			_subEditView.setCurrentLineHiLiting(!_pEditView->isCurrentLineHiLiting(), colour);
		}
		break;

#ifdef MOD_EDGE
		case IDM_VIEW_EDGE:
		{
			_pEditView->execute(SCI_SETEDGEMODE, NppParameters::getInstance()->getSVP()._edgeMode);	// EDGE_LINE
		}
		break;
#else
		case IDM_VIEW_EDGEBACKGROUND:
		case IDM_VIEW_EDGELINE:
		case IDM_VIEW_EDGENONE:
		{
			int mode;
			switch (id)
			{
			case IDM_VIEW_EDGELINE:
			{
				mode = EDGE_LINE;
				break;
			}
			case IDM_VIEW_EDGEBACKGROUND:
			{
				mode = EDGE_BACKGROUND;
				break;
			}
			default:
				mode = EDGE_NONE;
			}
			_mainEditView.execute(SCI_SETEDGEMODE, mode);
			_subEditView.execute(SCI_SETEDGEMODE, mode);
		}
		break;
#endif

		case IDM_VIEW_LWDEF:
		case IDM_VIEW_LWALIGN:
		case IDM_VIEW_LWINDENT:
		{
			int mode = (id == IDM_VIEW_LWALIGN) ? SC_WRAPINDENT_SAME : \
				(id == IDM_VIEW_LWINDENT) ? SC_WRAPINDENT_INDENT : SC_WRAPINDENT_FIXED;
			_mainEditView.execute(SCI_SETWRAPINDENTMODE, mode);
			_subEditView.execute(SCI_SETWRAPINDENTMODE, mode);
		}
		break;
#ifdef JAPANESE_STYLE_WORDWRAP
		case IDM_WRAPMODE_WORD:
		case IDM_WRAPMODE_CHAR:
		case IDM_WRAPMODE_WHITESPACE:
		{
			bool isWraped = _pEditView->isWrap();
			_pEditView->wrap(!isWraped);
			_pEditView->wrap(isWraped);
		}
		break;
#endif
#ifdef LINE_SPACING
		case IDM_LINESPACINGRESTORE:
			process(0, NPPM_SETLINESPACING, 0, 0);
			break;
	//	case IDM_LINESPACINGDECREMENT:
	//		process(0, NPPM_SETLINESPACING, NppParameters::getInstance()->getSVP().lineSpacing - 1, 0);
	//		break;
	//	case IDM_LINESPACINGINCREMENT:
	//		process(0, NPPM_SETLINESPACING, NppParameters::getInstance()->getSVP().lineSpacing + 1, 0);
	//		break;
#endif
#ifdef MOD_ESCAPE
		case IDM_ESCAPE:
		{
			const auto recordingMacro = _pEditView->execute(SCI_STOPRECORD);	// SCI_CANCEL
			_pEditView->execute(SCI_CANCEL);
			_pEditView->execute(SCI_SETANCHOR, _pEditView->execute(SCI_GETCURRENTPOS));
			if (recordingMacro) _pEditView->execute(SCI_STARTRECORD);
		}
		break;
#endif

		default :
			if (id > IDM_FILEMENU_LASTONE && id < (IDM_FILEMENU_LASTONE + _lastRecentFileList.getMaxNbLRF() + 1))
			{
				BufferID lastOpened = doOpen(_lastRecentFileList.getItem(id));
				if (lastOpened != BUFFER_INVALID)
				{
					switchToFile(lastOpened);
				}
			}
			else if ((id > IDM_LANG_USER) && (id < IDM_LANG_USER_LIMIT))
			{
				TCHAR langName[langNameLenMax];
				::GetMenuString(_mainMenuHandle, id, langName, langNameLenMax, MF_BYCOMMAND);
				_pEditView->getCurrentBuffer()->setLangType(L_USER, langName);
				if (_pDocMap)
				{
					_pDocMap->setSyntaxHiliting();
				}
			}
			else if ((id >= IDM_LANG_EXTERNAL) && (id <= IDM_LANG_EXTERNAL_LIMIT))
			{
				setLanguage((LangType)(id - IDM_LANG_EXTERNAL + L_EXTERNAL));
				if (_pDocMap)
				{
					_pDocMap->setSyntaxHiliting();
				}
			}
			else if ((id >= ID_MACRO) && (id < ID_MACRO_LIMIT))
			{
				int i = id - ID_MACRO;
				vector<MacroShortcut> & theMacros = (NppParameters::getInstance())->getMacroList();
				macroPlayback(theMacros[i].getMacro());
			}
			else if ((id >= ID_USER_CMD) && (id < ID_USER_CMD_LIMIT))
			{
				int i = id - ID_USER_CMD;
				vector<UserCommand> & theUserCommands = (NppParameters::getInstance())->getUserCommandList();
				UserCommand ucmd = theUserCommands[i];

				Command cmd(ucmd.getCmd());
				cmd.run(_pPublicInterface->getHSelf());
			}
			else if ((id >= ID_PLUGINS_CMD) && (id < ID_PLUGINS_CMD_LIMIT))
			{
				int i = id - ID_PLUGINS_CMD;
				_pluginsManager.runPluginCommand(i);
			}
			else if (_pluginsManager.inDynamicRange(id)) // in the dynamic range allocated with NPPM_ALLOCATECMDID
			{
				_pluginsManager.relayNppMessages(WM_COMMAND, id, 0);
			}
/*UNLOAD
			else if ((id >= ID_PLUGINS_REMOVING) && (id < ID_PLUGINS_REMOVING_END))
			{
				int i = id - ID_PLUGINS_REMOVING;
				_pluginsManager.unloadPlugin(i, _pPublicInterface->getHSelf());
			}
*/
			else if ((id >= IDM_WINDOW_MRU_FIRST) && (id <= IDM_WINDOW_MRU_LIMIT))
			{
				activateDoc(id-IDM_WINDOW_MRU_FIRST);
			}
#ifdef MOD_MENU_DIRECT_SCI
			else if (2000 < id && id < 3000)	/*SCI_START ... SCI_OPTIONAL_START*/
				_pEditView->execute(id, 0, 0);
#endif
	}

	if (_recordingMacro)
		switch (id)
		{
 //TODO:	case IDM_EDIT_PASTE_AS_HTML:
 //TODO:	case IDM_EDIT_PASTE_AS_RTF:
 //TODO:	case IDM_EDIT_PASTE_BINARY:
 //TODO:	case IDM_SEARCH_FINDCHARINRANGE:
			case IDM_FILE_NEW :
			case IDM_FILE_CLOSE :
			case IDM_FILE_CLOSEALL :
			case IDM_FILE_CLOSEALL_BUT_CURRENT :
			case IDM_FILE_CLOSEALL_TOLEFT :
			case IDM_FILE_CLOSEALL_TORIGHT :
			case IDM_FILE_SAVE :
			case IDM_FILE_SAVEALL :
			case IDM_FILE_RELOAD:
			case IDM_EDIT_UNDO:
			case IDM_EDIT_REDO:
			case IDM_EDIT_CUT:
			case IDM_EDIT_COPY:
			//case IDM_EDIT_PASTE:
			case IDM_EDIT_DELETE:
			case IDM_SEARCH_FINDNEXT :
			case IDM_SEARCH_FINDPREV :
            case IDM_SEARCH_SETANDFINDNEXT :
			case IDM_SEARCH_SETANDFINDPREV :
			case IDM_SEARCH_GOTOMATCHINGBRACE :
			case IDM_SEARCH_SELECTMATCHINGBRACES :
			case IDM_SEARCH_TOGGLE_BOOKMARK :
			case IDM_SEARCH_NEXT_BOOKMARK:
			case IDM_SEARCH_PREV_BOOKMARK:
			case IDM_SEARCH_CLEAR_BOOKMARKS:
			case IDM_EDIT_SELECTALL:
#ifdef MOD_EDIT_SELECTWORD
			case IDM_EDIT_SELECTWORD:
#endif
			case IDM_EDIT_INS_TAB:
			case IDM_EDIT_RMV_TAB:
			case IDM_EDIT_DUP_LINE:
			case IDM_EDIT_TRANSPOSE_LINE:
			case IDM_EDIT_SPLIT_LINES:
			case IDM_EDIT_JOIN_LINES:
			case IDM_EDIT_LINE_UP:
			case IDM_EDIT_LINE_DOWN:
			case IDM_EDIT_REMOVEEMPTYLINES:
			case IDM_EDIT_REMOVEEMPTYLINESWITHBLANK:
			case IDM_EDIT_UPPERCASE:
			case IDM_EDIT_LOWERCASE:
			case IDM_EDIT_PROPERCASE_FORCE:
			case IDM_EDIT_PROPERCASE_BLEND:
			case IDM_EDIT_SENTENCECASE_FORCE:
			case IDM_EDIT_SENTENCECASE_BLEND:
			case IDM_EDIT_INVERTCASE:
			case IDM_EDIT_RANDOMCASE:
#ifdef MOD_EDIT_CONVERT_KANJI
			case IDM_EDIT_FULLWIDTH:
			case IDM_EDIT_HALFWIDTH:
			case IDM_EDIT_KATAKANA:
			case IDM_EDIT_HIRAGANA:
#endif
			case IDM_EDIT_BLOCK_COMMENT:
			case IDM_EDIT_BLOCK_COMMENT_SET:
			case IDM_EDIT_BLOCK_UNCOMMENT:
			case IDM_EDIT_STREAM_COMMENT:
			case IDM_EDIT_TRIMTRAILING:
			case IDM_EDIT_TRIMLINEHEAD:
			case IDM_EDIT_TRIM_BOTH:
			case IDM_EDIT_EOL2WS:
			case IDM_EDIT_TRIMALL:
			case IDM_EDIT_TAB2SW:
#ifdef MOD_EDIT_TAB2SW_LEADING
			case IDM_EDIT_TAB2SW_LEADING:
#endif
			case IDM_EDIT_SW2TAB_ALL:
			case IDM_EDIT_SW2TAB_LEADING:
			case IDM_EDIT_SETREADONLY :
			case IDM_EDIT_FULLPATHTOCLIP :
			case IDM_EDIT_FILENAMETOCLIP :
			case IDM_EDIT_CURRENTDIRTOCLIP :
			case IDM_EDIT_CLEARREADONLY :
			case IDM_EDIT_RTL :
			case IDM_EDIT_LTR :
			case IDM_EDIT_BEGINENDSELECT:
			case IDM_EDIT_SORTLINES_LEXICOGRAPHIC_ASCENDING:
			case IDM_EDIT_SORTLINES_LEXICOGRAPHIC_DESCENDING:
#ifdef MOD_SORTLINES_FILENAME
			case IDM_EDIT_SORTLINES_FILENAME_ASCENDING:
			case IDM_EDIT_SORTLINES_FILENAME_DESCENDING:
#endif
			case IDM_EDIT_SORTLINES_INTEGER_ASCENDING:
			case IDM_EDIT_SORTLINES_INTEGER_DESCENDING:
			case IDM_EDIT_SORTLINES_DECIMALCOMMA_ASCENDING:
			case IDM_EDIT_SORTLINES_DECIMALCOMMA_DESCENDING:
			case IDM_EDIT_SORTLINES_DECIMALDOT_ASCENDING:
			case IDM_EDIT_SORTLINES_DECIMALDOT_DESCENDING:
			case IDM_EDIT_BLANKLINEABOVECURRENT:
			case IDM_EDIT_BLANKLINEBELOWCURRENT:
			case IDM_VIEW_FULLSCREENTOGGLE :
			case IDM_VIEW_ALWAYSONTOP :
#ifdef MOD_EACH_VIEW_TABSIZE
			case IDM_VIEW_TABSIZE_2:
			case IDM_VIEW_TABSIZE_4:
			case IDM_VIEW_TABSIZE_8:
#endif
#ifdef MOD_EDGE
			case IDM_VIEW_WRAP_NONE:
			case IDM_VIEW_WRAP_EDGE:
			case IDM_VIEW_WRAP_WINDOW:
#else
			case IDM_VIEW_WRAP :
#endif
			case IDM_VIEW_FOLD_CURRENT :
			case IDM_VIEW_UNFOLD_CURRENT :
			case IDM_VIEW_TOGGLE_FOLDALL:
			case IDM_VIEW_TOGGLE_UNFOLDALL:
			case IDM_VIEW_FOLD_1:
			case IDM_VIEW_FOLD_2:
			case IDM_VIEW_FOLD_3:
			case IDM_VIEW_FOLD_4:
			case IDM_VIEW_FOLD_5:
			case IDM_VIEW_FOLD_6:
			case IDM_VIEW_FOLD_7:
			case IDM_VIEW_FOLD_8:
			case IDM_VIEW_UNFOLD_1:
			case IDM_VIEW_UNFOLD_2:
			case IDM_VIEW_UNFOLD_3:
			case IDM_VIEW_UNFOLD_4:
			case IDM_VIEW_UNFOLD_5:
			case IDM_VIEW_UNFOLD_6:
			case IDM_VIEW_UNFOLD_7:
			case IDM_VIEW_UNFOLD_8:
			case IDM_VIEW_GOTO_ANOTHER_VIEW:
			case IDM_VIEW_SYNSCROLLV:
			case IDM_VIEW_SYNSCROLLH:
			case IDM_VIEW_TAB1:
			case IDM_VIEW_TAB2:
			case IDM_VIEW_TAB3:
			case IDM_VIEW_TAB4:
			case IDM_VIEW_TAB5:
			case IDM_VIEW_TAB6:
			case IDM_VIEW_TAB7:
			case IDM_VIEW_TAB8:
			case IDM_VIEW_TAB9:
#ifdef MOD_VIEW_TAB_RECENT
			case IDM_VIEW_TAB_RECENT:
#endif
			case IDM_VIEW_TAB_NEXT:
			case IDM_VIEW_TAB_PREV:
			case IDM_VIEW_TAB_MOVEFORWARD:
			case IDM_VIEW_TAB_MOVEBACKWARD:
			case IDC_PREV_DOC :
			case IDC_NEXT_DOC :
			case IDM_SEARCH_GOPREVMARKER1   :
			case IDM_SEARCH_GOPREVMARKER2   :
			case IDM_SEARCH_GOPREVMARKER3   :
			case IDM_SEARCH_GOPREVMARKER4   :
			case IDM_SEARCH_GOPREVMARKER5   :
			case IDM_SEARCH_GOPREVMARKER_DEF:
			case IDM_SEARCH_GONEXTMARKER1   :
			case IDM_SEARCH_GONEXTMARKER2   :
			case IDM_SEARCH_GONEXTMARKER3   :
			case IDM_SEARCH_GONEXTMARKER4   :
			case IDM_SEARCH_GONEXTMARKER5   :
			case IDM_SEARCH_GONEXTMARKER_DEF:
			case IDM_SEARCH_VOLATILE_FINDNEXT:
			case IDM_SEARCH_VOLATILE_FINDPREV:
			case IDM_SEARCH_CUTMARKEDLINES   :
			case IDM_SEARCH_COPYMARKEDLINES     :
			case IDM_SEARCH_PASTEMARKEDLINES    :
			case IDM_SEARCH_DELETEMARKEDLINES   :
			case IDM_SEARCH_DELETEUNMARKEDLINES :
			case IDM_SEARCH_MARKALLEXT1      :
			case IDM_SEARCH_UNMARKALLEXT1    :
			case IDM_SEARCH_MARKALLEXT2      :
			case IDM_SEARCH_UNMARKALLEXT2    :
			case IDM_SEARCH_MARKALLEXT3      :
			case IDM_SEARCH_UNMARKALLEXT3    :
			case IDM_SEARCH_MARKALLEXT4      :
			case IDM_SEARCH_UNMARKALLEXT4    :
			case IDM_SEARCH_MARKALLEXT5      :
			case IDM_SEARCH_UNMARKALLEXT5    :
			case IDM_SEARCH_CLEARALLMARKS    :
#ifdef MOD_INDIRECT_JUMP
			case IDM_SEARCH_INDIRECTJUMP:
#endif
#ifdef MOD_SEARCH_GOTOBRACKET
			case IDM_SEARCH_GOTOLEFTCURLYBRACKET:
			case IDM_SEARCH_GOTORIGHTCURLYBRACKET:
#endif
#ifdef LINE_SPACING
			case IDM_LINESPACINGRESTORE:
		//	case IDM_LINESPACINGDECREMENT:
		//	case IDM_LINESPACINGINCREMENT:
#endif
#ifdef MOD_ESCAPE
			case IDM_ESCAPE:
#endif
#ifdef ADD_BY_JOJO // macro
			case IDM_FILE_RESTORELASTCLOSEDFILE:
			case IDM_VIEW_SWITCHTO_OTHER_VIEW:
#endif
				_macro.push_back(recordedMacroStep(id));
				break;
		}
}
