/**
 * Live2D SDK for DirectX 
 *
 *  You can modify and use this source freely
 *  only for the development of application related Live2D.
 *
 *  (c) Live2D Inc. All rights reserved.
 */

#include <Windowsx.h>
#include "main.h"
#include "Live2D.h"
#include "util/UtSystem.h"
#include "Live2DModelD3D11.h"

// Oo̓}N
#define LOGD(fmt, ...) live2d::UtDebug::println(fmt, __VA_ARGS__)
#define LOGE(fmt, ...) live2d::UtDebug::error(fmt, __VA_ARGS__)

// HRESULT̃G[̏ڍׂ𕶎ƂĕԂ
static const char* getDXErrorString(HRESULT hr)
{
	static char buffer[512];
	DWORD errOrLen = ::FormatMessageA(FORMAT_MESSAGE_FROM_SYSTEM,
		NULL, hr, 0, buffer, sizeof(buffer) / sizeof(buffer[0]), NULL);

	if (errOrLen > 0)
	{
		// trim '\r\n'
		if (errOrLen > 2) {
			if (buffer[errOrLen - 1] == '\n' && buffer[errOrLen - 2] == '\r')
				buffer[errOrLen - 2] = '\0';
		}
		return buffer;
	}
	else
	{
		return "";
	}
}

// Utility Template
template<typename T>
inline static void SAFE_RELEASE(T& x)
{
	if (x) { x->Release(); x = NULL; }
}

template<typename T>
inline static void SAFE_RELEASE_ARRAY(T& array, size_t size)
{
	for (size_t i = 0; i < size; i++)
	{
		if (array[i]) { array[i]->Release(); array[i] = NULL; }
	}
}

INT WINAPI WinMain(HINSTANCE hInstance,HINSTANCE,LPSTR,INT)
{
	// fobO q[v }l[Wɂ郁蓖Ă̒ǐՕ@ݒ
	_CrtSetDbgFlag(_CRTDBG_ALLOC_MEM_DF | _CRTDBG_LEAK_CHECK_DF);

	Application* app = new Application;
	if (app != NULL)
	{
		if (SUCCEEDED(app->initWindow(hInstance, 0, 0, 
			INITIAL_WINDOW_SIZE, FULLSCREEN_SIZE, APP_NAME)))
		{
			if (SUCCEEDED(app->initGraphics()))
			{
				if (SUCCEEDED(app->initLive2D()))
				{
					if (SUCCEEDED(app->initModel()))
					{
						app->mainLoop();					
					}
				}
			}
		}

		app->destroyModel();
		app->destroyLive2D();
		app->destroyGraphics();
		delete app;
	}
	return 0;
}


LRESULT Application::wndProcWrapper(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	Application* app = reinterpret_cast<Application*>(GetWindowLongPtr(hWnd, GWLP_USERDATA));

	if (!app)
	{
		if (uMsg == WM_CREATE)
		{
			app = static_cast<Application*>(reinterpret_cast<LPCREATESTRUCT>(lParam)->lpCreateParams);
			if (app)
			{
				SetWindowLongPtr(hWnd, GWLP_USERDATA, reinterpret_cast<LONG_PTR>(app));
			}
		}
	}
	else
	{
		return app->wndProc(hWnd, uMsg, wParam, lParam);
	}

	return DefWindowProc(hWnd, uMsg, wParam, lParam);
}


Application::Application()
	: _hWnd(0)
	, _initialWindowSize()
	, _fullscreenSize()
	, _device(NULL)
	, _deviceContext(NULL)
	, _swapChain(NULL)
	, _backBufferRTV(NULL)
	, _standbyMode(false)
	, _prevWindowSize()
	, _renderer(NULL)
	, _live2DMgr(NULL)
{
}


Application::~Application()
{
}


LRESULT Application::wndProc(HWND hWnd, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
	switch (iMsg)
	{
	case WM_DESTROY:
		//  EChE
		PostQuitMessage(0);
		return 0;

		//  EChEETCY̕ύX
	case WM_SIZE:
		// N邢͍ŏ̏ꍇ̓TCY͍sȂ
		if (!_device || wParam == SIZE_MINIMIZED)
			break;
		if (wParam == SIZE_MAXIMIZED || wParam == SIZE_RESTORED)
		{
			int width = LOWORD(lParam);
			int height = HIWORD(lParam);
			resizeBackBuffer(width, height);
		}
		break;

	case WM_KEYDOWN:
		//  L[͂̏
		switch (wParam)
		{
		case VK_ESCAPE:	//  [ESCAPE]L[ŃEChE
			 PostMessage(hWnd, WM_CLOSE, 0, 0);
			break;

		case 'W':	//  ʃ[h̐؂ւ
			toggleFullscreen();
			break;
		}
		break;
#if ENABLE_LIVE2D
	case WM_RBUTTONDOWN:
		_live2DMgr->changeModel();
		break;

	case WM_LBUTTONDOWN:
		if ((wParam & MK_LBUTTON) != 0){
			int xPos = GET_X_LPARAM(lParam);
			int yPos = GET_Y_LPARAM(lParam);

			_renderer->mousePress(xPos, yPos);
		}
		break;

	case WM_MOUSEMOVE:
		if ((wParam & MK_LBUTTON) != 0){
			int xPos = GET_X_LPARAM(lParam);
			int yPos = GET_Y_LPARAM(lParam);

			_renderer->mouseDrag(xPos, yPos);
		}
		break;

	case WM_MOUSEWHEEL:
	{
		// zC[
		int delta = GET_WHEEL_DELTA_WPARAM(wParam);

		// XN[W烍[JWɕϊ
		POINT cursor;
		cursor.x = GET_X_LPARAM(lParam);
		cursor.y = GET_Y_LPARAM(lParam);
		ScreenToClient(hWnd, &cursor);

		_renderer->mouseWheel(delta, cursor.x, cursor.y);
	}
	break;
#endif
	}
	return DefWindowProc (hWnd, iMsg, wParam, lParam);
}


HRESULT Application::initWindow(HINSTANCE hInstance,
                                INT x, INT y, SIZE windowSize, SIZE fullscreenSize, LPCWSTR windowName)
{
	// EBhE̒`
	WNDCLASSEX wc;
	ZeroMemory(&wc, sizeof(wc));
	wc.cbSize = sizeof(wc);
	wc.style = CS_HREDRAW | CS_VREDRAW;
	wc.lpfnWndProc = Application::wndProcWrapper;
	wc.hInstance = hInstance;
	wc.hIcon = LoadIcon(NULL, IDI_APPLICATION);
	wc.hCursor = LoadCursor(NULL, IDC_ARROW);
	wc.hbrBackground = static_cast<HBRUSH>(GetStockObject(LTGRAY_BRUSH));
	wc.lpszClassName = windowName;
	wc.hIconSm = LoadIcon(NULL, IDI_APPLICATION);
	RegisterClassEx(&wc);

	_prevWindowSize = _initialWindowSize = windowSize;
	_fullscreenSize = fullscreenSize;

	// EBhE̍쐬
	_hWnd = CreateWindow(windowName, windowName, WS_OVERLAPPEDWINDOW,
		CW_USEDEFAULT, CW_USEDEFAULT, windowSize.cx, windowSize.cy, 0, 0, hInstance, this);
	if (!_hWnd)
	{
		return E_FAIL;
	}

	// EChE̕\
	ShowWindow(_hWnd, SW_SHOW);
	UpdateWindow(_hWnd);

	return S_OK;
}


HRESULT Application::initGraphics()
{
	// foCXƃXbv`F[̍쐬
	DXGI_SWAP_CHAIN_DESC sd;
	ZeroMemory(&sd, sizeof(sd));
	sd.BufferCount = 1;
	sd.BufferDesc.Width = _initialWindowSize.cx;
	sd.BufferDesc.Height = _initialWindowSize.cy;
	sd.BufferDesc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
	sd.BufferDesc.RefreshRate.Numerator = 60;
	sd.BufferDesc.RefreshRate.Denominator = 1;
	sd.BufferUsage = DXGI_USAGE_RENDER_TARGET_OUTPUT;
	sd.OutputWindow = _hWnd;
	sd.SampleDesc.Count = 1;
	sd.SampleDesc.Quality = 0;
	sd.Windowed = TRUE;
#if FULLSCREEN_FIXED_RESOLUTION
	sd.Flags = DXGI_SWAP_CHAIN_FLAG_ALLOW_MODE_SWITCH;
#endif	
	// Cubism SDK for DirectX11  Feature Level 10.0ȏT|[g
	// Feature Level 9.1`9.3͔T|[g
	D3D_FEATURE_LEVEL featureLevels[] = {
		D3D_FEATURE_LEVEL_11_0,
		D3D_FEATURE_LEVEL_10_1,
		D3D_FEATURE_LEVEL_10_0,
	};
	const int numFeatureLevel = sizeof(featureLevels) / sizeof(featureLevels[0]);
	D3D_FEATURE_LEVEL* featureLevel = NULL;

#if defined(DEBUG) || defined(_DEBUG)
	UINT createDeviceFlags = D3D11_CREATE_DEVICE_DEBUG; // fobOOo͂
#else
	UINT createDeviceFlags = 0;
#endif

	HRESULT hr;
	hr = D3D11CreateDeviceAndSwapChain(NULL, D3D_DRIVER_TYPE_HARDWARE, NULL,
		createDeviceFlags, featureLevels, numFeatureLevel, D3D11_SDK_VERSION, &sd,
		&_swapChain, &_device, featureLevel, &_deviceContext);
	if (FAILED(hr))
	{
		LOGE("initGraphics D3D11CreateDeviceAndSwapChain [%s]", getDXErrorString(hr));
		return hr;
	}

	hr = initBackBuffer();

	return hr;
}


HRESULT Application::initLive2D()
{
#if ENABLE_LIVE2D
	// Live2Df̃[h
	_live2DMgr = new LAppLive2DManager();
	_live2DMgr->setGraphicsContext(_device, _deviceContext);

	// Live2D`pNX
	_renderer = new LAppRenderer();
	_renderer->setLive2DManager(_live2DMgr);

	// _̃TCYw
	_renderer->setDeviceSize(_initialWindowSize.cx, _initialWindowSize.cy);

	_live2DMgr->changeModel();
#endif
	return S_OK;
}


HRESULT Application::initModel()
{
	// NOP
	return S_OK;
}


HRESULT Application::initBackBuffer()
{
	D3D11_TEXTURE2D_DESC descBackBuffer;
	HRESULT hr;

	// obNobt@[eNX`擾(ɂ̂ō쐬ł͂Ȃ)
	{
		ID3D11Texture2D *backBufferTex;
		hr = _swapChain->GetBuffer(0, __uuidof(ID3D11Texture2D), reinterpret_cast<LPVOID*>(&backBufferTex));
		if (FAILED(hr))
		{
			LOGE("initBackBuffer GetBuffer [%s]", getDXErrorString(hr));
			return hr;
		}

		// ̃eNX`[ɑ΂_[^[Qbgr[(RTV)쐬
		_device->CreateRenderTargetView(backBufferTex, NULL, &_backBufferRTV);

		// obNobt@̏
		backBufferTex->GetDesc(&descBackBuffer);

		SAFE_RELEASE(backBufferTex);
	}

	// _[^[Qbg̐ݒ(fvXXeV͗pȂ)
	_deviceContext->OMSetRenderTargets(1, &_backBufferRTV, NULL);

	// r[|[g̐ݒ
	{
		D3D11_VIEWPORT vp;
		vp.Width = static_cast<float>(descBackBuffer.Width);
		vp.Height = static_cast<float>(descBackBuffer.Height);
		vp.MinDepth = 0.0f;
		vp.MaxDepth = 1.0f;
		vp.TopLeftX = 0;
		vp.TopLeftY = 0;
		_deviceContext->RSSetViewports(1, &vp);
	}

#if ENABLE_LIVE2D
	if (_renderer)
		_renderer->setDeviceSize(descBackBuffer.Width, descBackBuffer.Height);
#endif

	return S_OK;
}


void Application::mainLoop()
{
	MSG msg={0};
	ZeroMemory(&msg, sizeof(msg));
	while(msg.message != WM_QUIT)
	{
		if(PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
		{
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		}
		else
		{
			// ACh
			if (!appProcess())
			{
				// G[ꍇAAvP[VI
				DestroyWindow(_hWnd);
			}
		}
	}
}


bool Application::appProcess()
{
	HRESULT hr;

	// foCX̏
	hr = isDeviceRemoved();
	if (FAILED(hr))
		return false;

	// X^oC [h
	if (_standbyMode) {
		hr = _swapChain->Present(0, DXGI_PRESENT_TEST);
		if (hr != S_OK) {
			Sleep(100);	// 0.1b҂
			return true;
		}
		_standbyMode = false; // X^oC [h
		LOGD("come out of standby");
	}

	// ʂ̍XV
	hr = render();
	if (hr == DXGI_STATUS_OCCLUDED) {
		LOGD("enter standby");
		_standbyMode = true;  // X^oC [hɓ
	}

	return true;
}


HRESULT Application::isDeviceRemoved()
{
	HRESULT hr;

	// foCX̏mF
	hr = _device->GetDeviceRemovedReason();
	switch (hr) {
	case S_OK:
		break;         // 

	case DXGI_ERROR_DEVICE_HUNG:
	case DXGI_ERROR_DEVICE_RESET:
		LOGE("isDeviceRemoved _device->GetDeviceRemovedReason [%s]", getDXErrorString(hr));
		hr = recoverGraphics();

		if (FAILED(hr))
			return hr; // sAAvP[VI
		break;

	case DXGI_ERROR_DEVICE_REMOVED:
	case DXGI_ERROR_DRIVER_INTERNAL_ERROR:
	case DXGI_ERROR_INVALID_CALL:
	default:
		LOGE("isDeviceRemoved _device->GetDeviceRemovedReason [%s]", getDXErrorString(hr));
		return hr;   // s\AAvP[VIB
	};

	return S_OK;  // 
}


HRESULT Application::recoverGraphics()
{
	destroyModel();
	destroyLive2D();
	destroyGraphics();

	HRESULT hr;
	hr = initGraphics();
	if (FAILED(hr)) return hr;
	hr = initLive2D();
	if (FAILED(hr)) return hr;
	hr = initModel();
	if (FAILED(hr)) return hr;

	return S_OK;
}


HRESULT Application::render()
{
	// ʃNA
	float ClearColor[4] = {1.0f, 1.0f, 0.75f, 1.0f}; // NAF쐬 RGBȀ
	_deviceContext->ClearRenderTargetView(_backBufferRTV,ClearColor); // J[obt@NA
#if ENABLE_LIVE2D
	//  Tvł́AʂLeft Top (-1,1) , Right Bottom (1,-1) , z = 0 ƂȂViewOLive2D`悵܂B
	//  AvP[V̎dlƂȂꍇ́ALive2D̕`̑OɃ[hWϊAL̋Ԃɍ悤
	//  ϊĂĂяoĉB

	// f`
	_renderer->draw();
#endif
	// ʍXV(obNobt@tgobt@ɃXbv)
	HRESULT hr = _swapChain->Present(0, 0); 

	return hr;
}


HRESULT Application::resizeBackBuffer(int width, int height)
{
	if (_deviceContext == NULL) return E_FAIL;

	DXGI_SWAP_CHAIN_DESC sd;
	_swapChain->GetDesc(&sd);

	// obNobt@ƃfvXXeVobt@̎QƂ؂
	_deviceContext->OMSetRenderTargets(0, NULL, NULL);
	SAFE_RELEASE(_backBufferRTV);
	HRESULT hr = _swapChain->ResizeBuffers(1,
		width,
		height,
		sd.BufferDesc.Format,
		sd.Flags);

	if (FAILED(hr))
	{
		LOGE("resizeBackBuffer _swapChain->ResizeBuffers [%s]", getDXErrorString(hr));
		return hr;
	}

	hr = initBackBuffer();
	return hr;
}


HRESULT Application::toggleFullscreen()
{
	BOOL isFullscreen;
	_swapChain->GetFullscreenState(&isFullscreen, NULL);

	SIZE size;

	// ݂̏ԂtXN[Ԃł΃EBhE[h̎̃TCY𕜌
	if (isFullscreen)
	{
		size = _prevWindowSize;
	}
	else
	{
		// EBhE[h̃TCYۑ
		DXGI_SWAP_CHAIN_DESC sd;
		_swapChain->GetDesc(&sd);
		_prevWindowSize = SIZE{ LONG(sd.BufferDesc.Width), LONG(sd.BufferDesc.Height) };
		size = _fullscreenSize;
	}

	_swapChain->SetFullscreenState(!isFullscreen, NULL);
	requestResizeTarget(size.cx, size.cy);

	return S_OK;
}


HRESULT Application::requestResizeTarget(int width, int height)
{
	DXGI_SWAP_CHAIN_DESC sd;
	_swapChain->GetDesc(&sd);

	DXGI_MODE_DESC desc = sd.BufferDesc;
	desc.RefreshRate.Numerator = 0;
	desc.RefreshRate.Denominator = 0;
	desc.Width = width;
	desc.Height = height;

	HRESULT hr = _swapChain->ResizeTarget(&desc);
	if (FAILED(hr))
	{
		LOGE("requestResizeTarget _swapChain->ResizeTarget [%s]", getDXErrorString(hr));
		return hr;
	}

	return S_OK;
}


void Application::destroyModel()
{
	// NOP
}


void Application::destroyLive2D()
{
#if ENABLE_LIVE2D
	delete _renderer; _renderer = NULL;
	delete _live2DMgr; _live2DMgr = NULL;
#endif
}


void Application::destroyGraphics()
{
	// foCXXe[gNA
	if (_deviceContext)
		_deviceContext->ClearState();

	// tXN[ł̓Xbv`F[łȂꍇ
	// 邽߁AEChE [hɂ
	if (_swapChain)
		_swapChain->SetFullscreenState(FALSE, NULL);

	SAFE_RELEASE(_backBufferRTV);
	SAFE_RELEASE(_deviceContext);
	SAFE_RELEASE(_device);
	SAFE_RELEASE(_swapChain);
}
