/**
 * Live2D SDK for DirectX 
 *
 *  You can modify and use this source freely
 *  only for the development of application related Live2D.
 *
 *  (c) Live2D Inc. All rights reserved.
 */

#include "main.h"
#include "Live2D.h"
#include "util/UtSystem.h"
#include "Live2DModelD3D11.h"
#include <DirectXMath.h>
#include <WICTextureLoader.h>

// Utility Template
template<typename T>
inline static void SAFE_RELEASE(T& x)
{
	if (x) { x->Release(); x = NULL; }
}


template<typename T>
inline static void SAFE_RELEASE_ARRAY(T& array, size_t size)
{
	for (size_t i = 0; i < size; i++)
	{
		if (array[i]) { array[i]->Release(); array[i] = NULL; }
	}
}


INT WINAPI WinMain(HINSTANCE hInstance,HINSTANCE,LPSTR,INT)
{
	Application* app = new Application;
	if (app != NULL)
	{
		if (SUCCEEDED(app->initWindow(hInstance, 0, 0, WINDOW_SIZE, APP_NAME)))
		{
			if (SUCCEEDED(app->initGraphics()))
			{
				if (SUCCEEDED(app->initLive2D()))
				{
					if (SUCCEEDED(app->initModel()))
					{
						app->mainLoop();					
					}
				}
			}
		}

		app->destroyModel();
		app->destroyLive2D();
		app->destroyGraphics();
		delete app;
	}
	return 0;
}


LRESULT Application::wndProcWrapper(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	Application* app = reinterpret_cast<Application*>(GetWindowLongPtr(hWnd, GWLP_USERDATA));

	if (!app)
	{
		if (uMsg == WM_CREATE)
		{
			app = static_cast<Application*>(reinterpret_cast<LPCREATESTRUCT>(lParam)->lpCreateParams);
			if (app)
			{
				SetWindowLongPtr(hWnd, GWLP_USERDATA, reinterpret_cast<LONG_PTR>(app));
			}
		}
	}
	else
	{
		return app->wndProc(hWnd, uMsg, wParam, lParam);
	}

	return DefWindowProc(hWnd, uMsg, wParam, lParam);
}


Application::Application()
	: _hWnd(0)
	, _windowSize()
	, _device(NULL)
	, _deviceContext(NULL)
	, _swapChain(NULL)
	, _backBufferRTV(NULL)
	, live2DModel(NULL)
	, _textures(NULL)
{
}


Application::~Application()
{
}


LRESULT Application::wndProc(HWND hWnd, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
	switch (iMsg)
	{
	case WM_KEYDOWN:
		switch ((char)wParam)
		{
		case VK_ESCAPE:
			PostQuitMessage(0);
			break;
		}
		break;
	case WM_DESTROY:
		PostQuitMessage(0);
		break;
	}
	return DefWindowProc (hWnd, iMsg, wParam, lParam);
}


HRESULT Application::initWindow(HINSTANCE hInstance, INT x, INT y, 
	SIZE windowSize, LPCWSTR windowName)
{
	// EBhE̒`
	WNDCLASSEX wc;
	ZeroMemory(&wc, sizeof(wc));
	wc.cbSize = sizeof(wc);
	wc.style = CS_HREDRAW | CS_VREDRAW;
	wc.lpfnWndProc = Application::wndProcWrapper;
	wc.hInstance = hInstance;
	wc.hIcon = LoadIcon(NULL, IDI_APPLICATION);
	wc.hCursor = LoadCursor(NULL, IDC_ARROW);
	wc.hbrBackground = static_cast<HBRUSH>(GetStockObject(LTGRAY_BRUSH));
	wc.lpszClassName = windowName;
	wc.hIconSm = LoadIcon(NULL, IDI_APPLICATION);
	RegisterClassEx(&wc);

	_windowSize = windowSize;

	// EBhE̍쐬
	_hWnd = CreateWindow(windowName, windowName, WS_OVERLAPPEDWINDOW,
		CW_USEDEFAULT, CW_USEDEFAULT, windowSize.cx, windowSize.cy, 0, 0, hInstance, this);
	if (!_hWnd)
	{
		return E_FAIL;
	}

	// EChE̕\
	ShowWindow(_hWnd, SW_SHOW);
	UpdateWindow(_hWnd);

	return S_OK;
}


void Application::mainLoop()
{
	MSG msg={0};
	ZeroMemory(&msg, sizeof(msg));
	while(msg.message != WM_QUIT)
	{
		if(PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
		{
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		}
		else
		{
			process();
		}
	}
}


void Application::process()
{
	render();
}


HRESULT Application::initGraphics()
{
	HRESULT hr;

	// foCXƃXbv`F[̍쐬
	{
		DXGI_SWAP_CHAIN_DESC sd;
		ZeroMemory(&sd, sizeof(sd));
		sd.BufferCount = 1;
		sd.BufferDesc.Width = _windowSize.cx;
		sd.BufferDesc.Height = _windowSize.cy;
		sd.BufferDesc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
		sd.BufferDesc.RefreshRate.Numerator = 60;
		sd.BufferDesc.RefreshRate.Denominator = 1;
		sd.BufferUsage = DXGI_USAGE_RENDER_TARGET_OUTPUT;
		sd.OutputWindow = _hWnd;
		sd.SampleDesc.Count = 1;
		sd.SampleDesc.Quality = 0;
		sd.Windowed = TRUE;
	
		// Cubism SDK for DirectX11  Feature Level 10.0ȏT|[g
		// Feature Level 9.1`9.3͔T|[g
		D3D_FEATURE_LEVEL featureLevels[] = {
			D3D_FEATURE_LEVEL_11_0,
			D3D_FEATURE_LEVEL_10_1,
			D3D_FEATURE_LEVEL_10_0,
		};
		const int numFeatureLevel = sizeof(featureLevels) / sizeof(featureLevels[0]);
		D3D_FEATURE_LEVEL* featureLevel = NULL;
	
		hr = D3D11CreateDeviceAndSwapChain(NULL, D3D_DRIVER_TYPE_HARDWARE, NULL,
			0, featureLevels, numFeatureLevel, D3D11_SDK_VERSION, &sd, &_swapChain, &_device,
			featureLevel, &_deviceContext);
		if (FAILED(hr))
		{
			return hr;
		}
	}

	// obNobt@[eNX`擾iɂ̂ō쐬ł͂Ȃj
	{
		ID3D11Texture2D *backBufferTex;
		_swapChain->GetBuffer(0, __uuidof(ID3D11Texture2D), reinterpret_cast<LPVOID*>(&backBufferTex));
		// ̃eNX`[ɑ΂_[^[Qbgr[(RTV)쐬
		_device->CreateRenderTargetView(backBufferTex, NULL, &_backBufferRTV);
		SAFE_RELEASE(backBufferTex);
	}

	// _[^[Qbg̐ݒ(fvXXeV͗pȂ)
	_deviceContext->OMSetRenderTargets(1, &_backBufferRTV, NULL);

	// r[|[g̐ݒ
	{
		D3D11_VIEWPORT vp;
		vp.Width	= static_cast<float>(_windowSize.cx);
		vp.Height	= static_cast<float>(_windowSize.cy);
		vp.MinDepth	= 0.0f;
		vp.MaxDepth	= 1.0f;
		vp.TopLeftX	= 0;
		vp.TopLeftY	= 0;
		_deviceContext->RSSetViewports(1, &vp);
	}

	return S_OK;
}


HRESULT Application::initLive2D()
{
	// Live2D̏
	live2d::Live2D::init();

	// foCXReLXgݒ(fԂŋL)
	live2d::Live2DModelD3D11::setGraphicsContext(_device, _deviceContext);

	return S_OK;
}


HRESULT Application::initModel()
{
	// f̃[h
	const char* mocPath = "res\\epsilon\\Epsilon2.1.moc";
	const LPCTSTR texturePaths[] = {
		L"res\\epsilon\\Epsilon2.1.2048\\texture_00.png",
	};
	const int numTexturePath = sizeof(texturePaths) / sizeof(LPCTSTR);

	live2DModel = live2d::Live2DModelD3D11::loadModel(mocPath);

	// eNX`̃[h
	ID3D11ShaderResourceView* textureSRV;
	for (int i = 0; i < numTexturePath; i++)
	{
		if (SUCCEEDED(DirectX::CreateWICTextureFromFileEx(_device, _deviceContext, texturePaths[i], 0, 
			D3D11_USAGE_DEFAULT, D3D11_BIND_SHADER_RESOURCE, 0, 0, false, NULL, &textureSRV)))
		{
			_textures.push_back(textureSRV);
			live2DModel->setTexture(i, textureSRV);			
		}
		else
		{
			live2d::UtDebug::error("Could not create texture [%d]", i);
			return E_FAIL;
		}
	}

	// Wϊ
	// ݂Live2Df́A摜̂悤ɉ(0,0,w,h)ɔzu
	// Z^OƏ㉺t]Kv
	{
		float aspect = static_cast<float>(_windowSize.cy) / _windowSize.cx;
		float modelWidth = live2DModel->getModelImpl()->getCanvasWidth() ;
		float modelHeight = live2DModel->getModelImpl()->getCanvasHeight() ;

		DirectX::XMMATRIX ortho = DirectX::XMMatrixOrthographicLH(modelHeight, - modelHeight * aspect, -1.0f, 1.0f);
		DirectX::XMMATRIX trans = DirectX::XMMatrixTranslation(-modelWidth/2, -modelHeight/2, 0);

		DirectX::XMFLOAT4X4 wvp;
		DirectX::XMStoreFloat4x4(&wvp, DirectX::XMMatrixMultiplyTranspose(trans, ortho));
		live2DModel->setMatrix(reinterpret_cast<float*>(wvp.m));
	}

	return S_OK;
}


void Application::render()
{
	// ʃNA
	float ClearColor[4] = { 1.0f, 1.0f, 1.0f, 1.0f }; // NAF쐬 RGBȀ
	_deviceContext->ClearRenderTargetView(_backBufferRTV,ClearColor); // J[obt@NA

	// f̃p[^𒼐ڐݒ肷Tv
	double t = (live2d::UtSystem::getUserTimeMSec()/1000.0) * 2 * 3.14 ;

	// ӂ
	live2DModel->setParamFloat("PARAM_ANGLE_X", (float)(30 * sin( t/3.0 )) );

	// ڃp`
	live2DModel->setParamFloat("PARAM_EYE_L_OPEN",(float)(1.0 + sin(t/3)));
	live2DModel->setParamFloat("PARAM_EYE_R_OPEN",(float)(1.0 + sin(t/3)));

	// f̕`
	live2DModel->update();
	live2DModel->draw();

	// ʍXV(obNobt@tgobt@ɃXbv)
	_swapChain->Present(0,0); 
}


void Application::destroyModel()
{
	if (live2DModel)
	{
		delete live2DModel;
	}

	SAFE_RELEASE_ARRAY(_textures, _textures.size());
	_textures.clear();
}


void Application::destroyLive2D()
{
	live2d::Live2D::dispose();
}


void Application::destroyGraphics()
{
	SAFE_RELEASE(_backBufferRTV);
	SAFE_RELEASE(_deviceContext);
	SAFE_RELEASE(_device);
	SAFE_RELEASE(_swapChain);
}
