// -*- mode:c++; indent-tabs-mode:nil; tab-width:2; -*-
// 
// FileDescriptor.h
//  $Id: UssFileDescriptor.h,v 1.14 2001/08/02 07:20:26 seagull Exp $
//

#if !defined(USS_H_FileDescriptor)
#define USS_H_FileDescriptor

#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <stdio.h>
#include <stdint.h>

#include "UssException.h"
#include "UssIO.h"


class UssFileDescriptor;


class EUssFileNotOpened : public EUssException
{
public:
  EUssFileNotOpened(UssFileDescriptor* p)
    : EUssException(static_cast<void*>(p))
  {
  }
};


/**
   ޡȤʥեǥץ
 */
class UssFileDescriptor : public IUssIO
{
public:
  UssFileDescriptor(int fd = -1)
  {
    m_fd = fd;
    m_AutoClose = false;
  }

  virtual ~UssFileDescriptor()
  {
    if (isOpened())
      close(); // ˥Ƥ
  }

  int open(char const* fname, int flags)
  {
    m_fd = ::open(fname, flags);
    if (m_fd < 0)
      throw new EUssOSError();

    return m_fd;
  }


  int open(char const* fname, int flags, mode_t mode)
  {
    m_fd = ::open(fname, flags, mode);
    if (m_fd < 0)
      throw new EUssOSError();

    return m_fd;
  }

  virtual void close()
  {
    if (isOpened())
      {
        ::close(getFD());
        m_fd = -1;
      }
  }
  
  virtual int read(void* buf, int len)
  {
    testOpened();

    int result = ::read(getFD(), buf, len);
    if (result < 0)
      {
        if (errno == ECONNRESET)
          result = 0;
        else
          throw new EUssOSError();
      }
    
    /**
       auto close this handle,
       if reaches to EOF and AutoClose.
    */
    if (result == 0 && m_AutoClose)
      close();
    
    return result;
  }

  int getchar()
  {
    uint8_t ch;
    if (read(&ch, 1) == 0)
      return -1;
    return ch;
  }

  virtual int write(void const* buf, int len)
  {
    testOpened();

    int result = ::write(getFD(), const_cast<void*>(buf), len);
    if (result < 0)
      throw new EUssOSError();
    
    return result;
  }

  int putchar(uint8_t ch)
  {
    return write((void*)&ch, 1);
  }





  int fcntrl(int cmd) { return ::fcntl(getFD(), cmd); }
  int fcntrl(int cmd, long arg)
  {
    return ::fcntl(getFD(), cmd, arg);
  }
  int fcntrl(int cmd, struct flock* lock)
  {
    return ::fcntl(getFD(), cmd, lock);
  }

  //  operator int() const { return getFD(); }
  int getFD() const { return m_fd; }
  void setFD(int fd)
  {
    attach(fd);
  }
  int detach()
  {
    int fd = m_fd;
    m_fd = -1;
    return fd;
  }
  void attach(int fd)
  {
    if (fd < 0)
      throw new EUssOSError(EBADFD);
    if (getFD() >= 0)
      close();

    m_fd = fd;
  }

  bool isOpened() { return getFD() >= 0; }
  void testOpened()
  {
    if (! isOpened() )
      throw new EUssFileNotOpened(this);
  }
  
  bool getAutoClose() const { return m_AutoClose; }
  void setAutoClose(bool autoclose = true)
  {
    m_AutoClose = autoclose;
  }

protected:
  int m_fd;
  bool m_AutoClose;
};


  
#endif


// EOF
