﻿/*
 * MidiEvent.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Lib.Vsq.
 *
 * Boare.Lib.Vsq is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Lib.Vsq is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;

namespace Boare.Lib.Vsq {

    [Serializable]
    public class MidiEvent : IComparable<MidiEvent>, ICloneable {
        public int index;
        public MidiEventType type;
        public int[] intValue;
        public string stringValue;
        public byte[] byteValue;

        public static MidiEvent TempoChange( int clock, int tempo ) {
            MidiEvent res = new MidiEvent();
            res.index = clock;
            res.type = MidiEventType.tempo;
            res.intValue = new int[1];
            res.intValue[0] = tempo;
            return res;
        }

        public static MidiEvent TimeSig( int clock, int numerator, int denominator ) {
            MidiEvent res = new MidiEvent();
            res.index = clock;
            res.type = MidiEventType.time_signal;
            res.intValue = new int[2];
            res.intValue[0] = numerator;
            res.intValue[1] = denominator;
            return res;
        }

        public object Clone() {
            MidiEvent res = new MidiEvent();
            res.index = index;
            res.type = type;
            if ( intValue != null ) {
                res.intValue = new int[intValue.Length];
                for ( int i = 0; i < intValue.Length; i++ ) {
                    res.intValue[i] = intValue[i];
                }
            }
            res.stringValue = stringValue;
            if ( byteValue != null ) {
                res.byteValue = new byte[byteValue.Length];
                for ( int i = 0; i < byteValue.Length; i++ ) {
                    res.byteValue[i] = byteValue[i];
                }
            }
            return res;
        }

        public int CompareTo( MidiEvent obj ) {
            return this.index - obj.index;
        }

        public bool Equals( MidiEvent obj ) {
            if ( this.index == obj.index ) {
                return true;
            } else {
                return false;
            }
        }

        private MidiEvent() {
        }

        /// <summary>
        /// かきかけ。メタテキスト以外のmidiイベントを取り扱う。
        /// </summary>
        /// <param name="line"></param>
        public MidiEvent( string line ) {
            index = -1;
            type = MidiEventType.unknown;
            //intValue = new int[1];
            //intValue[0] = -9000;
            stringValue = "";
            byteValue = null;

            string[] spl = line.Split( new char[] { ' ' } );
            index = int.Parse( spl[0] );
            switch ( spl[1] ) {
                case "Tempo":
                    type = MidiEventType.tempo;
                    intValue = new int[1];
                    intValue[0] = int.Parse( spl[2] );
                    break;
                case "TimeSig":
                    type = MidiEventType.time_signal;
                    intValue = new int[4];
                    string[] spl2 = spl[2].Split( new char[] { '/' } );
                    intValue[0] = int.Parse( spl2[0] );
                    intValue[1] = int.Parse( spl2[1] );
                    intValue[2] = int.Parse( spl[3] );
                    intValue[3] = int.Parse( spl[4] );
                    break;
                case "Par":
                    type = MidiEventType.parameter;
                    intValue = new int[3];
                    string[] spl3 = spl[2].Split( new char[] { '=' } );
                    intValue[0] = int.Parse( spl3[1] );
                    spl3 = spl[3].Split( new char[] { '=' } );
                    intValue[1] = int.Parse( spl3[1] );
                    spl3 = spl[4].Split( new char[] { '=' } );
                    intValue[2] = int.Parse( spl3[1] );
                    break;
                default:
                    type = MidiEventType.unknown;
                    stringValue = spl[2];
                    for ( int i = 1; i < spl.Length; i++ ) {
                        stringValue += " " + spl[i];
                    }
                    break;
            }

        }
    }

    public enum MidiEventType {
        /// <summary>
        /// channel = 0～15の値
        /// </summary>
        channel,
        /// <summary>
        /// note = 0～127の値
        /// </summary>
        note,
        /// <summary>
        /// dtime = 0～268,435,455 (0x0FFFFFFF)の値
        /// </summary>
        dtime,
        /// <summary>
        /// velocity = 0～127の値
        /// </summary>
        velocity,
        /// <summary>
        /// patch = 0～127の値
        /// </summary>
        patch,
        /// <summary>
        /// sequence = 0-65,535 (0xFFFF)の値
        /// </summary>
        sequence,
        /// <summary>
        /// text = 0byte以上のASCII文字列
        /// </summary>
        text,
        /// <summary>
        /// raw = 0byte以上のバイナリデータの文字列
        /// </summary>
        raw,
        /// <summary>
        /// pitch_wheel = -8192～8191 (0x1FFF)の値
        /// </summary>
        pitch_wheel,
        /// <summary>
        /// song_pos = 0～16,383 (0x3FFF)の値
        /// </summary>
        song_pos,
        /// <summary>
        /// song_number = 0～127の値
        /// </summary>
        song_number,
        /// <summary>
        /// tempo = マイクロ秒, 0～16,777,215 (0x00FFFFFF)の値
        /// </summary>
        tempo,
        time_signal,
        unknown,
        /// <summary>
        /// 勝手に追加。スイマセン
        /// </summary>
        parameter,
    }

}
