﻿/*
 * VsqBPList.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Lib.Vsq.
 *
 * Boare.Lib.Vsq is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Lib.Vsq is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;

namespace Boare.Lib.Vsq {

    /// <summary>
    /// BPListのデータ部分を取り扱うためのクラス。
    /// </summary>
    [Serializable]
    public class VsqBPList: ICloneable {
        private SortedList<int, int> m_list = new SortedList<int, int>();
        private int m_default = 0;
        private int m_maximum = 127;
        private int m_minimum = 0;

        /// <summary>
        /// このVsqBPListの同一コピーを作成します
        /// </summary>
        /// <returns></returns>
        public object Clone() {
            VsqBPList res = new VsqBPList( m_default, m_minimum, m_maximum );
            foreach( int key in m_list.Keys ){
                res.m_list.Add( key, m_list[key] );
            }
            return res;
        }
        

        /// <summary>
        /// コンストラクタ。デフォルト値はココで指定する。
        /// </summary>
        /// <param name="default_value"></param>
        public VsqBPList( int default_value, int minimum, int maximum ) {
            m_default = default_value;
            m_maximum = maximum;
            m_minimum = minimum;
        }


        /// <summary>
        /// このリストに設定された最大値を取得します。
        /// </summary>
        public int Maximum {
            get {
                return m_maximum;
            }
        }


        /// <summary>
        /// このリストに設定された最小値を取得します
        /// </summary>
        public int Minimum {
            get {
                return m_minimum;
            }
        }


        public IEnumerable<int> GetKeyClockEnumerator() {
            for ( int i = 0; i < m_list.Keys.Count; i++ ) {
                yield return m_list.Keys[i];
            }
        }


        public void Remove( int clock ) {
            if ( m_list.ContainsKey( clock ) ) {
                m_list.Remove( clock );
            }
        }


        public bool ContainsKey( int clock ) {
            return m_list.ContainsKey( clock );
        }


        public int Count {
            get {
                return m_list.Count;
            }
        }


        public int[] Keys {
            get {
                List<int> t = new List<int>( m_list.Keys );
                return t.ToArray();
            }
        }


        public void Clear() {
            m_list.Clear();
        }


        /// <summary>
        /// 新しいデータ点を追加します。
        /// </summary>
        /// <param name="clock"></param>
        /// <param name="value"></param>
        public void Add( int clock, int value ) {
            lock ( m_list ) {
                if ( m_list.ContainsKey( clock ) ) {
                    m_list[clock] = value;
                } else {
                    m_list.Add( clock, value );
                }
            }
        }


        public int this[int clock] {
            get {
                if ( m_list.Count == 0 ) {
                    return Default;
                } else {
                    if ( m_list.ContainsKey( clock ) ) {
                        return m_list[clock];
                    } else {
                        int index = 0;
                        int prev = 0;
                        foreach ( int key in m_list.Keys ) {
                            if ( clock < key ) {
                                index = prev;
                                break;
                            }
                            prev = key;
                        }
                        if ( m_list.ContainsKey( index ) ) {
                            return m_list[index];
                        } else {
                            return m_default;
                        }
                    }
                }
            }
        }


        /// <summary>
        /// このBPListのデフォルト値を取得します
        /// </summary>
        public int Default {
            get {
                return m_default;
            }
        }


        /// <summary>
        /// このBPListの内容をテキストファイルに書き出します
        /// </summary>
        /// <param name="writer"></param>
        public void Print( StreamWriter writer ) {
            bool first = true;
            foreach ( int key in m_list.Keys ) {
                int val = m_list[key];
                if ( first ) {
                    writer.WriteLine( key + "=" + val );
                    first = false;
                } else {
                    writer.WriteLine( key + "=" + val );
                }
            }
        }


        /// <summary>
        /// このBPListの内容をテキストファイルに書き出します
        /// </summary>
        /// <param name="writer"></param>
        public void Print( TextMemoryStream writer, int start, string header ) {
            bool first = true;
            foreach ( int key in m_list.Keys ) {
                if ( start <= key ) {
                    if ( first ) {
                        writer.WriteLine( header );
                        first = false;
                    }
                    int val = m_list[key];
                    writer.WriteLine( key + "=" + val );
                }
            }
        }

        
        /// <summary>
        /// テキストファイルからデータ点を読込み、現在のリストに追加します
        /// </summary>
        /// <param name="reader"></param>
        /// <returns></returns>
        public string Read( TextMemoryStream reader ) {
            string last_line = reader.ReadLine();
            while ( !last_line.StartsWith( "[" ) ) {
                string[] spl = last_line.Split( new char[] { '=' } );
                int i1 = int.Parse( spl[0] );
                int i2 = int.Parse( spl[1] );
                m_list.Add( i1, i2 );
                if ( reader.Peek() < 0 ) {
                    break;
                } else {
                    last_line = reader.ReadLine();
                }
            }
            return last_line;
        }

#if DEBUG
        public static void test() {
            VsqBPList list = new VsqBPList( 0, 0, 127 );
            int min = 27595;
            int max = 45960;
            list.Add( 27595, 0 );
            list.Add( 27597, 546 );
            list.Add( 27599, 1092 );
            list.Add( 27601, 1638 );
            list.Add( 27603, 2184 );
            list.Add( 27605, 2730 );
            list.Add( 27607, 3276 );
            list.Add( 27609, 3822 );
            list.Add( 27611, 4368 );
            list.Add( 27613, 4914 );
            list.Add( 27615, 5460 );
            list.Add( 27617, 6006 );
            list.Add( 27619, 6552 );
            list.Add( 27621, 7098 );
            list.Add( 27623, 7644 );
            list.Add( 27625, 8191 );
            list.Add( 27790, 7000 );
            list.Add( 27792, 6404 );
            list.Add( 27794, 6107 );
            list.Add( 27796, 5809 );
            list.Add( 27798, 5511 );
            list.Add( 27800, 5213 );
            list.Add( 27802, 4915 );
            list.Add( 27804, 4617 );
            list.Add( 27806, 4319 );
            list.Add( 27808, 4022 );
            list.Add( 27810, 3724 );
            list.Add( 27812, 3426 );
            list.Add( 27814, 3128 );
            list.Add( 27816, 2830 );
            list.Add( 27818, 2532 );
            list.Add( 27820, 2234 );
            list.Add( 27822, 1937 );
            list.Add( 27824, 1639 );
            list.Add( 27826, 1341 );
            list.Add( 27828, 1043 );
            list.Add( 27830, 745 );
            list.Add( 27832, 447 );
            list.Add( 27834, 149 );
            list.Add( 27835, 0 );
            list.Add( 27840, 0 );
            list.Add( 30963, 0 );
            list.Add( 30965, 712 );
            list.Add( 30967, 1424 );
            list.Add( 30969, 2136 );
            list.Add( 30971, 2849 );
            list.Add( 30973, 3561 );
            list.Add( 30975, 4273 );
            list.Add( 30977, 4985 );
            list.Add( 30979, 5698 );
            list.Add( 30981, 6410 );
            list.Add( 30983, 7122 );
            list.Add( 30985, 8191 );
            list.Add( 30986, 8191 );
            list.Add( 30990, 8191 );
            list.Add( 31286, 8191 );
            list.Add( 31288, 7561 );
            list.Add( 31290, 6931 );
            list.Add( 31292, 6301 );
            list.Add( 31294, 5671 );
            list.Add( 31296, 5041 );
            list.Add( 31298, 4411 );
            list.Add( 31300, 3781 );
            list.Add( 31302, 3151 );
            list.Add( 31304, 2521 );
            list.Add( 31306, 1891 );
            list.Add( 31308, 1261 );
            list.Add( 31310, 631 );
            list.Add( 31312, 0 );
            list.Add( 34281, 0 );
            list.Add( 34283, 273 );
            list.Add( 34285, 546 );
            list.Add( 34287, 819 );
            list.Add( 34289, 1092 );
            list.Add( 34291, 1365 );
            list.Add( 34293, 1638 );
            list.Add( 34295, 1911 );
            list.Add( 34297, 2184 );
            list.Add( 34299, 2457 );
            list.Add( 34301, 2730 );
            list.Add( 34303, 3003 );
            list.Add( 34305, 3276 );
            list.Add( 34307, 3549 );
            list.Add( 34309, 3822 );
            list.Add( 34311, 4095 );
            list.Add( 34313, 4368 );
            list.Add( 34315, 4641 );
            list.Add( 34317, 4914 );
            list.Add( 34319, 5187 );
            list.Add( 34321, 5460 );
            list.Add( 34323, 5733 );
            list.Add( 34325, 6006 );
            list.Add( 34327, 6279 );
            list.Add( 34329, 6552 );
            list.Add( 34331, 6825 );
            list.Add( 34333, 7098 );
            list.Add( 34335, 7371 );
            list.Add( 34337, 7644 );
            list.Add( 34339, 7917 );
            list.Add( 34341, 8191 );
            list.Add( 34534, 8191 );
            list.Add( 34536, 7411 );
            list.Add( 34538, 6631 );
            list.Add( 34540, 5851 );
            list.Add( 34542, 5071 );
            list.Add( 34544, 4291 );
            list.Add( 34546, 3511 );
            list.Add( 34548, 2731 );
            list.Add( 34550, 1951 );
            list.Add( 34552, 1171 );
            list.Add( 34554, 391 );
            list.Add( 34555, 0 );
            list.Add( 34561, 37 );
            list.Add( 34565, 74 );
            list.Add( 45900, 0 );
            list.Add( 45960, 0 );
            using ( StreamWriter sw = new StreamWriter( "VsqBPList_test_result.txt" ) ) {
                for ( int i = min; i <= max; i++ ) {
                    sw.WriteLine( i + " " + list[i] );
                }
            }
            using ( StreamWriter sw = new StreamWriter( "VsqBPList_test_original.txt" ) ) {
                foreach ( int key in list.Keys ) {
                    sw.WriteLine( key + " " + list[key] );
                }
            }
        }
#endif
    }

}
