﻿/*
 * VsqMetaText/Handle.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Lib.Vsq.
 *
 * Boare.Lib.Vsq is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Lib.Vsq is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;

namespace Boare.Lib.Vsq {

    [Serializable]
    public class IconHandle : VsqHandle, ICloneable {
        public object Clone() {
            IconHandle ret = new IconHandle();
            ret.Caption = m_caption;
            ret.IconID = m_icon_id;
            ret.IDS = m_ids;
            ret.Index = Index;
            ret.Language = Language;
            ret.Length = Length;
            ret.Original = Original;
            ret.Program = Program;
            ret.Type = Type;
            return ret;
        }


        public int Program {
            get {
                return m_program;
            }
            set {
                m_program = value;
            }
        }


        public int Language {
            get {
                return m_language;
            }
            set {
                m_language = value;
            }
        }


        public int Length {
            get {
                return m_length;
            }
            set {
                m_length = value;
            }
        }


        public string Caption {
            get {
                return m_caption;
            }
            set {
                m_caption = value;
            }
        }


        public string IconID {
            get {
                return m_icon_id;
            }
            set {
                m_icon_id = value;
            }
        }


        public string IDS {
            get {
                return m_ids;
            }
            set {
                m_ids = value;
            }
        }


        public int Original {
            get {
                return m_original;
            }
            set {
                m_original = value;
            }
        }
    }


    [Serializable]
    public class LyricHandle : VsqHandle, ICloneable {
        internal LyricHandle() {
        }


        /// <summary>
        /// type = Lyric用のhandleのコンストラクタ
        /// </summary>
        /// <param name="phrase">歌詞</param>
        /// <param name="phonetic_symbol">発音記号</param>
        public LyricHandle( string phrase, string phonetic_symbol ) {
            Type = VsqHandleType.Lyric;
            m_lyric = new Lyric( phrase, phonetic_symbol );
        }

        
        public Lyric L0 {
            get {
                return m_lyric;
            }
            set {
                m_lyric = value;
            }
        }


        public object Clone() {
            LyricHandle ret = new LyricHandle();
            ret.Type = Type;
            ret.Index = Index;
            ret.m_lyric = (Lyric)m_lyric.Clone();
            return ret;
        }
    }


    [Serializable]
    public class VibratoHandle : VsqHandle, ICloneable {
        public object Clone() {
            VibratoHandle result = new VibratoHandle();
            result.Type = Type;
            result.Index = Index;
            result.IconID = IconID;
            result.IDS = this.IDS;
            result.Original = this.Original;
            result.Caption = this.Caption;
            result.Length = this.Length;
            result.StartDepth = this.StartDepth;
            result.DepthBPNum = this.DepthBPNum;
            if ( this.DepthBPX != null ) {
                result.DepthBPX = (float[])this.DepthBPX.Clone();
            }
            if ( this.DepthBPY != null ) {
                result.DepthBPY = (int[])this.DepthBPY.Clone();
            }
            result.StartRate = this.StartRate;
            result.RateBPNum = this.RateBPNum;
            if ( this.RateBPX != null ) {
                result.RateBPX = (float[])this.RateBPX.Clone();
            }
            if ( this.RateBPY != null ) {
                result.RateBPY = (int[])this.RateBPY.Clone();
            }
            return result;
        }


        public int Original {
            get {
                return m_original;
            }
            set {
                m_original = value;
            }
        }


        public int Length {
            get {
                return m_length;
            }
            set {
                m_length = value;
            }
        }


        public string Caption {
            get {
                return m_caption;
            }
            set {
                m_caption = value;
            }
        }


        public string IDS {
            get {
                return m_ids;
            }
            set {
                m_ids = value;
            }
        }


        public string IconID {
            get {
                return m_icon_id;
            }
            set {
                m_icon_id = value;
            }
        }


        public int StartDepth {
            get {
                return m_start_depth;
            }
            set {
                m_start_depth = value;
            }
        }


        public int StartRate {
            get {
                return m_start_rate;
            }
            set {
                m_start_rate = value;
            }
        }


        public int DepthBPNum {
            get {
                return m_depth_bp_num;
            }
            set {
                m_depth_bp_num = value;
            }
        }


        public int RateBPNum {
            get {
                return m_rate_bp_num;
            }
            set {
                m_rate_bp_num = value;
            }
        }


        public float[] DepthBPX {
            get {
                return m_depth_bp_x;
            }
            set {
                m_depth_bp_x = value;
            }
        }


        public int[] DepthBPY {
            get {
                return m_depth_bp_y;
            }
            set {
                m_depth_bp_y = value;
            }
        }


        public float[] RateBPX {
            get {
                return m_rate_bp_x;
            }
            set {
                m_rate_bp_x = value;
            }
        }


        public int[] RateBPY {
            get {
                return m_rate_bp_y;
            }
            set {
                m_rate_bp_y = value;
            }
        }
    }


    /// <summary>
    /// ハンドルを取り扱います。ハンドルにはLyricHandle、VibratoHandleおよびIconHandleがある
    /// </summary>
    [Serializable]
    public class VsqHandle {
        private VsqHandleType m_type;
        private int m_index;
        protected string m_icon_id;
        protected string m_ids;
        protected Lyric m_lyric;
        protected int m_original;
        protected string m_caption;
        protected int m_length;
        protected int m_start_depth;
        protected int m_depth_bp_num;
        protected float[] m_depth_bp_x;
        protected int[] m_depth_bp_y;
        protected int m_start_rate;
        protected int m_rate_bp_num;
        protected float[] m_rate_bp_x;
        protected int[] m_rate_bp_y;
        protected int m_language;
        protected int m_program;

        public LyricHandle ConvertToLyricHandle() {
            LyricHandle ret = new LyricHandle();
            ret.m_lyric = (Lyric)m_lyric;
            ret.m_type = m_type;
            ret.m_index = m_index;
            return ret;
        }


        public VibratoHandle ConvertToVibratoHandle() {
            VibratoHandle ret = new VibratoHandle();
            ret.m_type = m_type;
            ret.m_index = m_index;
            ret.Caption = m_caption;
            ret.DepthBPNum = m_depth_bp_num;
            ret.DepthBPX = m_depth_bp_x;
            ret.DepthBPY = m_depth_bp_y;
            ret.IconID = m_icon_id;
            ret.IDS = m_ids;
            ret.Index = m_index;
            ret.Length = m_length;
            ret.Original = m_original;
            ret.RateBPNum = m_rate_bp_num;
            ret.RateBPX = m_rate_bp_x;
            ret.RateBPY = m_rate_bp_y;
            ret.StartDepth = m_start_depth;
            ret.StartRate = m_start_rate;
            return ret;
        }


        public IconHandle ConvertToIconHandle() {
            IconHandle ret = new IconHandle();
            ret.m_type = m_type;
            ret.m_index = m_index;
            ret.Caption = m_caption;
            ret.IconID = m_icon_id;
            ret.IDS = m_ids;
            ret.Index = m_index;
            ret.Language = m_language;
            ret.Length = m_length;
            ret.Original = m_original;
            ret.Program = m_program;
            return ret;
        }


        public VsqHandleType Type {
            get {
                return m_type;
            }
            set {
                m_type = value;
            }
        }


        public int Index {
            get {
                return m_index;
            }
            set {
                m_index = value;
            }
        }


        /*// <summary>
        /// このインスタンスの簡易コピーを取得します。
        /// </summary>
        /// <returns></returns>
        public object Clone() {
            VsqHandle result = new VsqHandle();
            result.m_type = m_type;
            result.m_index = m_index;
            result.m_icon_id = m_icon_id;
            result.IDS = this.IDS;
            if ( this.L0 != null ) {
                result.L0 = this.L0.Clone();
            }
            result.Original = this.Original;
            result.Caption = this.Caption;
            result.Length = this.Length;
            result.StartDepth = this.StartDepth;
            result.DepthBPNum = this.DepthBPNum;
            if ( this.DepthBPX != null ) {
                result.DepthBPX = (float[])this.DepthBPX.Clone();
            }
            if ( this.DepthBPY != null ) {
                result.DepthBPY = (int[])this.DepthBPY.Clone();
            }
            result.StartRate = this.StartRate;
            result.RateBPNum = this.RateBPNum;
            if ( this.RateBPX != null ) {
                result.RateBPX = (float[])this.RateBPX.Clone();
            }
            if ( this.RateBPY != null ) {
                result.RateBPY = (int[])this.RateBPY.Clone();
            }
            result.Language = this.Language;
            result.Program = this.Program;
            return result;
        }*/


        public VsqHandle() {
        }

        
        /// <summary>
        /// インスタンスをストリームに書き込みます。
        /// encode=trueの場合、2バイト文字をエンコードして出力します。
        /// </summary>
        /// <param name="sw">書き込み対象</param>
        /// <param name="encode">2バイト文字をエンコードするか否かを指定するフラグ</param>
        public void write( TextMemoryStream sw, bool encode ) {
            sw.WriteLine( this.ToString( encode ) );
        }


        /// <summary>
        /// FileStreamから読み込みながらコンストラクト
        /// </summary>
        /// <param name="sr">読み込み対象</param>
        public VsqHandle( TextMemoryStream sr, int value, ref string last_line ) {
            this.Index = value;
            string[] spl;
            string[] spl2;

            // default値で梅
            this.Type = VsqHandleType.Vibrato;
            m_icon_id = "";
            m_ids = "normal";
            m_lyric = new Lyric( "" );
            m_original = 0;
            m_caption = "";
            m_length = 0;
            m_start_depth = 0;
            m_depth_bp_num = 0;
            m_depth_bp_x = null;
            m_depth_bp_y = null;
            m_start_rate = 0;
            m_rate_bp_num = 0;
            m_rate_bp_x = null;
            m_rate_bp_y = null;
            m_language = 0;
            m_program = 0;

            string tmpDepthBPX = "";
            string tmpDepthBPY = "";
            string tmpRateBPX = "";
            string tmpRateBPY = "";

            // "["にぶち当たるまで読込む
            last_line = sr.ReadLine();
            while ( !last_line.StartsWith( "[" ) ) {
                spl = last_line.Split( new char[] { '=' } );
                switch ( spl[0] ) {
                    case "Language":
                        m_language = int.Parse( spl[1] );
                        break;
                    case "Program":
                        m_program = int.Parse( spl[1] );
                        break;
                    case "IconID":
                        m_icon_id = spl[1];
                        break;
                    case "IDS":
                        m_ids = spl[1];
                        break;
                    case "Original":
                       m_original = int.Parse( spl[1] );
                        break;
                    case "Caption":
                        m_caption = spl[1];
                        for ( int i = 2; i < spl.Length; i++ ) {
                            m_caption += "=" + spl[i];
                        }
                        break;
                    case "Length":
                        m_length = int.Parse( spl[1] );
                        break;
                    case "StartDepth":
                        m_start_depth = int.Parse( spl[1] );
                        break;
                    case "DepthBPNum":
                        m_depth_bp_num = int.Parse( spl[1] );
                        break;
                    case "DepthBPX":
                        tmpDepthBPX = spl[1];
                        break;
                    case "DepthBPY":
                        tmpDepthBPY = spl[1];
                        break;
                    case "StartRate":
                        m_start_rate = int.Parse( spl[1] );
                        break;
                    case "RateBPNum":
                        m_rate_bp_num = int.Parse( spl[1] );
                        break;
                    case "RateBPX":
                        tmpRateBPX = spl[1];
                        break;
                    case "RateBPY":
                        tmpRateBPY = spl[1];
                        break;
                    case "L0":
                        m_type = VsqHandleType.Lyric;
                        m_lyric = new Lyric( spl[1] );
                        break;
                }
                if ( sr.Peek() < 0 ) {
                    break;
                }
                last_line = sr.ReadLine();
            }
            if ( m_ids != "normal" ) {
                Type = VsqHandleType.Singer;
            } else if ( m_icon_id != "" ) {
                Type = VsqHandleType.Vibrato;
            } else {
                Type = VsqHandleType.Lyric;
            }

            // RateBPX, RateBPYの設定
            if ( this.Type == VsqHandleType.Vibrato ) {
                if ( m_rate_bp_num > 0 ) {
                    m_rate_bp_x = new float[m_rate_bp_num];
                    spl2 = tmpRateBPX.Split( new char[] { ',' } );
                    for ( int i = 0; i < m_rate_bp_num; i++ ) {
                        m_rate_bp_x[i] = float.Parse( spl2[i] );
                    }

                    m_rate_bp_y = new int[m_rate_bp_num];
                    spl2 = tmpRateBPY.Split( new char[] { ',' } );
                    for ( int i = 0; i < m_rate_bp_num; i++ ) {
                        m_rate_bp_y[i] = int.Parse( spl2[i] );
                    }
                } else {
                    m_rate_bp_x = null;
                    m_rate_bp_y = null;
                }

                // DepthBPX, DepthBPYの設定
                if ( m_depth_bp_num > 0 ) {
                    m_depth_bp_x = new float[m_depth_bp_num];
                    spl2 = tmpDepthBPX.Split( new char[] { ',' } );
                    for ( int i = 0; i < m_depth_bp_num; i++ ) {
                        m_depth_bp_x[i] = float.Parse( spl2[i] );
                    }

                    m_depth_bp_y = new int[m_depth_bp_num];
                    spl2 = tmpDepthBPY.Split( new char[] { ',' } );
                    for ( int i = 0; i < m_depth_bp_num; i++ ) {
                        m_depth_bp_y[i] = int.Parse( spl2[i] );
                    }
                } else {
                    m_depth_bp_x = null;
                    m_depth_bp_y = null;
                }
            }

        }


        /// <summary>
        /// VsqHandleを構築するテストを行います
        /// </summary>
        /// <returns>テストに成功すればtrue、そうでなければfalseを返します</returns>
        public static bool test() {
            string fpath = Path.GetTempFileName();
            TextMemoryStream sr = null;
            LyricHandle vsqHandle;
            string last_line = "";

            // type Lyricのコンストラクト
            Console.WriteLine( "VsqHandle; type Lyric" );
            StreamWriter sw = new StreamWriter( fpath, false, Encoding.Unicode );
            sw.WriteLine( "L0=\\\"\\x82\\xe7\\\",\\\"4 a\\\",1.000000,64,0,1" );
            sw.WriteLine( "[h#0579]" );
            sw.Close();

            bool result;
            sr = new TextMemoryStream( fpath, Encoding.Unicode );
            vsqHandle = (LyricHandle)(new VsqHandle( sr, 578, ref last_line ));
            if ( vsqHandle.Type == VsqHandleType.Lyric &&
                 vsqHandle.L0.Phrase == "ら" &&
                 vsqHandle.L0.PhoneticSymbol == "4 a" &&
                 vsqHandle.L0.UnknownFloat == 1.0 &&
                 vsqHandle.L0.ConsonantAdjustment[0] == 64 &&
                 vsqHandle.L0.ConsonantAdjustment[1] == 0 &&
                 vsqHandle.L0.PhoneticSymbolProtected ) {
                result = true;
            } else {
                sr.Close();
                File.Delete( fpath );
                return false;
            }
            sr.Close();

            // type Vibrato
            Console.WriteLine( "VsqHandle; type Vibrato" );
            sw = new StreamWriter( fpath, false, Encoding.Unicode );
            sw.WriteLine( "IconID=$04040001" );
            sw.WriteLine( "IDS=normal" );
            sw.WriteLine( "Original=1" );
            sw.WriteLine( "Caption=foo" );
            sw.WriteLine( "Length=880" );
            sw.WriteLine( "StartDepth=64" );
            sw.WriteLine( "DepthBPNum=15" );
            sw.WriteLine( "DepthBPX=0.001300,0.003800,0.566300,0.568800,0.658800,0.746300,0.836300,0.851300,0.853800,0.883800,0.913800,0.943800,0.971300,0.997500,1.000000" );
            sw.WriteLine( "DepthBPY=64,25,25,30,29,27,26,26,25,26,27,29,30,31,64" );
            sw.WriteLine( "StartRate=50" );
            sw.WriteLine( "RateBPNum=57" );
            sw.WriteLine( "RateBPX=0.214800,0.215900,0.236400,0.252300,0.270500,0.286400,0.304500,0.320500,0.338600,0.354500,0.370500,0.388600,0.406800,0.422700,0.438600,0.445500,0.447700,0.472700,0.476100,0.477300,0.667000,0.669300,0.670500,0.681800,0.686400,0.693200,0.700000,0.706800,0.713600,0.720500,0.727200,0.734100,0.740900,0.747700,0.752300,0.761400,0.768200,0.772700,0.779500,0.788600,0.795500,0.800000,0.806800,0.815900,0.820500,0.827300,0.831800,0.835200,0.836400,0.838600,0.840900,0.870500,0.897700,0.925000,0.952300,0.977300,1.000000" );
            sw.WriteLine( "RateBPY=50,54,55,57,58,59,60,61,62,64,65,66,67,68,69,69,71,71,71,71,69,50,69,71,72,73,74,75,76,78,79,80,81,82,83,85,86,87,88,89,90,92,93,94,95,96,97,50,50,50,71,69,68,67,66,65,65" );
            sw.WriteLine( "[h#0127]" );
            sw.Close();

            float[] dbpx = { 0.001300f, 0.003800f, 0.566300f, 0.568800f, 0.658800f, 0.746300f, 0.836300f, 0.851300f, 0.853800f, 0.883800f, 0.913800f, 0.943800f, 0.971300f, 0.997500f, 1.000000f };
            int[] dbpy = { 64, 25, 25, 30, 29, 27, 26, 26, 25, 26, 27, 29, 30, 31, 64 };
            float[] rbpx = { 0.214800f, 0.215900f, 0.236400f, 0.252300f, 0.270500f, 0.286400f, 0.304500f, 0.320500f, 0.338600f, 0.354500f, 0.370500f, 0.388600f, 0.406800f, 0.422700f, 0.438600f, 0.445500f, 0.447700f, 0.472700f, 0.476100f, 0.477300f, 0.667000f, 0.669300f, 0.670500f, 0.681800f, 0.686400f, 0.693200f, 0.700000f, 0.706800f, 0.713600f, 0.720500f, 0.727200f, 0.734100f, 0.740900f, 0.747700f, 0.752300f, 0.761400f, 0.768200f, 0.772700f, 0.779500f, 0.788600f, 0.795500f, 0.800000f, 0.806800f, 0.815900f, 0.820500f, 0.827300f, 0.831800f, 0.835200f, 0.836400f, 0.838600f, 0.840900f, 0.870500f, 0.897700f, 0.925000f, 0.952300f, 0.977300f, 1.000000f };
            int[] rbpy = { 50, 54, 55, 57, 58, 59, 60, 61, 62, 64, 65, 66, 67, 68, 69, 69, 71, 71, 71, 71, 69, 50, 69, 71, 72, 73, 74, 75, 76, 78, 79, 80, 81, 82, 83, 85, 86, 87, 88, 89, 90, 92, 93, 94, 95, 96, 97, 50, 50, 50, 71, 69, 68, 67, 66, 65, 65 };
            sr = new TextMemoryStream( fpath, Encoding.Unicode );
            VibratoHandle vibHandle = (VibratoHandle)(new VsqHandle( sr, 126, ref last_line ));
            if ( vibHandle.IconID == "$04040001" &&
                vibHandle.IDS == "normal" &&
                vibHandle.Original == 1 &&
                vibHandle.Caption == "foo" &&
                vibHandle.Length == 880 &&
                vibHandle.StartDepth == 64 &&
                vibHandle.DepthBPNum == 15 &&
                vibHandle.StartRate == 50 &&
                vibHandle.RateBPNum == 57 &&
                last_line == "[h#0127]") {
                for ( int i = 0; i < vibHandle.DepthBPNum; i++ ) {
                    if ( vibHandle.DepthBPX[i] != dbpx[i] ) {
                        result =  false;
                    }
                    if ( vibHandle.DepthBPY[i] != dbpy[i] ) {
                        result =  false;
                    }
                }
                for ( int i = 0; i < vibHandle.RateBPNum; i++ ) {
                    if ( vibHandle.RateBPX[i] != rbpx[i] ) {
                        result =  false;
                    }
                    if ( vibHandle.RateBPY[i] != rbpy[i] ) {
                        result =  false;
                    }
                }
                result =  result & true;
            } else {
                result =  result & false;
            }
            sr.Close();
            File.Delete( fpath );
            return result;
        }


        /// <summary>
        /// ハンドル指定子（例えば"h#0123"という文字列）からハンドル番号を取得します
        /// </summary>
        /// <param name="_string">ハンドル指定子</param>
        /// <returns>ハンドル番号</returns>
        public static int HandleIndexFromString( string _string ) {
            string[] spl = _string.Split( new char[] { '#' } );
            return int.Parse( spl[1] );
        }


        /// <summary>
        /// インスタンスをテキストファイルに出力します
        /// </summary>
        /// <param name="sw">出力先</param>
        public void Print( StreamWriter sw ) {
            string result = this.ToString();
            sw.WriteLine( result );
        }


        /// <summary>
        /// インスタンスをコンソール画面に出力します
        /// </summary>
        private void Print() {
            string result = this.ToString();
            Console.WriteLine( result );
        }


        /// <summary>
        /// インスタンスを文字列に変換します
        /// </summary>
        /// <param name="encode">2バイト文字をエンコードするか否かを指定するフラグ</param>
        /// <returns>インスタンスを変換した文字列</returns>
        public string ToString( bool encode ) {
            string result = "";
            result += "[h#" + Index.ToString( "0000" ) + "]";
            switch ( Type ) {
                case VsqHandleType.Lyric:
                    result += Environment.NewLine + "L0=" + m_lyric.ToString( encode );
                    break;
                case VsqHandleType.Vibrato:
                    result += Environment.NewLine + "IconID=" + m_icon_id + Environment.NewLine;
                    result += "IDS=" + m_ids + Environment.NewLine;
                    result += "Original=" + m_original + Environment.NewLine;
                    result += "Caption=" + m_caption + Environment.NewLine;
                    result += "Length=" + m_length + Environment.NewLine;
                    result += "StartDepth=" + m_start_depth + Environment.NewLine;
                    result += "DepthBPNum=" + m_depth_bp_num + Environment.NewLine;
                    if ( m_depth_bp_num > 0 ) {
                        result += "DepthBPX=" + m_depth_bp_x[0].ToString( "0.000000" );
                        for ( int i = 1; i < m_depth_bp_num; i++ ) {
                            result += "," + m_depth_bp_x[i].ToString( "0.000000" );
                        }
                        result += Environment.NewLine + "DepthBPY=" + m_depth_bp_y[0];
                        for ( int i = 1; i < m_depth_bp_num; i++ ) {
                            result += "," + m_depth_bp_y[i];
                        }
                        result += Environment.NewLine;
                    }
                    result += "StartRate=" + m_start_rate + Environment.NewLine;
                    result += "RateBPNum=" + m_rate_bp_num;
                    if ( m_rate_bp_num > 0 ) {
                        result += Environment.NewLine + "RateBPX=" + m_rate_bp_x[0].ToString( "0.000000" );
                        for ( int i = 1; i < m_rate_bp_num; i++ ) {
                            result += "," + m_rate_bp_x[i].ToString( "0.000000" );
                        }
                        result += Environment.NewLine + "RateBPY=" + m_rate_bp_y[0];
                        for ( int i = 1; i < m_rate_bp_num; i++ ) {
                            result += "," + m_rate_bp_y[i];
                        }
                    }
                    break;
                case VsqHandleType.Singer:
                    result += Environment.NewLine + "IconID=" + m_icon_id + Environment.NewLine;
                    result += "IDS=" + m_ids + Environment.NewLine;
                    result += "Original=" + m_original + Environment.NewLine;
                    result += "Caption=" + m_caption + Environment.NewLine;
                    result += "Length=" + m_length + Environment.NewLine;
                    result += "Language=" + m_language + Environment.NewLine;
                    result += "Program=" + m_program;
                    break;
                default:
                    break;
            }
            return result;

        }

    }
}
