﻿/*
 * VsqMetaText/Lyric.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Lib.Vsq.
 *
 * Boare.Lib.Vsq is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Lib.Vsq is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Windows.Forms;

using bocoree;

namespace Boare.Lib.Vsq {

    /// <summary>
    /// VsqHandleに格納される歌詞の情報を扱うクラス。
    /// </summary>
    [Serializable]
    public class Lyric {
        private string m_phrase;
        private string[] m_phonetic_symbol;
        private float d1;
        private int[] m_consonant_adjustment;
        private bool m_protected;

        public bool PhoneticSymbolProtected {
            get {
                return m_protected;
            }
            set {
                m_protected = value;
            }
        }


        public float UnknownFloat {
            get {
                return d1;
            }
            set {
                d1 = value;
            }
        }


        public int[] ConsonantAdjustment {
            get {
                return m_consonant_adjustment;
            }
        }


        /// <summary>
        /// このオブジェクトの簡易コピーを取得します。
        /// </summary>
        /// <returns>このインスタンスの簡易コピー</returns>
        public Lyric Clone() {
            Lyric result = new Lyric();
            result.m_phrase = this.m_phrase;
            result.m_phonetic_symbol = (string[])this.m_phonetic_symbol.Clone();
            result.d1 = this.d1;
            result.m_consonant_adjustment = (int[])this.m_consonant_adjustment.Clone();
            result.m_protected = m_protected;
            return result;
        }

        
        /// <summary>
        /// 歌詞、発音記号を指定したコンストラクタ
        /// </summary>
        /// <param name="phrase">歌詞</param>
        /// <param name="phonetic_symbol">発音記号</param>
        public Lyric( string phrase, string phonetic_symbol ) {
            m_phrase = phrase;
            PhoneticSymbol = phonetic_symbol;
            d1 = 0.000000f;
        }


        private Lyric() {
        }


        /// <summary>
        /// この歌詞のフレーズを取得または設定します。
        /// </summary>
        [Obsolete]
        public string jp {
            get {
                return Phrase;
            }
            set {
                Phrase = value;
            }
        }

        
        /// <summary>
        /// この歌詞のフレーズを取得または設定します。
        /// </summary>
        public string Phrase {
            get {
                return m_phrase;
            }
            set {
                m_phrase = value;
            }
        }


        /// <summary>
        /// この歌詞の発音記号を取得または設定します。
        /// </summary>
        public string PhoneticSymbol {
            get {
                string ret = m_phonetic_symbol[0];
                for ( int i = 1; i < m_phonetic_symbol.Length; i++ ) {
                    ret += " " + m_phonetic_symbol[i];
                }
                return ret;
            }
            set {
                string s = value.Replace( "  ", " " );
                m_phonetic_symbol = s.Split( " ".ToCharArray(), 16 );
                for ( int i = 0; i < m_phonetic_symbol.Length; i++ ) {
                    m_phonetic_symbol[i] = m_phonetic_symbol[i].Replace( @"\\", @"\" );
                }
                m_consonant_adjustment = new int[m_phonetic_symbol.Length];
                for ( int i = 0; i < m_phonetic_symbol.Length; i++ ) {
                    if ( VsqPhoneticSymbol.IsConsonant( m_phonetic_symbol[i] ) ) {
                        m_consonant_adjustment[i] = 64;
                    } else {
                        m_consonant_adjustment[i] = 0;
                    }
                }
            }
        }

        public string[] PhoneticSymbolList{
            get{
                string[] ret = new string[m_phonetic_symbol.Length];
                for ( int i = 0; i < m_phonetic_symbol.Length; i++ ) {
                    ret[i] = m_phonetic_symbol[i];
                }
                return ret;
            }
        }


        /// <summary>
        /// この歌詞の発音記号を取得または設定します。
        /// </summary>
        [Obsolete]
        public string en {
            get {
                return PhoneticSymbol;
            }
            set {
                PhoneticSymbol = value;
            }
        }


        /// <summary>
        /// 文字列からのコンストラクタ
        /// </summary>
        /// <param name="_line">生成元の文字列</param>
        public Lyric( string _line ) {
            if ( _line.Length <= 0 ) {
                m_phrase = "a";
                PhoneticSymbol = "a";
                d1 = 1.0f;
                m_protected = false;
            } else {
                string[] spl = _line.Split( new char[] { ',' } );
                int c_length = spl.Length - 3;
                if ( spl.Length < 4 ) {
                    m_phrase = "a";
                    PhoneticSymbol = "a";
                    d1 = 0.0f;
                    m_protected = false;
                } else {
                    m_phrase = decode( spl[0] );
                    PhoneticSymbol = decode( spl[1] );
                    d1 = float.Parse( spl[2] );
                    m_protected = (spl[spl.Length - 1] == "0") ? false : true;
                }
            }
        }


        /// <summary>
        /// mIndexOfのテストメソッド。search, valueをいろいろ変えてテストする事。
        /// </summary>
        /// <returns></returns>
        public static bool test_mIndexOf() {
            byte[] search = { 0, 12, 3, 5, 16, 34 };
            byte[] value = { 16, 34 };
            if ( mIndexOf( search, value ) == 4 ) {
                return true;
            } else {
                return false;
            }
        }


        /// <summary>
        /// バイト並びsearchの中に含まれるバイト並びvalueの位置を探します。
        /// </summary>
        /// <param name="search">検索対象のバイト並び</param>
        /// <param name="value">検索するバイト並び</param>
        /// <returns>valueが見つかればそのインデックスを、見つからなければ-1を返します</returns>
        private static int mIndexOf( byte[] search, byte[] value ) {
            int i, j;
            int search_length = search.Length;
            int value_length = value.Length;
            
            // 検索するバイト並びが、検索対象のバイト並びより長いとき。
            // 見つかるわけない
            if ( value_length > search_length ) {
                return -1;
            }

            // i : 検索の基点
            for ( i = 0; i <= search_length - value_length; i++ ) {
                bool failed = false;
                for ( j = 0; j < value_length; j++ ) {
                    if ( search[i + j] != value[j] ) {
                        failed = true;
                        break;
                    }
                }
                if ( !failed ) {
                    return i;
                }
            }
            return -1;
        }


        /// <summary>
        /// エスケープされた\"や、\x**を復帰させます
        /// </summary>
        /// <param name="_string">デコード対象の文字列</param>
        /// <returns>デコード後の文字列</returns>
        public static string decode( string _string ) {
            string result = _string;
            result = result.Replace( "\\\"", "" );
            //Encoding sjis = Encoding.GetEncoding( 932 );
            //Encoding sjis = Encoding.GetEncoding( "Shift_JIS" );
            byte[] str = Encoding.ASCII.GetBytes( result );

            //Console.WriteLine( "Lyric.decode; sjis.GetString( str )=" + sjis.GetString( str ) );
            byte[] x16 = Encoding.ASCII.GetBytes( "\\x" );
            int index = mIndexOf( str, x16 );
            while ( index >= 0 ) {
                //Console.WriteLine( "Lyric.decode; index=" + index );
                byte[] chr_byte = new byte[2];
                chr_byte[0] = str[index + 2];
                chr_byte[1] = str[index + 3];
                string chr = Encoding.ASCII.GetString( chr_byte );
                //Console.WriteLine( "Lyric.decode; chr=" + chr );
                int chrcode = Convert.ToInt32( chr, 16 );
                str[index] = (byte)chrcode;
                for ( int i = index + 4; i < str.Length; i++ ) {
                    str[i - 3] = str[i];
                }
                int length = str.Length - 3;
                byte[] new_str = new byte[length];
                for ( int i = 0; i < length; i++ ) {
                    new_str[i] = str[i];
                }
                Array.Resize( ref str, length );
                str = new_str;
                //search = Encoding.Unicode.GetString( str );
                //Console.WriteLine( "Lyric.decode; sjis.GetString( str )=" + sjis.GetString( str ) );
                index = mIndexOf( str, x16 );
            }
            //return sjis.GetString( str );
            return cp932.convert( str );
        }


        /// <summary>
        /// 与えられた文字列の中の2バイト文字を\x**の形式にエンコードします。
        /// </summary>
        /// <param name="item">エンコード対象</param>
        /// <returns>エンコードした文字列</returns>
        public static char[] encode( string item ) {
            //Encoding sjis = Encoding.GetEncoding( 932 );
            byte[] bytea = cp932.convert( item );//            sjis.GetBytes( item );
            string result = "";
            for ( int i = 0; i < bytea.Length; i++ ) {
                if ( isprint( (char)bytea[i] ) ) {
                    result += (char)bytea[i];
                } else {
                    result += "\\x" + Convert.ToString( bytea[i], 16 );
                }
            }
            char[] res = result.ToCharArray();
            return res;
        }


        /// <summary>
        /// 与えられた文字列をShift_JISとみなし、byte[]に変換しさらにchar[]に変換したもの返します
        /// </summary>
        /// <param name="item">変換元の文字列</param>
        /// <returns>変換後のchar[]</returns>
        public static char[] encodeEx( string item ) {
            //Encoding sjis = Encoding.GetEncoding( 932 );
            byte[] dat = cp932.convert( item );//            sjis.GetBytes( item );
            char[] result = new char[dat.Length];
            for ( int i = 0; i < dat.Length; i++ ) {
                result[i] = (char)dat[i];
            }
            return result;
        }


        /// <summary>
        /// このインスタンスを文字列に変換します
        /// </summary>
        /// <param name="a_encode">2バイト文字をエンコードするか否かを指定するフラグ</param>
        /// <returns>変換後の文字列</returns>
        public string ToString( bool a_encode ) {
            string result;
            if ( a_encode ) {
                string njp = new string( encode( this.Phrase ) );
                result = "\"" + njp + "\",\"" + this.PhoneticSymbol + "\"," + d1.ToString( "0.000000" );
            } else {
                result = "\"";
                //Encoding sjis = Encoding.GetEncoding( 932 );
                byte[] dat = cp932.convert( this.Phrase );//                sjis.GetBytes( this.Phrase );
                for ( int i = 0; i < dat.Length; i++ ) {
                    result += (char)dat[i];
                }
                result += "\",\"" + this.PhoneticSymbol + "\"," + d1.ToString( "0.000000" );
                result = result.Replace( @"\\", @"\" );
            }
            for ( int i = 0; i < m_consonant_adjustment.Length; i++ ) {
                result += "," + m_consonant_adjustment[i];
            }
            if ( m_protected ) {
                result += ",1";
            } else {
                result += ",0";
            }
            return result;
        }


        /// <summary>
        /// 文字がプリント出力可能かどうかを判定します
        /// </summary>
        /// <param name="ch"></param>
        /// <returns></returns>
        private static bool isprint( char ch ) {
            if ( 32 <= (int)ch && (int)ch <= 126 ) {
                return true;
            } else {
                return false;
            }
        }


        /// <summary>
        /// Lyricインスタンスを構築するテストを行います
        /// </summary>
        /// <returns>テストに成功すればtrue、そうでなければfalseを返します</returns>
        public static bool test() {

            string line = "\\\"\\x82\\xe7\\\",\\\"4 a\\\",1.000000,64,1,1";
            //Console.WriteLine( "Lyric.test; line=" + line );
            Lyric lyric = new Lyric( line );
            if ( lyric.Phrase == "ら" &&
                lyric.PhoneticSymbol == "4 a" &&
                lyric.d1 == 1.0 &&
                lyric.m_consonant_adjustment[0] == 64 &&
                lyric.m_consonant_adjustment[1] == 1 &&
                lyric.m_consonant_adjustment[2] == 1 ) {
                return true;
            } else {
                return false;
            }

        }

    }
}
