package org.lixm.core.common;

/**
 * <p>LIXMの仕様で定められたLIXMアプリケーションに関する例外を表すスーパークラスです。<br>
 * このクラスにはより一般的な例外や警告といった情報を含むことができます。</p>
 *
 *<h3>一般情報</h3>
 *
 * <p>一般的な情報を含む場合それはLIXM例外の発生する原因となった情報となります。</p>
 *
 * <p>これを<em>一般情報と言います</em>が、
 * この情報が存在する場合{@link #getMessage()}や{@link #toString()}は<br>
 * この例外クラス自身のメッセージや文字列表現ではなく
 * 原因となった一般情報に基づいたものが返されます。</p>
 *
 * @since 1.0M1
 */
public class LIXMException extends Exception{

    /**
	 * 
	 */
	private static final long serialVersionUID = -3724973378468914835L;
	private Exception         exception;

    /**
     * メッセージも一般情報も何も持たない新しいLIXM例外を作ります。
     */
    public LIXMException() {
        super( );
        exception = null;
    }

    /**
     * 一般情報を持たず直接的なメッセージを持つLIXM例外を作ります。
     * @param message LIXM例外が発生した直接的なメッセージ
     */
    public LIXMException(
        String message)
    {
        super(message);
        exception = null;
    }

    /**
     * 一般情報を持ち直接的なメッセージを持たないLIXM例外を作ります。
     * @param e 一般情報となる例外
     */
    public LIXMException(
        Exception e)
    {
        this( );
        exception = e;
    }

    /**
     */
    public LIXMException(
        String message, Exception e)
    {
        super(message);
        exception = e;
    }

    /**
     * <p>この例外のメッセージです。<br>
     * 例外に一般情報が含まれる場合はそれに基づいたメッセージが返されますが
     * 返されるメッセージは実装依存です。</p>
     * 
     * @return 例外メッセージ。一般情報を持つ場合はそれに基づく。
     */
    public String getMessage() {

    	//FIXME 一般例外がメッセージを持ってない場合、他の情報を返すべき
        if (exception != null) {
        	String mess = (exception.getMessage( ).length( ) == 0) ?
        		exception.toString( ) : exception.getMessage( );
            return mess;
        } else {
            return super.getMessage( );
        }
    }

    /**
     * ラップされた一般情報を返します。
     * @return LIXM例外にラップされた一般情報。存在しない場合は<code>null</code>。
     */
    public Exception getException() {
        return exception;
    }

    /**
     * <p>この例外の文字列表現です。<br>
     * 例外に一般情報が含まれる場合はそれに基づいた文字列表現が返されますが
     * 返される文字列は実装依存です。</p>
     * 
     * @return この例外の文字列表現。一般情報を持つ場合はそれに基づく。
     */
    public String toString() {

        if (exception != null) {
            return exception.toString( ) + "->" + super.toString( );
        } else {
            return super.toString( );
        }
    }
}
