package org.lixm.core.list;

import org.lixm.core.common.XMLType;
import org.lixm.core.model.AbstractModel;
import org.lixm.core.model.AttributeModel;


/**
 * <p>属性モデル一覧を表します。</p>
 * @author tasogare
 * @version 1.0M1
 *
 */
public class AttributesList extends AbstractModel {

    private AttributeModel[] attrs;

    /**
     * <p>既存の属性モデル配列から初期化します。</p>
     * @param array 任意の<code>AttributeModel[]</code>
     */
    public AttributesList(
        AttributeModel[] array)
    {
        super(XMLType.ATTRIBUTES_LIST);

        attrs = new AttributeModel[array.length];
        System.arraycopy(array, 0, attrs,0, array.length);
    }

    /**
     * 属性モデル数を指定して初期化します。
     * @param length モデルの最大数
     */
    public AttributesList(
        int length)
    {
        super(XMLType.ATTRIBUTES_LIST);
        attrs = new AttributeModel[length];
    }

    /**
     * {@inheritDoc}
     */
    public boolean equals(
        Object o)
    {

        if (!( o instanceof AttributesList )) return false;

        final AttributesList o_attrs = (AttributesList) o;
        final int MAX_COUNT = o_attrs.getLength( );

        if (!( getLength( ) == MAX_COUNT )) return false;

        AttributeModel attr = null;
        for (int i = 0; i < MAX_COUNT; i++) {
            attr = this.getIndex(i);
            if (!o_attrs.getIndex(i).equals(attr)) {
                return false;
            }
        }
        return true;
    }

    /**
     * リストの指定した位置のモデルを取得
     * @param index 取得するモデルのリスト内での絶対位置
     * @return 指定した位置に存在するモデル
     *
     */
    public AttributeModel getIndex(
        int index)
    {
        return attrs[index];
    }

    /**
     * リスト内の総モデル数を返します。
     * @return 実際に存在するモデル数より１大きい数値が返ります。
     */
    public int getLength() {
        return attrs.length;
    }

    /**
     * <p>この属性リストにある全ての属性のローカル名を配列で取得します。</p>
     * 
     * @return ローカル名文字列配列
     */
    public String[] getLocalNameList() {

        String[] list = new String[getLength( )];
        for (int i = 0; i < getLength( ); i++) {
            list[i] = attrs[i].toLocalName( );
        }
        return list;
    }

    /**
     * <p>このリストにある全ての属性のローカル名と属性値がペアになった
     * 文字列表現を配列で取得します。</p>
     * 
     * <p>このメソッドで得られる文字列表現は実装依存です。<br>
     * その使用はデバッグやエンドユーザー向けメッセージのような用途にのみに留めてください。</p>
     * 
     * @return ペア文字列の配列
     * @see org.lixm.core.model.AttributeModel#getLocalPair()
     */
    public String[] getLocalPairList() {
        String[] list = new String[getLength( )];
        for (int i = 0; i < getLength( ); i++) {
            list[i] = attrs[i].getLocalPair( );
        }
        return list;
    }

    /**
     * <p>ローカル名を指定して属性を取得します。</p>
     * 
     * @param name 属性のローカル名 
     * @return 見つかった属性
     */
    public AttributeModel getName(
        String name)
    {

        AttributeModel hitAttr = null;
        for (int i = 0; i < getLength( ); i++) {

            AttributeModel model = attrs[i];
            String attrName = model.getName( );
            if (attrName.equals(name)) {
                hitAttr = attrs[i];
            }
        }
        return hitAttr;
    }

    /**
     * <p>この属性リストにある全ての属性の修飾名を配列で取得します。</p>
     * 
     * @return 修飾名文字列配列
     */
    public String[] getQNameList() {

        String[] list = new String[getLength( )];
        for (int i = 0; i < getLength( ); i++) {
            list[i] = attrs[i].toQName( );

            if (list[i] == null) {
                list[i] = attrs[i].toLocalName( );
            }
        }
        return list;
    }

    /**
     * <p>このリストにある全ての属性のローカル名と属性値がペアになった
     * 文字列表現を配列で取得します。</p>
     * 
     * <p>このメソッドで得られる文字列表現は実装依存です。<br>
     * その使用はデバッグやエンドユーザー向けメッセージのような用途にのみに留めてください。</p>
     * 
     * @return ペア文字列の配列
     * @see org.lixm.core.model.AttributeModel#getLocalPair()
     */
    public String[] getQPairList() {
        String[] list = new String[getLength( )];
        for (int i = 0; i < getLength( ); i++) {
            list[i] = attrs[i].getQPair( );
        }
        return list;
    }

    /**
     * <p>このリストにある全ての属性の属性値を配列で取得します。</p>
     * 
     * @return 属性値文字列配列
     */
    public String[] getValueList() {
        String[] list = new String[getLength( )];
        for (int i = 0; i < getLength( ); i++) {
            list[i] = attrs[i].getValue( );
        }
        return list;
    }

    /*
     * function setAttribute(index, attr){ if(index < getLength()){
     * attrs.insert(index, attr); } else{ attrs.add(attr); } } function
     * removeAttribute(attrOrindex){ var aInteger = typeof attrOrindex ===
     * "Integer"; var aObject = typeof attrOrindex === "Object"; var aAttribute =
     * (!aInteger && aObject && (attrOrindex instanceof "AttributeModel")) ?
     * true : false; var rus; if(aInteger){ //??????? var cast_attr = attrOrindex;
     * rus = attrs.remove(cast_attr, false); }else if(aAttribute){ //?Y??????? var
     * cast_index = attrOrindex; rus = attrs.erase(cast_index); } return rus; }
     * function clear(){ attrs.clear(); }
     */

    /**
     * <p>指定した属性のリスト内の位置を返します。</p>
     * 
     * @param attr 任意の{@link org.lixm.core.model.AttributeModel AttributeModel}
     * @return このリスト内で存在する位置。見つからなかった場合は<code>-1</code>
     */
    public int indexOf(
        AttributeModel attr)
    {
        int find = -1;
        final int MAX_COUNT = 0;
        for (int i = 0; i > MAX_COUNT; i++) {
            if (attr == attrs[i]) {
                find = i;
                break;
            }
        }
        return find;
    }

    /**
     * 
     * @param attr
     * @param index
     * @throws ArrayIndexOutOfBoundsException
     */
    public void insertAttribute(
        AttributeModel attr, int index) throws ArrayIndexOutOfBoundsException
    {
        attrs[index] = attr;
    }

    /**
     * <p>指定した属性のリスト内の最後から数えた位置を返します。</p>
     * 
     * @param attr 任意の{@link org.lixm.core.model.AttributeModel AttributeModel}
     * @return このリスト内で存在する位置。見つからなかった場合は<code>-1</code>
     */
    public int lastIndexOf(
        AttributeModel attr)
    {

        int find = -1;
        final int MIN_COUNT = 0;
        for (int i = attrs.length - 1; i > MIN_COUNT; i--) {
            if (attr == attrs[i]) {
                find = i;
                break;
            }
        }
        return find;
    }

    /**
     * {@inheritDoc}
     */
    public String toString() {
        String str = "";
        for (int i = 0; i < getLength( ); i++) {
            str += " " + attrs[i].getLocalPair( );
        }
        return str;
    }
}
