package org.lixm.core.list;

import org.lixm.core.common.XMLType;
import org.lixm.core.model.AbstractModel;

/**
 * <p>カーソルはリスト内のモデルの位置情報をカプセル化します。</p>
 * @author tasogare
 * @version 1.0M1
 *
 */
public class XMLCursor {

    private XMLModelList list;

    private int          current = -1;

    private int          mark    = -1;

    private XMLCursor() {}

    /**
     * <p>カーソルを生成します。</p>
     * 
     * @param list モデルリスト
     */
    public XMLCursor(
        XMLModelList list)
    {
        this.list = list;
    }

    /**
     * <p>リストの初めにあるモデルの位置情報を取得します。</p>
     * 
     * @return この戻り値は常に0になる筈です。
     */
    public int getFirst() {
        return 0;
    }

    /**
     * <p>リストの最後にあるモデルの位置情報を取得します。</p>
     * 
     * @return この戻り値は常に
     * {@link org.lixm.core.list.XMLModelList#size() size()} - 1 になる筈です。
     * @see org.lixm.core.list.XMLModelList
     */
    public int getLast() {
        return ( list.size( ) - 1 );
    }


    /**
     * <p>現在位置の一つ次にあるモデルの位置情報を取得します。</p>
     * <p>この時、位置情報を更新します。
     * リスト内の次の位置にモデルが無い場合、
     * <code>IndexOutOfBoundsException</code>を投げます。</p>
     * 
     * @return <code>current + 1</code>となる値が返る。
     * 
     * @throws IndexOutOfBoundsException hasNext()が偽を返す場合
     * 
     * @see #next()
     * @see #hasNext()
     */
    public int inc()  throws IndexOutOfBoundsException{
        if(!hasNext( )) throw new IndexOutOfBoundsException("");
        return ++current;
    }


    /**
     * <p>現在位置の一つ次にあるモデルの位置情報を取得します。
     * {@link #inc()}との違いは位置情報を更新しません。</p>
     * @return <code>current + 1</code>となる値が返る。
     */
    public int next() {
        return ( current + 1 );
    }

    /**
     * <p>次に存在する開始タグまたは終了タグの位置情報を取得します。</p>
     * @return タグモデルを見つけた場合その位置
     */
    public int nextTag() {

        if(!hasNext( )) return -1;

        int tempCurrent = getIndex( );
        AbstractModel nextModel = null;

        final int MAX_COUNT = list.size( );
        for (int i = 1; tempCurrent < MAX_COUNT; i++) {
            nextModel = list.get(getIndex( ) + i);

            boolean nextModelOfStartOrEndTag =
                ( nextModel.matchTypes(XMLType.START_TAG) || nextModel
                    .matchTypes(XMLType.END_TAG) );

            if (nextModelOfStartOrEndTag) {
                return (getIndex( ) + i);
            }
        }
        return -1;
    }


    /**
     * <p>現在位置から見て一つ次の位置にまだモデルが存在しているか調べます。</p>
     * 
     * @return 次にモデルが存在しているなら真、 そうで無いなら偽を返します。
     * 
     * @see #inc()
     * @see #next()
     */
    public boolean hasNext() {
        return ( next( ) < list.size( ) );
    }

 
    /**
     *  <p>現在位置の一つ前にあるモデルの位置情報を取得します。</p>
     *  <p>この時、位置情報を更新します。
     *  リスト内の前の位置にモデルが無い場合、
     *  <code>IndexOutOfBoundsException</code>を投げます。</p>
     * 
     * @return <code>current - 1</code>となる値が返る。
     * 
     * @throws IndexOutOfBoundsException
     * 
     * @see #previous()
     * @see #hasPrevious()
     */
    public int dec()  throws IndexOutOfBoundsException{
        if(!hasPrevious( )) throw new IndexOutOfBoundsException("");
        return --current;
    }


    /**
     * <p>現在位置の一つ前にあるモデルの位置情報を取得します。
     * {@link #dec()}との違いは位置情報を更新しません。</p>
     * 
     * @return <code>current - 1</code>となる値が返る。
     */
    public int previous() {
        return ( current - 1 );
    }


    /**
     * <p>以前に存在する開始タグまたは終了タグの位置情報を取得します。</p>
     * 
     * @return タグモデルを見つけた場合その位置
     */
    public int previousTag() {

        if(!hasPrevious( )) return -1;

        AbstractModel previousModel = null;

        final int MAX_COUNT = getIndex( );
        for (int i = 1; i != MAX_COUNT; i++) {
            previousModel = list.get(getIndex( ) - i);

            boolean previousModelOfStartOrEndTag =
                ( previousModel.getXMLType( ) == XMLType.START_TAG || previousModel
                    .getXMLType( ) == XMLType.END_TAG );

            if (previousModelOfStartOrEndTag) {
                return (getIndex( ) - i);
            }
        }
        return -1;
    }


    /**
     * <p>現在位置から見て一つ前の位置にまだモデルが存在しているか調べます。</p>
     * 
     * @return 前にモデルが存在しているなら真、そうで無いなら偽を返します。
     * 
     * @see #dec()
     * @see #previous()
     */
    public boolean hasPrevious() {
        return ( previous( ) > -1 );
    }


    /**
     * <p>カーソルの現在位置を設定します。
     * この値は{@link #getIndex()}で取得する場合は絶対的な位置情報となり、
     * {@link #inc()}などで取得する場合はこのメソッドで設定した
     * 位置からの相対的な位置情報となります。</p>
     * 
     * @param index 設定する位置情報
     * @throws IndexOutOfBoundsException リストの範囲を超える場合に投げられます。
     * 
     * @see #getIndex()
     * @see #resetIndex()
     * @see #inc()
     * @see #next()
     * @see #dec()
     * @see #previous()
     */
    public void setIndex(
        int index) throws IndexOutOfBoundsException
    {
        if (-1 <= index && index < list.size( )) {
            current = index;
        } else {
            throw new IndexOutOfBoundsException( );
        }
    }


    /**
     * <p>現在位置を返します。</p>
     * 
     * @return 現在の絶対的な位置情報
     * 
     * @see #setIndex(int) 
     * @see #resetIndex()
     */
    public int getIndex() {
        return current;
    }


    /**
     * <p>現在位置を初期化されたときの状態に戻します。</p>
     * 
     * @see #setIndex(int)
     * @see #getIndex()
     *
     */
    public void resetIndex() {
        current = -1;
    }


    /**
     * <p>任意の位置情報を記憶します。</p>
     * 
     * @param index 記憶する位置情報。
     * @throws IndexOutOfBoundsException 実引数がリストの範囲外の場合。
     * 
     * @see #getMark()
     */
    public void setMark(
        int index) throws IndexOutOfBoundsException
    {
        if (-1 <= index && index < list.size( )) {
            mark = index;
        } else {
            throw new ArrayIndexOutOfBoundsException( );
        }
    }


    /**
     * <p>記憶済み位置情報を返します。</p>
     * 
     * @return 記憶済みの位置情報
     * 
     * @see #setMark(int)
     */
    public int getMark() {
        return mark;
    }


    /**
     * <p>記憶済み位置情報を初期化されたときの状態に戻します。</p>
     * 
     * @see #getMark()
     */
    public void resetMark() {
        mark = -1;
    }


    /**
     * <p>現在位置を記憶し、任意の位置を現在位置に設定しなおします。</p>
     * 
     * @param pos 設定する現在位置。
     * @throws IndexOutOfBoundsException リストの範囲を超える場合。
     */
    public void seek(
        int pos) throws IndexOutOfBoundsException
    {
        setMark(current);
        setIndex(pos);
    }


    /**
     * <p>このカーソルに対応するリストを返します。</p>
     * 
     * @return このカーソルに対応するリスト
     * 
     * @see org.lixm.core.list.XMLModelList
     */
    public XMLModelList getList() {
        return list;
    }
}
