/*
# copyright (C) 2013-2014 FUJITSU LIMITED All Rights Reserved

# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; version 2
# of the License.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  
# 02110-1301, USA.
*/

/*
 * lxc resource command
 *	copyright (C) 2014 Fujitsu Limited.
 *
 * ex)
 *	lxcf-resource1 show       lxc-bench1 
 *	lxcf-resource1 initrsc    lxc-bench1 
 *	lxcf-resource1 cpun       lxc-bench1 0
 *	lxcf-resource1 cpurate    lxc-bench1 30
 *	lxcf-resource1 memlimit   lxc-bench1 100M
 *	lxcf-resource1 memswlimit lxc-bench1 200M
 *	lxcf-resource1 numa       lxc-bench1 0-3
 *	lxcf-resource1 blkio_r    lxc-bench1 8:0 10M
 *	lxcf-resource1 blkio_w    lxc-bench1 8:0 10M
 *	lxcf-resource1 blkiops_r  lxc-bench1 8:0 10
 *	lxcf-resource1 blkiops_w  lxc-bench1 8:0 10
 *	lxcf-resource1 net_cls    lxc-bench1 010000f00
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/file.h>
#include <sys/stat.h>

//#define DEBUG	1
int check_libvirt_F20(char *name);
int check_libvirt_F19(char *name);
int show_net_cls(FILE* fd, char *name);
int show_blkio(FILE* fd, char *name);

/* cgroup initial rsc directories */
#define	CPUSET_CPUS_INIT		\
		"/sys/fs/cgroup/cpuset/cpuset.cpus"
#define	CPUSET_MEMS_INIT		\
		"/sys/fs/cgroup/cpuset/cpuset.mems"

/* Libvirt used is chosen */
char*	libvirtname;

char* 	CPUSET_CPUS;
char* 	CPUSET_CPUS_VCPU;
char* 	CPUSET_MEMS;
char* 	CPUSET_MEMORY_MIGRATE;
char* 	CPUSET_MEMS_VCPU;
char* 	CPUSET_MEMORY_MIGRATE_VCPU;
char*	CPU_CFS_QUOTA_US;
char*	CPU_CFS_PERIOD_US;
char*	MEMORY_LIMIT_IN_BYTES;
char*	MEMORY_MEMSW_LIMIT_IN_BYTES;
char*	BLKIO_THROTTLE_READ_BPS_DEVICES;
char*	BLKIO_THROTTLE_WRITE_BPS_DEVICES;
char*	BLKIO_THROTTLE_READ_IOPS_DEVICES;
char*	BLKIO_THROTTLE_WRITE_IOPS_DEVICES;
char*	NET_CLS_CLASSID;


char*	LXCNAME;
char*	KVMNAME;

char*	NAME1;
char*	NAME2;

int (*check_libvirt)(char *name);


//#ifdef __RHEL7__

/* cgroup directories */
#define	CPUSET_CPUS_F20		\
		"/sys/fs/cgroup/cpuset/machine.slice/%s%s.scope/cpuset.cpus"
#define	CPUSET_CPUS_VCPU_F20	\
		"/sys/fs/cgroup/cpuset/machine.slice/%s%s.scope/vcpu%d/cpuset.cpus"
#define	CPUSET_MEMS_F20		\
		"/sys/fs/cgroup/cpuset/machine.slice/%s%s.scope/cpuset.mems"
#define	CPUSET_MEMORY_MIGRATE_F20		\
		"/sys/fs/cgroup/cpuset/machine.slice/%s%s.scope/cpuset.memory_migrate"
#define	CPUSET_MEMS_VCPU_F20		\
		"/sys/fs/cgroup/cpuset/machine.slice/%s%s.scope/vpu%d/cpuset.mems"
#define	CPUSET_MEMORY_MIGRATE_VCPU_F20		\
		"/sys/fs/cgroup/cpuset/machine.slice/%s%s.scope/vpu%d/cpuset.memory_migrate"
#define CPU_CFS_QUOTA_US_F20	\
		"/sys/fs/cgroup/cpu/machine.slice/%s%s.scope/cpu.cfs_quota_us"
#define CPU_CFS_PERIOD_US_F20	\
		"/sys/fs/cgroup/cpu/machine.slice/%s%s.scope/cpu.cfs_period_us"
#define MEMORY_LIMIT_IN_BYTES_F20	\
		"/sys/fs/cgroup/memory/machine.slice/%s%s.scope/memory.limit_in_bytes"
#define MEMORY_MEMSW_LIMIT_IN_BYTES_F20	\
		"/sys/fs/cgroup/memory/machine.slice/%s%s.scope/memory.memsw.limit_in_bytes"
#define BLKIO_THROTTLE_READ_BPS_DEVICES_F20 \
	"/sys/fs/cgroup/blkio/machine.slice/%s%s.scope/blkio.throttle.read_bps_device"
#define BLKIO_THROTTLE_WRITE_BPS_DEVICES_F20 \
	"/sys/fs/cgroup/blkio/machine.slice/%s%s.scope/blkio.throttle.write_bps_device"
#define BLKIO_THROTTLE_READ_IOPS_DEVICES_F20 \
	"/sys/fs/cgroup/blkio/machine.slice/%s%s.scope/blkio.throttle.read_iops_device"
#define BLKIO_THROTTLE_WRITE_IOPS_DEVICES_F20 \
	"/sys/fs/cgroup/blkio/machine.slice/%s%s.scope/blkio.throttle.write_iops_device"
#define NET_CLS_CLASSID_F20	\
		"/sys/fs/cgroup/net_cls/machine.slice/%s%s.scope/net_cls.classid"


/* LXC libvirt name and KVM libvirt name */
char *LXCNAME_F20 = "machine-lxc\\x2d";
char *KVMNAME_F20 = "machine-qemu\\x2d";

#define	NAME1_F20		libvirtname
#define NAME2_F20		x2dname

char x2dname[1024];
void (*x2d)(char *name);


void x2d_F20(char *name)
{
	int	i, j;
	int	len = strlen(name);

	if (len >= 1024/4) {
		len = 1024/4-1;
	}

	for (i=0, j=0; i < len; i++, j++) {
		if (name[i] == 0) {
			break;
		} else if (name[i] == '-') {
			x2dname[j++] = '\\';
			x2dname[j++] = 'x';
			x2dname[j++] = '2';
			x2dname[j]   = 'd';
		} else {
			x2dname[j] = name[i];
		}
	}
	x2dname[1024-1] = 0;

}

//#else /* __RHEL7__ */

/* cgroup directories */
#define	CPUSET_CPUS_F19		\
		"/sys/fs/cgroup/cpuset/machine/%s.%s/cpuset.cpus"
#define	CPUSET_CPUS_VCPU_F19	\
		"/sys/fs/cgroup/cpuset/machine/%s.%s/vcpu%d/cpuset.cpus"
#define	CPUSET_MEMS_F19		\
		"/sys/fs/cgroup/cpuset/machine/%s.%s/cpuset.mems"
#define	CPUSET_MEMORY_MIGRATE_F19		\
		"/sys/fs/cgroup/cpuset/machine/%s.%s/cpuset.memory_migrate"
#define	CPUSET_MEMS_VCPU_F19		\
		"/sys/fs/cgroup/cpuset/machine/%s.%s/vpu%d/cpuset.mems"
#define	CPUSET_MEMORY_MIGRATE_VCPU_F19		\
		"/sys/fs/cgroup/cpuset/machine/%s.%s/vpu%d/cpuset.memory_migrate"
#define CPU_CFS_QUOTA_US_F19	\
		"/sys/fs/cgroup/cpu/machine/%s.%s/cpu.cfs_quota_us"
#define CPU_CFS_PERIOD_US_F19	\
		"/sys/fs/cgroup/cpu/machine/%s.%s/cpu.cfs_period_us"
#define MEMORY_LIMIT_IN_BYTES_F19	\
		"/sys/fs/cgroup/memory/machine/%s.%s/memory.limit_in_bytes"
#define MEMORY_MEMSW_LIMIT_IN_BYTES_F19	\
		"/sys/fs/cgroup/memory/machine/%s.%s/memory.memsw.limit_in_bytes"
#define BLKIO_THROTTLE_READ_BPS_DEVICES_F19 \
	"/sys/fs/cgroup/blkio/machine/%s.%s/blkio.throttle.read_bps_device"
#define BLKIO_THROTTLE_WRITE_BPS_DEVICES_F19 \
	"/sys/fs/cgroup/blkio/machine/%s.%s/blkio.throttle.write_bps_device"
#define BLKIO_THROTTLE_READ_IOPS_DEVICES_F19 \
	"/sys/fs/cgroup/blkio/machine/%s.%s/blkio.throttle.read_iops_device"
#define BLKIO_THROTTLE_WRITE_IOPS_DEVICES_F19 \
	"/sys/fs/cgroup/blkio/machine/%s.%s/blkio.throttle.write_iops_device"
#define NET_CLS_CLASSID_F19	\
		"/sys/fs/cgroup/net_cls/machine/%s.%s/net_cls.classid"

/* LXC libvirt name and KVM libvirt name */
char *LXCNAME_F19 = "libvirt-lxc";
char *KVMNAME_F19 = "libvirt-qemu";

#define	NAME1_F19		x2dname
#define NAME2_F19		libvirtname

void x2d_F19(char *name) 
{
	int	i, j;
	int	len = strlen(name);

	if (len >= 1024/4) {
		len = 1024/4-1;
	}
	
	for (i=0, j=0; i < len; i++, j++) {
		if (name[i] == 0) {
			break;
		} else {
			x2dname[j] = name[i];
		}
	}
	x2dname[1024-1] = 0;
}

//#endif /* __RHEL7__ */

/* switch F20 or F19 */
void set_F20()
{
	CPUSET_CPUS = CPUSET_CPUS_F20;
	CPUSET_CPUS_VCPU = CPUSET_CPUS_VCPU_F20;
	CPUSET_MEMS = CPUSET_MEMS_F20;
	CPUSET_MEMORY_MIGRATE = CPUSET_MEMORY_MIGRATE_F20;
	CPUSET_MEMS_VCPU = CPUSET_MEMS_VCPU_F20;
	CPUSET_MEMORY_MIGRATE_VCPU = CPUSET_MEMORY_MIGRATE_VCPU_F20;
	CPU_CFS_QUOTA_US = CPU_CFS_QUOTA_US_F20;
	CPU_CFS_PERIOD_US = CPU_CFS_PERIOD_US_F20;
	MEMORY_LIMIT_IN_BYTES = MEMORY_LIMIT_IN_BYTES_F20;
	MEMORY_MEMSW_LIMIT_IN_BYTES = MEMORY_MEMSW_LIMIT_IN_BYTES_F20;
	BLKIO_THROTTLE_READ_BPS_DEVICES = BLKIO_THROTTLE_READ_BPS_DEVICES_F20;
	BLKIO_THROTTLE_WRITE_BPS_DEVICES = BLKIO_THROTTLE_WRITE_BPS_DEVICES_F20;
	BLKIO_THROTTLE_READ_IOPS_DEVICES = BLKIO_THROTTLE_READ_IOPS_DEVICES_F20;
	BLKIO_THROTTLE_WRITE_IOPS_DEVICES = BLKIO_THROTTLE_WRITE_IOPS_DEVICES_F20;
	NET_CLS_CLASSID = NET_CLS_CLASSID_F20;

}

void set_F19()
{
	CPUSET_CPUS = CPUSET_CPUS_F19;
	CPUSET_CPUS_VCPU = CPUSET_CPUS_VCPU_F19;
	CPUSET_MEMS = CPUSET_MEMS_F19;
	CPUSET_MEMORY_MIGRATE = CPUSET_MEMORY_MIGRATE_F19;
	CPUSET_MEMS_VCPU = CPUSET_MEMS_VCPU_F19;
	CPUSET_MEMORY_MIGRATE_VCPU = CPUSET_MEMORY_MIGRATE_VCPU_F19;
	CPU_CFS_QUOTA_US = CPU_CFS_QUOTA_US_F19;
	CPU_CFS_PERIOD_US = CPU_CFS_PERIOD_US_F19;
	MEMORY_LIMIT_IN_BYTES = MEMORY_LIMIT_IN_BYTES_F19;
	MEMORY_MEMSW_LIMIT_IN_BYTES = MEMORY_MEMSW_LIMIT_IN_BYTES_F19;
	BLKIO_THROTTLE_READ_BPS_DEVICES = BLKIO_THROTTLE_READ_BPS_DEVICES_F19;
	BLKIO_THROTTLE_WRITE_BPS_DEVICES = BLKIO_THROTTLE_WRITE_BPS_DEVICES_F19;
	BLKIO_THROTTLE_READ_IOPS_DEVICES = BLKIO_THROTTLE_READ_IOPS_DEVICES_F19;
	BLKIO_THROTTLE_WRITE_IOPS_DEVICES = BLKIO_THROTTLE_WRITE_IOPS_DEVICES_F19;
	NET_CLS_CLASSID = NET_CLS_CLASSID_F19;

}




/* resource save path */
#define RSCSAVEPATH	"/etc/lxcf/rsc"


#define OSHOW		"show"
#define OINITRSC	"initrsc"
#define OCPUN		"cpun"
#define OCPURATE	"cpurate"
#define OMEMLIMIT	"memlimit"
#define OMEMSWLIMIT	"memswlimit"
#define ONUMA		"numa"
#define OBLKIO_R	"blkio_r"
#define OBLKIO_W	"blkio_w"
#define OBLKIOPS_R	"blkiops_r"
#define OBLKIOPS_W	"blkiops_w"
#define ONET_CLS	"net_cls"

/* check cgroup */
int check_cgroup_F20()
{
	FILE *fd;
	fd = fopen("/sys/fs/cgroup/cpuset/machine.slice/cpuset.cpus", "r");
	if (fd != NULL) {
		fclose(fd);
		set_F20();
		return 1;
	}
	
	fd = fopen("/sys/fs/cgroup/cpuset/machine/cpuset.cpus", "r");
	if (fd != NULL) {
		fclose(fd);
		set_F19();
		return 0;
	}

	set_F20();
	return 1;
}



/* check the unit of KMG of input data */
long long check_KMG(char *val)
{
	char *endptrp;
	double valnum;
		
	valnum = strtod(val, &endptrp);

	if (*endptrp == 0) {
		return (long long)valnum;
	}

	switch (*endptrp) {
		case 'k' :
		case 'K' : {
			valnum = valnum * 1024.;
			return (long long)valnum;
		}
		case 'm' :
		case 'M' : {
			valnum = valnum * 1024. * 1024.;
			return (long long)valnum;
		}
		case 'g' :
		case 'G' : {
			valnum = valnum * 1024. * 1024. * 1024. ;
			return (long long)valnum;
		}
		case 't' :
		case 'T' : {
			valnum = valnum * 1024. * 1024. * 1024. * 1024.;
			return (long long)valnum;
		}



		default : return -1LL;
	}

	return -1LL;
}

void printKMG(FILE* fd, char *val)
{
	char 	*endptrp;
	double 	valnum;
		
	valnum = strtod(val, &endptrp);

	if (valnum >= 1024.0*1024.0*1024.0*1024.0*80000) {
		fprintf(fd, "-");
	} else if (valnum >= 1024.0*1024.0*1024.0*1024.0) {
		fprintf(fd, "%.1fT", valnum/1024.0/1024.0/1024.0/1024.0);
	} else if (valnum >= 1024.0*1024.0*1024.0) {
		fprintf(fd, "%.1fG", valnum/1024.0/1024.0/1024.0);
	} else if (valnum >= 1024.0*1024.0) {
		fprintf(fd, "%.1fM", valnum/1024.0/1024.0);
	} else if (valnum >= 1024.0) {
		fprintf(fd, "%.1fk", valnum/1024.0);
	} else {
		fprintf(fd, "%d", (int)valnum);
	}
}

/* The value is written in name path. */
int writecg(char *name, char *val)
{
#ifndef DEBUG
	FILE* fd = fopen(name, "w");
	if (fd == NULL) {
		fprintf(stderr, "can't open %s \n", name);
		return 0;
	}

	fprintf(fd, "%s", val);

	fclose(fd);
#else
	printf("%s : %s", name, val);

#endif

	return 1;
}


#define	MAXGETLINE	1024
char readcg_buf[MAXGETLINE];

/* The value is read from name path. */
int readcg(char *name)
{
	char	*r;
	FILE* fd = fopen(name, "r");
	if (fd == NULL) {
		fprintf(stderr, "can't open %s \n", name);
		return 0;
	}

	r = fgets(readcg_buf, MAXGETLINE-1, fd);
	if (r == NULL) {
		fprintf(stderr, "can't read value : %s\n", name);
		fclose(fd);
		return 0;
	}
	readcg_buf[MAXGETLINE - 1] = 0;

	if (readcg_buf[strlen(readcg_buf)-1] < 0x20) {
		readcg_buf[strlen(readcg_buf)-1] = 0;
	}
	
	fclose(fd);

	return 1;
}

/* read the blkio value  */
int readblkio_lines(FILE* pfd, char *name, char *lxcname, char *filename)
{
	char	*adr;
	int	flg = 0;

	FILE* fd = fopen(name, "r");
	if (fd == NULL) {
		fprintf(stderr, "can't open %s \n", name);
		return 0;
	}

	while (fgets(readcg_buf, MAXGETLINE-1, fd) != NULL) {
		readcg_buf[MAXGETLINE - 1] = 0;

		if (readcg_buf[strlen(readcg_buf)-1] < 0x20) {
			readcg_buf[strlen(readcg_buf)-1] = 0;
		}

		if (readcg_buf[0] != 0) {
			if (flg == 0) {
				flg++;
			} else {
				fprintf(pfd, ", ");
			}
				
			adr = strtok(readcg_buf, " ");
			fprintf(pfd, "\"%s ", adr);
			adr = strtok(NULL, " ");
			if ((strncmp(filename, "initblkiops_r_data", strlen("initblkiops_r_data")) != 0) &&
			    (strncmp(filename, "initblkiops_w_data", strlen("initblkiops_w_data")) != 0)) {
				printKMG(pfd, adr);
			}
			fprintf(pfd, "\"");
		}
	}
	
	fclose(fd);

	return 1;
}

/* The existence of name path is checked. */
int readpath(char *name)
{
	FILE* fd = fopen(name, "r");
	if (fd == NULL) {
		return 0;
	}

	fclose(fd);
	return 1;
}

#define RSCSAVEFILE	"/etc/lxcf/rsc/%s/resource.val"

int cat_rsc(FILE* fd, char *name)
{
	FILE*	rfd;
	char 	buf[1024];
	int	c;

	snprintf(buf, 1024-1, RSCSAVEFILE, name);

	rfd = fopen(buf, "r");
	if (rfd == NULL) {
		return (0);
	}

	while ((c = fgetc(rfd)) != EOF) {
		fputc(c, fd);
	}

	fclose(rfd);

	return 1;
}



//#ifdef __RHEL7__

/* Whether the object is LXC or KVM */
int check_libvirt_F20(char *name)
{
	char buf[1024];

	(*x2d)(name);

	snprintf(buf, 1024-1, CPUSET_CPUS, LXCNAME, x2dname);
	buf[1024-1] = 0;

	if (readpath(buf)) {
		libvirtname = LXCNAME;
	} else {
		snprintf(buf, 1024-1, CPUSET_CPUS, KVMNAME, x2dname);
		buf[1024-1] = 0;
		
		if (readpath(buf)) {
			libvirtname = KVMNAME;
		} else {
			return (0);
		}
	}
	return(1);
}

//#else /* __RHEL7__ */

/* Whether the object is LXC or KVM */
int check_libvirt_F19(char *name)
{
	char buf[1024];

	snprintf(buf, 1024-1, CPUSET_CPUS, name, LXCNAME);
	buf[1024-1] = 0;

	if (readpath(buf)) {
		libvirtname = LXCNAME;
	} else {
#if 0 /* Qemu is not targeted.  */
		snprintf(buf, 1024-1, CPUSET_CPUS, name, KVMNAME);
		buf[1024-1] = 0;
		
		if (readpath(buf)) {
			libvirtname = KVMNAME;
		} else {
			return (0);
		}
#else /* 0 */
		return 0;
#endif /* 0 */
	}
	return(1);
}

//#endif /* __RHEL7__ */

/* inirialize resources */
int initrsc(char *name)
{
	FILE*	rfd;
	char 	rbuf[1024];
	char 	buf[1024];

	snprintf(rbuf, 1024-1, RSCSAVEFILE, name);
	rbuf[1024-1] = 0;

	rfd = fopen(rbuf, "w");
	if (rfd == NULL) {
		return (0);
	}

	/* display lxc or kvm name */
	fprintf(rfd, "{\"name\" : \"%s\", ", name);
	fprintf(rfd, "\n ");
	
	/* display cpu set */
	snprintf(buf, 1024-1, CPUSET_CPUS_INIT);
	buf[1024-1] = 0;

	if (!readcg(buf)) {
		fclose(rfd);
		return -1;
	}

	fprintf(rfd, "\"cpun\" : \"%s\", ", readcg_buf);
	fprintf(rfd, "\n ");


	/* display cpu rate */
	snprintf(buf, 1024-1, CPUSET_CPUS_INIT);
	buf[1024-1] = 0;

	if (!readcg(buf)) {
		fclose(rfd);
		return -1;
	}

	fprintf(rfd, "\"cpurate\" : \"-\", ");
	fprintf(rfd, "\n ");

	/* display numa */
	snprintf(buf, 1024-1, CPUSET_MEMS_INIT);
	buf[1024-1] = 0;

	if (!readcg(buf)) {
		fclose(rfd);
		return -1;
	}

	fprintf(rfd, "\"numa\" : \"%s\", ", readcg_buf);
	fprintf(rfd, "\n ");

	fprintf(rfd, "\"memlimit\" : \"1.0G\", ");
	fprintf(rfd, "\n ");

	/* display memory+swap limit */
	fprintf(rfd, "\"memswlimit\" : \"-\", ");
	fprintf(rfd, "\n ");

	fprintf(rfd, "\"blkio_r\" : [  ],\n \"blkio_w\" : [  ],\n \"blkiops_r\" : [  ],\n \"blkiops_w\" : [  ],\n ");

	fprintf(rfd, "\"net_cls\" : \"00000000\" }");
	fprintf(rfd, "\n");

	fclose(rfd);

	cat_rsc(stdout, name);
	return 1;
}

/* show the Information */
int show(FILE* fd, char *name)
{
	long int cpurate;
	long int cpuquota;
	long int cpuperiod;
	char buf[1024];

//#ifdef __RHEL7__
	(*x2d)(name);
//#endif /* __RHEL7__ */

	/* display lxc or kvm name */
	fprintf(fd, "{\"name\" : \"%s\", ", name);

	fprintf(fd, "\n ");
	
	/* display cpu set */
	if (libvirtname == LXCNAME) {
		snprintf(buf, 1024-1, CPUSET_CPUS, NAME1, NAME2);
	} else if (libvirtname == KVMNAME) {
		snprintf(buf, 1024-1, CPUSET_CPUS_VCPU, NAME1, NAME2, 0);
	}
	buf[1024-1] = 0;

	if (!readcg(buf)) {
		return -1;
	}

	fprintf(fd, "\"cpun\" : \"%s\", ", readcg_buf);

	fprintf(fd, "\n ");

	/* get a cpu.cfs_period_us */	
	snprintf(buf, 1024-1, CPU_CFS_PERIOD_US, NAME1, NAME2);
	buf[1024-1] = 0;

	if (!readcg(buf)) {
		return -1;
	}

	/* display cpu rate */
	cpuperiod = strtol(readcg_buf, NULL, 10);

	snprintf(buf, 1024-1, CPU_CFS_QUOTA_US,	NAME1, NAME2);
	buf[1024-1] = 0;

	if (!readcg(buf)) {
		return -1;
	}

	cpuquota = strtol(readcg_buf, NULL, 10);
	if (cpuquota < 0) {
		fprintf(fd, "\"cpurate\" : \"-\", ");
	} else {
		cpurate = cpuquota*100 / cpuperiod;
		fprintf(fd, "\"cpurate\" : \"%ld\", ", cpurate);
	}

	fprintf(fd, "\n ");

	/* display numa */
	if (libvirtname == LXCNAME) {
		snprintf(buf, 1024-1, CPUSET_MEMS, NAME1, NAME2);
	} else if (libvirtname == LXCNAME) {
		snprintf(buf, 1024-1, CPUSET_MEMS_VCPU, NAME1, NAME2, 0);
	}
	buf[1024-1] = 0;

	if (!readcg(buf)) {
		return -1;
	}

	fprintf(fd, "\"numa\" : \"%s\", ", readcg_buf);

	fprintf(fd, "\n ");

	/* display memory limit */
	snprintf(buf, 1024-1, MEMORY_LIMIT_IN_BYTES, NAME1, NAME2);
	buf[1024-1] = 0;
	
	if (!readcg(buf)) {
		return -1;
	}

	fprintf(fd, "\"memlimit\" : \"");
	printKMG(fd, readcg_buf);
	fprintf(fd, "\", ");

	fprintf(fd, "\n ");

	/* display memory+swap limit */
	snprintf(buf, 1024-1, MEMORY_MEMSW_LIMIT_IN_BYTES, NAME1, NAME2);
	buf[1024-1] = 0;
	
	if (!readcg(buf)) {
		return -1;
	}

	fprintf(fd, "\"memswlimit\" : \"");
	printKMG(fd, readcg_buf);
	fprintf(fd, "\", ");

	fprintf(fd, "\n");

	show_blkio(fd, name);

	fprintf(fd, " ");

	show_net_cls(fd, name);
	
	fprintf(fd, "\n");

	return 1;
}

/* show the Information of blkio */
int show_blkio(FILE* fd, char *name)
{
	char buf[1024];

//#ifdef __RHEL7__
	(*x2d)(name);
//#endif /* __RHEL7__ */

	/* display read_bps */
	fprintf(fd, " \"blkio_r\" : [ ");
	snprintf(buf, 1024-1, BLKIO_THROTTLE_READ_BPS_DEVICES, 
							NAME1, NAME2);
	buf[1024-1] = 0;

	readblkio_lines(fd, buf, name, "initblkio_r_data");
	fprintf(fd, " ], \n");

	/* display write_bps */
	fprintf(fd, " \"blkio_w\" : [ ");
	snprintf(buf, 1024-1, BLKIO_THROTTLE_WRITE_BPS_DEVICES, 
							NAME1, NAME2);
	buf[1024-1] = 0;

	readblkio_lines(fd, buf, name, "initblkio_w_data");
	fprintf(fd, " ], \n");
	
	/* display read_iops */
	fprintf(fd, " \"blkiops_r\" : [ "); 
	snprintf(buf, 1024-1, BLKIO_THROTTLE_READ_IOPS_DEVICES, 
							NAME1, NAME2);
	buf[1024-1] = 0;

	readblkio_lines(fd, buf, name, "initblkiops_r_data");
	fprintf(fd, " ], \n");

	/* display write_iops */
	fprintf(fd, " \"blkiops_w\" : [ "); 
	snprintf(buf, 1024-1, BLKIO_THROTTLE_WRITE_IOPS_DEVICES, 
							NAME1, NAME2);
	buf[1024-1] = 0;

	readblkio_lines(fd, buf, name, "initblkiops_w_data");
	fprintf(fd, " ], \n");

	return 1;
}

/* show the Information on NET_CLS */
int show_net_cls(FILE* fd, char *name)
{
	long int net_cls;
	char buf[1024];

//#ifdef __RHEL7__
	(*x2d)(name);
//#endif /* __RHEL7__ */

	/* display net_cls value */
	snprintf(buf, 1024-1, NET_CLS_CLASSID, NAME1, NAME2);

	buf[1024-1] = 0;

	if (!readcg(buf)) {
		return -1;
	}

	net_cls = strtol(readcg_buf, NULL, 10);
	
	fprintf(fd, "\"net_cls\" : \"%08lx\" ", net_cls);

	fprintf(fd, "}\n");

	return 1;
}

/* save the Information */
int save(char *name)
{
	FILE*	fd;
	int	f;
	char	buf[1024];
	
	snprintf(buf, 1024-1, "%s/%s", RSCSAVEPATH, name);
	buf[1024-1] = 0;

	mkdir(buf, 0x755);
	
	snprintf(buf, 1024-1, "%s/%s/resource.val", RSCSAVEPATH, name);
	buf[1024-1] = 0;

	fd = fopen(buf, "w");
	if (fd == NULL) {
		fprintf(stderr, "can't open %s\n", buf);
		exit(-1);
	}

	/* lock save file */
	f = fileno(fd);
	flock(f, LOCK_EX);
	
	show(fd, name);

	/* unlock save file */
	flock(f, LOCK_UN);

	fclose(fd);

	return 0;	
}

/* allocate CPU */
int cpuset_cpus(char *name, char *val)
{
	char buf[1024];

//#ifdef __RHEL7__
	(*x2d)(name);
//#endif /* __RHEL7__ */

	if (libvirtname == LXCNAME) {
		snprintf(buf, 1024-1, CPUSET_CPUS, NAME1, NAME2);
		buf[1024-1] = 0;

		writecg(buf, val);
	} else if (libvirtname == KVMNAME) {
		int i;
		for (i = 0; ; i++) {
			snprintf(buf, 1024-1, CPUSET_CPUS_VCPU, 
						NAME1, NAME2, i);
			buf[1024-1] = 0;

			if (readpath(buf)) {
				writecg(buf, val);
			} else {
				break;
			}
		}
	}
	
	return 1;
}

/* allocate NUMA */
int cpuset_mems(char *name, char *val)
{
	char buf[1024];

//#ifdef __RHEL7__
	(*x2d)(name);
//#endif /* __RHEL7__ */

	if (libvirtname == LXCNAME) {
		// set memory_migrate
		snprintf(buf, 1024-1, CPUSET_MEMORY_MIGRATE, NAME1, NAME2);
		buf[1024-1] = 0;
	
		if ((strlen(val) == 1) && (val[0] == '-')) {
			writecg(buf, "0");
		} else {
			writecg(buf, "1");
		}

		// set numa number
		snprintf(buf, 1024-1, CPUSET_MEMS, NAME1, NAME2);
		buf[1024-1] = 0;
	
		writecg(buf, val);
	} else if (libvirtname == KVMNAME) {
		int i;
		for (i = 0; ; i++) {
			// set memory_migrate
			snprintf(buf, 1024-1, CPUSET_MEMORY_MIGRATE_VCPU, 
						NAME1, NAME2, i);
			buf[1024-1] = 0;

			if ((strlen(val) == 1) && (val[0] == '-')) {
				writecg(buf, "0");
			} else {
				writecg(buf, "1");
			}

			// set numa number
			snprintf(buf, 1024-1, CPUSET_MEMS_VCPU, 
						NAME1, NAME2, i);
			buf[1024-1] = 0;

			if (readpath(buf)) {
				writecg(buf, val);
			} else {
				break;
			}
		}
	}
	
	return 1;
}

/* The interval when the time of CPU is checked is returned. */
int cpu_cfs_period_us(char *name)
{
	char buf[1024];

//#ifdef __RHEL7__
	(*x2d)(name);
//#endif /* __RHEL7__ */

	snprintf(buf, 1024-1, CPU_CFS_PERIOD_US, NAME1, NAME2);
	buf[1024-1] = 0;
	
	writecg(buf, "100000");
	
	return 1;
}

/* The time of CPU that can be spent in the time of cpu.cfs_period_us is 
  returned. */
int cpu_cfs_quota_us(char *name, char *val)
{
	char buf[1024];
	char valbuf[10];

//#ifdef __RHEL7__
	(*x2d)(name);
//#endif /* __RHEL7__ */

	long int v = strtol(val, NULL, 10);
	if (v > 100) {
		fprintf(stderr, "It is invalid to exceed 100%%. \n");
		exit(-1);
	}
	
	snprintf(buf, 1024-1, CPU_CFS_QUOTA_US,	NAME1, NAME2);
	buf[1024-1] = 0;

	if ((val[0] == '-') && (val[1] == 0)) {
		valbuf[0] = '-'; valbuf[1] = '1'; valbuf[2] = 0;
	} else {
		snprintf(valbuf, 10-1, "%ld", v*1000);
		valbuf[10-1] = 0;
	}
	
	writecg(buf, valbuf);

	return 1;
}

/* set the limit of the memory  */
int memory_limit_in_bytes(char *name, char *val)
{
	char buf[1024];
	char valbuf[100];
	long long memlimit;

//#ifdef __RHEL7__
	(*x2d)(name);
//#endif /* __RHEL7__ */

	if (libvirtname == KVMNAME) {
		fprintf(stderr, "memlimit cannot be set with kvm. \n");
		exit(-1);
	}
	
	snprintf(buf, 1024-1, MEMORY_LIMIT_IN_BYTES, NAME1, NAME2);
	buf[1024-1] = 0;

	memlimit = check_KMG(val);
	if (memlimit < 0) {
		fprintf(stderr, "illegal value : %s\n", val);
		return 0;
	}

	snprintf(valbuf, 100-1, "%lld", memlimit);
	valbuf[100-1] = 0;
	
	writecg(buf, valbuf);
	
	return 1;
}

/* set the limit of the memory  */
int memory_memsw_limit_in_bytes(char *name, char *val)
{
	char buf[1024];
	char valbuf[100];
	long long memswlimit;

//#ifdef __RHEL7__
	(*x2d)(name);
//#endif /* __RHEL7__ */

	if (libvirtname == KVMNAME) {
		fprintf(stderr, "memswlimit cannot be set with kvm. \n");
		exit(-1);
	}
	
	snprintf(buf, 1024-1, MEMORY_MEMSW_LIMIT_IN_BYTES, NAME1, NAME2);
	buf[1024-1] = 0;

	if (val[0] == '-') {
		memswlimit = 87960930222080000LL;
	} else {
		memswlimit = check_KMG(val);
	}
	
	if (memswlimit < 0) {
		fprintf(stderr, "illegal value : %s\n", val);
		return 0;
	}

	snprintf(valbuf, 100-1, "%lld", memswlimit);
	valbuf[100-1] = 0;
	
	writecg(buf, valbuf);
	
	return 1;
}

/* set the read bps of blkio  */
int blkio_throttle_read_bps_devices(char *name, char *devval, char* val)
{
	char buf[1024];
	char valbuf[100];
	long long bps;

//#ifdef __RHEL7__
	(*x2d)(name);
//#endif /* __RHEL7__ */

	snprintf(buf, 1024-1, BLKIO_THROTTLE_READ_BPS_DEVICES, 
						NAME1, NAME2);
	buf[1024-1] = 0;

	if ((val[0] == '-') && (val[1] == 0)) {
		bps = 0;
	} else {
		bps = check_KMG(val);
	}
	
	if (bps < 0) {
		fprintf(stderr, "illegal value : %s\n", val);
		return 0;
	}

	snprintf(valbuf, 100-1, "%s %lld", devval, bps);
	valbuf[100-1] = 0;
	
	writecg(buf, valbuf);
	
	return 1;
}

/* set the write bps of blkio  */
int blkio_throttle_write_bps_devices(char *name, char *devval, char* val)
{
	char buf[1024];
	char valbuf[100];
	long long bps;

//#ifdef __RHEL7__
	(*x2d)(name);
//#endif /* __RHEL7__ */

	snprintf(buf, 1024-1, BLKIO_THROTTLE_WRITE_BPS_DEVICES, 
						NAME1, NAME2);
	buf[1024-1] = 0;

	if ((val[0] == '-') && (val[1] == 0)) {
		bps = 0;
	} else {
		bps = check_KMG(val);
	}
	
	if (bps < 0) {
		fprintf(stderr, "illegal value : %s\n", val);
		return 0;
	}

	snprintf(valbuf, 100-1, "%s %lld", devval, bps);
	valbuf[100-1] = 0;
	
	writecg(buf, valbuf);
	
	return 1;
}

/* set the read iops of blkio  */
int blkio_throttle_read_iops_devices(char *name, char *devval, char* val)
{
	char buf[1024];
	char valbuf[100];
	long long iops;

//#ifdef __RHEL7__
	(*x2d)(name);
//#endif /* __RHEL7__ */

	snprintf(buf, 1024-1, BLKIO_THROTTLE_READ_IOPS_DEVICES, 
						NAME1, NAME2);
	buf[1024-1] = 0;

	if ((val[0] == '-') && (val[1] == 0)) {
		iops = 0;
	} else {
		iops = check_KMG(val);
	}
	
	if (iops < 0) {
		fprintf(stderr, "illegal value : %s\n", val);
		return 0;
	}

	snprintf(valbuf, 100-1, "%s %lld", devval, iops);
	valbuf[100-1] = 0;
	
	writecg(buf, valbuf);
	
	return 1;
}

/* set the write iops of blkio  */
int blkio_throttle_write_iops_devices(char *name, char *devval, char* val)
{
	char buf[1024];
	char valbuf[100];
	long long iops;

//#ifdef __RHEL7__
	(*x2d)(name);
//#endif /* __RHEL7__ */

	snprintf(buf, 1024-1, BLKIO_THROTTLE_WRITE_IOPS_DEVICES, 
						NAME1, NAME2);
	buf[1024-1] = 0;

	if ((val[0] == '-') && (val[1] == 0)) {
		iops = 0;
	} else {
		iops = check_KMG(val);
	}
	
	if (iops < 0) {
		fprintf(stderr, "illegal value : %s\n", val);
		return 0;
	}

	snprintf(valbuf, 100-1, "%s %lld", devval, iops);
	valbuf[100-1] = 0;
	
	writecg(buf, valbuf);
	
	return 1;
}

/* set the net_cls.classid */
int net_cls_classid(char *name, char *val)
{
	char buf[1024];
	char hexbuf[1024];
	long int hexval;

//#ifdef __RHEL7__
	(*x2d)(name);
//#endif /* __RHEL7__ */

	snprintf(buf, 1024-1, NET_CLS_CLASSID, NAME1, NAME2);
	buf[1024-1] = 0;

	hexval=strtol(val, NULL, 16);

	snprintf(hexbuf, 1024-1, "0x%08lx", hexval);
	hexbuf[1024-1] = 0;
	
	writecg(buf, hexbuf);
	
	return 1;
}


void usage()
{
	fprintf(stderr, 
	  "usage : lxcf resource [ OPTIONS ] lxcname value\n");
	fprintf(stderr, 
	  "        OPTIONS := { show | \n");
	fprintf(stderr, 
	  "                     cpun | cpurate | memlimit | memswlimit | numa\n");
	fprintf(stderr, 
	  "                     blkio_r | blkio_w | blkiops_r | blkiops_w | \n");
	fprintf(stderr, 
	  "                     net_cls }\n");
	exit(-1);
}

int main(int argc, char **argv)
{
	char	*name;
	char	*val, *val2;
	int	flg = 0;
	int	r;

	if (geteuid() != 0) {
		fprintf(stderr, "error: Because you are not root, you cannot execute this command. \n");
		exit(-1);
	}
	
	if (argc < 3) {
		usage();
	}
	
	name = argv[2];

	r = check_cgroup_F20();
	
	if (r == 1) {

		LXCNAME = LXCNAME_F20;
		KVMNAME = KVMNAME_F20;

		libvirtname = LXCNAME;	

		NAME1 = NAME1_F20;
		NAME2 = NAME2_F20;

		check_libvirt = check_libvirt_F20;
		x2d = x2d_F20;
	} else {
		LXCNAME = LXCNAME_F19;
		KVMNAME = KVMNAME_F19;

		libvirtname = LXCNAME;	

		NAME1 = NAME1_F19;
		NAME2 = NAME2_F19;

		check_libvirt = check_libvirt_F19;
		x2d = x2d_F19;
	}
	
	if (!(*check_libvirt)(name)) {
		flg = 1;
	}

	if (argc == 3) {
		if (strncmp(argv[1], OSHOW, strlen(OSHOW)) == 0) {
			if (flg) {
				if (!cat_rsc(stdout, name)) {
					fprintf(stderr, "%s is not found.\n", name);
					exit(-1);
				} else {	
					exit(0);
				}
			}
			
			// show command
			show(stdout, name);
			save(name);

			exit (0);
			
		} else if (strncmp(argv[1], OINITRSC, strlen(OINITRSC)) == 0) {
			initrsc(name);
			exit(0);
		} else {
			fprintf(stderr, "illegal operation : %s \n", argv[1]);
			usage();
			exit (-1);
		}
	}
	
	if (flg) {
		fprintf(stderr, "%s is not started.\n", name);
		exit(-1);
	}

	if (argc == 4) {
		val = argv[3];
	
		if (strncmp(argv[1], OCPUN, strlen(OCPUN)) == 0) {
			// cpun command
			cpuset_cpus(name, val);
			save(name);
		} else if (strncmp(argv[1], OCPURATE, strlen(OCPURATE)) == 0) {
			// cpurae command
			cpu_cfs_period_us(name);
			cpu_cfs_quota_us(name, val);
			save(name);
		} else if (strncmp(argv[1], OMEMLIMIT, strlen(OMEMLIMIT)) == 0) {
			// memorylimit command
			memory_limit_in_bytes(name, val);
			save(name);
		} else if (strncmp(argv[1], OMEMSWLIMIT, strlen(OMEMSWLIMIT)) == 0) {
			// memsw command
			memory_memsw_limit_in_bytes(name, val);
			save(name);
		} else if (strncmp(argv[1], ONUMA, strlen(ONUMA)) == 0) {
			// numa command
			cpuset_mems(name, val);
			save(name);
		} else if (strncmp(argv[1], ONET_CLS, strlen(ONET_CLS)) == 0) {
			// netcls command
			net_cls_classid(name, val);
			save(name);
		} else {
			fprintf(stderr, "illegal operation : %s \n", argv[1]);
		}
	} else if (argc == 5) {
		val  = argv[3];
		val2 = argv[4];

		if (strncmp(argv[1], OBLKIO_R, strlen(OBLKIO_R)) == 0) {
			// iobps_r command
			blkio_throttle_read_bps_devices(name, val, val2);
			save(name);
		} else if (strncmp(argv[1], OBLKIO_W, strlen(OBLKIO_W)) == 0) {
			// iobps_w command
			blkio_throttle_write_bps_devices(name, val, val2);
			save(name);
		} else if (strncmp(argv[1], OBLKIOPS_R, strlen(OBLKIOPS_R)) == 0) {
			// iops_r command
			blkio_throttle_read_iops_devices(name, val, val2);
			save(name);
		} else if (strncmp(argv[1], OBLKIOPS_W, strlen(OBLKIOPS_W)) == 0) {
			// iops_w command
			blkio_throttle_write_iops_devices(name, val, val2);
			save(name);
		} else {
			fprintf(stderr, "illegal operation : %s \n", argv[1]);
			usage();
		}
	} else {
		usage();
	}

	exit(0);
}

