
import java.awt.BorderLayout;
import java.awt.Polygon;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.Collection;
import java.util.Properties;

import javax.swing.JFrame;
import javax.swing.JMenuBar;
import javax.swing.UIManager;

import map.CityMap;
import map.IsjFactory;
import map.KsjFactory;
import map.Sdf25kFactory;
import map.SdfStorage;
import map.cell.CellSearch;
import map.store.Store;
import storage.CityNameDatabase;
import util.Log;
import util.Loader;
import util.Version;
import view.DialogFactory;
import view.MapMenu;
import view.MapPanel;
import view.Setting;
import view.StatusBar;
import controller.Controller;

/*
 * Digital Map
 * 国土地理院の数値地図25000（空間データ基盤）を表示するプログラムです。
 * Copyright (C) 2005-2006 Masayasu Fujiwara
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
/**
 * 地図描画アプリケーション起動のためのクラス
 * 
 * @author Masayasu Fujiwara
 */
public class Main {

	public static final String SETTING = ".setting";
	
	/**
	 * フレーム高さ
	 */
	private static final int F_WIDTH = 800;

	/**
	 * フレーム幅
	 */
	private static final int F_HEIGHT = 600;

	/**
	 * 都道府県データの相対位置リスト
	 */
	private static final String prefData = "/data/prefecture.csv";

	/**
	 * 都道府県界のデータ
	 */
	private static final String prefBorderData = "/data/prefecture.dat";

	/**
	 * 都道府県セルのデータ
	 */
	private static final String cellData = "/data/prefecture.cell";
	
	/**
	 * args[0]には地図データのルートディレクトリの位置を指定します。
	 * args[1]にはプラグインのディレクトリの位置を指定します。
	 * 
	 * @param args 引数
	 */
	public static void main(String[] args) {
		String mapDir = ".digital_map" + File.separatorChar;
		String pluginDir = "plugin" + File.separatorChar;
		String libDir = "lib" + File.separatorChar;
		switch (args.length) {
			case 3: libDir = args[2];
			case 2: pluginDir = args[1];
			case 1: mapDir = args[0];
		}

		// 外部JARをロードします。
		Loader.addExternalJar(libDir);
		
		try {
			Properties properties = System.getProperties();
			if (((String) properties.get("java.version")).toLowerCase().contains("1.6")) {
				if (((String) properties.get("sun.desktop")).toLowerCase().contains("windows")){
					properties.setProperty("sun.java2d.d3d", "true");
					Log.out(Main.class, "setProperty sun.java2d.d3d=true");
				}
			}
		} catch (Exception e) {
			Log.err(Main.class, e);
		}
		try {
			UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
		} catch (Exception e) {
			Log.err(Main.class, e);
		}
		
		JFrame frame = null;
		String version = null;
		try {
			version = Version.get("/history.txt");
		} catch (IOException e) {
			Log.err(Main.class, e);
		}
		if (version == null) {
			frame = new JFrame("Digital Map");
		} else {
			frame = new JFrame("Digital Map ver." + version);
		}
		frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		frame.setLayout(new BorderLayout());


		CityNameDatabase cityNameDB;
		StatusBar statusbar = null;
		try {
			cityNameDB = new CityNameDatabase("/data/city.csv", mapDir + "city.idx", mapDir + "city.dat");
			statusbar = new StatusBar(" ", cityNameDB);
		} catch (IOException e) {
			e.printStackTrace();
			cityNameDB = null;
			statusbar = new StatusBar("IO ERROR");
		}
		MapPanel panel = new MapPanel(cityNameDB);
		Log.out(Main.class, "created panel.");

		Log.out(Main.class, "created statusbar");

		Setting setting = Setting.readSetting(mapDir);

		// Listener
		Controller controller = new Controller(panel, setting);
		JMenuBar menu = new MapMenu(controller);
		Log.out(Main.class, "created listener.");

		frame.add(statusbar, BorderLayout.SOUTH);
		frame.add(panel, BorderLayout.CENTER);
		frame.add(menu, BorderLayout.NORTH);

		frame.setSize(Main.F_WIDTH, Main.F_HEIGHT);
		
		frame.setVisible(true);

		if (setting.isCheckUpdate()) {
			URL url = null;
			try {
				String latest = Version.sourceforge("ma38su", "Digital Map");
				if (latest != null && version != null && !version.equals(latest)) {
					url = Version.getURL("ma38su", "Digital Map", latest);
					DialogFactory.versionDialog(latest, url, frame, controller);
				}
			} catch (IOException ex) {
				Log.err(Main.class, ex);
			}
		}

		Log.out(Main.class, "frame set visible.");

		try {
			statusbar.setRdFile("初期設定");

			Collection<Store> store = Loader.loadStorePlugin(pluginDir, mapDir + "store" + File.separatorChar);
			Log.out(Main.class, "loaded plugins.");
			
			final CityMap maps = new CityMap(panel, new CellSearch(Main.cellData),
					new Sdf25kFactory(new SdfStorage(mapDir, Main.prefData, statusbar)),
					new IsjFactory(mapDir + "isj" + File.separatorChar), store, statusbar);
			statusbar.clearReading();
			Log.out(Main.class, "created maps.");

			statusbar.setRdFile("国土数値情報 都道府県");
			final Polygon[] prefecture = KsjFactory.readPolygon(Main.prefBorderData);
			Log.out(Main.class, "created prefecture.");

			// 湖などを繰りぬく
			final int[] coast = new int[] { 84, 288, 505, 941, 953, 1008, 1013, 1014, 1593, 1916, 1595, 2345, 5193 };
			final Polygon[] prefectureCoast = new Polygon[coast.length];
			for (int i = 0; i < coast.length; i++) {
				prefectureCoast[i] = prefecture[coast[i]];
			}
			final int[] island = new int[] { 82, 83, 1592, 1914 };
			final Polygon[] prefectureIsland = new Polygon[island.length];
			for (int i = 0; i < island.length; i++) {
				prefectureIsland[i] = prefecture[island[i]];
			}
			statusbar.clearReading();

			panel.init(setting, maps, prefecture, prefectureCoast, prefectureIsland);
			maps.start();
			Log.out(Main.class, "started maps.");

		} catch (IOException e) {
			statusbar.setRdFile(e.getMessage());
			Log.err(Main.class, e);
		}
		statusbar.clearReading();
		frame.addKeyListener(controller);
		panel.addKeyListener(controller);
		panel.addMouseListener(controller);
		panel.addMouseMotionListener(controller);
		panel.addMouseWheelListener(controller);

		statusbar.setThreadPriority(Thread.MIN_PRIORITY);
		
		Log.out(Main.class, "finished main().");
	}
	
	/**
	 * インスタンス生成不要
	 */
	private Main() {
	}
}
