package controller;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.io.IOException;
import java.net.URL;

import javax.print.PrintException;
import javax.swing.AbstractButton;
import javax.swing.JCheckBoxMenuItem;

import util.Log;
import util.Output;
import util.Version;
import view.DialogFactory;
import view.MapPanel;
import view.Setting;
import map.model.Road;

/**
 * マウス、キーボードによる操作クラス
 * @author Masayasu Fujiwara
 */
public class Controller implements MouseListener, MouseMotionListener,
		MouseWheelListener, KeyListener, ActionListener {

	/**
	 * キーボードによる平行移動感度
	 */
	private static final int MOVE_SENSE = 8;

	/**
	 * マウスがWindow内に入っているかどうか
	 */
	private boolean isMouseEntered;

	/**
	 * 地図表示パネル
	 */
	private final MapPanel panel;

	/**
	 * 設定
	 */
	private Setting setting;

	private JCheckBoxMenuItem updateCheckItem;

	/**
	 * 操作の中心位置（マウス位置）のX座標
	 */
	private int x;

	/**
	 * 操作の中心位置（マウス位置）のY座標
	 */
	private int y;
	
	/**
	 * 
	 * @param panel 地図表示のためのパネル
	 * @param setting 設定
	 */
	public Controller(final MapPanel panel, Setting setting) {
		this.panel = panel;
		this.setting = setting;
		this.x = panel.getWidth() / 2;
		this.y = panel.getHeight() / 2;
	}

	public void actionPerformed(ActionEvent e) {
		String command = e.getActionCommand();
		if (command.equals("exit")) {
			// 即終了しても大丈夫だろうか・・・
			System.exit(0);
		} else if (command.equals("menu_help")) {
			this.updateCheckItem.setVisible(true);
		} else if (command.equals("export")) {
			try {
				while (!this.panel.isLoaded()) {
						Thread.sleep(2L);
				}
				Output.export(this.panel);
			} catch (InterruptedException ex) {
				Log.err(this, ex);
			}
		} else if (command.equals("print")) {
			try {
				while (!this.panel.isLoaded()) {
						Thread.sleep(1L);
				}
				Output.print(this.panel);
			} catch (InterruptedException ex) {
				Log.err(this, ex);
			} catch (PrintException ex) {
				Log.err(this, ex);
			}
		} else if (command.startsWith("path_")) {
			this.panel.switchShortestPathAlgorithm(command);
		} else if (command.startsWith("label")) {
			if (command.endsWith("Station")) {
				this.panel.switchLabel(1);
			} else if (command.endsWith("City")) {
				this.panel.switchLabel(2);
			} else if (command.endsWith("Facility")) {
				this.panel.switchLabel(4);
			}
		} else if (command.equals("node")) {
			this.panel.switchNodeView();
		} else if (command.equals("alias")) {
			this.panel.switchRendering();
		} else if (command.equals("fill")) {
			this.panel.switchFill();
		} else if (command.equals("highway")) {
			Road.switchHighway();
			this.panel.reroute();
		} else if (command.equals("help")) {
			this.setting.switchManualVisible();
			this.panel.repaint();
		} else if (command.startsWith("move_")) {
			if (command.equals("move_location")) {
				DialogFactory.locationDialog(this.panel, this);
			} else if (command.equals("move_home")) {
				this.panel.moveDefault();
			} else if (command.equals("move_sdf")) {
				this.panel.zoomSdf(this.x, this.y);
			}
		} else if (command.equals("version")) {
			String version = null;
			URL url = null;
			try {
				version = Version.getVersion();
				if (version != null) {
					url = Version.getURL("ma38su", "Digital Map", version);
				}
			} catch (IOException ex) {
				Log.err(this, ex);
			}
			DialogFactory.versionDialog(version, url, this.panel, this);
		} else if (command.equals("check")) {
			Object source = e.getSource();
			boolean state = ((AbstractButton) source).isSelected();
			this.setting.setCheckUpdate(state);
			this.updateCheckItem.setSelected(state);
		}
	}

	public Setting getSetting() {
		return this.setting;
	}

	public void keyPressed(KeyEvent e) {
		this.panel.setOperation(true);
		switch (e.getKeyCode()) {
		case KeyEvent.VK_RIGHT:
		case KeyEvent.VK_KP_RIGHT:
			this.panel.moveLocation(-Controller.MOVE_SENSE, 0);
			break;
		case KeyEvent.VK_LEFT:
		case KeyEvent.VK_KP_LEFT:
			this.panel.moveLocation(Controller.MOVE_SENSE, 0);
			break;
		case KeyEvent.VK_UP:
		case KeyEvent.VK_KP_UP:
			this.panel.moveLocation(0, Controller.MOVE_SENSE);
			break;
		case KeyEvent.VK_DOWN:
		case KeyEvent.VK_KP_DOWN:
			this.panel.moveLocation(0, -Controller.MOVE_SENSE);
			break;
		case KeyEvent.VK_PLUS:
		case KeyEvent.VK_SEMICOLON:
		case KeyEvent.VK_PAGE_UP:
			this.panel.zoom(this.x, this.y, 1);
			break;
		case KeyEvent.VK_MINUS:
		case KeyEvent.VK_PAGE_DOWN:
		case KeyEvent.VK_EQUALS:
			this.panel.zoom(this.x, this.y, -1);
			break;
		}
	}

	public void keyReleased(KeyEvent e) {
		this.panel.setOperation(false);
	}

	public void keyTyped(KeyEvent e) {
	}

	public void mouseClicked(MouseEvent e) {
		if (e.getButton() == MouseEvent.BUTTON3) {
			this.panel.searchBoundary(e.getX(), e.getY(),
					(e.getModifiersEx() & InputEvent.SHIFT_DOWN_MASK) != 0);
		}
	}

	public void mouseDragged(MouseEvent e) {
		// ポインター座標変化量を計算
		int dx = e.getX() - this.x;
		int dy = e.getY() - this.y;
		// 変化時のポインター座標
		this.x = e.getX();
		this.y = e.getY();
		// 平行移動
		this.panel.moveLocation(dx, dy);
		this.panel.setOperation(true);
	}

	public void mouseEntered(MouseEvent e) {
		this.isMouseEntered = true;
	}

	public void mouseExited(MouseEvent e) {
		this.isMouseEntered = false;
		if (!this.panel.isOperation()) {
			this.x = this.panel.getWidth() / 2;
			this.y = this.panel.getHeight() / 2;
		}
	}

	public void mouseMoved(MouseEvent e) {
		if (this.isMouseEntered) {
			this.x = e.getX();
			this.y = e.getY();
		}
	}

	public void mousePressed(MouseEvent e) {
		this.panel.setOperation(true);
	}

	public void mouseReleased(MouseEvent e) {
		if (!this.isMouseEntered
				&& (e.getModifiersEx() & (InputEvent.BUTTON1_DOWN_MASK
						| InputEvent.BUTTON2_DOWN_MASK | InputEvent.BUTTON3_DOWN_MASK)) == 0) {
			this.x = this.panel.getWidth() / 2;
			this.y = this.panel.getHeight() / 2;
		}
		this.panel.setOperation(false);
	}

	public void mouseWheelMoved(MouseWheelEvent e) {
		int d = e.getWheelRotation();
		this.panel.zoom(e.getX(), e.getY(), d);
	}

	/**
	 * 起動時に更新を確認するためのメニュー
	 * @param item
	 */
	public void setUpdateCheck(JCheckBoxMenuItem item) {
		this.updateCheckItem = item;
	}

	public float getLocationMouseX() {
		return (float) this.panel.getLocationX(this.x);
	}

	public float getLocationMouseY() {
		return (float) this.panel.getLocationY(this.y);
	}
}
