package view;

import java.io.File;

import javax.swing.JLabel;

import storage.CityNameDatabase;
import util.Log;

import map.model.City;
import map.route.SearchThread;

/**
 * JLabelを継承したステータスバー
 * 
 * @author ma38su
 *
 */
public class StatusBar extends JLabel implements Runnable {

	/**
	 * 表示範囲内の地図データであるか確認している市区町村番号
	 */
	private String checkCode;

	/**
	 * インターネットから取得している市区町村番号
	 */
	private String downloadCode;

	/**
	 * インターネットから取得しているファイル
	 */
	private File file;

	/**
	 * インターネットから取得するファイルのサイズ
	 */
	private float fileSize;

	/**
	 * ファイル入出力エラーの有無
	 */
	private boolean ioException;

	/**
	 * 使用メモリ量（MB）
	 */
	private float memory;

	/**
	 * メモリ容量オーバーのフラグ
	 */
	private boolean outOfMemory;

	/**
	 * 補助記憶装置（ハードディスク）を読み込んでいる市区町村番号
	 */
	private String readFile;

	/**
	 * 経路探索を行うスレッド
	 */
	private SearchThread search;

	/**
	 * ステータスバーの更新のためのスレッド
	 */
	private Thread thread;

	private CityNameDatabase db;

	/**
	 * コンストラクタ
	 */
	public StatusBar(String msg) {
		super(msg);
	}

	/**
	 * コンストラクタ
	 * @param msg 初期メッセージ
	 */
	public StatusBar(String msg, CityNameDatabase db) {
		super(msg);
		this.db = db;
		this.thread = new Thread(this);
		this.thread.start();
		this.fileSize = 0;
	}

	/**
	 * 表示範囲内かどうかのチェックが終了したことを設定します。
	 *
	 */
	public void clearCheckCode() {
		this.checkCode = null;
	}

	public void clearDLFile() {
		this.downloadCode = null;
		this.file = null;
	}

	/**
	 * 読み込みが終了したことを設定します。
	 *
	 */
	public void clearReading() {
		this.readFile = null;
		Log.out(this, "cleared reading.");
	}

	/**
	 * ファイル入出力エラーを表示
	 */
	public void ioException() {
		this.ioException = true;
	}
	private void loop() {
		boolean flag = false;
		StringBuilder sb = new StringBuilder();
		sb.append("MEMORY : " + this.memory + "MB");
		if (this.outOfMemory) {
			sb.append(" OUT OF MEMORY");
		}
		if (this.search != null) {
			sb.append(this.search);
			flag = true;
		}
		if (this.readFile != null) {
			flag = true;
			sb.append(" / READ " + this.readFile);
		} else if (this.checkCode != null) {
			flag = true;
			sb.append(" / CHECK " + this.checkCode);
		}
		if (this.downloadCode != null) {
			sb.append(" / DOWNLOAD " + this.downloadCode);
			if (this.ioException) {
				sb.append(" : I/O ERROR");
				this.file = null;
			} else if (this.fileSize > 0) {
				sb.append(" : ");
				if (this.file.length() < this.fileSize) {
					sb.append((int) (this.file.length() * 100 / this.fileSize + 0.5f));
				} else {
					sb.append(100);
					this.file = null;
					this.fileSize = 0;
				}
				sb.append('%');
			}
			flag = true;
		}
		if (sb.length() == 0) {
			super.setText(" ");
		} else {
			super.setText(sb.toString());
		}
		try {
			if (!flag || this.outOfMemory || this.ioException) {
				Thread.sleep(2000L);
				this.outOfMemory = false;
				this.ioException = false;
			} else {
				Thread.sleep(250L);
			}
		} catch (InterruptedException e) {
			Log.err(this, e);
		}
	}
	/**
	 * OutOfMemoryを表示
	 */
	public void memoryError() {
		this.outOfMemory = true;
	}
	
	public void run() {
		while (true) {
			this.memory = (float) (int) ((Runtime.getRuntime().totalMemory() - Runtime
					.getRuntime().freeMemory()) / 1024) / 1024;
			this.loop();
		}
	}

	/**
	 * 表示範囲内であるかどうかチェックしている市区町村番号を設定します。
	 * @param code 市区町村番号
	 */
	public void setCheckCode(String code) {
		this.checkCode = code;
	}

	/**
	 * インターネット上から取得しているファイルを設定します。
	 * 
	 * @param code 市区町村番号
	 */
	public void setDLFile(int code) {
		this.downloadCode = this.db.getName(code);
		if (this.downloadCode == null) {
			this.downloadCode = City.cityFormat(code);
		}
	}

	/**
	 * インターネット上から取得しているファイルを設定します。
	 * 
	 * @param code 市区町村番号
	 * @param length ファイル長
	 * @param file ダウンロード中のファイル
	 */
	public void setDLFile(int code, int length, File file) {
		this.downloadCode = this.db.getName(code);
		if (this.downloadCode == null) {
			this.downloadCode = City.cityFormat(code);
		}
		this.fileSize = length;
		this.file = file;
	}

	public void setRdFile(String file) {
		this.readFile = file;
	}
	/**
	 * ローカルの補助記憶装置（ハードディスク）からの読み込んでいる市区町村番号を設定します。
	 * @param code 市区町村番号
	 */
	public void setRdFile(int code) {
		this.readFile = this.db.getName(code);
		if (this.readFile == null) {
			this.readFile = City.cityFormat(code);
		}
		Log.out(this, "this.readCode = " + code);
	}

	/**
	 * 経路探索のためのスレッドを設定します。
	 * @param thread 経路探索のスレッド
	 */
	public void setThread(SearchThread thread) {
		this.search = thread;
	}

	/**
	 * ステータスバーの更新スレッドの優先度を設定します。
	 * @param priority 優先度（0から5）
	 */
	public void setThreadPriority(int priority) {
		this.thread.setPriority(priority);
	}
}
