package map;

import java.awt.Point;
import java.awt.Rectangle;
import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import labeling.Label;
import map.model.City;
import map.model.Curve;
import map.model.ExtractNode;
import map.model.Facility;
import map.model.Mesh;
import map.model.Node;
import map.model.Road;
import map.model.Station;
import util.Common;
import util.Log;
import database.SdfDatabase;

/**
 * 数値地図25000のデータファクトリー
 * @author ma38su
 */
public class Sdf25kReader extends Thread {

	public static String CODE = "SJIS";

	private final int DEFAULT_BORDER_CAPACITY = 20;

	private final int DEFAULT_MESH_CAPACITY = 20000;

	private final int DEFAULT_NODE_CAPACITY = 1000;

	private final int DEFAULT_SLP_CAPACITY = 20000;

	private final String KEY_BORDER = "GK.sal";

	private final String KEY_COAST = "SK.sal";
	
	private final String KEY_MESH = "mesh";

	private final String KEY_NODE = "DS.sal";

	private final String KEY_RAIL = "rail";

	private final String KEY_RIVER = "KK.sal";

	private final String KEY_SLP = "slp";

	private final String KEY_STATION = "station";

	/**
	 * ファイル読み込みのための正規表現
	 */
	private final Pattern SAL_PARSER = Pattern.compile("([A-Z]{2})" + "(?:"
			+ "(?:" + "\\(" + "([^)]*)" + "\\)" + ")" + "|" + "\\{"
			+ "([^\\}\\{]*)" + "\\}" + ")" + "(?:" + "\\{" + "(?:" + "("
			+ "(?:" + "\\d{6},)*" + "\\d{6}" + ")" + "|" + "(" + "(?:"
			+ "[A-Z]{2}" + "\\(" + "[^\\)]*" + "\\)" + ")*" + ")" + ")" + "\\}"
			+ ")?");

	/**
	 * 必要なファイルを取得するためのクラス
	 */
	private final SdfDatabase storage;

	/**
	 * コンストラクタ
	 * 
	 * @param storage
	 *            数値地図25000のデータ取得管理クラス
	 * @throws IOException 
	 */
	public Sdf25kReader(SdfDatabase storage) {
		this.storage = storage;
	}

	private Curve[] getBorder(int code, Map<String, File> fileMap, Point[] slp, String key) throws IOException {
		Curve[] curve = null;
		if (this.storage.hasSerializable(code, key)) {
			curve = (Curve[]) this.storage.readSerializable(code, key);
		}
		if (curve == null) {
			curve = this.readBorder(fileMap.get(key), slp);
			this.storage.writeSerializable(code, key, curve);
		}
		return curve;
	}

	/**
	 * 水域を読み込みます。
	 * @param code 市区町村番号
	 * @param fileMap
	 * @param slp
	 * @return 水域
	 * @throws IOException
	 */
	@SuppressWarnings("unchecked")
	private Curve[][] getCoast(int code, Map<String, File> fileMap, Point[] slp) throws IOException {
		Curve[][] coast = null;
		if (this.storage.hasSerializable(code, this.KEY_COAST)) {
			try {
				coast = (Curve[][]) this.storage.readSerializable(code, this.KEY_COAST);
			} catch (Throwable e) {
				coast = null;
			}
		} else {
			coast = new Curve[2][];
		}
		if (coast[0] == null || coast[1] == null) {
			coast = this.readCoast(fileMap.get(this.KEY_COAST), slp);
			this.storage.writeSerializable(code, this.KEY_COAST, coast);
		}
		return coast;
	}

	private Facility[] getFacility(int code, Map<String, File> fileMap, Point[] slp, String key) throws IOException {
		Facility[] facility = null;
		if (this.storage.hasSerializable(code, key)) {
			facility = (Facility[]) this.storage.readSerializable(code, key);
		}
		if (facility == null) {
			facility = this.readFacility(fileMap.get(key), slp);
			this.storage.writeSerializable(code, key, facility);
		}
		return facility;
	}

	/**
	 * 市区町村番号からファイルを取得
	 * @param code 市区町村番号
	 * @param fileMap
	 * @return slmファイル
	 * @throws IOException 入出力エラー
	 */
	private File getFile(int code, Map<String, File> fileMap) throws IOException {
		File slm = null;
		for (File file : this.storage.getSdf25k(code)) {
			Log.out(this, "get "+  file.getCanonicalPath());
			String path = file.getPath();
			if (path.endsWith("slm")) {
				slm = file;
				if (fileMap == null) {
					break;
				}
			} else if (fileMap != null) {
				if (path.endsWith(".sal")) {
					int length = path.length();
					String label = path.substring(length - 6);
					fileMap.put(label, file);
				} else if (path.endsWith(".slp")) {
					if (path.endsWith("MH.slp")) {
						// メッシュ標高座標
						fileMap.put("MH.slp", file);
					} else {
						// 座標
						fileMap.put(".slp", file);
					}
				}
			}
		}
		return slm;
	}

	public Mesh[] getMesh(int code, Rectangle area, Map<String, File> fileMap) throws IOException {
		Mesh[] mesh = null;
		if (this.storage.hasSerializable(code, this.KEY_MESH)) {
			try {
				mesh = (Mesh[]) this.storage.readSerializable(code, "mesh");
			} catch (Throwable e) {
				mesh = null;
			}
		}
		if (mesh == null) {
			mesh = this.readMesh(fileMap.get("MH.slp"), fileMap.get("MH.sal"), area).toArray(new Mesh[] {});
			this.storage.writeSerializable(code, "mesh", mesh);
		}
		return mesh;
	}

	@SuppressWarnings("unchecked")
	public Map<Integer, Node> getNode(int code, Map<String, File> fileMap, Point[] slp) throws IOException {
		Map<Integer, Node> nodes = null;
		if (this.storage.hasSerializable(code, this.KEY_NODE)) {
			try {
				nodes = (Map<Integer, Node>) this.storage.readSerializable(code, this.KEY_NODE);
			} catch (Throwable e) {
				nodes = null;
			}
		}
		if (nodes == null) {
			nodes = this.readNode(code, fileMap.get(this.KEY_NODE), slp);
			this.storage.writeSerializable(code, this.KEY_NODE, nodes);
		}
		return nodes;
	}

	public Point[] getSLP(int code, Rectangle area, Map<String, File> map) throws IOException {
		Point[] slp = null;
		if (this.storage.hasSerializable(code, this.KEY_SLP)) {
			try {
				slp = (Point[]) this.storage.readSerializable(code, this.KEY_SLP);
			} catch (Throwable e) {
				slp = null;
			}
		}
		if (slp == null) {
			slp = this.readSLP(map.get(".slp"), area);
			this.storage.writeSerializable(code, this.KEY_SLP, slp);
		}
		return slp;
	}
	
	/**
	 * グラフをつなぎあわせて頂点数を減らします。
	 * 次数2の頂点は削除して、次数1と、次数3の頂点のみ残します。
	 * @param nodes つなぎあわせる頂点
	 * @param set
	 */
	private void jointNode(Collection<ExtractNode> nodes, Collection<Curve> set) {
		Set<Curve> borderSet = new HashSet<Curve>();
		for (ExtractNode node : nodes) {
			if (node.getBorder().size() == 2) {
				for (int i = 0; i < 4; i++) {
					Curve border = node.connect(i);
					if (border != null) {
						set.add(border);
					}
				}
			}
		}
		for (ExtractNode node : nodes) {
			for (Curve border : node.getBorder()) {
				borderSet.add(border);
			}
		}
		for (Curve border : borderSet) {
			set.add(border);
		}
	}

	/**
	 * 多角形面の向き
	 * 
	 * @param x
	 *            多角形面を構成するx座標の配列
	 * @param y
	 *            多角形面を構成するy座標の配列
	 * @return 多角形から計算した外積のZ座標の値
	 */
	public int polygonDirection(int[] x, int[] y) {
		int vector = 0;
		for (int i = 1; i < x.length - 1; i++) {
			int dx0 = x[i] - x[0];
			int dy0 = y[i] - y[0];

			int dx1 = x[i + 1] - x[0];
			int dy1 = y[i + 1] - y[0];

			vector += dx1 * dy0 - dy1 * dx0;
		}
		return (vector >= 0) ? 1 : -1;
	}

	/**
	 * 市区町村の範囲を取得します。
	 * 
	 * @param city
	 *            市区町村データ
	 * @throws IOException
	 *             入出力エラー
	 */
	public void preproduct(City city) throws IOException {
		Rectangle area = this.readSLM(city.getCode(), null);
		city.setArea(area);
	}

	/**
	 * 市区町村データを取得
	 * 
	 * @param code
	 *            市区町村番号
	 * @return 市区町村データ
	 * @throws IOException
	 *             入出力エラー
	 */
	public City preproduct(int code) throws IOException {
		Rectangle area = this.readSLM(code, null);
		return new City(code, area);
	}
	
	public void product(City city) throws IOException {
		Map<String, File> fileMap = new HashMap<String, File>();
		this.getFile(city.getCode(), fileMap);
		this.product(city, fileMap);
	}

	/**
	 * 市区町村データを生成
	 * 
	 * @param city 市区町村データ
	 * @param fileMap 
	 * @throws IOException 入出力エラー
	 */
	public void product(City city, Map<String, File> fileMap) throws IOException {
		int code = city.getCode();
		if (!city.hasSdk25Bounds()) {
			this.preproduct(city);
		}
		Mesh[] mesh = this.getMesh(code, city.getArea(), fileMap);

		// ファイルの読み込み
		Point[] slp = this.getSLP(code, city.getArea(), fileMap);

		Map<String, Facility[]> label = new HashMap<String, Facility[]>();
		label.put(Label.KEY_CM, this.getFacility(code, fileMap, slp, "CM.sal"));
		label.put(Label.KEY_KO, this.getFacility(code, fileMap, slp, "KO.sal"));

		// 道路接点の読み込み
		Map<Integer, Node> nodes = this.getNode(code, fileMap, slp);

		// 道路区間の読み込み
		Road[][] road = this.readRoad(code, fileMap.get("DK.sal"), slp, nodes, true);

		Curve[][] rail = null;
		Station[] station = null;
		if (this.storage.hasSerializable(code, this.KEY_RAIL)
				&& this.storage.hasSerializable(city.getCode(),
						this.KEY_STATION)) {
			try {
				rail = (Curve[][]) this.storage.readSerializable(code,
						this.KEY_RAIL);
				station = (Station[]) this.storage.readSerializable(code,
						this.KEY_STATION);
			} catch (Throwable e) {
				rail = null;
				station = null;
			}
		}
		if (rail == null || station == null) {
			Map<Integer, Curve> railMap = new HashMap<Integer, Curve>();
			// 鉄道区間の読み込み
			rail = this.readRail(fileMap.get("TK.sal"), slp, railMap);
			// 駅区間の読み込み
			station = this.readStation(fileMap.get("EK.sal"), railMap);
			this.storage.writeSerializable(code, this.KEY_RAIL, rail);
			this.storage.writeSerializable(code, this.KEY_STATION, station);
		}

		// 河川区間の読み込み
		Curve[] river = this.getBorder(code, fileMap, slp, this.KEY_RIVER);;

		// 水域界の読み込み
		Curve[][] coast = this.getCoast(code, fileMap, slp);

		// 行政界の読み込み
		Curve[] border = this.getBorder(code, fileMap, slp, this.KEY_BORDER);

		city.setData(border, coast, river, rail, station, nodes, road, label, mesh);
	}

	/**
	 * 市区町村データを生成
	 * 
	 * @param code
	 *            市区町村番号
	 * @return 市区町村データ
	 * @throws IOException
	 */
	public City product(int code) throws IOException {
		Map<String, File> fileMap = new HashMap<String, File>();
		Rectangle area = this.readSLM(code, fileMap);
		City city = new City(code, area);
		this.product(city, fileMap);
		return city;
	}

	/**
	 * 市区町村データに頂点データを付加します。
	 * 
	 * @param city
	 *            市区町村データ
	 * @param isDetailRoad
	 *            詳細な道路データを作成する場合はtrue
	 * @throws IOException
	 *             入出力エラー
	 */
	public void productNode(City city, boolean isDetailRoad) throws IOException {

		Map<String, File> fileMap = new HashMap<String, File>();

		int code = city.getCode();

		if (city.hasData()) {
			this.getFile(code, fileMap);
		} else {
			city.setArea(this.readSLM(code, fileMap));
		}

		// ファイルの読み込み
		Point[] slp = this.getSLP(code, city.getArea(), fileMap);

		// 道路接点の読み込み
		Map<Integer, Node> nodes = this.getNode(code, fileMap, slp);

		this.readRoad(code, fileMap.get("DK.sal"), slp, nodes, isDetailRoad);
		city.setData(nodes);
	}

	/**
	 * 頂点データを生成します。
	 * 
	 * @param code
	 *            市区町村番号
	 * @param isDetailRoad
	 *            詳細な道路データを作成する場合はtrue
	 * @return 頂点データ
	 * @throws IOException
	 *             入出力エラー
	 */
	public City productNode(int code, boolean isDetailRoad) throws IOException {
		Map<String, File> fileMap = new HashMap<String, File>();
		Rectangle area = this.readSLM(code, fileMap);
		City map = new City(code, area);

		// ファイルの読み込み
		Point[] slp = this.getSLP(code, area, fileMap);

		// 道路接点の読み込み
		Map<Integer, Node> nodes = this.getNode(code, fileMap, slp);
		this.readRoad(code, fileMap.get("DK.sal"), slp, nodes, isDetailRoad);

		map.setData(nodes);
		return map;
	}

	/**
	 * 境界の読み込み
	 * 
	 * @param file
	 * @param slp
	 * @return 境界の配列
	 * @throws IOException
	 */
	private Curve[] readBorder(File file, Point[] slp) throws IOException {
		List<Curve> border = new ArrayList<Curve>(this.DEFAULT_BORDER_CAPACITY);
		BufferedReader bi = null;
		try {
			bi = new BufferedReader(new InputStreamReader(new FileInputStream(
					file), Sdf25kReader.CODE));
			while (bi.ready()) {
				String line = bi.readLine();
				Matcher matcher = this.SAL_PARSER.matcher(line.substring(17));
				int[] curveX = null;
				int[] curveY = null;
				while (matcher.find()) {
					String id = matcher.group(1);
					if (id.equals("CV")) {
						// カーブメモリ
						String[] ref = Common.PATTERN_CSV.split(matcher.group(4));
						curveX = new int[ref.length];
						curveY = new int[ref.length];
						for (int i = 0; i < ref.length; i++) {
							Point point = slp[Integer.parseInt(ref[i]) - 1];
							curveX[i] = point.x;
							curveY[i] = point.y;
						}
					}
				}
				Curve b = new Curve(curveX, curveY, 0);
				border.add(b);
			}
		} finally {
			if (bi != null) {
				bi.close();
			}
		}
		return border.toArray(new Curve[] {});
	}

	/**
	 * 水域界の読み込み
	 * @param seaborder 海岸線
	 * @param file 水域界のファイル
	 * @param slp
	 * @return 水域界データ
	 * @throws IOException
	 */
	private Curve[][] readCoast(File file, Point[] slp) throws IOException {
		List<Curve> coast = new ArrayList<Curve>();
		List<Curve> seaborder = new ArrayList<Curve>();
		BufferedReader bi = null;
		try {
			bi = new BufferedReader(new InputStreamReader(new FileInputStream(
					file), Sdf25kReader.CODE));
			while (bi.ready()) {
				String line = bi.readLine();
				Matcher matcher = this.SAL_PARSER.matcher(line.substring(17));
				int type = 0;
				int[] curveX = null;
				int[] curveY = null;
				while (matcher.find()) {
					String id = matcher.group(1);
					if (id.equals("CV")) {
						// カーブメモリ
						String[] ref = Common.PATTERN_CSV.split(matcher.group(4));
						curveX = new int[ref.length];
						curveY = new int[ref.length];
						for (int i = 0; i < ref.length; i++) {
							Point point = slp[Integer.parseInt(ref[i]) - 1];
							curveX[i] = point.x;
							curveY[i] = point.y;
						}
					} else if (id.equals("SR")) {
						/**
						 * 51 水涯線または湖岸線 52 海岸線 53 河口 54 湖沼と河川の境界
						 */
						type = Integer.parseInt(matcher.group(3)) - 51;
						if (type != 1) {
							type = 0;
						}
						if (type == 3) {
							Log.out(this, "SR 54 ? " + file);
						}
					}
				}
				if (type != 1) {
					coast.add(new Curve(curveX, curveY, type));
				} else {
					seaborder.add(new Curve(curveX, curveY, type));
				}
			}
		} finally {
			if (bi != null) {
				bi.close();
			}
		}
		return new Curve[][]{coast.toArray(new Curve[]{}), seaborder.toArray(new Curve[]{})};
	}

	/**
	 * 施設情報の読み込み
	 * 
	 * @param file
	 * @param slp
	 * @throws IOException
	 * @return 施設情報配列
	 */
	private Facility[] readFacility(File file, Point[] slp) throws IOException {
		List<Facility> facility = new ArrayList<Facility>();
		BufferedReader bi = null;
		try {
			bi = new BufferedReader(new InputStreamReader(new FileInputStream(
					file), Sdf25kReader.CODE));
			while (bi.ready()) {
				String line = bi.readLine();
				Matcher matcher = this.SAL_PARSER.matcher(line.substring(17));
				Point point = null;
				boolean isNM = false;
				String name = null;
				while (matcher.find()) {
					String param = matcher.group(1);
					if (param.equals("NM")) {
						name = matcher.group(3);
						isNM = true;
					} else if (param.equals("PT")) {
						point = slp[Integer.parseInt(matcher.group(4)) - 1];
					}
					if (isNM && point != null) {
						facility.add(new Facility(name, point.x, point.y));
					}
				}
			}
		} finally {
			if (bi != null) {
				bi.close();
			}
		}
		return facility.toArray(new Facility[] {});
	}

	/**
	 * 
	 * メッシュ標高の読み込み
	 * 
	 * @param slp
	 * @param sal
	 * @param area
	 * @return メッシュ標高のリスト
	 * @throws IOException
	 * 
	 */
	private List<Mesh> readMesh(File slp, File sal, Rectangle area)
			throws IOException {
		List<Point> points = new ArrayList<Point>(this.DEFAULT_MESH_CAPACITY);
		BufferedReader bi = null;
		try {
			bi = new BufferedReader(new FileReader(slp));
			while (bi.ready()) {
				String line = bi.readLine();
				int x = Integer.parseInt(line.substring(0, 7)) + area.x;
				int y = Integer.parseInt(line.substring(8, 15)) + area.y;
				points.add(new Point(x, y));
			}
		} finally {
			if (bi != null) {
				bi.close();
			}
		}
		List<Mesh> mesh = new ArrayList<Mesh>(points.size());
		try {
			bi = new BufferedReader(new InputStreamReader(new FileInputStream(
					sal), Sdf25kReader.CODE));
			while (bi.ready()) {
				String line = bi.readLine();
				// String category = line.substring(0, 2);
				Matcher matcher = this.SAL_PARSER.matcher(line.substring(17));
				Point point = null;
				int height = 0;
				while (matcher.find()) {
					String param = matcher.group(1);
					if (param.equals("PT")) {
						point = points
								.get(Integer.parseInt(matcher.group(4)) - 1);
					} else if (param.equals("HK")) {
						height = Integer.parseInt(matcher.group(3));
					}
				}
				mesh.add(new Mesh(point.x, point.y, height));
			}
		} finally {
			if (bi != null) {
				bi.close();
			}
		}
		return mesh;
	}

	/**
	 * 接点の読み込み
	 * 
	 * @param code
	 *            市区町村番号
	 * @param file
	 * @param slp
	 * @return 頂点番号をkey、頂点をvalueとしたMap
	 * @throws IOException
	 *             入出力エラー
	 */
	private Map<Integer, Node> readNode(int code, File file, Point[] slp)
			throws IOException {
		Map<Integer, Node> node = new HashMap<Integer, Node>(
				this.DEFAULT_NODE_CAPACITY);
		BufferedReader bi = null;
		try {
			bi = new BufferedReader(new InputStreamReader(new FileInputStream(
					file), Sdf25kReader.CODE));
			while (bi.ready()) {
				String line = bi.readLine();
				Matcher matcher = this.SAL_PARSER.matcher(line.substring(17));
				String id = null;
				Point point = null;
				int slpindex = -1;
				while (matcher.find()) {
					String param = matcher.group(1);
					if (param.equals("PT")) {
						slpindex = Integer.parseInt(matcher.group(4)) - 1;
						point = slp[slpindex];
					} else if (param.equals("ND")) {
						id = matcher.group(2).substring(5, 11);
					}
				}
				if (point != null && id != null) {
					if (code == 40101) {
						if (id.equals("000985")) {
							point.setLocation(471453115, 122261236);
						} else if (id.equals("001020")) {
							point.setLocation(471457816, 122265788);
						}
					} else if (code == 38202 && id.equals("001867")) {
						point.setLocation(478795248, 122834185);
					}
					node.put(Integer.parseInt(id), new Node(Long.parseLong(code
							+ id), point));
				}
			}
		} finally {
			if (bi != null) {
				bi.close();
			}
		}
		return node;
	}

	/**
	 * 水域界の読み込み
	 * 
	 * @param file
	 *            水域界のファイル
	 * @param slp
	 * @param map
	 * @return 水域界データ
	 * @throws IOException
	 */
	private Curve[][] readRail(File file, Point[] slp, Map<Integer, Curve> map)
			throws IOException {
		Collection<Curve> jr = new HashSet<Curve>();
		Collection<Curve> other = new HashSet<Curve>();
		Map<Point, ExtractNode> points = new HashMap<Point, ExtractNode>();
		BufferedReader bi = null;
		try {
			bi = new BufferedReader(new InputStreamReader(new FileInputStream(
					file), Sdf25kReader.CODE));
			while (bi.ready()) {
				String line = bi.readLine();
				Matcher matcher = this.SAL_PARSER.matcher(line.substring(17));
				int type = 0;
				int[] curveX = null;
				int[] curveY = null;
				while (matcher.find()) {
					String id = matcher.group(1);
					if (id.equals("CV")) {
						// カーブメモリ
						String[] ref = Common.PATTERN_CSV.split(matcher.group(4));
						curveX = new int[ref.length];
						curveY = new int[ref.length];
						for (int i = 0; i < ref.length; i++) {
							Point point = slp[Integer.parseInt(ref[i]) - 1];
							curveX[i] = point.x;
							curveY[i] = point.y;
						}
					} else if (id.equals("SB")) {
						type = Integer.parseInt(matcher.group(3)) - 43;
					}
				}
				Curve border = new Curve(curveX, curveY, type);
				map.put(Integer.valueOf(line.substring(8, 14)), border);
				ExtractNode p1 = new ExtractNode(curveX[0], curveY[0]);
				ExtractNode p2 = new ExtractNode(curveX[curveX.length - 1],
						curveY[curveX.length - 1]);
				if (border.getType() != 0) {
					other.add(border);
				} else if (p1.equals(p2)) {
					jr.add(border);
				} else {
					if (points.containsKey(p1)) {
						p1 = points.get(p1);
					} else {
						points.put(p1, p1);
					}
					if (points.containsKey(p2)) {
						p2 = points.get(p2);
					} else {
						points.put(p2, p2);
					}
					Curve b = p1.put(p2, border);
					if (b != null) {
						jr.add(b);
					}
				}
			}
			this.jointNode(points.values(), jr);
		} finally {
			if (bi != null) {
				bi.close();
			}
		}
		return new Curve[][] { jr.toArray(new Curve[] {}),
				other.toArray(new Curve[] {}) };
	}

	/**
	 * 道路の読み込み
	 * 
	 * @param code
	 * @param file
	 *            道路情報のファイル
	 * @param slp
	 *            slpファイルの読み込んだもの
	 * @param node
	 * @param isDetail
	 * @return 道路ファイル
	 * @throws IOException
	 */
	private Road[][] readRoad(int code, File file, Point[] slp, Map<Integer, Node> node, boolean isDetail) throws IOException {
		List<List<Curve>> border = new ArrayList<List<Curve>>(6);
		List<List<Curve>> highway = new ArrayList<List<Curve>>(6);
		if (isDetail) {
			for (int i = 0; i < 6; i++) {
				border.add(new ArrayList<Curve>());
				highway.add(new ArrayList<Curve>());
			}
		}
		BufferedReader bi = null;
		try {
			bi = new BufferedReader(new InputStreamReader(new FileInputStream(file), Sdf25kReader.CODE));
			while (bi.ready()) {
				String line = bi.readLine();
				Matcher matcher = this.SAL_PARSER.matcher(line.substring(17));
				int[] curveX = null;
				int[] curveY = null;
				int type = 0;
				int width = 0;
				int n0 = -1;
				int n1 = -1;
				String[] ref = null;
				double cost = 0;
				while (matcher.find()) {
					String id = matcher.group(1);
					if (id.equals("CV")) {
						// カーブメモリ
						ref = Common.PATTERN_CSV.split(matcher.group(4));
						curveX = new int[ref.length];
						curveY = new int[ref.length];
						Point p0 = null;
						for (int i = 0; i < ref.length; i++) {
							Point point = slp[Integer.parseInt(ref[i]) - 1];
							curveX[i] = point.x;
							curveY[i] = point.y;
							if (i == 0) {
								p0 = point;
							} else {
								long dx = p0.x - point.x;
								long dy = p0.y - point.y;
								cost += Math.sqrt(dx * dx + dy * dy);
								p0 = point;
							}
						}
					} else if (id.equals("EG")) {
						String edge = matcher.group(5);
						n0 = Integer.parseInt(edge.substring(13, 19));
						n1 = Integer.parseInt(edge.substring(34, 40));
					} else if (id.equals("SB")) {
						type = Integer.parseInt(matcher.group(3)) - 13;
						if (type < 0) {
							type = 0;
						}
					} else if (id.equals("FI")) {
						// 17から16進数で与えられる
						width = Integer.parseInt(matcher.group(3), 16) - 23;
						if (width == 7) {
							width = 5;
						}
					}
				}
				Road b;
				if (isDetail) {
					b = new Road(curveX, curveY, type, width, (float) cost);
				} else {
					b = new Road(type, width, (float) cost);
				}
				Node node0 = node.get(n0);
				Node node1 = node.get(n1);
				node0.connect(code * 1000000L + n1, b);
				node1.connect(code * 1000000L + n0, b);
				if (isDetail) {
					if (type == 3) {
						highway.get(width).add(b);
					} else {
						border.get(width).add(b);
					}
				}
			}
		} finally {
			if (bi != null) {
				bi.close();
			}
		}
		InputStream stream = this.storage.getBoundaryNode(code);
		if (stream != null) {
			DataInputStream disc = null;
			try {
				disc = new DataInputStream(new BufferedInputStream(stream));
				while (disc.available() >= 12) {
					node.get(disc.readInt()).connect(disc.readLong(), null);
				}
			} finally {
				if (disc != null) {
					disc.close();
				}
			}
		}
		Road[][] road = new Road[12][];
		if (isDetail) {
			for (int i = 0; i < 6; i++) {
				road[i] = border.get(i).toArray(new Road[] {});
			}
			for (int i = 0; i < 6; i++) {
				road[i + 6] = highway.get(i).toArray(new Road[] {});
			}
			return road;
		} else {
			return null;
		}
	}

	/**
	 * 領域などの読み込み
	 * 
	 * @param code
	 *            市区町村番号
	 * @param map
	 * @return 市区町村の領域
	 * @throws IOException
	 */
	private Rectangle readSLM(int code, Map<String, File> map) throws IOException {
		Rectangle rect = null;
		BufferedReader bi = null;
		Log.out(this, "read slm "+ code);
		File file = this.getFile(code, map);
		try {
			bi = new BufferedReader(new FileReader(file));
			// 原点読み込み
			StringTokenizer st = new StringTokenizer(bi.readLine(), ",");
			int x = (int) (Long.parseLong(st.nextToken()) / 10);
			int y = (int) (Long.parseLong(st.nextToken()) / 10);

			// 領域読み込み
			st = new StringTokenizer(bi.readLine(), ",");
			int width = Integer.parseInt(st.nextToken()) / 10;
			int height = Integer.parseInt(st.nextToken()) / 10;
			rect = new Rectangle(x, y, width, height);
		} finally {
			if (bi != null) {
				bi.close();
			}
		}
		return rect;
	}

	/**
	 * SLPファイルの読み込み
	 * 
	 * @param file
	 * @param area
	 * @return Point配列
	 * @throws IOException
	 * @throws NumberFormatException
	 */
	private Point[] readSLP(File file, Rectangle area) throws NumberFormatException, IOException {
		List<Point> slp = new ArrayList<Point>(this.DEFAULT_SLP_CAPACITY);
		BufferedReader bi = null;
		try {
			bi = new BufferedReader(new FileReader(file));
			while (bi.ready()) {
				String line = bi.readLine();
				int x = Integer.parseInt(line.substring(0, 7)) + area.x;
				int y = Integer.parseInt(line.substring(8, 15)) + area.y;
				Point p = new Point(x, y);
				slp.add(p);
			}
		} finally {
			if (bi != null) {
				bi.close();
			}
		}
		return slp.toArray(new Point[] {});
	}

	/**
	 * 駅区間の読み込み
	 * 
	 * @param file
	 * @param railway
	 * @return 駅区間の配列
	 * @throws IOException
	 */
	public Station[] readStation(File file, Map<Integer, Curve> railway) throws IOException {
		List<Station> station = new ArrayList<Station>();
		Map<String, List<Curve>> stationMap = new HashMap<String, List<Curve>>();
		BufferedReader bi = null;
		try {
			bi = new BufferedReader(new InputStreamReader(new FileInputStream(file), Sdf25kReader.CODE));
			while (bi.ready()) {
				String line = bi.readLine();
				Matcher matcher = this.SAL_PARSER.matcher(line.substring(17));
				String name = null;
				Curve curve = null;
				while (matcher.find()) {
					String id = matcher.group(1);
					if (id.equals("NM")) {
						// 駅名
						name = matcher.group(3);
					} else if (id.equals("KN")) {
						curve = railway.get(Integer.valueOf(matcher.group(2).substring(10, 16)));
					}
				}
				List<Curve> cache = stationMap.get(name);
				if (cache == null) {
					List<Curve> list = new ArrayList<Curve>();
					list.add(curve);
					stationMap.put(name, list);
				} else {
					boolean isConnect = false;
					for (Curve c : cache) {
						if(c.connect(curve)) {
							isConnect = true;
							break;
						}
					}
					if (!isConnect) {
						cache.add(curve);
					}
				}
			}
		} finally {
			if (bi != null) {
				bi.close();
			}
		}
		for (Map.Entry<String, List<Curve>> entry : stationMap.entrySet()) {
			boolean isConnect;
			do {
				isConnect = false;
				List<Curve> curves = entry.getValue();
				for (int i = 0; i < curves.size(); i++) {
					Curve curve = curves.get(i);
					ListIterator<Curve> itr = curves.listIterator(i + 1);
					while (itr.hasNext()) {
						if (curve.connect(itr.next())) {
							itr.remove();
							isConnect = true;
						}
					}
				}
			} while (isConnect);
			for (Curve curve : entry.getValue()) {
				station.add(new Station(entry.getKey(), curve));
			}
		}
		return station.toArray(new Station[] {});
	}
}
