package view;

import java.io.File;

import javax.swing.JLabel;

import map.model.City;
import map.route.SearchThread;
import util.Log;
import database.NameDatabase;

/**
 * JLabelを継承したステータスバー
 * 
 * @author ma38su
 *
 */
public class StatusBar extends JLabel implements Runnable {

	/**
	 * 数値地図2500の識別子
	 */
	public static final int SDF2500 = 1;

	/**
	 * 数値地図25000の識別子
	 */
	public static final int SDF25000 = 2;
	
	/**
	 * 数値地図25000の識別子
	 */
	public static final int ISJ = 3;

	/**
	 * 市区町村名データベース
	 */
	private NameDatabase db;

	/**
	 * インターネットから取得している市区町村情報
	 * 市区町村名または番号
	 */
	private String downloadCode;

	/**
	 * インターネットから取得しているファイル
	 */
	private File file;

	/**
	 * インターネットから取得するファイルのサイズ
	 */
	private float fileSize;

	/**
	 * 使用メモリ量（MB）
	 */
	private float memory;

	/**
	 * 補助記憶装置（ハードディスク）を読み込んでいる市区町村番号
	 */
	private String readCode;

	/**
	 * 経路探索を行うスレッド
	 */
	private SearchThread search;

	/**
	 * ステータスバーの更新のためのスレッド
	 */
	private Thread thread;

	
	/**
	 * 読み込んでいる地図の種類
	 */
	private int type;

	/**
	 * 読み込んでいる地図の種類
	 */
	private int dlType;

	
	/**
	 * コンストラクタ
	 * @param msg 初期メッセージ
	 * @param db 市区町村名データベース
	 */
	public StatusBar(String msg, NameDatabase db) {
		super(msg);
		this.db = db;
		this.thread = new Thread(this);
		this.thread.start();
		this.fileSize = 0;
	}

	/**
	 * ダウンロードが終了したことを通知します。
	 */
	public void finishDownload() {
		this.downloadCode = null;
		this.file = null;
	}

	/**
	 * 読み込みが終了したことを通知します。
	 */
	public void finishReading() {
		this.readCode = null;
		Log.out(this, "cleared reading.");
	}
	
	private void loop() {
		boolean flag = false;
		StringBuilder sb = new StringBuilder();
		sb.append("MEMORY : " + this.memory + "MB");
		if (this.search != null) {
			sb.append(this.search);
			flag = true;
		}
		if (this.readCode != null && !this.readCode.equals(this.downloadCode)) {
			/* ダウンロードしているファイルと読み込んでいるファイルが異なる場合 */
			flag = true;
			sb.append(" / READ ");
			switch (this.type) {
				case SDF2500 : 
					sb.append("SDF2500 ");
					break;
				case SDF25000 :
					sb.append("SDF25K ");
					break;
				case ISJ :
					sb.append("ISJ ");
					break;
			}
			sb.append(this.readCode);
		}
		if (this.downloadCode != null) {
			sb.append(" / DOWNLOAD ");
			switch (this.dlType) {
				case SDF2500 : 
					sb.append("SDF2500 ");
					break;
				case SDF25000 :
					sb.append("SDF25K ");
					break;
			}
			sb.append(this.downloadCode);
			if (this.file != null && this.fileSize > 0) {
				sb.append(" : ");
				if (this.file.length() < this.fileSize) {
					sb.append((int) (this.file.length() * 100 / this.fileSize + 0.5f));
				} else {
					sb.append(100);
					this.file = null;
					this.fileSize = 0;
				}
				sb.append('%');
			}
			flag = true;
		}
		if (sb.length() == 0) {
			super.setText(" ");
		} else {
			super.setText(sb.toString());
		}
		try {
			if (!flag) {
				Thread.sleep(2500L);
			} else {
				Thread.sleep(500L);
			}
		} catch (InterruptedException e) {
			Log.err(this, e);
		}
	}
	
	public void run() {
		while (true) {
			this.memory = (float) (int) ((Runtime.getRuntime().totalMemory() - Runtime
					.getRuntime().freeMemory()) / 1024) / 1024;
			this.loop();
		}
	}

	/**
	 * ダウンロードするファイル
	 * @param file
	 * @param size
	 */
	public void setDownloadFile(File file, int size) {
		this.file = file;
		this.fileSize = size;
	}

	/**
	 * 経路探索のためのスレッドを設定します。
	 * @param thread 経路探索のスレッド
	 */
	public void setThread(SearchThread thread) {
		this.search = thread;
	}

	/**
	 * ステータスバーの更新スレッドの優先度を設定します。
	 * @param priority 優先度（0から5）
	 */
	public void setThreadPriority(int priority) {
		this.thread.setPriority(priority);
	}
	
	/**
	 * インターネット上から取得しているファイルを設定します。
	 * @param code 市区町村番号
	 * @param type 読み込む地図の種類
	 */
	public void startDownload(int code, int type) {
		this.dlType = type;
		this.downloadCode = this.db.getName(code);
		if (this.downloadCode == null) {
			this.downloadCode = City.cityFormat(code);
		}
	}

	/**
	 * 読み込みを設定します。
	 * ローカルの補助記憶装置（ハードディスク）からの読み込んでいる市区町村番号を設定します。
	 * @param code 市区町村番号
	 * @param type 読み込む地図の種類
	 */
	public void startReading(int code, int type) {
		this.type = type;
		this.readCode = this.db.getName(code);
		if (this.readCode == null) {
			this.readCode = City.cityFormat(code);
		}
	}

	/**
	 * 読み込みを設定します。
	 * @param arg 表示する文字列
	 */
	public void startReading(String arg) {
		this.readCode = arg;
	}
}
