package pre.index;

import index.CellBounds;

import java.awt.Rectangle;
import java.io.BufferedReader;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.RandomAccessFile;

/**
 * 求めた外接長方形を分割数を指定してセルに分割する。
 * セルメソッドのインデックスを作成する。
 * @author ma38su
 */
public class CellPreprocessingSDF25K {

	public static void main(String[] args) throws IOException {
		CellPreprocessingSDF25K.splitCell(256, 256);	
	}
	
	/**
	 * 47都道府県すべてを含む外接長方形を求める。
	 * /data/prefecture.cellから求めます。
	 * @return 外接長方形
	 */
	public static Rectangle getBounds() {
		Rectangle rect = null;
		try {
			BufferedReader in = null;
			try {
				in = new BufferedReader(new InputStreamReader(CellPreprocessingSDF25K.class.getResourceAsStream("/.data/prefecture.cell")));
				String line;
				int minX = Integer.MAX_VALUE;
				int maxX = Integer.MIN_VALUE;
				int minY = Integer.MAX_VALUE;
				int maxY = Integer.MIN_VALUE;
				while ((line = in.readLine()) != null) {
					String[] param = line.split(",");
					int x = Integer.parseInt(param[0]);
					int y = Integer.parseInt(param[1]);
					int w = Integer.parseInt(param[2]);
					int h = Integer.parseInt(param[3]);
					if (minX > x) {
						minX = x;
					}
					if (minY > y) {
						minY = y;
					}
					if (maxX < x + w) {
						maxX = x + w;
					}
					if (maxY < y + h) {
						maxY = y + h;
					}
				}
				rect = new Rectangle(minX, minY, maxX - minX, maxY - minY);
				System.out.printf("x: %d, y: %d, w: %d, h: %d\n", minX, minY, maxX - minX, maxY - minY);
			} finally {
				if (in != null) {
					in.close();
				}
			}
		} catch (Throwable e) {
			rect = null;
		}
		return rect;
	}

	/**
	 * 分割数を指定して日本全体の外接長方形を分割してセルメソッドのインデックスを作成する。
	 * 出力は、cell.idxと、city.cell
	 * cell.idxはセルのインデックス
	 * city.cellはセルと重なる市区町村の情報
	 * @param splitWidth 横方向の分割数
	 * @param splitHeight 縦方向の分割数
	 * @throws IOException 入出力エラー
	 */
	public static void splitCell(int splitWidth, int splitHeight) throws IOException {
		RandomAccessFile disc = null;
		BufferedReader in = null;
		try {
			in = new BufferedReader(new InputStreamReader(CellPreprocessingSDF25K.class.getResourceAsStream("/.data/prefecture.cell")));
			Rectangle rect = CellPreprocessingSDF25K.getBounds();
			Rectangle cell = new Rectangle();
			if ((rect.width % splitWidth) > 0) {
				cell.width = rect.width / splitWidth + 1;
			} else {
				cell.width = rect.width / splitWidth;
			}
			if ((rect.height % splitHeight) > 0) {
				cell.height = rect.height / splitHeight + 1;
			} else {
				cell.height = rect.height / splitHeight;
			}
			System.out.println("cell:" + cell);
			DataOutputStream index = new DataOutputStream(new FileOutputStream("cell.idx"));
			index.writeInt(rect.x);
			index.writeInt(rect.y);
			index.writeInt(rect.width);
			index.writeInt(rect.height);
			index.writeShort(splitWidth);
			index.writeShort(splitHeight);
			RandomAccessFile out = new RandomAccessFile("city.cell", "rw");
			CacheCell[] cache = new CacheCell[splitWidth * splitHeight];
			for (int i = 0; i < cache.length; i++) {
				cache[i] = new CacheCell();
			}
			BufferedReader bi = new BufferedReader(new InputStreamReader(System.class.getResourceAsStream("/.data/city.csv"), "SJIS"));
			String line;
			while ((line = bi.readLine()) != null) {
				String[] param = line.split(",");
				int code = Integer.parseInt(param[1]);
				File file = new File("D:/java/map/cell-bounds"+ File.separatorChar + code + ".clb");
				try {
					System.out.println(file.getCanonicalPath());
					disc = new RandomAccessFile(file, "rw");
					CellBounds bounds = new CellBounds(disc);
					cell.y = rect.y;
					for (int i = 0; i < splitHeight; i++, cell.y += cell.height) {
						cell.x = rect.x;
						for (int j = 0; j < splitWidth; j++, cell.x += cell.width) {
							if (bounds.intersects(cell)) {
								cache[i * splitWidth + j].add(code, bounds);
							}
						}
					}
				} catch (Exception e) {
					System.out.println(code);
					e.printStackTrace();
					System.exit(0);
				} finally {
					if (disc != null) {
						disc.close();
					}
				}
			}
			for (CacheCell c : cache) {
				/* セルのインデックスのポインタ */
				index.writeLong(out.getFilePointer());
				index.writeByte(c.size());
				index.flush();
				/* セル */
				c.writeDisc(out);
			}
			index.close();
			out.close();
		} finally {
			if (in != null) {
				in.close();
			}
		}
	}
}
