package view;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.Stroke;
import java.awt.geom.GeneralPath;
import java.awt.print.Printable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import labeling.Label;
import labeling.Labeling;
import labeling.Labels;
import labeling.SimpleLabeling;
import map.CityMap;
import map.model.City;
import map.model.Curve;
import map.model.Mesh;
import map.model.Node;
import map.model.Road;
import map.route.DirectDistance;
import map.route.SearchThread;
import map.route.ShortestPathAlgorithm;
import map.route.UnidirectShortestPathSearch;
import util.Log;
import database.NameDatabase;

/**
 * 地図データを表示するためのパネル
 * 
 * 2005/09/23
 * 
 * @author ma38su
 */
public class MapPanel extends ExportablePanel implements Printable {
	
	/**
	 * 国土数値情報による塗りつぶし色
	 */
	private static final Color COLOR_GROUND = Color.GREEN.brighter();

	/**
	 * 高速道路の塗りつぶし色
	 */
	private static final Color COLOR_HIGHWAY = new Color(0x7777FF);
	
	/**
	 * 高速道路の境界色
	 */
	private static final Color COLOR_HIGHWAY_BORDER = new Color(0x000088);
	
	/**
	 * 一般道のルート色
	 */
	private static final Color COLOR_ROUTE_GENERAL = Color.YELLOW;

	/**
	 * 高速道路のルート色
	 */
	private static final Color COLOR_ROUTE_HIGHWAY = Color.RED;
	
	/**
	 * 水域を塗りつぶす色
	 */
	private static final Color COLOR_SEA = Color.CYAN;
	
	/**
	 * 水域の境界色
	 */
	private static final Color COLOR_SEA_BORDER = Color.CYAN.darker();

	/**
	 * 駅の塗りつぶし色
	 */
	private static final Color COLOR_STATION = new Color(0xFFAA44);

	/**
	 * フォントの種類（論理フォント）
	 * Serif, SansSerif, Monospaced, Dialog, DialogInput
	 */
	private static final String FONT_FAMILY = "SansSerif";

	/**
	 * ラベリングに用いるフォント
	 */
	public static final Font FONT_INFO = new Font(MapPanel.FONT_FAMILY, Font.PLAIN, 12);

	/**
	 * ラベリングに用いるフォント
	 */
	private static final Font FONT_LABEL = new Font(MapPanel.FONT_FAMILY, Font.PLAIN, 11);

	/**
	 * 最大フォントサイズ
	 */
	private static final int FONT_MAX_SIZE = 16;

	private static final Font FONT_STATION = new Font(MapPanel.FONT_FAMILY, Font.BOLD, 14);

	private static final int MAX_SCREEN_X = 154 * 3600000;

	private static final int MAX_SCREEN_Y = 46  * 3600000;

	private static final int MIN_SCREEN_X = 122 * 3600000;

	private static final int MIN_SCREEN_Y = 20  * 3600000;

	/**
	 * 市町村まで表示する倍率
	 */
	private static final float SCALE_CITY = 0.0002f;
	
	/**
	 * 表示倍率の上限
	 */
	private static final float SCALE_MAX = 0.1f;

	/**
	 * 表示倍率の下限
	 */
	private static final float SCALE_MIN = 0.000005f;

	/**
	 * 国土数値情報と数値地図25000を切り替える倍率
	 */
	private static final float SCALE_SDF = 0.0020f;

	/**
	 * 表示倍率の変更精度
	 */
	private static final float SCALE_SENSE = 0.08f;

	/**
	 * 道路区間，鉄道区間の描画の基本の幅
	 */
	private static final int STROKE_WIDTH = 130;

	/**
	 * 最短経路探索アルゴリズム
	 */
	private ShortestPathAlgorithm algorithm;

	/**
	 * 市区町村番号から市区町村名を検索するためのクラス
	 * ハッシュマップを持つ
	 */
	private NameDatabase db;

	/**
	 * 描画する道路のレベル
	 */
	private int drawlevel;

	/**
	 * アンチエイリアスのフラグ
	 */
	private boolean isAntialias;

	/**
	 * ラベル表示のフラグ
	 */
	private int isLabel;

	private boolean isLabelAntialias;
	
	/**
	 * 頂点表示のフラグ
	 */
	private boolean isNodeView;

	/**
	 * マウス操作のフラグ
	 */
	private boolean isOperation;

	/**
	 * 再描画フラグ
	 */
	private boolean isRepaint;
	
	/**
	 * ラベリングアルゴリズム
	 */
	private Labeling labeling;

	/**
	 * 地図情報管理マップ
	 */
	private CityMap maps;

	/**
	 * 頂点のサイズ
	 */
	private int nodeSize = 3;
	
	/**
	 * オフスクリーンイメージ
	 */
	private Image offs;

	/**
	 * 都道府県ポリゴン
	 */
	private Polygon[] prefecture;

	/**
	 * 水域
	 */
	private Polygon[] prefectureCoast;

	/**
	 * 都道府県ポリゴンの島
	 */
	private Polygon[] prefectureIsland;

	/**
	 * 地図の表示倍率
	 */
	private float scale = 0.005f;
	
	/**
	 * スクリーンサイズ
	 */
	private Rectangle screen;

	/**
	 * 探索の始点
	 */
	private Node start;

	/**
	 * 探索の終点
	 */
	private Node terminal;

	/**
	 * 探索を行うスレッド
	 */
	private SearchThread thread;
	
	/**
	 * 地図パネル
	 * @param db 市区町村名検索のためのクラス
	 */
	public MapPanel(NameDatabase db) {
		this.db = db;
		this.screen = new Rectangle();
		this.labeling = new SimpleLabeling(this.screen);
	}

	/**
	 * 地図の描画
	 * @param g
	 */
	@Override
	public void draw(Graphics2D g) {
		// スクリーン情報を指定する
		this.screen.width = (int) (this.getWidth() / this.scale);
		this.screen.height = (int) (this.getHeight() / this.scale);

		if(this.scale > MapPanel.SCALE_CITY) {
			if (!this.isOperation) {
				if (this.isAntialias) {
					g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
				}
			}
			if (this.scale >= MapPanel.SCALE_SDF) {
				this.drawMap(g);
			} else {
				g.setStroke(new BasicStroke(2, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
				this.fillPolygon(g, this.prefecture, MapPanel.COLOR_GROUND, Color.BLACK);
				this.fillPolygon(g, this.prefectureCoast, MapPanel.COLOR_SEA, Color.BLACK);
				this.fillPolygon(g, this.prefectureIsland, MapPanel.COLOR_GROUND, Color.BLACK);
				g.setStroke(new BasicStroke(1, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
				g.setColor(Color.BLACK);
				synchronized (this.maps) {
					for (City city : this.maps.values()) {
						this.drawBorder(g, city, Color.BLACK);
					}
				}
				this.drawRoute(g);
			}
		} else {
			/* 都道府県のみ表示 */
			
			g.setColor(MapPanel.COLOR_GROUND);
			this.fillPolygon(g, this.prefecture);
			g.setColor(Color.BLACK);
			this.drawPolygon(g, this.prefecture);
			
			g.setColor(MapPanel.COLOR_SEA);
			this.fillPolygon(g, this.prefectureCoast);
			g.setColor(Color.BLACK);
			this.drawPolygon(g, this.prefectureCoast);

			g.setColor(MapPanel.COLOR_GROUND);
			this.fillPolygon(g, this.prefectureIsland);
			g.setColor(Color.BLACK);
			this.drawPolygon(g, this.prefectureIsland);
		}
	}

	/**
	 * 市区町村の行政界と市区町村名を描画します。
	 * @param g 描画するGraphics2D
	 * @param city 描画する行政界
	 * @param color 
	 * @param labeling 
	 * @param bg 背景色
	 * @param line 境界色
	 */
	private void drawBorder(Graphics2D g, City city, Color color) {
		Polygon[] polygons = city.getPolygon();
		if (polygons != null) {
			g.setColor(color);
			int x = 0;
			int y = 0;
			int maxN = 0;
			for (int i = 0; i < polygons.length; i++) {
				if(polygons[i].getBounds().intersects(this.screen)) {
					int[] aryX = polygons[i].xpoints;
					int[] aryY = polygons[i].ypoints;
	
					int[] polygonX = new int[polygons[i].npoints];
					int[] polygonY = new int[polygons[i].npoints];
	
					for (int j = 0; j < polygons[i].npoints; j++) {
						polygonX[j] = (int)((aryX[j] - this.screen.x) * this.scale);
						polygonY[j] = this.getHeight() - (int)((aryY[j] - this.screen.y) * this.scale);
					}
					g.drawPolygon(polygonX, polygonY, polygonX.length);
					if (maxN < polygons[i].npoints) {
						maxN = polygons[i].npoints;
						Rectangle rect = polygons[i].getBounds();
						x = (int)((rect.x + rect.width / 2 - this.screen.x) * this.scale);
						y = this.getHeight() - (int)((rect.y + rect.height / 2 - this.screen.y) * this.scale);
					}
				}
			}
			if (maxN > 0) {
				String name = city.getName();
				if (name == null) {
					name = this.db.get(city.getCode());
					if (name == null) {
						name = "";
						throw new UnknownError();
					}
					city.setName(name);
				}
				this.labeling.add(name, x, y);
			}
		}
	}
	
	/**
	 * 曲線の描画
	 * 
	 * @param g 描画するGraphics
	 * @param curves 描画する行政界
	 */
	private void drawCurve(final Graphics2D g, final Curve[] curves) {
		if (curves != null) {
			for (final Curve curve : curves) {
				final int[] aryX = curve.getArrayX();
				final int[] aryY = curve.getArrayY();
				int x0 = (int) ((aryX[0] - this.screen.x) * this.scale);
				int y0 = this.getHeight() - (int) ((aryY[0] - this.screen.y) * this.scale);
				for (int k = 1; k < aryX.length; k++) {
					int x = (int) ((aryX[k] - this.screen.x) * this.scale);
					int y = this.getHeight() - (int) ((aryY[k] - this.screen.y) * this.scale);
					// 表示領域内
					if (this.screen.intersectsLine(aryX[k - 1], aryY[k - 1], aryX[k], aryY[k])) {
						g.drawLine(x0, y0, x, y);
					}
					x0 = x;
					y0 = y;
				}
			}
		}
	}

	/**
	 * 折れ線の描画をおこないます。
	 * @param g
	 * @param path
	 * @param color
	 * @param stroke
	 */
	private void drawGeneralPath(Graphics2D g, GeneralPath path, Color color, Stroke stroke) {
		g.setColor(color);
		g.setStroke(stroke);
		g.draw(path);
	}
	
	/**
	 * 数値地図を描画する
	 * 
	 * @param g
	 */
	public void drawMap(Graphics2D g) {

		int meshSize = (int) (Mesh.SIZE * this.scale) + 1;
		// メッシュの描画
		synchronized (this.maps) {
			for (final City map : this.maps.values()) {
				if(!map.hasData()) {
					this.fillBorder(g, map);
				} else {
					int halfMesh = Mesh.SIZE / 2;
					for (Mesh mesh : map.getMesh()) {
						int meshX = mesh.getX() - halfMesh;
						if (!this.screen.intersects(meshX, mesh.getY() - halfMesh, Mesh.SIZE, Mesh.SIZE)) {
							continue;
						}
						int x = (int) ((meshX - this.screen.x) * this.scale);
						int y = this.getHeight() - (int) ((mesh.getY() + halfMesh - this.screen.y) * this.scale);
		
						g.setColor(mesh.getColor());
						g.fillRect(x, y, meshSize, meshSize);
					}
				}
			}
			for (final City map : this.maps.values()) {
	
				if (!map.hasData()) {
					continue;
				}
				// 行政界を描画
				g.setColor(MapPanel.COLOR_GROUND);
				this.drawCurve(g, map.getBorder());
	
				// 海岸線を描画
				g.setColor(MapPanel.COLOR_SEA_BORDER);
				this.drawCurve(g, map.getSeaBorder());

				// 数値地図2500
				if (map.isSdk2500()) {
					g.setColor(MapPanel.COLOR_SEA);
					this.fillPolygon(g, map.getMizu());

					g.setColor(MapPanel.COLOR_SEA_BORDER);
					this.drawCurve(g, map.getMizuArc());
				} else {
					// 沿岸を描画
					g.setColor(MapPanel.COLOR_SEA_BORDER);
					this.drawCurve(g, map.getCoast());

					// 河川を描画
					g.setColor(MapPanel.COLOR_SEA_BORDER);
					this.drawCurve(g, map.getRiver());
				}
			}
		}
		{
			// 道路を描画する
			int[] roadWidth = new int[6];
			
			GeneralPath[] road = new GeneralPath[12];
			for (int i = 5; i >= 0; i--) {
				roadWidth[i] = (int)((i + 1) * MapPanel.STROKE_WIDTH * this.scale);
				if(roadWidth[i] > 0) {
					road[i] = new GeneralPath();
					this.drawlevel = i;
				}
				road[i + 6] = new GeneralPath();
			}
			
			GeneralPath jr = new GeneralPath();
			List<int[]> otherX = new ArrayList<int[]>();
			List<int[]> otherY = new ArrayList<int[]>();
			List<int[]> stationX = new ArrayList<int[]>();
			List<int[]> stationY = new ArrayList<int[]>();

			synchronized (this.maps) {
				for (final City map : this.maps.values()) {
					if (!map.hasData()) {
						continue;
					}
					for (int i = 5; i >= 0; i--) {
						this.extractRoadway(road[i], map.getRoad()[i]);
						this.extractRoadway(road[i + 6], map.getRoad()[i + 6]);
					}
					this.extractGeneralPath(map.getJRRailway(), jr);
					this.extractPolyLine(map.getOtherRailway(), otherX, otherY);
					this.extractPolyLine(map.getStation(), stationX, stationY);
				}
			}
			this.drawRoadway(g, road, roadWidth);
			int w = (int)(MapPanel.STROKE_WIDTH * this.scale * 4f + 0.5f);
			if (w == 0) {
				w = 1;
			}

			Stroke stroke = new BasicStroke(w + 2, BasicStroke.CAP_SQUARE, BasicStroke.JOIN_ROUND);
			this.drawPolyLine(g, otherX, otherY, Color.DARK_GRAY, stroke);
			this.drawGeneralPath(g, jr, Color.BLACK, stroke);

			this.drawPolyLine(g, otherX, otherY, Color.BLACK, new BasicStroke(w, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
			this.drawGeneralPath(g, jr, Color.WHITE, new BasicStroke(w, BasicStroke.CAP_BUTT, BasicStroke.JOIN_ROUND, 10f, new float[]{w * 8, w * 8}, 0.25f));

			w *= 3;
			this.drawPolyLine(g, stationX, stationY, Color.BLACK, new BasicStroke(w + 2, BasicStroke.CAP_SQUARE, BasicStroke.JOIN_ROUND));
			this.drawPolyLine(g, stationX, stationY, MapPanel.COLOR_STATION, new BasicStroke(w, BasicStroke.CAP_SQUARE, BasicStroke.JOIN_ROUND));
		}
		synchronized (this.maps) {
			for (final City map : this.maps.values()) {
				if (!map.hasData()) {
					continue;
				}	
				if (this.isNodeView) {
					int nodeSize = (int) (this.nodeSize / this.scale);
	
					g.setColor(Color.GRAY);
	
					for (Node node : map.getNode()) {
						// 表示領域外であれば次へ
						if (!this.screen.intersects(node.getX() - nodeSize, node.getY() - nodeSize, nodeSize * 2, nodeSize * 2)) {
							continue;
						}
	
						int x = (int) ((node.getX() - this.screen.x) * this.scale);
						int y = this.getHeight() - (int) ((node.getY() - this.screen.y) * this.scale);
	
						// draw node
						g.fillRect(x - this.nodeSize, y - this.nodeSize, this.nodeSize * 2, this.nodeSize * 2);
					}
				}
			}
		}
		this.drawRoute(g);
		g.setStroke(new BasicStroke(1, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));

		g.setFont(MapPanel.FONT_LABEL);
		if (this.isLabel > 0) {
			synchronized (this.maps) {
				for (City map : this.maps.values()) {
					if (map.hasData()) {	
						if ((this.isLabel & 1) != 0) {
							Font df = g.getFont();
							g.setFont(MapPanel.FONT_STATION);
							this.labeling.add(map.getStation());
							g.setFont(df);
						}
						if ((this.isLabel & 2) != 0) {
							this.labeling.add(map.getLabel(Label.KEY_CM));
						}
						if ((this.isLabel & 4) != 0) {
							this.labeling.add(map.getLabel(Label.KEY_KO));
						}
						for (Labels labels : map.getLabels()) {
							this.labeling.add(labels);
						}
					}
				}
			}
		}
	}

	/**
	 * ポリゴンを描画します。
	 * @param g 描画するGraphics2D
	 * @param polygons 描画するポリゴン
	 */
	private void drawPolygon(Graphics2D g, Polygon[] polygons) {
		if (polygons == null) {
			return;
		}
		for (int i = 0; i < polygons.length; i++) {
			if(polygons[i].intersects(this.screen)) {
				int[] aryX = polygons[i].xpoints;
				int[] aryY = polygons[i].ypoints;

				int[] polygonX = new int[polygons[i].npoints];
				int[] polygonY = new int[polygons[i].npoints];

				for (int j = 0; j < polygons[i].npoints; j++) {
					polygonX[j] = (int)((aryX[j] - this.screen.x) * this.scale);
					polygonY[j] = this.getHeight() - (int)((aryY[j] - this.screen.y) * this.scale);
				}
				g.drawPolygon(polygonX, polygonY, polygons[i].npoints);
			}
		}
	}

	/**
	 * 折れ線の描画をおこないます。
	 * @param g
	 * @param listX
	 * @param listY
	 * @param color
	 * @param stroke
	 */
	private void drawPolyLine(Graphics2D g, List<int[]> listX, List<int[]> listY, Color color, Stroke stroke) {
		g.setColor(color);
		g.setStroke(stroke);
		for (int i = 0; i < listX.size(); i++) {
			int[] aryX = listX.get(i);
			g.drawPolyline(aryX, listY.get(i), aryX.length);
		}
	}

	/**
	 * 道路の描画
	 * 
	 * @param g Graphics2D
	 * @param path 
	 * @param width 
	 */
	private void drawRoadway(Graphics2D g, GeneralPath[] path, int[] width) {
		g.setColor(Color.LIGHT_GRAY);
		for (int i = 0; i < 6; i++) {
			if(path[i] != null) {
				g.setStroke(new BasicStroke(width[i] + 2, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
				g.draw(path[i]);
			}
		}
		g.setColor(Color.WHITE);
		for (int i = 0; i < 6; i++) {
			if(path[i] != null) {
				g.setStroke(new BasicStroke(width[i], BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
				g.draw(path[i]);
			}
		}
		g.setColor(COLOR_HIGHWAY_BORDER);
		for (int i = 0; i < 6; i++) {
			g.setStroke(new BasicStroke(width[i] + 3, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
			g.draw(path[i + 6]);
		}
		/* 高速の塗りつぶし */
		g.setColor(COLOR_HIGHWAY);
		for (int i = 0; i < 6; i++) {
			g.setStroke(new BasicStroke(width[i] + 1, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
			g.draw(path[i + 6]);
		}
	}

	/**
	 * 探索済み頂点を描画する
	 * 
	 * @param g
	 */
	private void drawRoute(Graphics2D g) {
		if (this.thread != null) {
			Collection<Road> route = this.thread.getRoute();
			if (route != null) {
				int[] width = new int[12];
				for (int i = 0; i < 6; i++) {
					width[i] = (int)((i + 1) * MapPanel.STROKE_WIDTH * this.scale) + 1;
				}
				for (Road road : route) {
					int[] aryX = road.getArrayX();
					int[] aryY = road.getArrayY();
		
					int x0 = (int) ((aryX[0] - this.screen.x) * this.scale);
					int y0 = this.getHeight() - (int) ((aryY[0] - this.screen.y) * this.scale);

					if (road.getType() == 3) {
						g.setColor(MapPanel.COLOR_ROUTE_HIGHWAY);
					} else {
						g.setColor(MapPanel.COLOR_ROUTE_GENERAL);
					}
					g.setStroke(new BasicStroke(width[road.getWidth()], BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
		
					for (int k = 1; k < aryX.length; k++) {
		
						int x = (int) ((aryX[k] - this.screen.x) * this.scale);
						int y = this.getHeight() - (int) ((aryY[k] - this.screen.y) * this.scale);

						// 表示領域内
						if (this.screen.intersectsLine(aryX[k - 1], aryY[k - 1], aryX[k], aryY[k])) {
							g.drawLine(x0, y0, x, y);
						}
						x0 = x;
						y0 = y;
					}
				}
			}
		}
		int r = (int) (this.scale * 500) + 2;
		g.setStroke(new BasicStroke(r / 2f));
		if (this.start != null) {
			int x = (int) ((this.start.getX() - this.screen.x) * this.scale) - r;
			int y = this.getHeight() - (int) ((this.start.getY() - this.screen.y) * this.scale) - r;
			g.setColor(Color.YELLOW);
			g.fillOval(x, y, r * 2, r * 2);
			g.setColor(Color.BLACK);
			g.drawOval(x, y, r * 2, r * 2);
		}
		if (this.terminal != null) {
			int x = (int) ((this.terminal.getX() - this.screen.x) * this.scale) - r;
			int y = this.getHeight() - (int) ((this.terminal.getY() - this.screen.y) * this.scale) - r;
			g.setColor(Color.RED);
			g.fillOval(x, y, r * 2, r * 2);
			g.setColor(Color.BLACK);
			g.drawOval(x, y, r * 2, r * 2);
		}
	}

	/**
	 * 白抜き文字を描画します。
	 * @param g
	 * @param msg
	 * @param x
	 * @param y
	 * @param color
	 */
	public void drawString(Graphics g, String msg, int x, int y, Color color) {
		g.setColor(Color.BLACK);
		g.drawString(msg, x + 1, y + 1);
		g.drawString(msg, x + 1, y - 1);
		g.drawString(msg, x - 1, y + 1);
		g.drawString(msg, x - 1, y - 1);
		g.drawString(msg, x - 1, y);
		g.drawString(msg, x, y - 1);
		g.drawString(msg, x + 1, y);
		g.drawString(msg, x, y + 1);
		g.setColor(color);
		g.drawString(msg, x, y);
	}

	/**
	 * GeneralPathに展開
	 * 
	 * @param curves 展開する曲線の配列
	 * @param path 展開されたGeneralPath
	 */
	private void extractGeneralPath(final Curve[] curves, final GeneralPath path) {
		if (curves != null) {
			for (final Curve curve : curves) {
				int[] aryX = curve.getArrayX();
				int[] aryY = curve.getArrayY();

				int x = (int) ((aryX[0] - this.screen.x) * this.scale);
				int y = this.getHeight() - (int) ((aryY[0] - this.screen.y) * this.scale);
				path.moveTo(x, y);

				for (int i = 1; i < aryX.length; i++) {
					x = (int) ((aryX[i] - this.screen.x) * this.scale);
					y = this.getHeight() - (int) ((aryY[i] - this.screen.y) * this.scale);
					path.lineTo(x, y);
				}
			}
		}
	}

	/**
	 * 折れ線に展開
	 * 
	 * @param curves
	 * @param polyX 
	 * @param polyY 
	 */
	private void extractPolyLine(final Curve[] curves, final List<int[]> polyX, final List<int[]> polyY) {
		if (curves != null) {
			for (final Curve curve : curves) {
				int[] aryX = curve.getArrayX();
				int[] aryY = curve.getArrayY();
				int[] lineX = new int[aryX.length];
				int[] lineY = new int[aryY.length];
				for (int i = 0; i < aryX.length; i++) {
					lineX[i] = (int) ((aryX[i] - this.screen.x) * this.scale);
					lineY[i] = this.getHeight() - (int) ((aryY[i] - this.screen.y) * this.scale);
				}
				polyX.add(lineX);
				polyY.add(lineY);
			}
		}
	}

	/**
	 * 道路データをGeneralPathに展開する
	 * @param path 展開先のGeneralPath
	 * @param curves 道路データ
	 */
	private void extractRoadway(GeneralPath path, Curve[] curves) {
		if (path != null && curves != null) {
			for (Curve curve : curves) {
				int[] aryX = curve.getArrayX();
				int[] aryY = curve.getArrayY();
	
				int x0 = (int) ((aryX[0] - this.screen.x) * this.scale);
				int y0 = this.getHeight()
						- (int) ((aryY[0] - this.screen.y) * this.scale);
	
				path.moveTo(x0, y0);
				
				for (int k = 1; k < aryX.length; k++) {
	
					int x = (int) ((aryX[k] - this.screen.x) * this.scale);
					int y = this.getHeight() - (int) ((aryY[k] - this.screen.y) * this.scale);

					// 表示領域外であれば次へ
					if (this.screen.intersectsLine(aryX[k - 1], aryY[k - 1], aryX[k], aryY[k])) {
						path.lineTo(x, y);
					} else {
						path.moveTo(x, y);
					}
				}
			}
		}
	}

	/**
	 * 市区町村の行政界を描画します。
	 * @param g 描画するGraphics2D
	 * @param city 描画する行政界
	 * @param bg 背景色
	 * @param line 境界色
	 */
	private void fillBorder(Graphics2D g, City city) {
		Polygon[] polygons = city.getPolygon();
		if (polygons != null) {
			Polygon[] tmp = new Polygon[polygons.length];
			g.setColor(MapPanel.COLOR_GROUND);
			int x = 0;
			int y = 0;
			int maxN = 0;
			for (int i = 0; i < polygons.length; i++) {
				if(polygons[i].getBounds().intersects(this.screen)) {
					int[] aryX = polygons[i].xpoints;
					int[] aryY = polygons[i].ypoints;
	
					int[] polygonX = new int[polygons[i].npoints];
					int[] polygonY = new int[polygons[i].npoints];
	
					for (int j = 0; j < polygons[i].npoints; j++) {
						polygonX[j] = (int)((aryX[j] - this.screen.x) * this.scale);
						polygonY[j] = this.getHeight() - (int)((aryY[j] - this.screen.y) * this.scale);
						x += polygonX[j];
						y += polygonY[j];
					}
					tmp[i] = new Polygon(polygonX, polygonY, polygonX.length);
					g.fillPolygon(tmp[i]);
				}
				if (maxN < polygons[i].npoints) {
					maxN = polygons[i].npoints;
					Rectangle rect = polygons[i].getBounds();
					x = (int)((rect.x + rect.width / 2 - this.screen.x) * this.scale);
					y = this.getHeight() - (int)((rect.y + rect.height / 2 - this.screen.y) * this.scale);
				}
			}
			g.setColor(Color.BLACK);
			for (Polygon p : tmp) {
				if (p != null) {
					g.drawPolygon(p);
				}
			}
			if (maxN > 0) {
				String name = city.getName();
				if (name == null) {
					name = this.db.get(city.getCode());
					if (name == null) {
						name = "";
					}
					city.setName(name);
				}
				this.labeling.add(name, x, y);
			}
		}
	}
	
	/**
	 * ポリゴンを描画します。
	 * @param g 描画するGraphics2D
	 * @param polygons 描画するポリゴン
	 */
	private void fillPolygon(Graphics2D g, Polygon[] polygons) {
		if (polygons == null) {
			return;
		}
		for (int i = 0; i < polygons.length; i++) {
			if(polygons[i].intersects(this.screen)) {
				int[] aryX = polygons[i].xpoints;
				int[] aryY = polygons[i].ypoints;

				int[] polygonX = new int[polygons[i].npoints];
				int[] polygonY = new int[polygons[i].npoints];

				for (int j = 0; j < polygons[i].npoints; j++) {
					polygonX[j] = (int)((aryX[j] - this.screen.x) * this.scale);
					polygonY[j] = this.getHeight() - (int)((aryY[j] - this.screen.y) * this.scale);
				}
				g.fillPolygon(polygonX, polygonY, polygonX.length);
			}
		}
	}

	/**
	 * ポリゴンを描画します。
	 * @param g 描画するGraphics2D
	 * @param polygons 描画するポリゴン
	 * @param bg 背景色
	 * @param line 境界色
	 */
	private void fillPolygon(Graphics2D g, Polygon[] polygons, Color bg, Color line) {
		if (polygons == null) {
			return;
		}
		g.setColor(bg);
		Polygon[] tmp = new Polygon[polygons.length];
		for (int i = 0; i < polygons.length; i++) {
			if(polygons[i].intersects(this.screen)) {
				int[] aryX = polygons[i].xpoints;
				int[] aryY = polygons[i].ypoints;

				int[] polygonX = new int[polygons[i].npoints];
				int[] polygonY = new int[polygons[i].npoints];

				for (int j = 0; j < polygons[i].npoints; j++) {
					polygonX[j] = (int)((aryX[j] - this.screen.x) * this.scale);
					polygonY[j] = this.getHeight() - (int)((aryY[j] - this.screen.y) * this.scale);
				}
				tmp[i] = new Polygon(polygonX, polygonY, polygonX.length);
				g.fillPolygon(tmp[i]);
			}
		}
		// 境界を描画つぶします。
		g.setColor(line);
		for (Polygon p : tmp) {
			if (p != null) {
				g.drawPolygon(p);
			}
		}
	}

	public double getLocationX(int x) {
		return (this.screen.x + x / this.scale) / 3600000;
	}
	
	public double getLocationY(int y) {
		return (this.screen.y + (this.getHeight() - y) / this.scale) / 3600000;
	}

	/**
	 * スクリーン情報の取得
	 * 
	 * @return スクリーンのRectangle
	 */
	public Rectangle getScreen() {
		return this.screen;
	}

	/**
	 * 初期設定
	 * @param setting 設定
	 * @param map 地図データ管理クラス
	 * @param prefecture 都道府県ポリゴン
	 * @param coast 都道府県ポリゴンと重なる水域ポリゴン
	 * @param island 水域ポリゴンと重なる島などのポリゴン
	 */
	public void init(CityMap map, Polygon[] prefecture, Polygon[] coast, Polygon[] island) {
		Log.out(this, "init called.");
		this.algorithm = new UnidirectShortestPathSearch(new DirectDistance(), map);

		this.maps = map;
		
		this.prefecture = prefecture;
		this.prefectureCoast = coast;
		this.prefectureIsland = island;
		
		this.isLabel = 7;
		this.isNodeView = false;
		this.isAntialias = true;
		this.isOperation = false;
		this.moveDefault();
		super.setBackground(MapPanel.COLOR_SEA);
	}
	/**
	 * 地図データの読み込みを確認します。
	 * @return 地図データを読み込んでいればtrueを返します。
	 */
	public boolean isLoaded() {
		return this.maps != null;
	}

	/**
	 * 操作しているかどうか確認します。
	 * @return 操作していればtrue
	 */
	public boolean isOperation() {
		return this.isOperation;
	}
	
	/**
	 * 地図の表示状態を確認します。
	 * 2 : 数値地図25000
	 * 1 : 国土数値情報（市区町村）
	 * 0 : 国土数値情報（都道府県）
	 * @return 地図の表示状態
	 */
	public int mode() {
		if(this.scale > MapPanel.SCALE_SDF) {
			return 2;
		} else if(this.scale > MapPanel.SCALE_CITY) {
			return 1;
		} else {
			return 0;
		}
	}

	/**
	 * 表示位置を初期値へ
	 */
	public void moveDefault() {
		this.screen.x = MapPanel.MIN_SCREEN_X;
		this.screen.y = MapPanel.MIN_SCREEN_Y;

		double widthScale = (double) this.getWidth() / (MapPanel.MAX_SCREEN_X - MapPanel.MIN_SCREEN_X);
		double heightScale = (double) this.getHeight() / (MapPanel.MAX_SCREEN_Y - MapPanel.MIN_SCREEN_Y);
		this.scale = (float) ((widthScale < heightScale) ? widthScale : heightScale);
		this.repaint();
	}
	
	/**
	 * 表示位置を平行移動をさせます。
	 * @param dx X軸方向の移動量
	 * @param dy Y軸方向の移動量
	 */
	public void moveLocation(int dx, int dy) {
		this.screen.x -= dx / this.scale;
		this.screen.y += dy / this.scale;
		this.repaint();
	}
	
	/**
	 * 地図の中心経緯度を指定して表示位置を移動させます。
	 * @param x 東経
	 * @param y 北緯
	 */
	public void moveToLocation(double x, double y) {
		this.scale = MapPanel.SCALE_SDF * 1.5f;
		this.screen.x = (int)(x * 3600000) - (int)(this.getWidth() / 2 / this.scale + 0.5f);
		this.screen.y = (int)(y * 3600000) - (int)(this.getHeight() / 2 / this.scale + 0.5f);
		this.repaint();
	}

	@Override
	public void paintComponent(Graphics g) {
		if(this.maps == null) {
			this.isRepaint = false;
			this.offs = this.createImage(this.getWidth(), this.getHeight());
			Graphics2D offg = (Graphics2D) this.offs.getGraphics();
			super.paintComponent(offg);
			// オフスクリーンバッファ
			offg.setFont(new Font(MapPanel.FONT_FAMILY, Font.PLAIN, 20));
			String msg = "Now Loading...";
			FontMetrics metrics = offg.getFontMetrics();
			offg.drawString(msg, (this.getWidth() - metrics.stringWidth(msg)) / 2, (this.getHeight() - metrics.getHeight()) / 2);
			offg.dispose();
		} else if (this.isRepaint) {
			this.isRepaint = false;
			if (this.offs == null) {
				this.offs = this.createImage(this.getWidth(), this.getHeight());
			}
			if (this.getWidth() != this.offs.getWidth(null) || this.getHeight() != this.offs.getHeight(null)) {
				this.offs.flush();
				// オフスクリーンバッファ
				this.offs = super.createImage(this.getWidth(), this.getHeight());
			}
			Graphics2D offg = (Graphics2D) this.offs.getGraphics();
			this.labeling.init(offg, this.scale, this.getWidth(), this.getHeight());

			String mode = null;
			switch (this.mode()) {
			case 0 :
				mode = "MODE: 国土数値情報（都道府県）";
				break;
			case 1 : 
				mode = "MODE: 国土数値情報（都道府県＋市区町村）";
				break;
			default :
				mode = "MODE: 数値地図25000＋数値地図2500";
			}
			offg.setFont(MapPanel.FONT_INFO);
			this.labeling.set(mode, 5, 2);
			this.labeling.set("SCALE: " + Float.toString((int)(this.scale / 10 * 1000000 * 10) / 10f) + "µ", 5, 17);
			
			int fontSize = 9 + (int)(this.scale * 5000);
			if (fontSize > MapPanel.FONT_MAX_SIZE) {
				fontSize = MapPanel.FONT_MAX_SIZE;
			}
			offg.setFont(new Font("Serif", Font.PLAIN, fontSize));

			offg.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_OFF);
			
			offg.setColor(MapPanel.COLOR_SEA);
			offg.fillRect(0, 0, this.getWidth(), this.getHeight());
			this.draw(offg);

			offg.setColor(Color.BLACK);
			if (!this.isOperation) {
				this.labeling.draw(this.rendering);
			} else {
				this.labeling.draw();
			}
			offg.dispose();
		}
		g.drawImage(this.offs, 0, 0, null);
	}

	private int rendering;
	public void setLabelRendering(int rendering) {
		this.rendering = rendering;
	}

	@Override
	public void repaint() {
		if (this.maps == null || !this.isRepaint) {
			this.isRepaint = true;
			super.repaint();
		}
	}

	/**
	 * ルートの再計算を行います。
	 *
	 */
	public void reroute() {
		if(this.start != null && this.terminal != null && this.start != this.terminal) {
			if (this.thread != null) {
				this.thread.kill();
			}
			this.thread = this.algorithm.search(this.start, this.terminal);
		}
		this.repaint();
	}

	/**
	 * 探索端点の変更を行います。
	 * @param ex マウスのX座標
	 * @param ey マウスのY座標
	 * @param flag trueなら終点の変更、falseなら始点の変更
	 */
	public void searchBoundary(final int ex, final int ey, final boolean flag) {
		int x = (int)(ex / this.scale + this.screen.x);
		int y = (int)((this.getHeight() - ey) / this.scale + this.screen.y);
		
		Node point = null;
		double nodeDist = Double.POSITIVE_INFINITY;
		synchronized (this.maps) {
			for (final City map : this.maps.values()) {
				if (!map.hasData() || !map.getArea().contains(x, y)) {
					continue;
				}
				for (final Node node : map.getNode()) {
					final double d = node.distance(x, y, this.drawlevel);
					if (nodeDist > d) {
						point = node;
						nodeDist = d;
					}
				}
			}
		}
		if (point != null) {
			if (flag) {
				this.terminal = point;
				this.start = null;
				if (this.thread != null) {
					this.thread.kill();
				}
			} else {
				this.start = point;
				if (this.terminal != null) {
					if (this.thread != null) {
						this.thread.kill();
					}
					if (this.start != this.terminal) {
						this.thread = this.algorithm.search(this.start, this.terminal);
					}
				}
			}
		}
	}

	/**
	 * マウス操作の状態を設定する
	 * @param flag マウス操作していればtrue
	 */
	public void setOperation(boolean flag) {
		this.isOperation = flag;
		if(!flag) {
			this.repaint();
		}
	}

	/**
	 * ラベル表示を切り替える
	 * @param n 
	 */
	public void switchLabel(int n) {
		if ((this.isLabel & n) == 0) {
			this.isLabel += n;
		} else {
			this.isLabel -= n;
		}
	}

	public void switchLabelAntialias() {
		this.isLabelAntialias = !this.isLabelAntialias;
		this.repaint();
	}
	
	/**
	 * 頂点表示を切り替える
	 */
	public void switchNodeView() {
		this.isNodeView = !this.isNodeView;
		this.repaint();
	}
	
	/**
	 * アンチエイリアスを切り替える
	 */
	public void switchRendering() {
		this.isAntialias = !this.isAntialias;
		this.repaint();
	}

	/**
	 * ラベルの影を切り替えます。
	 */
	public void switchLabelShadow() {
		this.isLabelShadow = !this.isLabelShadow;
		this.repaint();
	}
	
	private boolean isLabelShadow;
	
	/**
	 * 最短経路を求めるアルゴリズムを切り替える
	 * @param algorithm 切り替えるための文字列
	 */
	public void switchShortestPathAlgorithm(String algorithm) {
		if(algorithm.endsWith("_a*")) {
			this.algorithm = new UnidirectShortestPathSearch(new DirectDistance(), this.maps);
		} else if (algorithm.endsWith("_dijkstra")) {
			this.algorithm = new UnidirectShortestPathSearch(null, this.maps);
		}
		this.reroute();
	}

	/**
	 * 拡大縮小を行う
	 * 
	 * @param x
	 * @param y
	 * @param d
	 */
	public void zoom(int x, int y, int d) {
		float newScale = this.scale * (1 + d * MapPanel.SCALE_SENSE);
		if (newScale > MapPanel.SCALE_MAX) {
			newScale = MapPanel.SCALE_MAX;
		} else if (newScale < MapPanel.SCALE_MIN) {
			newScale = MapPanel.SCALE_MIN;
		}
		y = this.getHeight() - y;
		this.screen.x = (int) (this.screen.x + x / this.scale - x / newScale);
		this.screen.y = (int) (this.screen.y + y / this.scale - y / newScale);

		this.scale = newScale;

		this.repaint();
	}
	
	/**
	 * 数値地図25000を表示する倍率まで拡大縮小
	 * @param x
	 * @param y
	 * @param d
	 */
	public void zoomSdf(int x, int y) {
		float newScale = MapPanel.SCALE_SDF * 1.5f;
		if (newScale > MapPanel.SCALE_MAX) {
			newScale = MapPanel.SCALE_MAX;
		} else if (newScale < MapPanel.SCALE_MIN) {
			newScale = MapPanel.SCALE_MIN;
		}
		y = this.getHeight() - y;
		this.screen.x = (int) (this.screen.x + x / this.scale - x / newScale);
		this.screen.y = (int) (this.screen.y + y / this.scale - y / newScale);
		this.scale = newScale;
		this.repaint();
	}
}