package view;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Point;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.Stroke;
import java.awt.geom.GeneralPath;
import java.awt.print.Printable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import util.Setting;

import labeling.Label;
import labeling.Labeling;
import labeling.Labels;
import labeling.SimpleLabeling;
import map.CityMap;
import map.model.City;
import map.model.Curve;
import map.model.Mesh;
import map.model.Node;
import map.model.Road;
import map.route.DirectDistance;
import map.route.SearchThread;
import map.route.ShortestPathAlgorithm;
import map.route.UnidirectShortestPathSearch;
import database.CodeDatabase;

import jp.sourceforge.ma38su.gui.ExportableComponent;
import jp.sourceforge.ma38su.util.Log;

/**
 * 地図データを表示するためのパネル
 * @author ma38su
 */
public class MapPanel extends ExportableComponent implements Printable {
	
	/**
	 * 市区町村の行政界
	 */
	private static Color COLOR_CITY_BORDER = new Color(161, 159, 156);

	/**
	 * 国土数値情報による塗りつぶし色
	 */
	private static Color COLOR_GROUND = new Color(242, 239, 233);
	
	/**
	 * 領域の境界
	 */
	private static Color COLOR_GROUND_BORDER = new Color(128, 128, 128);
	
	/**
	 * 高速道路の塗りつぶし色
	 */
	private static Color COLOR_HIGHWAY = new Color(150, 163, 230);

	/**
	 * 高速道路の境界色
	 */
	private static Color COLOR_HIGHWAY_BORDER = new Color(104, 118, 190);
	
	/**
	 * 名称の（データが）ある道路の塗りつぶし色
	 */
	private static Color COLOR_MAINROAD = new Color(255, 247, 165);

	/**
	 * 名称の（データが）ある道路の境界色
	 */
	private static Color COLOR_MAINROAD_BORDER = new Color(175, 163, 143);

	/**
	 * JR以外の鉄道の塗りつぶし色
	 */
	private static Color COLOR_OTHER_RAIL = new Color(110, 110, 110);

	/**
	 * 一般道のルート色
	 */
	private static Color COLOR_ROUTE = Color.YELLOW;

	/**
	 * 高速道路のルート色
	 */
	private static Color COLOR_ROUTE_HIGHWAY = Color.RED;

	/**
	 * 水域を塗りつぶす色
	 */
	private static Color COLOR_SEA = new Color(153, 179, 204);

	/**
	 * 水域の境界色
	 */
	private static Color COLOR_SEA_BORDER = MapPanel.COLOR_SEA.darker();

	/**
	 * 駅の塗りつぶし色
	 */
	private static Color COLOR_STATION = new Color(242, 133, 133);
	
	/**
	 * フォントの種類（論理フォント）
	 * Serif, SansSerif, Monospaced, Dialog, DialogInput
	 */
	private static final String FONT_FAMILY = "SansSerif";
	
	/**
	 * ラベリングに用いるフォント
	 */
	public static final Font FONT_INFO = new Font(MapPanel.FONT_FAMILY, Font.PLAIN, 12);

	/**
	 * ラベリングに用いるフォント
	 */
	private static final Font FONT_LABEL = new Font(MapPanel.FONT_FAMILY, Font.PLAIN, 11);

	/**
	 * 駅のフォント
	 */
	private static final Font FONT_STATION = new Font(MapPanel.FONT_FAMILY, Font.BOLD, 14);
	
	/**
	 * 市区町村名表示フォントの最大サイズ
	 */
	private static final int FONTSIZE_CITY_MAX = 38;
	
	/**
	 * 都道府県名表示フォントの最大サイズ
	 */
	private static final int FONTSIZE_PREFECTURE_MAX = 60;

	/**
	 * 施設
	 */
	public static final int LABEL_FACILITY = 4;

	/**
	 * 地名
	 */
	public static final int LABEL_PLACE = 2;

	/**
	 * 市区町村名、都道府県名
	 */
	public static final int lABEL_PLACE_GOVT = 8;

	/**
	 * 駅名
	 */
	public static final int LABEL_STATION = 1;

	private static final int MAX_SCREEN_X = 154 * 3600000;

	private static final int MAX_SCREEN_Y = 46  * 3600000;

	private static final int MIN_SCREEN_X = 122 * 3600000;

	private static final int MIN_SCREEN_Y = 20  * 3600000;

	/**
	 * この倍率以下で国土数値情報の都道府県界を表示します。
	 */
	private static final float MODE_PREF_SCALE = 0.00016f;

	/**
	 * 国土数値情報と数値地図25000を切り替える倍率
	 */
	private static final float MODE_SDF_SCALE = 0.0020f;

	/**
	 * この倍率以下でFreeGISを表示します。
	 */
	private static final float MODE_WORLD_SCALE = 0.000025f;

	/**
	 * 表示倍率の上限
	 */
	private static final float SCALE_MAX = 0.05f;

	/**
	 * 表示倍率の下限
	 */
	private static final float SCALE_MIN = 0.000001f;

	/**
	 * 表示倍率の変更精度
	 */
	private static final float SCALE_SENSE = 0.08f;

	/**
	 * 道路区間，鉄道区間の描画の基本の幅
	 */
	private static final int STROKE_WIDTH = 130;
	
	/**
	 * 最短経路探索アルゴリズム
	 */
	private ShortestPathAlgorithm algorithm;

	/**
	 * 市区町村境界のストローク
	 */
	private final Stroke border;

	/**
	 * ポリゴン描画のためのキャッシュ
	 */
	private List<Polygon> cachePolygon = new ArrayList<Polygon>();

	/**
	 * ポリゴン、折れ線描画のためのX座標のキャッシュ
	 */
	private int[] cacheX = new int[15000];

	/**
	 * ポリゴン、折れ線描画のためのY座標のキャッシュ
	 */
	private int[] cacheY = new int[15000];

	/**
	 * 市区町村番号から市区町村名を検索するためのクラス
	 * ハッシュマップを持つ
	 */
	private CodeDatabase db;

	/**
	 * 描画する道路のレベル
	 */
	private int drawlevel;

	/**
	 * アンチエイリアスのフラグ
	 */
	private boolean isAntialias;

	/**
	 * 経度・緯度の表示
	 */
	private boolean isAxis;
	
	/**
	 * 高速道路標示のフラグ
	 */
	private boolean isHighway;

	/**
	 * ラベル表示のフラグ
	 */
	private int isLabel;

	private boolean isLabelFailure;
	
	/**
	 * メッシュ表示のフラグ
	 */
	private boolean isMesh;

	/**
	 * 頂点表示のフラグ
	 */
	private boolean isNodeView;

	/**
	 * マウス操作のフラグ
	 */
	private boolean isOperation;
	
	/**
	 * 鉄道表示のフラグ
	 */
	private boolean isRailway;

	/**
	 * 再描画フラグ
	 */
	private boolean isRepaint;
	
	/**
	 * 河川の表示
	 */
	private boolean isRiver;

	/**
	 * 道路の表示
	 */
	private boolean isRoadway;

	/**
	 * レンダリング
	 */
	private boolean isShadow;

	/**
	 * ラベリングアルゴリズム
	 */
	private Labeling labeling;

	/**
	 * 地図情報管理マップ
	 */
	private CityMap maps;

	private final String[] name = new String[]{"北海道", "青森県", "岩手県", "宮城県", "秋田県", "山形県", "福島県", "茨城県", "栃木県", "群馬県", "埼玉県", "千葉県", "東京都", "神奈川県", "新潟県", "富山県", "石川県", "福井県", "山梨県", "長野県", "岐阜県", "静岡県", "愛知県", "三重県", "滋賀県", "京都府", "大阪府", "兵庫県", "奈良県", "和歌山県", "鳥取県", "島根県", "岡山県", "広島県", "山口県", "徳島県", "香川県", "愛媛県", "高知県", "福岡県", "佐賀県", "長崎県", "熊本県", "大分県", "宮崎県", "鹿児島県", "沖縄県"};

	/**
	 * 頂点のサイズ
	 */
	private int nodeSize = 3;
	
	/**
	 * オフスクリーンイメージ
	 */
	private Image offs;

	/**
	 * 都道府県ポリゴン
	 */
	private Polygon[][] prefectures;

	/**
	 * 地図の表示倍率
	 */
	private float scale = 0.005f;
	
	/**
	 * スクリーンサイズ
	 */
	private Rectangle screen;

	/**
	 * 探索の始点
	 */
	private Node start;
	
	/**
	 * 探索の終点
	 */
	private Node terminal;

	/**
	 * 探索を行うスレッド
	 */
	private SearchThread thread;
	
	/**
	 * 世界地図ポリゴン
	 */
	private Polygon[][] world;
	
	/**
	 * 世界地図の標準範囲
	 */
	private final Rectangle WORLD_SCREEN = new Rectangle(-648000000, -324000000, 1296000000, 648000000);

	private final Rectangle WORLD_SCREEN_EAST = new Rectangle(648000000, -324000000, 1296000000, 648000000);

	/**
	 * 地図パネル
	 * @param db 市区町村番号データベース
	 * @param styledir スタイルファイルのディレクトリ
	 * @param ver バージョン
	 */
	public MapPanel(CodeDatabase db, String styledir) {
		this.isShadow = true;
		this.border = new BasicStroke(0.5f);
		this.isAxis = true;
		this.isRailway = true;
		this.isMesh = true;
		this.isRiver = true;
		this.isRoadway = true;
		this.db = db;
		this.screen = new Rectangle();
		this.labeling = new SimpleLabeling(this.screen);
		this.isHighway = true;
		this.isLabelFailure = false;
		this.readStyle(styledir + "style.ini");
	}

	private void readStyle(String stylefile) {
		System.out.println(stylefile);
		Setting setting = new Setting(stylefile);
		try {
			Color CITY_BORDER = new Color(Integer.parseInt(setting.get("CITY_BORDER"), 16));
			Color GROUND = new Color(Integer.parseInt(setting.get("GROUND"), 16));
			Color GROUND_BORDER = new Color(Integer.parseInt(setting.get("GROUND_BORDER"), 16));
			Color HIGHWAY = new Color(Integer.parseInt(setting.get("HIGHWAY"), 16));
			Color HIGHWAY_BORDER = new Color(Integer.parseInt(setting.get("HIGHWAY_BORDER"), 16));
			Color MAINROAD = new Color(Integer.parseInt(setting.get("MAINROAD"), 16));
			Color MAINROAD_BORDER = new Color(Integer.parseInt(setting.get("MAINROAD_BORDER"), 16));
			Color OTHER_RAIL = new Color(Integer.parseInt(setting.get("OTHER_RAIL"), 16));
			Color ROUTE = new Color(Integer.parseInt(setting.get("ROUTE"), 16));
			Color ROUTE_HIGHWAY = new Color(Integer.parseInt(setting.get("ROUTE_HIGHWAY"), 16));
			Color SEA = new Color(Integer.parseInt(setting.get("SEA"), 16));
			Color SEA_BORDER = new Color(Integer.parseInt(setting.get("SEA_BORDER"), 16));
			Color STATION = new Color(Integer.parseInt(setting.get("STATION"), 16));

			MapPanel.COLOR_CITY_BORDER = CITY_BORDER;
			MapPanel.COLOR_GROUND = GROUND;
			MapPanel.COLOR_GROUND_BORDER = GROUND_BORDER;
			MapPanel.COLOR_HIGHWAY = HIGHWAY;
			MapPanel.COLOR_HIGHWAY_BORDER = HIGHWAY_BORDER;
			MapPanel.COLOR_MAINROAD = MAINROAD;
			MapPanel.COLOR_MAINROAD_BORDER = MAINROAD_BORDER;
			MapPanel.COLOR_OTHER_RAIL = OTHER_RAIL;
			MapPanel.COLOR_ROUTE = ROUTE;
			MapPanel.COLOR_ROUTE_HIGHWAY = ROUTE_HIGHWAY;
			MapPanel.COLOR_SEA = SEA;
			MapPanel.COLOR_SEA_BORDER = SEA_BORDER;
			MapPanel.COLOR_STATION = STATION;
		} catch (Throwable e) {
			e.printStackTrace();
		}
	}
	
	private final int STROKE_JOIN = BasicStroke.JOIN_ROUND;
	private final int STROKE_CAP = BasicStroke.CAP_BUTT;

	/**
	 * 地図の描画
	 * @param g
	 */
	@Override
	public void draw(Graphics2D g) {
		this.labeling.init(g, this.scale, this.getWidth(), this.getHeight(), this.isShadow, this.isLabelFailure);
		Stroke defaultStroke = g.getStroke();
		String mode = null;
		switch (this.mode()) {
		case 0 :
			mode = "MODE: 世界地図";
			break;
		case 1 : 
			mode = "MODE: 国土数値情報（都道府県）";
			break;
		case 2 :
			mode = "MODE: 国土数値情報（都道府県＋市区町村）";
			break;
		default :
			mode = "MODE: 数値地図25000＋数値地図2500";
		}
		g.setFont(MapPanel.FONT_INFO);
		this.labeling.set(mode, 5, 2);
		
		StringBuilder sb = new StringBuilder("SCALE: ");
		sb.append((int)(this.scale / 10 * 1000000 * 10) / 10f);
		sb.append("µ");
		this.labeling.set(sb.toString(), 5, 17);

		g.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_OFF);

		g.setColor(this.getBackground());
		g.fillRect(0, 0, this.getWidth(), this.getHeight());
		g.setColor(MapPanel.COLOR_SEA);
		this.drawBackground(g);

		this.setPrefectrueFont(g);

		g.setColor(MapPanel.COLOR_GROUND);
		this.fillPolygonWorld(g, this.world[0]);
		
		if (this.scale < MapPanel.MODE_WORLD_SCALE) {

			g.setColor(MapPanel.COLOR_GROUND);
			this.fillPolygonWorld(g, this.world[1]);
			
		} else if(this.scale < MapPanel.MODE_PREF_SCALE) {
			if ((this.isLabel & MapPanel.lABEL_PLACE_GOVT) != 0) {
				for (int i = 0; i < this.prefectures.length; i++) {
					this.fillPrefectures(g, this.name[i], this.prefectures[i]);
				}
			} else {
				for (int i = 0; i < this.prefectures.length; i++) {
					this.fillPolygon(g, this.prefectures[i], MapPanel.COLOR_GROUND, MapPanel.COLOR_GROUND_BORDER);
				}
			}
		} else {
			if (!this.isOperation) {
				if (this.isAntialias) {
					g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
				}
			}
			g.setColor(MapPanel.COLOR_GROUND);
			if (this.scale >= MapPanel.MODE_SDF_SCALE) {
				this.setCityFont(g);
				this.drawSdf(g);
			} else {
				g.setStroke(new BasicStroke(1.5f, this.STROKE_CAP, this.STROKE_JOIN));
				if ((this.isLabel & MapPanel.lABEL_PLACE_GOVT) != 0) {
					for (int i = 0; i < this.prefectures.length; i++) {
						this.fillPrefectures(g, this.name[i], this.prefectures[i]);
					}
				} else {
					for (int i = 0; i < this.prefectures.length; i++) {
						this.fillPolygon(g, this.prefectures[i], MapPanel.COLOR_GROUND, MapPanel.COLOR_GROUND_BORDER);
					}
				}
				this.setCityFont(g);
				g.setStroke(this.border);
				synchronized (this.maps) {
					for (City city : this.maps.values()) {
						this.drawBorder(g, city);
					}
				}
				g.setStroke(defaultStroke);
				this.drawRoute(g);
			}
		}

		g.setColor(Color.BLACK);
		if (this.isAxis) {
			this.drawAxis(g);
		}
		this.labeling.draw();
	}
	
	/**
	 * 経度緯度を描画します。
	 * @param g 
	 */
	private void drawAxis(Graphics2D g) {
		int step;
		switch (this.mode()) {
			case 0: step = 10;
			break;
			case 3: step = 1;
			break;
			default: step = 5;
		}
		step *= 3600000;
		int height = this.getHeight();
		int width = this.getWidth();
		int sy = height + (int)((324000000 + this.screen.y) * this.scale);
		int ey = height - (int)((324000000 - this.screen.y) * this.scale);
		for (int y = -324000000; y <= 324000000; y += step) {
			int ty = height - (int)((y - this.screen.y) * this.scale);
			if (ty < 0) {
				break;
			} else if (ty > height){
				continue;
			}
			g.drawLine(0, ty, width, ty);
		}
		for (int x = -648000000;; x += step) {
			int tx = (int)((x - this.screen.x) * this.scale);
			if (tx < 0) {
				continue;
			} else if (tx > width){
				break;
			}
			g.drawLine(tx, sy, tx, ey);
		}
	}

	/**
	 * 背景の描画
	 * @param g
	 */
	private void drawBackground(Graphics2D g) {
		int height = this.getHeight();
		int sy = height - (int)((90 * 3600000 - this.screen.y) * this.scale);
		if (sy < 0) {
			sy = 0;
		}
		int ey = height - (int)((-90 * 3600000 - this.screen.y) * this.scale);
		if (ey > height) {
			ey = height;
		}
		g.fillRect(0, sy, this.getWidth(), (ey - sy));
	}

	/**
	 * 市区町村の行政界と市区町村名を描画します。
	 * 塗りつぶしは行いません。
	 * @param g 描画するGraphics2D
	 * @param city 描画する行政界
	 */
	private void drawBorder(Graphics2D g, City city) {
		Polygon[] polygons = city.getPolygon();
		if (polygons != null) {
			g.setColor(MapPanel.COLOR_GROUND_BORDER);
			for (Polygon polygon : polygons) {
				if(polygon.intersects(this.screen)) {
					int[] aryX = polygon.xpoints;
					int[] aryY = polygon.ypoints;

					for (int j = 0; j < polygon.npoints; j++) {
						this.cacheX[j] = (int)((aryX[j] - this.screen.x) * this.scale);
						this.cacheY[j] = this.getHeight() - (int)((aryY[j] - this.screen.y) * this.scale);
					}
					g.drawPolygon(this.cacheX, this.cacheY, polygon.npoints);
				}
			}
			g.setColor(Color.BLACK);
			String name = city.getName();
			if (name == null) {
				name = this.db.get(city.getCode());
				if (name == null) {
					name = "";
					throw new UnknownError();
				}
				city.setName(name);
			}
			if ((this.isLabel & MapPanel.lABEL_PLACE_GOVT) != 0 && !"".equals(name)) {
				int x = (int) ((city.getX() - this.screen.x) * this.scale);
				int y = this.getHeight() - (int) ((city.getY() - this.screen.y) * this.scale);
				this.labeling.add(name, x, y, true);
			}
		}
	}
	
	/**
	 * 曲線の描画
	 * 
	 * @param g 描画するGraphics
	 * @param curves 描画する行政界
	 */
	private void drawCurve(Graphics2D g, final Curve[] curves) {
		if (curves != null) {
			for (final Curve curve : curves) {
				final int[] aryX = curve.getArrayX();
				final int[] aryY = curve.getArrayY();
				int x0 = (int) ((aryX[0] - this.screen.x) * this.scale);
				int y0 = this.getHeight() - (int) ((aryY[0] - this.screen.y) * this.scale);
				for (int k = 1; k < aryX.length; k++) {
					int x = (int) ((aryX[k] - this.screen.x) * this.scale);
					int y = this.getHeight() - (int) ((aryY[k] - this.screen.y) * this.scale);
					// 表示領域内
					if (this.screen.intersectsLine(aryX[k - 1], aryY[k - 1], aryX[k], aryY[k])) {
						g.drawLine(x0, y0, x, y);
					}
					x0 = x;
					y0 = y;
				}
			}
		}
	}
	
	/**
	 * 線分を描画します。
	 * @param g 描画するGraphics2D
	 * @param x1 始点のX座標
	 * @param y1 始点のY座標
	 * @param x2 終点のX座標
	 * @param y2 終点のY座標
	 * @param line 境界色
	 */
	public void drawLine(Graphics2D g, int x1, int y1, int x2, int y2) {
		int tx1 = (int)((x1 - this.screen.x) * this.scale);
		int ty1 = this.getHeight() - (int)((y1 - this.screen.y) * this.scale);
		int tx2 = (int)((x2 - this.screen.x) * this.scale);
		int ty2 = this.getHeight() - (int)((y2 - this.screen.y) * this.scale);
		g.drawLine(tx1, ty1, tx2, ty2);
	}

	/**
	 * メッシュを描画します。
	 * @param g
	 * @param size サイズ
	 * @param area 描画範囲
	 * @param mesh メッシュ
	 */
	private void drawMesh(Graphics2D g, int size, Rectangle area, Mesh[][] mesh) {
		int meshX = area.x;
		for (int x = 0; x < mesh.length; x++) {
			int meshY = area.y;
			for (int y = 0; y < mesh[x].length; y++) {
				if (mesh[x][y] != null) {
					if (this.screen.intersects(meshX, meshY, Mesh.SIZE, Mesh.SIZE)) {
						int tx = (int) ((meshX - this.screen.x) * this.scale);
						int ty = this.getHeight() - (int) ((meshY + Mesh.SIZE - this.screen.y) * this.scale);
						g.setColor(mesh[x][y].getColor());
						g.fillRect(tx, ty, size, size);
					}
				}
				meshY += Mesh.SIZE;
			}
			meshX += Mesh.SIZE;
		}
	}
	
	/**
	 * ポリゴンを描画します。
	 * @param g 描画するGraphics2D
	 * @param polygons 描画するポリゴン
	 */
	void drawPolygon(Graphics2D g, Polygon[] polygons) {
		if (polygons == null) {
			return;
		}
		for (Polygon polygon : polygons) {
			if(polygon.intersects(this.screen)) {
				int[] aryX = polygon.xpoints;
				int[] aryY = polygon.ypoints;

				for (int j = 0; j < polygon.npoints; j++) {
					this.cacheX[j] = (int)((aryX[j] - this.screen.x) * this.scale);
					this.cacheY[j] = this.getHeight() - (int)((aryY[j] - this.screen.y) * this.scale);
				}
				g.drawPolygon(this.cacheX, this.cacheY, polygon.npoints);
			}
		}
	}

	/**
	 * 折れ線を描画します。
	 * @param g 
	 * @param curves 曲線
	 */
	private void drawPolyline(Graphics2D g, Curve[] curves) {
		if (curves != null) {
			for (Curve curve : curves) {
				int[] aryX = curve.getArrayX();
				int[] aryY = curve.getArrayY();
				for (int i = 0; i < aryX.length; i++) {
					this.cacheX[i] = (int) ((aryX[i] - this.screen.x) * this.scale);
					this.cacheY[i] = this.getHeight() - (int) ((aryY[i] - this.screen.y) * this.scale);
				}
				g.drawPolyline(this.cacheX, this.cacheY, aryX.length);
			}
		}
	}

	/**
	 * 折れ線の描画をおこないます。
	 * @param g
	 * @param listX
	 * @param listY
	 * @param color
	 * @param stroke
	 */
	private void drawPolyLine(Graphics2D g, List<int[]> listX, List<int[]> listY) {
		for (int i = 0; i < listX.size(); i++) {
			int[] aryX = listX.get(i);
			g.drawPolyline(aryX, listY.get(i), aryX.length);
		}
	}

	/**
	 * Rectangleを描画します。
	 * @param g 描画するGraphics2D
	 * @param rect 描画するRectangle
	 * @param polygons 描画するポリゴン
	 * @param bg 背景色
	 * @param line 境界色
	 */
	void drawRectangle(Graphics2D g, Rectangle rect) {
		int rectX = (int)((rect.x - this.screen.x) * this.scale);
		float height = rect.height * this.scale;
		int rectY = this.getHeight() - (int)((rect.y - this.screen.y) * this.scale + height);
		g.drawRect(rectX, rectY, (int)(rect.width * this.scale), (int) (height));
	}

	/**
	 * 道路の描画
	 * 
	 * @param g Graphics2D
	 * @param path 
	 * @param width 
	 */
	private void drawRoadway(Graphics2D g, GeneralPath[] path, int[] width) {
		if (this.isRoadway) {
			g.setColor(Color.LIGHT_GRAY);
			for (int i = 0; i < 6; i++) {
				if(path[i] != null) {
					g.setStroke(new BasicStroke(width[i] + 2, this.STROKE_CAP, this.STROKE_JOIN));
					g.draw(path[i]);
				}
			}
			g.setColor(MapPanel.COLOR_MAINROAD_BORDER);
			for (int i = 0; i < 6; i++) {
				if(path[i] != null) {
					g.setStroke(new BasicStroke(width[i] + 2, this.STROKE_CAP, this.STROKE_JOIN));
					g.draw(path[i + 6]);
				}
			}

			g.setColor(Color.WHITE);
			for (int i = 0; i < 6; i++) {
				if(path[i] != null) {
					g.setStroke(new BasicStroke(width[i], this.STROKE_CAP, this.STROKE_JOIN));
					g.draw(path[i]);
				}
			}

			g.setColor(MapPanel.COLOR_MAINROAD);
			for (int i = 0; i < 6; i++) {
				if(path[i] != null) {
					g.setStroke(new BasicStroke(width[i], this.STROKE_CAP, this.STROKE_JOIN));
					g.draw(path[i + 6]);
				}
			}
		}
		
		if (this.isHighway) {
			g.setColor(MapPanel.COLOR_HIGHWAY_BORDER);
			for (int i = 0; i < 6; i++) {
				g.setStroke(new BasicStroke(width[i] + 3, this.STROKE_CAP, this.STROKE_JOIN));
				g.draw(path[i + 12]);
			}
			g.setColor(MapPanel.COLOR_HIGHWAY);
			for (int i = 0; i < 6; i++) {
				g.setStroke(new BasicStroke(width[i] + 1, this.STROKE_CAP, this.STROKE_JOIN));
				g.draw(path[i + 12]);
			}
		}
	}

	/**
	 * 探索済み頂点を描画する
	 * 
	 * @param g
	 */
	private void drawRoute(Graphics2D g) {
		if (this.thread != null) {
			Collection<Road> route = this.thread.getRoute();
			if (route != null) {
				int[] width = new int[12];
				for (int i = 0; i < 6; i++) {
					width[i] = (int)((i + 1) * MapPanel.STROKE_WIDTH * this.scale) + 1;
				}
				for (Road road : route) {
					int[] aryX = road.getArrayX();
					int[] aryY = road.getArrayY();
		
					int x0 = (int) ((aryX[0] - this.screen.x) * this.scale);
					int y0 = this.getHeight() - (int) ((aryY[0] - this.screen.y) * this.scale);

					if (road.getType() == 3) {
						g.setColor(MapPanel.COLOR_ROUTE_HIGHWAY);
					} else {
						g.setColor(MapPanel.COLOR_ROUTE);
					}
					g.setStroke(new BasicStroke(width[road.getWidth()], this.STROKE_CAP, this.STROKE_JOIN));
		
					for (int k = 1; k < aryX.length; k++) {
		
						int x = (int) ((aryX[k] - this.screen.x) * this.scale);
						int y = this.getHeight() - (int) ((aryY[k] - this.screen.y) * this.scale);

						// 表示領域内
						if (this.screen.intersectsLine(aryX[k - 1], aryY[k - 1], aryX[k], aryY[k])) {
							g.drawLine(x0, y0, x, y);
						}
						x0 = x;
						y0 = y;
					}
				}
			}
		}
		int r = (int) (this.scale * 500) + 2;
		g.setStroke(new BasicStroke(r / 2f, this.STROKE_CAP, this.STROKE_JOIN));
		if (this.start != null) {
			int x = (int) ((this.start.getX() - this.screen.x) * this.scale) - r;
			int y = this.getHeight() - (int) ((this.start.getY() - this.screen.y) * this.scale) - r;
			g.setColor(Color.YELLOW);
			g.fillOval(x, y, r * 2, r * 2);
			g.setColor(Color.BLACK);
			g.drawOval(x, y, r * 2, r * 2);
		}
		if (this.terminal != null) {
			int x = (int) ((this.terminal.getX() - this.screen.x) * this.scale) - r;
			int y = this.getHeight() - (int) ((this.terminal.getY() - this.screen.y) * this.scale) - r;
			g.setColor(Color.RED);
			g.fillOval(x, y, r * 2, r * 2);
			g.setColor(Color.BLACK);
			g.drawOval(x, y, r * 2, r * 2);
		}
	}

	
	/**
	 * 数値地図を描画する
	 * @param g
	 */
	public void drawSdf(Graphics2D g) {
		synchronized (this.maps) {
			for (int code = 1; code <= 47; code++) {
				if (!this.maps.hasPrefecture(code)) {
					this.fillPrefectures(g, this.name[code - 1], this.prefectures[code - 1]);
				}
			}
		}
		// メッシュの描画
		synchronized (this.maps) {
			if (this.isMesh) {
				for (final City map : this.maps.values()) {
					if(map.hasData()) {
						int meshSize = (int) (Mesh.SIZE * this.scale) + 1;
						Rectangle area = map.getArea();
						Mesh[][] mesh = map.getMesh();
						this.drawMesh(g, meshSize, area, mesh);
					} else {
						this.fillBorder(g, map);
					}
				}
			}

			for (final City map : this.maps.values()) {
				if (!map.hasData()) {
					continue;
				}
	
				// 海岸線を描画
				g.setColor(MapPanel.COLOR_SEA_BORDER);
				this.drawCurve(g, map.getSeaBorder());

				// 数値地図2500
				if (this.isRiver) {
					if (map.isSdk2500()) {
						g.setColor(MapPanel.COLOR_SEA);
						this.fillPolygon(g, map.getMizu());
						
						g.setColor(MapPanel.COLOR_SEA_BORDER);
						this.drawCurve(g, map.getMizuArc());
					} else {
						g.setColor(MapPanel.COLOR_SEA_BORDER);
						this.drawCurve(g, map.getCoast());
	
						g.setColor(MapPanel.COLOR_SEA_BORDER);
						this.drawCurve(g, map.getRiver());
					}
				}
			}
		}
		{
			int[] roadWidth = new int[6];

			GeneralPath[] road = new GeneralPath[18];
			for (int i = 5; i >= 0; i--) {
				roadWidth[i] = (int)((i + 1) * MapPanel.STROKE_WIDTH * this.scale);
				if(roadWidth[i] > 0) {
					road[i] = new GeneralPath();
					this.drawlevel = i;
				}
				road[i + 6] = new GeneralPath();
				road[i + 12] = new GeneralPath();
			}

			GeneralPath jr = new GeneralPath();
			List<int[]> otherX = new ArrayList<int[]>();
			List<int[]> otherY = new ArrayList<int[]>();
			List<int[]> stationX = new ArrayList<int[]>();
			List<int[]> stationY = new ArrayList<int[]>();

			synchronized (this.maps) {
				for (final City map : this.maps.values()) {
					if (!map.hasData()) {
						continue;
					}
					for (int i = 17; i >= 0; i--) {
						this.extractRoadway(road[i], map.getRoad()[i]);
					}
					this.extractGeneralPath(map.getJRRailway(), jr);
					this.extractPolyLine(map.getOtherRailway(), otherX, otherY);
					this.extractPolyLine(map.getStation(), stationX, stationY);
				}
			}
			
			this.drawRoadway(g, road, roadWidth);

			if (this.isRailway) {
				int w = (int)(MapPanel.STROKE_WIDTH * this.scale * 4f + 0.5f);
				if (w == 0) {
					w = 1;
				}

				Stroke stroke = new BasicStroke(w + 2, this.STROKE_CAP, this.STROKE_JOIN);
				g.setColor(MapPanel.COLOR_OTHER_RAIL);
				g.setStroke(stroke);
				this.drawPolyLine(g, otherX, otherY);
				
				g.setColor(Color.BLACK);
				g.draw(jr);
				
				g.setColor(Color.WHITE);
				g.setStroke(new BasicStroke(w, this.STROKE_CAP, this.STROKE_JOIN, 10f, new float[]{w * 8, w * 8}, 0.25f));
				g.draw(jr);

				w *= 3;
				g.setColor(MapPanel.COLOR_STATION.darker());
				g.setStroke(new BasicStroke(w + 2, this.STROKE_CAP, this.STROKE_JOIN));
				this.drawPolyLine(g, stationX, stationY);
				
				g.setColor(MapPanel.COLOR_STATION);
				g.setStroke(new BasicStroke(w, this.STROKE_CAP, this.STROKE_JOIN));
				this.drawPolyLine(g, stationX, stationY);
			}
		}
		synchronized (this.maps) {
			g.setStroke(this.border);
			for (final City city : this.maps.values()) {
				if (!city.hasData()) {
					continue;
				}
				g.setColor(MapPanel.COLOR_CITY_BORDER);
				this.drawSdfBorder(g, city);
			}
		}
		if (this.isNodeView) {
			synchronized (this.maps) {
				for (final City map : this.maps.values()) {
					if (!map.hasData()) {
						continue;
					}	
					int nodeSize = (int) (this.nodeSize / this.scale);
	
					g.setColor(Color.GRAY);
	
					for (Node node : map.getNode()) {
						// 表示領域外であれば次へ
						if (!this.screen.intersects(node.getX() - nodeSize, node.getY() - nodeSize, nodeSize * 2, nodeSize * 2)) {
							continue;
						}
	
						int x = (int) ((node.getX() - this.screen.x) * this.scale);
						int y = this.getHeight() - (int) ((node.getY() - this.screen.y) * this.scale);
	
						// draw node
						g.fillRect(x - this.nodeSize, y - this.nodeSize, this.nodeSize * 2, this.nodeSize * 2);
					}
				}
			}
		}
		this.drawRoute(g);
		g.setStroke(new BasicStroke(1, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));

		g.setFont(MapPanel.FONT_LABEL);
		if (this.isLabel > 0) {
			synchronized (this.maps) {
				for (City map : this.maps.values()) {
					if (map.hasData()) {	
						if ((this.isLabel & MapPanel.LABEL_STATION) != 0) {
							Font df = g.getFont();
							g.setFont(MapPanel.FONT_STATION);
							this.labeling.add(map.getStation());
							g.setFont(df);
						}
						if ((this.isLabel & MapPanel.LABEL_PLACE) != 0) {
							this.labeling.add(map.getLabel(Label.KEY_CM));
						}
						if ((this.isLabel & MapPanel.LABEL_FACILITY) != 0) {
							this.labeling.add(map.getLabel(Label.KEY_KO));
						}
						for (Labels labels : map.getLabels()) {
							this.labeling.add(labels);
						}
					}
				}
			}
		}
	}

	/**
	 * 市区町村の行政界と市区町村名を描画します。
	 * @param g 描画するGraphics2D
	 * @param city 描画する行政界
	 * @param color 
	 * @param labeling 
	 * @param bg 背景色
	 * @param line 境界色
	 */
	private void drawSdfBorder(Graphics2D g, City city) {
		g.setColor(MapPanel.COLOR_GROUND_BORDER);
		this.drawPolyline(g, city.getBorder());
		String name = city.getName();
		if (name == null) {
			name = this.db.get(city.getCode());
			if (name == null) {
				name = "";
				throw new UnknownError();
			}
			city.setName(name);
		}
		if ((this.isLabel & MapPanel.lABEL_PLACE_GOVT) != 0 && !"".equals(name)) {
			int x = (int) ((city.getX() - this.screen.x) * this.scale);
			int y = this.getHeight() - (int) ((city.getY() - this.screen.y) * this.scale);
			this.labeling.add(name, x, y, false);
		}
	}

	/**
	 * 白抜き文字を描画します。
	 * @param g
	 * @param msg
	 * @param x
	 * @param y
	 * @param color
	 */
	void drawString(Graphics g, String msg, int x, int y, Color color) {
		g.setColor(color);
		g.drawString(msg, x + 1, y + 1);
		g.drawString(msg, x + 1, y - 1);
		g.drawString(msg, x - 1, y + 1);
		g.drawString(msg, x - 1, y - 1);
		g.drawString(msg, x - 1, y);
		g.drawString(msg, x, y - 1);
		g.drawString(msg, x + 1, y);
		g.drawString(msg, x, y + 1);
		g.setColor(Color.BLACK);
		g.drawString(msg, x, y);
	}

	/**
	 * GeneralPathに展開
	 * 
	 * @param curves 展開する曲線の配列
	 * @param path 展開されたGeneralPath
	 */
	private void extractGeneralPath(final Curve[] curves, final GeneralPath path) {
		if (curves != null) {
			for (final Curve curve : curves) {
				int[] aryX = curve.getArrayX();
				int[] aryY = curve.getArrayY();

				int x = (int) ((aryX[0] - this.screen.x) * this.scale);
				int y = this.getHeight() - (int) ((aryY[0] - this.screen.y) * this.scale);
				path.moveTo(x, y);

				for (int i = 1; i < aryX.length; i++) {
					x = (int) ((aryX[i] - this.screen.x) * this.scale);
					y = this.getHeight() - (int) ((aryY[i] - this.screen.y) * this.scale);
					path.lineTo(x, y);
				}
			}
		}
	}

	/**
	 * 折れ線に展開
	 * 
	 * @param curves
	 * @param polyX 
	 * @param polyY 
	 */
	private void extractPolyLine(final Curve[] curves, final List<int[]> polyX, final List<int[]> polyY) {
		if (curves != null) {
			for (final Curve curve : curves) {
				int[] aryX = curve.getArrayX();
				int[] aryY = curve.getArrayY();
				int[] lineX = new int[aryX.length];
				int[] lineY = new int[aryY.length];
				for (int i = 0; i < aryX.length; i++) {
					lineX[i] = (int) ((aryX[i] - this.screen.x) * this.scale);
					lineY[i] = this.getHeight() - (int) ((aryY[i] - this.screen.y) * this.scale);
				}
				polyX.add(lineX);
				polyY.add(lineY);
			}
		}
	}

	/**
	 * 道路データをGeneralPathに展開する
	 * @param path 展開先のGeneralPath
	 * @param curves 道路データ
	 */
	private void extractRoadway(GeneralPath path, Curve[] curves) {
		if (path != null && curves != null) {
			for (Curve curve : curves) {
				int[] aryX = curve.getArrayX();
				int[] aryY = curve.getArrayY();
	
				int x0 = (int) ((aryX[0] - this.screen.x) * this.scale);
				int y0 = this.getHeight() - (int) ((aryY[0] - this.screen.y) * this.scale);
	
				path.moveTo(x0, y0);
				
				for (int k = 1; k < aryX.length; k++) {
	
					int x = (int) ((aryX[k] - this.screen.x) * this.scale);
					int y = this.getHeight() - (int) ((aryY[k] - this.screen.y) * this.scale);

					// 表示領域外であれば次へ
					if (this.screen.intersectsLine(aryX[k - 1], aryY[k - 1], aryX[k], aryY[k])) {
						path.lineTo(x, y);
					} else {
						path.moveTo(x, y);
					}
				}
			}
		}
	}

	/**
	 * 市区町村の行政界を描画します。
	 * @param g 描画するGraphics2D
	 * @param city 描画する行政界
	 * @param bg 背景色
	 * @param line 境界色
	 */
	private void fillBorder(Graphics2D g, City city) {
		Polygon[] polygons = city.getPolygon();
		if (polygons != null) {
			Polygon[] tmp = new Polygon[polygons.length];
			g.setColor(MapPanel.COLOR_GROUND);
			for (int i = 0; i < polygons.length; i++) {
				if(polygons[i].getBounds().intersects(this.screen)) {
					int[] aryX = polygons[i].xpoints;
					int[] aryY = polygons[i].ypoints;
	
					int[] polygonX = new int[polygons[i].npoints];
					int[] polygonY = new int[polygons[i].npoints];
	
					for (int j = 0; j < polygons[i].npoints; j++) {
						polygonX[j] = (int)((aryX[j] - this.screen.x) * this.scale);
						polygonY[j] = this.getHeight() - (int)((aryY[j] - this.screen.y) * this.scale);
					}
					tmp[i] = new Polygon(polygonX, polygonY, polygonX.length);
					g.fillPolygon(tmp[i]);
				}
			}
			g.setColor(MapPanel.COLOR_GROUND_BORDER);
			for (Polygon p : tmp) {
				if (p != null) {
					g.drawPolygon(p);
				}
			}
			g.setColor(Color.BLACK);
			String name = city.getName();
			if (name == null) {
				name = this.db.get(city.getCode());
				if (name == null) {
					name = "";
				}
				city.setName(name);
			}
			if ((this.isLabel & MapPanel.lABEL_PLACE_GOVT) != 0 && !"".equals(name)) {
				int x = (int) ((city.getX() - this.screen.x) * this.scale);
				int y = this.getHeight() - (int) ((city.getY() - this.screen.y) * this.scale);
				this.labeling.add(name, x, y, true);
			}
		}
	}

	/**
	 * ポリゴンを描画します。
	 * @param g 描画するGraphics2D
	 * @param polygons 描画するポリゴン
	 */
	private void fillPolygon(Graphics2D g, Polygon[] polygons) {
		if (polygons == null) {
			return;
		}
		for (Polygon polygon : polygons) {
			if(polygon.intersects(this.screen)) {
				int[] aryX = polygon.xpoints;
				int[] aryY = polygon.ypoints;

				for (int j = 0; j < polygon.npoints; j++) {
					this.cacheX[j] = (int)((aryX[j] - this.screen.x) * this.scale);
					this.cacheY[j] = this.getHeight() - (int)((aryY[j] - this.screen.y) * this.scale);
				}
				g.fillPolygon(this.cacheX, this.cacheY, polygon.npoints);
			}
		}
	}

	/**
	 * ポリゴンを描画します。
	 * @param g 描画するGraphics2D
	 * @param polygons 描画するポリゴン
	 * @param bg 背景色
	 * @param line 境界色
	 */
	void fillPolygon(Graphics2D g, Polygon[] polygons, Color bg, Color line) {
		if (polygons == null) {
			return;
		}
		g.setColor(bg);
		for (Polygon polygon : polygons) {
			if(polygon.intersects(this.screen)) {
				int[] aryX = polygon.xpoints;
				int[] aryY = polygon.ypoints;
				for (int j = 0; j < polygon.npoints; j++) {
					this.cacheX[j] = (int)((aryX[j] - this.screen.x) * this.scale);
					this.cacheY[j] = this.getHeight() - (int)((aryY[j] - this.screen.y) * this.scale);
				}
				Polygon p = new Polygon(this.cacheX, this.cacheY, polygon.npoints);
				this.cachePolygon.add(new Polygon(this.cacheX, this.cacheY, polygon.npoints));
				g.fillPolygon(p);
			}
		}
		// 境界を描画つぶします。
		g.setColor(line);
		for (Polygon p : this.cachePolygon) {
			g.drawPolygon(p);
		}
		this.cachePolygon.clear();
	}

	/**
	 * ポリゴンを描画します。
	 * @param g 描画するGraphics2D
	 * @param polygons 描画するポリゴン
	 */
	private void fillPolygonWorld(Graphics2D g, Polygon[] polygons) {
		if (polygons == null) {
			return;
		}
		for (int i = 0; i < polygons.length; i++) {
			if(polygons[i].intersects(this.screen)) {
				int[] aryX = polygons[i].xpoints;
				int[] aryY = polygons[i].ypoints;
								
				for (int j = 0; j < polygons[i].npoints; j++) {
					this.cacheX[j] = (int)((aryX[j] - this.screen.x) * this.scale);
					this.cacheY[j] = this.getHeight() - (int)((aryY[j] - this.screen.y) * this.scale);
				}
				Polygon p = new Polygon(this.cacheX, this.cacheY, polygons[i].npoints);
				g.fillPolygon(p);
				this.cachePolygon.add(p);
			}
		}
		if (!this.WORLD_SCREEN.contains(this.screen)) {
			if (this.WORLD_SCREEN_EAST.intersects(this.screen)) {
				Rectangle screen = new Rectangle(this.screen);
				screen.x -= this.WORLD_SCREEN.width;
				long screenX = 1296000000 - this.screen.x;
				for (Polygon polygon : polygons) {
					if(polygon.intersects(screen)) {
						int[] aryX = polygon.xpoints;
						int[] aryY = polygon.ypoints;
						for (int j = 0; j < polygon.npoints; j++) {
							float x = (aryX[j] + screenX) * this.scale;
							this.cacheX[j] = (x >= Integer.MAX_VALUE) ? Integer.MAX_VALUE : (int) x;
							this.cacheY[j] = this.getHeight() - (int)((aryY[j] - this.screen.y) * this.scale);
						}
						Polygon p = new Polygon(this.cacheX, this.cacheY, polygon.npoints);
						g.fillPolygon(p);
						this.cachePolygon.add(p);
					}
				}
			}
		}
		g.setColor(MapPanel.COLOR_GROUND_BORDER);
		for (Polygon p : this.cachePolygon) {
			g.drawPolygon(p);
		}
		this.cachePolygon.clear();
	}
	
	/**
	 * ポリゴンを描画します。
	 * @param g 描画するGraphics2D
	 * @param name 
	 * @param prefecture 描画するポリゴン
	 */
	private void fillPrefectures(Graphics2D g, String name, Polygon[] prefecture) {
		g.setColor(MapPanel.COLOR_GROUND);
		int maxN = 0;
		int lx = 0;
		int ly = 0;
		for (Polygon polygon : prefecture) {
			if(polygon.intersects(this.screen)) {
				int centerX = 0;
				int centerY = 0;
				int[] aryX = polygon.xpoints;
				int[] aryY = polygon.ypoints;
				for (int j = 0; j < polygon.npoints; j++) {
					this.cacheX[j] = (int)((aryX[j] - this.screen.x) * this.scale);
					this.cacheY[j] = this.getHeight() - (int)((aryY[j] - this.screen.y) * this.scale);
					centerX += this.cacheX[j];
					centerY += this.cacheY[j];
				}
				if (maxN < polygon.npoints) {
					maxN = polygon.npoints;
					lx = centerX / maxN;
					ly = centerY / maxN;
				}
				Polygon p = new Polygon(this.cacheX, this.cacheY, polygon.npoints);
				this.cachePolygon.add(new Polygon(this.cacheX, this.cacheY, polygon.npoints));
				g.fillPolygon(p);
			}
		}
		g.setColor(MapPanel.COLOR_GROUND_BORDER);
		for (Polygon p : this.cachePolygon) {
			g.drawPolygon(p);
		}
		if (maxN > 0) {
			this.labeling.add(name, lx, ly, false);
		}
		this.cachePolygon.clear();
	}

	public double getLocationX(int x) {
		return (this.screen.x + x / this.scale) / 3600000;
	}
	
	public double getLocationY(int y) {
		return (this.screen.y + (this.getHeight() - y) / this.scale) / 3600000;
	}

	/**
	 * スクリーン情報の取得
	 * 
	 * @return スクリーンのRectangle
	 */
	public Rectangle getScreen() {
		return this.screen;
	}
	
	/**
	 * 初期設定
	 * @param setting 設定
	 * @param map 地図データ管理クラス
	 * @param world 世界（日本以外）国境ポリゴン
	 * @param japan 日本国境ポリゴン
	 * @param prefectures 都道府県ポリゴン
	 * @param coast 都道府県ポリゴンと重なる水域ポリゴン
	 * @param island 水域ポリゴンと重なる島などのポリゴン
	 */
	public void init(CityMap map, Polygon[][] world, Polygon[][] prefectures) {
		Log.out(this, "init called.");
		this.algorithm = new UnidirectShortestPathSearch(new DirectDistance(), map);

		this.maps = map;
		
		this.world = world;
		this.prefectures = prefectures;
		
		this.isLabel = 15;
		this.isNodeView = false;
		this.isAntialias = true;
		this.isOperation = false;
		this.moveDefault();
	}

	/**
	 * 地図データの読み込みを確認します。
	 * @return 地図データを読み込んでいればtrueを返します。
	 */
	public boolean isLoaded() {
		return this.maps != null;
	}

	/**
	 * 操作しているかどうか確認します。
	 * @return 操作していればtrue
	 */
	public boolean isOperation() {
		return this.isOperation;
	}
	
	/**
	 * 地図の表示状態を確認します。
	 * 3 : 数値地図25000
	 * 2 : 国土数値情報（市区町村）
	 * 1 : 国土数値情報（都道府県）
	 * 0 : FreeGIS（世界地図）
	 * @return 地図の表示状態
	 */
	public int mode() {
		if(this.scale > MapPanel.MODE_SDF_SCALE) {
			return 3;
		} else if(this.scale > MapPanel.MODE_PREF_SCALE) {
			return 2;
		} else if (this.scale > MapPanel.MODE_WORLD_SCALE) {
			return 1;
		} else {
			return 0;
		}
	}

	/**
	 * 表示位置を初期値へ
	 */
	public void moveDefault() {
		double widthScale = (double) this.getWidth() / (MapPanel.MAX_SCREEN_X - MapPanel.MIN_SCREEN_X);
		double heightScale = (double) this.getHeight() / (MapPanel.MAX_SCREEN_Y - MapPanel.MIN_SCREEN_Y);
		this.scale = (float) ((widthScale < heightScale) ? widthScale : heightScale);
		this.screen.x = MapPanel.MIN_SCREEN_X;
		this.screen.y = MapPanel.MIN_SCREEN_Y;
		this.screen.width = (int) (this.getWidth() / this.scale);
		this.screen.height = (int) (this.getHeight() / this.scale);
	}

	/**
	 * 表示位置を平行移動をさせます。
	 * @param dx X軸方向の移動量
	 * @param dy Y軸方向の移動量
	 */
	public void moveLocation(int dx, int dy) {
		this.screen.x -= dx / this.scale;
		this.screen.y += dy / this.scale;
		if (this.screen.x < this.WORLD_SCREEN.x) {
			this.screen.x += this.WORLD_SCREEN.width;
		} else if (this.screen.x > this.WORLD_SCREEN.x + this.WORLD_SCREEN.width) {
			this.screen.x -= this.WORLD_SCREEN.width;
		}
	}

	/**
	 * 市区町村番号に対応した地点を表示させます。
	 * @param code 市区町村番号
	 * @return 移動すれば、true
	 */
	public boolean moveMap(int code) {
		Point point = this.maps.getPoint(code);
		if (point != null) {
			this.scale = MapPanel.MODE_SDF_SCALE * 1.5f;
			this.setMapLocation(point.x, point.y);
			this.repaint();
			return true;
		}
		return false;
	}
	
	@Override
	protected void paintComponent(Graphics g) {
		if(this.maps == null) {
			this.isRepaint = false;
			this.offs = this.createImage(this.getWidth(), this.getHeight());
			Graphics2D offg = (Graphics2D) this.offs.getGraphics();
			super.paintComponent(offg);
			// オフスクリーンバッファ
			offg.setFont(new Font(MapPanel.FONT_FAMILY, Font.PLAIN, 20));
			String msg = "Now Loading...";
			FontMetrics metrics = offg.getFontMetrics();
			offg.drawString(msg, (this.getWidth() - metrics.stringWidth(msg)) / 2, (this.getHeight() - metrics.getHeight()) / 2);
			offg.dispose();
		} else if (this.isRepaint) {
			this.isRepaint = false;
			if (this.offs == null) {
				this.offs = this.createImage(this.getWidth(), this.getHeight());
			}
			if (this.getWidth() != this.offs.getWidth(null) || this.getHeight() != this.offs.getHeight(null)) {
				int centerX = this.screen.x + this.screen.width / 2;
				int centerY = this.screen.y + this.screen.height / 2;
				// Windowサイズが変わった
				this.offs.flush();
				// オフスクリーンバッファ
				this.offs = this.createImage(this.getWidth(), this.getHeight());
				this.setMapLocation(centerX, centerY);
			}
			Graphics2D offg = (Graphics2D) this.offs.getGraphics();
			this.draw(offg);
			offg.dispose();
		}
		g.drawImage(this.offs, 0, 0, null);
	}
	@Override
	public void repaint() {
		if (this.maps == null) {
			this.isRepaint = true;
			super.repaint();
		} else if (!this.isRepaint) {
			this.isRepaint = true;
			super.repaint();
			this.maps.start();
		}
	}

	/**
	 * ルートの再計算を行います。
	 */
	public void reroute() {
		if(this.start != null && this.terminal != null && this.start != this.terminal) {
			if (this.thread != null) {
				this.thread.kill();
			}
			this.thread = this.algorithm.search(this.start, this.terminal);
		}
		this.repaint();
	}

	/**
	 * 探索端点の変更を行います。
	 * @param ex マウスのX座標
	 * @param ey マウスのY座標
	 * @param flag trueなら終点の変更、falseなら始点の変更
	 */
	public void searchBoundary(final int ex, final int ey, final boolean flag) {
		int x = (int)(ex / this.scale + this.screen.x);
		int y = (int)((this.getHeight() - ey) / this.scale + this.screen.y);
		
		Node point = null;
		double nodeDist = Double.POSITIVE_INFINITY;
		synchronized (this.maps) {
			for (final City map : this.maps.values()) {
				if (!map.hasData() || !map.getArea().contains(x, y)) {
					continue;
				}
				for (final Node node : map.getNode()) {
					final double d = node.distance(x, y, this.drawlevel);
					if (nodeDist > d) {
						point = node;
						nodeDist = d;
					}
				}
			}
		}
		if (point != null) {
			if (flag) {
				this.terminal = point;
				this.start = null;
				if (this.thread != null) {
					this.thread.kill();
				}
			} else {
				this.start = point;
				if (this.terminal != null) {
					if (this.thread != null) {
						this.thread.kill();
					}
					if (this.start != this.terminal) {
						this.thread = this.algorithm.search(this.start, this.terminal);
					}
				}
			}
		}
	}

	/**
	 * 市区町村表示のフォントを設定します。
	 * @param g
	 */
	private void setCityFont(Graphics2D g) {
		int fontSize = 10 + (int)((this.scale - MapPanel.MODE_PREF_SCALE) * 4000);
		if (fontSize > MapPanel.FONTSIZE_CITY_MAX) {
			fontSize = MapPanel.FONTSIZE_CITY_MAX;
		}
		
		if (fontSize < 17) {
			g.setFont(new Font("Serif", Font.PLAIN, fontSize));
		} else {
			g.setFont(new Font("Serif", Font.BOLD, fontSize));
		}
	}

	public void setLabelRendering(boolean rendering) {
		this.isShadow = rendering;
	}

	/**
	 * 地図の中心経緯度を指定して表示位置を移動させます。
	 * @param x 東経
	 * @param y 北緯
	 */
	public void setMapLocation(double x, double y) {
		this.scale = MapPanel.MODE_SDF_SCALE * 1.5f;
		this.screen.x = (int)(x * 3600000) - (int)(this.getWidth() / 2 / this.scale + 0.5f);
		this.screen.y = (int)(y * 3600000) - (int)(this.getHeight() / 2 / this.scale + 0.5f);
	}
	
	/**
	 * 地図の中心座標を指定して表示位置を移動させます。
	 * @param x 中心のX座標
	 * @param y 中心のY座標
	 */
	public void setMapLocation(int x, int y) {
		this.screen.x = x - (int)(this.getWidth() / 2 / this.scale + 0.5f);
		this.screen.y = y - (int)(this.getHeight() / 2 / this.scale + 0.5f);
		this.screen.width = (int) (this.getWidth() / this.scale);
		this.screen.height = (int) (this.getHeight() / this.scale);
	}

	/**
	 * マウス操作の状態を設定する
	 * @param flag マウス操作していればtrue
	 */
	public void setOperation(boolean flag) {
		this.isOperation = flag;
		if(!flag) {
			this.repaint();
		}
	}

	/**
	 * 市区町村表示のフォントを設定します。
	 * @param g
	 */
	private void setPrefectrueFont(Graphics2D g) {
		int fontSize = 11 + (int)(this.scale * 100000);
		if (fontSize > MapPanel.FONTSIZE_PREFECTURE_MAX) {
			fontSize = MapPanel.FONTSIZE_PREFECTURE_MAX;
		}
		g.setFont(new Font("Serif", Font.BOLD, fontSize));
	}

	public void switchAxis() {
		this.isAxis = !this.isAxis;
	}

	/**
	 * 道路表示を切り替えます。
	 */
	public void switchHighway() {
		this.isHighway = !this.isHighway;
	}

	/**
	 * ラベル表示を切り替える
	 * @param n 
	 */
	public void switchLabel(int n) {
		if ((this.isLabel & n) == 0) {
			this.isLabel += n;
		} else {
			this.isLabel -= n;
		}
	}

	public void switchLabelFailure() {
		this.isLabelFailure = !this.isLabelFailure;
	}
	
	/**
	 * メッシュの表示を切り替えます。
	 */
	public void switchMesh() {
		this.isMesh = !this.isMesh;
	}

	/**
	 * 頂点表示を切り替える
	 */
	public void switchNodeView() {
		this.isNodeView = !this.isNodeView;
		this.repaint();
	}

	/**
	 * 鉄道表示を切り替えます。
	 */
	public void switchRailway() {
		this.isRailway = !this.isRailway;
	}
	
	/**
	 * アンチエイリアスを切り替える
	 */
	public void switchRendering() {
		this.isAntialias = !this.isAntialias;
		this.repaint();
	}
	
	/**
	 * 河川表示を切り替えます。
	 */
	public void switchRiver() {
		this.isRiver = !this.isRiver;
	}

	/**
	 * 道路表示を切り替えます。
	 */
	public void switchRoadway() {
		this.isRoadway = !this.isRoadway;
	}

	/**
	 * 最短経路を求めるアルゴリズムを切り替える
	 * @param algorithm 切り替えるための文字列
	 */
	public void switchShortestPathAlgorithm(String algorithm) {
		if(algorithm.endsWith("_a*")) {
			this.algorithm = new UnidirectShortestPathSearch(new DirectDistance(), this.maps);
		} else if (algorithm.endsWith("_dijkstra")) {
			this.algorithm = new UnidirectShortestPathSearch(null, this.maps);
		}
		this.reroute();
	}
	
	/**
	 * 拡大縮小を行う
	 * 
	 * @param x
	 * @param y
	 * @param d
	 */
	public void zoom(int x, int y, int d) {
		float newScale = this.scale * (1 + d * MapPanel.SCALE_SENSE);
		if (newScale > MapPanel.SCALE_MAX) {
			newScale = MapPanel.SCALE_MAX;
		} else if (newScale < MapPanel.SCALE_MIN) {
			newScale = MapPanel.SCALE_MIN;
		}
		y = this.getHeight() - y;
		this.screen.x = (int) (this.screen.x + x / this.scale - x / newScale);
		this.screen.y = (int) (this.screen.y + y / this.scale - y / newScale);
		this.screen.width = (int) (this.getWidth() / newScale);
		this.screen.height = (int) (this.getHeight() / newScale);
		this.scale = newScale;
	}


	/**
	 * 数値地図25000を表示する倍率まで拡大縮小
	 * @param x
	 * @param y
	 * @param d
	 */
	public void zoomSdf(int x, int y) {
		float newScale = MapPanel.MODE_SDF_SCALE * 1.5f;
		if (newScale > MapPanel.SCALE_MAX) {
			newScale = MapPanel.SCALE_MAX;
		} else if (newScale < MapPanel.SCALE_MIN) {
			newScale = MapPanel.SCALE_MIN;
		}
		y = this.getHeight() - y;
		this.screen.x = (int) (this.screen.x + x / this.scale - x / newScale);
		this.screen.y = (int) (this.screen.y + y / this.scale - y / newScale);
		this.screen.width = (int) (this.getWidth() / newScale);
		this.screen.height = (int) (this.getHeight() / newScale);
		this.scale = newScale;
	}
}