<?php
/**
 * DBクラス
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    Magic3 Framework
 * @author     平田直毅(Naoki Hirata) <naoki@aplo.co.jp>
 * @copyright  Copyright 2006-2008 Magic3 Project.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id$
 * @link       http://www.magic3.org
 */
require_once($gEnvManager->getDbPath() . '/baseDb.php');

class ec_menuDb extends BaseDb
{
	/**
	 * メニュー情報を取得
	 *
	 * @param string	$menuId				メニューID
	 * @param string	$lang				言語
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getMenu($menuId, $lang, &$row)
	{
		$retValue = 0;
		$queryStr  = 'select * from menu ';
		$queryStr .=   'WHERE me_id = ? ';
		$queryStr .=     'AND me_language_id = ? ';
		$ret = $this->selectRecord($queryStr, array($menuId, $lang), $row);
		return $ret;
	}
	/**
	 * メニューの項目を取得
	 *
	 * @param function	$callback			コールバック関数
	 * @param string	$menuId				メニューID
	 * @param string	$lang				言語
	 * @return 			なし
	 */
	function getMenuItems($callback, $menuId, $lang)
	{
		$queryStr = 'SELECT * FROM menu_item ';
		$queryStr .=  'WHERE mi_visible = true ';
		$queryStr .=    'AND mi_menu_id = ? ';
		$queryStr .=    'AND mi_language_id = ? ';
		$queryStr .=  'ORDER BY mi_index';
		$this->selectLoop($queryStr, array($menuId, $lang), $callback, null);
	}
	/**
	 * メニュー項目をシリアル番号で取得
	 *
	 * @param string	$serial				シリアル番号
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getMenuBySerial($serial, &$row)
	{
		$retValue = 0;
		$queryStr  = 'select * from menu_item ';
		$queryStr .=   'WHERE mi_serial = ? ';
		$ret = $this->selectRecord($queryStr, array($serial), $row);
		return $ret;
	}
	/**
	 * メニュー項目のデフォルトインデックス番号を取得
	 *
	 * @return int					デフォルトインデックス番号
	 */
	function getDefaultMenuIndex()
	{
		$default = 0;
		$queryStr = 'select max(mi_index) as ms from menu_item ';
		$ret = $this->selectRecord($queryStr, array(), $maxRow);
		if ($ret) $default = $maxRow['ms'] + 1;
		return $default;
	}
	/**
	 * 新規のメニュー項目IDを取得
	 *
	 * @return int					新規メニュー項目ID番号
	 */
	function getNewMenuId()
	{
		$id = 0;
		$queryStr = 'select max(mi_id) as ms from menu_item ';
		$ret = $this->selectRecord($queryStr, array(), $maxRow);
		if ($ret) $id = $maxRow['ms'] + 1;
		return $id;
	}
	/**
	 * メニューの項目を取得(管理用)
	 *
	 * @param function	$callback			コールバック関数
	 * @param string	$menuId				メニューID
	 * @param string	$lang				言語
	 * @return 			なし
	 */
	function getAllMenuItems($callback, $menuId, $lang)
	{
		$queryStr = 'SELECT * FROM menu_item ';
		$queryStr .=  'WHERE mi_menu_id = ? ';
		$queryStr .=    'AND mi_language_id = ? ';
		$queryStr .=  'ORDER BY mi_index';
		$this->selectLoop($queryStr, array($menuId, $lang), $callback, null);
	}
	/**
	 * メニュー項目の追加
	 *
	 * @param string  $menuId		メニューID
	 * @param string  $id			メニュー項目ID
	 * @param string  $lang			言語ID
	 * @param string  $name			メニュー名
	 * @param int     $index		インデックス番号
	 * @param int     $linkType		リンクタイプ
	 * @param string  $url			URL
	 * @param string  $param		パラメータ
	 * @param bool    $visible		表示状態
	 * @param bool    $enable		使用可否
	 * @param int     $userId		更新者ユーザID
	 * @param int     $newSerial	新規シリアル番号
	 * @return bool					true = 成功、false = 失敗
	 */
	function addMenuItem($menuId, $id, $lang, $name, $index, $linkType, $url, $param, $visible, $enable, $userId, &$newSerial)
	{
		// インデックスが0のときは、最大値を格納
		if (empty($index)){
			$default = 1;
			$queryStr = 'SELECT max(mi_index) as ms FROM menu_item ';
			$queryStr .=  'WHERE mi_menu_id = ? ';
			$queryStr .=    'AND mi_language_id = ? ';
			$ret = $this->selectRecord($queryStr, array($menuId, $lang), $maxRow);
			if ($ret) $default = $maxRow['ms'] + 1;
			$index = $default;
		}
		$queryStr = 'INSERT INTO menu_item ';
		$queryStr .=  '(mi_menu_id, mi_id, mi_language_id, mi_name, mi_index, mi_link_type, mi_link_url, mi_param, mi_visible, mi_enable, mi_update_user_id, mi_update_dt) ';
		$queryStr .=  'VALUES ';
		$queryStr .=  '(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, now())';
		$this->execStatement($queryStr, array($menuId, $id, $lang, $name, $index, $linkType, $url, $param, $visible, $enable, $userId));
		
		// 新規のシリアル番号取得
		$queryStr = 'select max(mi_serial) as ns from menu_item ';
		$ret = $this->selectRecord($queryStr, array(), $row);
		if ($ret) $newSerial = $row['ns'];
		
		return $ret;
	}
	/**
	 * メニュー項目の更新
	 *
	 * @param string  $menuId		メニューID
	 * @param string  $id			メニュー項目ID
	 * @param string  $lang			言語ID
	 * @param string  $name			メニュー名
	 * @param int     $index		インデックス番号
	 * @param int     $linkType		リンクタイプ
	 * @param string  $url			URL
	 * @param string  $param		パラメータ
	 * @param bool    $visible		表示状態
	 * @param bool    $enable		使用可否
	 * @param int     $userId		更新者ユーザID
	 * @return bool					true = 成功、false = 失敗
	 */
	function updateMenuItem($menuId, $id, $lang, $name, $index, $linkType, $url, $param, $visible, $enable, $userId)
	{
		$params = array();
		$queryStr = 'UPDATE menu_item ';
		$queryStr .=  'SET mi_menu_id = ?, ';		$params[] = $menuId;
		$queryStr .=    'mi_name = ?, ';			$params[] = $name;
		if (!empty($index)){
			$queryStr .=    'mi_index = ?, ';
			$params[] = $index;
		}
		$queryStr .=    'mi_link_type = ?, ';		$params[] = $linkType;
		$queryStr .=    'mi_link_url = ?, ';		$params[] = $url;
		$queryStr .=    'mi_param = ?, ';			$params[] = $param;
		$queryStr .=    'mi_visible = ?, ';			$params[] = $visible;
		$queryStr .=    'mi_enable = ?, ';			$params[] = $enable;
		$queryStr .=    'mi_update_user_id = ?, ';	$params[] = $userId;
		$queryStr .=    'mi_update_dt = now() ';
		$queryStr .=  'WHERE mi_id = ? AND ';		$params[] = $id;
		$queryStr .=    'mi_language_id = ?';		$params[] = $lang;
		$ret = $this->execStatement($queryStr, $params);
		return $ret;
	}
	/**
	 * メニュー項目の更新
	 *
	 * @param int     $serial		メニュー項目シリアル番号
	 * @param string  $menuId		メニューID
	 * @param string  $name			メニュー名
	 * @param int     $index		インデックス番号
	 * @param int     $linkType		リンクタイプ
	 * @param string  $url			URL
	 * @param string  $param		パラメータ
	 * @param bool    $visible		表示状態
	 * @param bool    $enable		使用可否
	 * @param int     $userId		更新者ユーザID
	 * @return bool					true = 成功、false = 失敗
	 */
	function updateMenuItemBySerial($serial, $menuId, $name, $index, $linkType, $url, $param, $visible, $enable, $userId)
	{
		$params = array();
		$queryStr = 'UPDATE menu_item ';
		$queryStr .=  'SET mi_menu_id = ?, ';		$params[] = $menuId;
		$queryStr .=    'mi_name = ?, ';			$params[] = $name;
		if (!empty($index)){
			$queryStr .=    'mi_index = ?, ';
			$params[] = $index;
		}
		$queryStr .=    'mi_link_type = ?, ';		$params[] = $linkType;
		$queryStr .=    'mi_link_url = ?, ';		$params[] = $url;
		$queryStr .=    'mi_param = ?, ';			$params[] = $param;
		$queryStr .=    'mi_visible = ?, ';			$params[] = $visible;
		$queryStr .=    'mi_enable = ?, ';			$params[] = $enable;
		$queryStr .=    'mi_update_user_id = ?, ';	$params[] = $userId;
		$queryStr .=    'mi_update_dt = now() ';
		$queryStr .=  'WHERE mi_serial = ? ';		$params[] = $serial;
		$ret = $this->execStatement($queryStr, $params);
		return $ret;
	}
	/**
	 * メニュー項目の削除
	 *
	 * @param int $serialNo			シリアルNo
	 * @return						true=成功、false=失敗
	 */
	function delMenuItem($serialNo)
	{
		// レコードを削除
		$queryStr  = 'DELETE FROM menu_item ';
		$queryStr .=   'WHERE mi_serial = ?';
		$ret = $this->execStatement($queryStr, array($serialNo));
		return $ret;
	}
	/**
	 * メニュー項目のすべて削除
	 *
	 * @param int $contentType		コンテンツタイプ
	 * @param int $userId			ユーザID(データ更新者)
	 * @return						true=成功、false=失敗
	 */
	function delMenuAllItem($contentType, $userId)
	{
		// トランザクション開始
		$this->startTransaction();
		
		// レコードを削除
		$queryStr  = 'DELETE FROM menu_item ';
		$this->execStatement($queryStr, array());
		
		// レコードを削除
		$queryStr  = 'UPDATE content ';
		$queryStr .=   'SET cn_deleted = true, ';	// 削除
		$queryStr .=     'cn_update_user_id = ?, ';
		$queryStr .=     'cn_update_dt = now() ';
		$queryStr .=   'WHERE cn_deleted = false ';
		$queryStr .=     'AND cn_type = ? ';
		$ret = $this->execStatement($queryStr, array($userId, $contentType));

		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * メニュー項目IDの重複チェック
	 *
	 * @param string  $id			メニュー項目ID
	 * @param string  $lang			言語ID
	 * @return bool					true = 存在する、false = 存在しない
	 */
	function isExistsMenuId($id, $lang)
	{
		$queryStr = 'SELECT * from menu_item ';
		$queryStr .=  'WHERE mi_id = ? ';
		$queryStr .=    'AND mi_language_id = ?';
		return $this->isRecordExists($queryStr, array($id, $lang));
	}
	/**
	 * ページIDのリストを取得
	 *
	 * @param function $callback	コールバック関数
	 * @param int $type				リストの種別
	 * @return						なし
	 */
	function getPageIdList($callback, $type)
	{
		$queryStr = 'select * from _page_id ';
		$queryStr .=  'where pg_type = ? ';
		$queryStr .=  'order by pg_priority';
		$this->selectLoop($queryStr, array($type), $callback);
	}
	/**
	 * メニュー項目の表示順を変更する
	 *
	 * @param string  $menuId			メニューID
	 * @param string  $lang				言語ID
	 * @param array $menuItemNoArray	並び順
	 * @return bool					true = 成功、false = 失敗
	 */
	function orderMenuItems($menuId, $lang, $menuItemNoArray)
	{
		global $gEnvManager;

		// メニュー項目をすべて取得
		$queryStr = 'SELECT * FROM menu_item ';
		$queryStr .=  'WHERE mi_visible = true ';		// 表示中の項目
		$queryStr .=    'AND mi_menu_id = ? ';
		$queryStr .=    'AND mi_language_id = ? ';
		$queryStr .=  'ORDER BY mi_index';
		$ret = $this->selectRecords($queryStr, array($menuId, $lang), $rows);
		if (!$ret) return false;
	
		// メニュー数をチェックし、異なっている場合はエラー
		$menuItemCount = count($rows);
		if ($menuItemCount != count($menuItemNoArray)) return false;
		
		$now = date("Y/m/d H:i:s");	// 現在日時
		$userId = $gEnvManager->getCurrentUserId();	// 現在のユーザ
		
		// トランザクション開始
		$this->startTransaction();

		for ($i = 0; $i < $menuItemCount; $i++){
			$serialNo = $rows[$menuItemNoArray[$i]]['mi_serial'];
			$index = $rows[$i]['mi_index'];

			// 既存項目を更新
			$queryStr  = 'UPDATE menu_item ';
			$queryStr .=   'SET ';
			$queryStr .=     'mi_index = ?, ';
			$queryStr .=     'mi_update_user_id = ?, ';
			$queryStr .=     'mi_update_dt = ? ';
			$queryStr .=   'WHERE mi_serial = ? ';
			$this->execStatement($queryStr, array($index, $userId, $now, $serialNo));
		}
										
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * 商品カテゴリー一覧を取得
	 *
	 * @param string	$lang				言語
	 * @param array		$rows				取得データ
	 * @return bool							true=取得、false=取得せず
	 */
	function getAllCategory($lang, &$rows)
	{
		$queryStr = 'SELECT * FROM product_category LEFT JOIN _login_user ON pc_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=  'WHERE pc_language_id = ? ';
		$queryStr .=    'AND pc_deleted = false ';		// 削除されていない
		$queryStr .=  'ORDER BY pc_sort_order';
		$retValue = $this->selectRecords($queryStr, array($lang), $rows);
		return $retValue;
	}
	/**
	 * コンテンツ項目をコンテンツIDで取得
	 *
	 * @param string	$contentType		コンテンツタイプ
	 * @param string	$contentId			コンテンツID
	 * @param string	$langId				言語ID
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getContentByContentId($contentType, $contentId, $langId, &$row)
	{
		$queryStr  = 'SELECT * FROM content ';
		$queryStr .=   'WHERE cn_deleted = false ';	// 削除されていない
		$queryStr .=    'AND cn_type = ? ';
		$queryStr .=   'AND cn_id = ? ';
		$queryStr .=   'AND cn_language_id = ? ';
		$ret = $this->selectRecord($queryStr, array($contentType, $contentId, $langId), $row);
		return $ret;
	}
	/**
	 * コンテンツ項目の更新
	 *
	 * @param string  $contentType	コンテンツタイプ
	 * @param string  $contentId	コンテンツID(0のとき新規)
	 * @param string  $lang			言語ID
	 * @param string  $name			コンテンツ名
	 * @param string  $html			HTML
	 * @param bool    $visible		表示状態
	 * @param string  $key			外部参照用キー
	 * @param int     $userId		更新者ユーザID
	 * @param int     $newContentId	新規コンテンツID
	 * @param int     $newSerial	新規シリアル番号
	 * @return bool					true = 成功、false = 失敗
	 */
	function updateContentItem($contentType, $contentId, $lang, $name, $html, $visible, $default, $key, $userId, &$newContentId, &$newSerial)
	{
		$historyIndex = 0;		// 履歴番号
		$now = date("Y/m/d H:i:s");	// 現在日時
		
		if (empty($contentId)){			// 新規コンテンツ追加のとき
			// コンテンツIDを決定する
			$queryStr = 'select max(cn_id) as mid from content ';
			$queryStr .=  'WHERE cn_type = ? ';
			$ret = $this->selectRecord($queryStr, array($contentType), $row);
			if ($ret){
				$contId = $row['mid'] + 1;
			} else {
				$contId = 1;
			}
			$desc = '';
		} else {
			// 前レコードの削除状態チェック
			$queryStr = 'SELECT * FROM content ';
			$queryStr .=  'WHERE cn_type = ? ';
			$queryStr .=    'AND cn_id = ? ';
			$queryStr .=    'AND cn_language_id = ? ';
			$queryStr .=  'ORDER BY cn_history_index DESC ';
			$ret = $this->selectRecord($queryStr, array($contentType, $contentId, $lang), $row);
			if ($ret){
				if ($row['cn_deleted']){		// レコードが削除されていれば終了
					return false;
				}
			} else {
				return false;
			}
			$historyIndex = $row['cn_history_index'] + 1;
			$contId = $row['cn_id'];
			$desc = $row['cn_description'];
				
			// 古いレコードを削除
			$queryStr  = 'UPDATE content ';
			$queryStr .=   'SET cn_deleted = true, ';	// 削除
			$queryStr .=     'cn_update_user_id = ?, ';
			$queryStr .=     'cn_update_dt = ? ';
			$queryStr .=   'WHERE cn_serial = ?';
			$this->execStatement($queryStr, array($userId, $now, $row['cn_serial']));
		}
		
		// 新規レコード追加
		$queryStr = 'INSERT INTO content ';
		$queryStr .=  '(cn_type, cn_id, cn_language_id, cn_history_index, cn_name, cn_description, cn_html, cn_visible, cn_default, cn_key, cn_create_user_id, cn_create_dt) ';
		$queryStr .=  'VALUES ';
		$queryStr .=  '(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)';
		$this->execStatement($queryStr, array($contentType, $contId, $lang, $historyIndex, $name, $desc, $html, $visible, $default, $key, $userId, $now));

		// 新規のシリアル番号取得
		$queryStr = 'SELECT max(cn_serial) as ns FROM content ';
		$queryStr .=  'WHERE cn_type = ? ';
		$ret = $this->selectRecord($queryStr, array($contentType), $row);
		if ($ret) $newSerial = $row['ns'];
		
		$newContentId = $contId;		// 新規コンテンツID
		return $ret;
	}
	/**
	 * コンテンツ項目の削除
	 *
	 * @param string  $contentType	コンテンツタイプ
	 * @param string  $contentId	コンテンツID(0のとき新規)
	 * @param string  $lang			言語ID
	 * @param int $userId			ユーザID(データ更新者)
	 * @return						true=成功、false=失敗
	 */
	function delContentItem($contentType, $contentId, $lang, $userId)
	{
		$now = date("Y/m/d H:i:s");	// 現在日時
		
		// 前レコードの削除状態チェック
		$queryStr = 'SELECT * FROM content ';
		$queryStr .=  'WHERE cn_type = ? ';
		$queryStr .=    'AND cn_id = ? ';
		$queryStr .=    'AND cn_language_id = ? ';
		$queryStr .=  'ORDER BY cn_history_index DESC ';
		$ret = $this->selectRecord($queryStr, array($contentType, $contentId, $lang), $row);
		if ($ret){
			if ($row['cn_deleted']){		// レコードが削除されていれば終了
				return false;
			}
		} else {
			return false;
		}
			
		// 古いレコードを削除
		$queryStr  = 'UPDATE content ';
		$queryStr .=   'SET cn_deleted = true, ';	// 削除
		$queryStr .=     'cn_update_user_id = ?, ';
		$queryStr .=     'cn_update_dt = ? ';
		$queryStr .=   'WHERE cn_serial = ?';
		$ret = $this->execStatement($queryStr, array($userId, $now, $row['cn_serial']));
		return $ret;
	}
	/**
	 * カテゴリーを取得(管理用)
	 *
	 * @param string	$lang			言語ID
	 * @param string	$categoryId		カテゴリーID
	 * @return			true=取得、false=取得せず
	 */
	function getCategory($lang, $categoryId, &$rows)
	{
		$params = array();
		$queryStr = 'SELECT * FROM product_category ';
		$queryStr .=  'WHERE pc_deleted = false ';	// 削除されていない
		$queryStr .=    'AND pc_language_id = ? ';
		if (!empty($categoryId)) $queryStr .=    'AND pc_id in (' . $categoryId . ') ';
		$queryStr .=  'ORDER BY pc_sort_order';
		$retValue = $this->selectRecords($queryStr, array($lang), $rows);
		return $retValue;
	}
}
?>
