<?php
/**
 * 環境取得用マネージャー
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    Magic3 Framework
 * @author     平田直毅(Naoki Hirata) <naoki@aplo.co.jp>
 * @copyright  Copyright 2006-2008 Magic3 Project.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id$
 * @link       http://www.magic3.org
 */
require_once(M3_SYSTEM_INCLUDE_PATH . '/core/version.php');	// システムバージョンクラス
require_once(M3_SYSTEM_INCLUDE_PATH . '/common/userInfo.php');		// ユーザ情報クラス

class EnvManager
{
	public  $sysVersion;		// システムバージョンオブジェクト
	private $currentTemplateId;	// 現在のテンプレートID(ディレクトリ名)
	private $currentTemplateType;	// 現在のテンプレートのタイプ
	private $currentWidgetId;	// 現在作成中のウィジェットId
	private $currentWidgetConfigId;	// 現在作成中のウィジェットの定義ID
	private $currentIWidgetConfigId;	// 現在作成中のインナーウィジェットの定義ID
	private $currentIWidgetId;	// 現在作成中のインナーウィジェットId
	private $currentPageId;		// 現在のページId
	private $currentPageSubId;	// 現在のページサブId
	private $currentPageDefSerial;	// 現在処理を行っているページ定義のレコードのシリアル番号
	private $defaultPageSubId;	// デフォルトのページサブId
	private $currentWidgetPrefix;	// 現在作成中のウィジェットのプレフィックス文字列
	private $defaultLanguage;	// デフォルト言語(システムで固定)
	private $currentLanguage;	// 現在の言語(ユーザによって可変)
	private $multiLanguage;		// 多言語対応かどうか
 	private $db;				// DBオブジェクト
	private $canUseDbSession;	// DBセッションが使用できるかどうか
	private $canUseDb;			// DBが使用可能状態にあるかどうか
	private $canChangeLang;		// 言語変更可能かどうか
	private $siteName;			// サイト名称
	private $siteOwner;			// サイト所有者
	private $siteCopyRight;		// サイトコピーライト
	private $siteEmail;			// サイトEメール
	private $widgetLog;			// ウィジェット実行ログを残すかどうか
	private $isMobile;			// 携帯アクセスかどうか
	private $mobileEncoding;	// 携帯用の入力、出力エンコーディング
	const DEFAULT_CSV_DELIM_CODE = 'csv_delim_code';		// デフォルトのCSV区切り文字コード
	const DEFAULT_CSV_NL_CODE = 'csv_nl_code';		// デフォルトのCSV改行コード
	const DEFAULT_CSV_FILE_SUFFIX = 'csv_file_suffix';		// デフォルトのCSVファイル拡張子
	const MULTI_LANGUAGE = 'multi_language';		// 多言語対応かどうか
	const MOBILE_ENCODING = 'mobile_encoding';		// 携帯用入出力エンコーディング
	const DEFAULT_PAGE_ID = 'index';					// デフォルトのページID
	const DEFAULT_MOBILE_PAGE_ID = 'm_index';					// 携帯用デフォルトのページID
	const DEFAULT_ADMIN_PAGE_ID = 'admin_index';		// デフォルトの管理機能用ページID
	
	/**
	 * コンストラクタ
	 */
	function __construct()
	{
		global $gInstanceManager;
		
		// システムバージョンオブジェクト作成
		$this->sysVersion = new m3Version();
		
		// システムDBオブジェクト取得
		$this->db = $gInstanceManager->getSytemDbObject();
		
		// ######## DBの接続チェック ########
		if (defined('M3_STATE_IN_INSTALL')){		// システムがインストールモードで起動のとき
			$this->canUseDb = false;			// DBは使用できない
		} else {
			// システム名称、バージョンを取得
			$status = $this->db->getDisplayErrMessage();	// 出力状態を取得
			$this->db->displayErrMessage(false);		// 画面へのエラー出力を抑止
			$value = $this->db->getSystemConfig(M3_TB_FIELD_SYSTEM_NAME);
			$this->db->displayErrMessage($status);		// 抑止解除
			// 値が取得できたときは、セッションDBテーブルも作成されているとする
			if ($value == ''){
				$this->canUseDbSession = false;
				$this->canUseDb = false;			// DBは使用できない
			} else {
				$this->canUseDbSession = true;
				$this->canUseDb = true;			// DBは使用可能
				
				// システム関係のパラメータを取得
				$this->loadSystemParams();
			}
		}
		// 日本語処理関係
		if (extension_loaded('mbstring')){	// mbstring使用可能
			ini_set('mbstring.http_input',                  'pass');
			ini_set('mbstring.http_output',                 'pass');
			ini_set('mbstring.encoding_translation',        'Off');
			ini_set('mbstring.func_overload',               '0');
			
			if (function_exists('mb_language')) mb_language("Japanese");
			if (function_exists('mb_internal_encoding')) mb_internal_encoding("UTF-8");
		}
	}
	/**
	 * システム関係のパラメータを再取得
	 */
	public function loadSystemParams()
	{
		// デフォルト値取得
		$this->defaultLanguage = $this->db->getDefaultLanguage();
		$this->currentLanguage = $this->defaultLanguage;
		
		// 言語変更可能かどうか
		$this->canChangeLang = $this->db->getCanChangeLang();
		$this->multiLanguage = $this->db->getSystemConfig(self::MULTI_LANGUAGE);		// 多言語対応かどうか
		
		$this->mobileEncoding = $this->db->getSystemConfig(self::MOBILE_ENCODING);	// 携帯用の入力、出力エンコーディング
	}
	/**
	 * デバッグ出力を行うかどうか
	 */
	public function getSystemDebugOut()
	{
		return M3_SYSTEM_DEBUG_OUT;
	}

	// ##################### システム全体のパス環境 #####################
	/**
	 * システムルートディレクトリへのパスを取得
	 */
	public function getSystemRootPath()
	{
		return M3_SYSTEM_ROOT_PATH;
	}
	/**
	 * 管理用ディレクトリへのパスを取得
	 */
	public function getAdminPath()
	{
		return M3_SYSTEM_ROOT_PATH . DIRECTORY_SEPARATOR . M3_DIR_NAME_ADMIN;
	}
	/**
	 * includeディレクトリへのパスを取得
	 */
	public function getIncludePath()
	{
		return M3_SYSTEM_INCLUDE_PATH;
	}
	/**
	 * インナーウィジェット用ディレクトリへのパスを取得
	 */
	public function getIWidgetsPath()
	{
		return M3_SYSTEM_INCLUDE_PATH . DIRECTORY_SEPARATOR . 'iwidgets';
	}
	/**
	 * addonsディレクトリへのパスを取得
	 */
	public function getAddonsPath()
	{
		return M3_SYSTEM_INCLUDE_PATH . DIRECTORY_SEPARATOR . 'addons';
	}
	/**
	 * コンテナクラス用ディレクトリへのパスを取得
	 */
	public function getContainerPath()
	{
		return M3_SYSTEM_INCLUDE_PATH . DIRECTORY_SEPARATOR . 'container';
	}
	/**
	 * DBクラス用ディレクトリへのパスを取得
	 */
	public function getDbPath()
	{
		return M3_SYSTEM_INCLUDE_PATH . DIRECTORY_SEPARATOR . 'db';
	}
	/**
	 * ライブラリ用ディレクトリへのパスを取得
	 */
	public function getLibPath()
	{
		return M3_SYSTEM_INCLUDE_PATH . DIRECTORY_SEPARATOR . 'lib';
	}
	/**
	 * SQL格納用ディレクトリへのパスを取得
	 */
	public function getSqlPath()
	{
		return M3_SYSTEM_INCLUDE_PATH . DIRECTORY_SEPARATOR . 'sql';
	}
	/**
	 * テーブル作成用SQL格納用ディレクトリへのパスを取得
	 */
	public function getTablesPath()
	{
		return M3_SYSTEM_INCLUDE_PATH . DIRECTORY_SEPARATOR . 'tables';
	}
	/**
	 * Coreディレクトリへのパスを取得
	 */
	public function getCorePath()
	{
		return M3_SYSTEM_INCLUDE_PATH . DIRECTORY_SEPARATOR . 'core';
	}
	/**
	 * Commonディレクトリへのパスを取得
	 */
	public function getCommonPath()
	{
		return M3_SYSTEM_INCLUDE_PATH . DIRECTORY_SEPARATOR . 'common';
	}
	/**
	 * Joomla用ルートディレクトリへのパスを取得
	 */
	public function getJoomlaRootPath()
	{
		return M3_SYSTEM_INCLUDE_PATH . DIRECTORY_SEPARATOR . 'mos';
	}
	/**
	 * スクリプトファイルディレクトリへのパスを取得
	 */
	public function getScriptsPath()
	{
		return M3_SYSTEM_ROOT_PATH . DIRECTORY_SEPARATOR . 'scripts';
	}
	/**
	 * テンプレート用ディレクトリへのパスを取得
	 */
	public function getTemplatesPath()
	{
		return M3_SYSTEM_ROOT_PATH . DIRECTORY_SEPARATOR . 'templates';
	}
	/**
	 * リソース用ディレクトリへのパスを取得
	 */
	public function getResourcePath()
	{
		return M3_SYSTEM_ROOT_PATH . DIRECTORY_SEPARATOR . 'resource';
	}
	/**
	 * widgetsディレクトリへのパスを取得
	 *
	 * 例) /var/www/html/magic3/widgets
	 */
	public function getWidgetsPath()
	{
		return M3_SYSTEM_ROOT_PATH . DIRECTORY_SEPARATOR . 'widgets';
	}
	/**
	 * システムのルートURLを取得
	 */
	public function getRootUrl()
	{
		return M3_SYSTEM_ROOT_URL;
	}
	/**
	 * widgetsディレクトリへのURLを取得
	 *
	 * 例) http://www.magic3.org/magic3/widgets
	 */
	public function getWidgetsUrl()
	{
		return M3_SYSTEM_ROOT_URL . '/widgets';
	}
	/**
	 * リソース用ディレクトリへのパスを取得
	 */
	public function getResourceUrl()
	{
		return M3_SYSTEM_ROOT_URL . '/resource';
	}
	/**
	 * scriptsディレクトリ(共通スクリプトディレクトリ)へのURLを取得
	 *
	 * 例) http://www.magic3.org/magic3/scripts
	 */
	public function getScriptsUrl()
	{
		return M3_SYSTEM_ROOT_URL . '/scripts';
	}
	/**
	 * templatesディレクトリ(テンプレートディレクトリ)へのURLを取得
	 *
	 * 例) http://www.magic3.org/magic3/templates
	 */
	public function getTemplatesUrl()
	{
		return M3_SYSTEM_ROOT_URL . '/templates';
	}
	/**
	 * 管理用ディレクトリへのパスを取得
	 */
	public function getAdminUrl()
	{
		return M3_SYSTEM_ROOT_URL . '/' . M3_DIR_NAME_ADMIN;
	}
	/**
	 * システムのデフォルトindexのURLを取得
	 */
	public function getDefaultUrl()
	{
		return M3_SYSTEM_ROOT_URL . '/index.php';
	}
	/**
	 * システムの携帯用デフォルトindexのURLを取得
	 */
	public function getDefaultMobileUrl()
	{
		return M3_SYSTEM_ROOT_URL . '/' . M3_DIR_NAME_MOBILE . '/index.php';
	}
	/**
	 * システムのデフォルトの管理用indexのURLを取得
	 */
	public function getDefaultAdminUrl()
	{
		return M3_SYSTEM_ROOT_URL . '/' . M3_DIR_NAME_ADMIN . '/index.php';
	}
	/**
	 * ドキュメントルートからのリソース用ディレクトリへの相対パスを取得
	 */
	public function getRelativeResourcePathToDocumentRoot()
	{
		// 相対パスを得る
		$res = parse_url($this->getResourceUrl());
		return $res['path'];
	}
	/**
	 * アプリケーションルートから指定ディレクトリへの相対パスを取得
	 *
	 * @param string $url	指定URL
	 */
	public function getRelativePathToSystemRootUrl($url)
	{
		// URLから相対パスを得る
		$root = parse_url($this->getRootUrl());
		$target = parse_url($url);		
		//return $this->calcRelativePath($this->getSystemRootPath(), $path);
		return $this->calcRelativePath($root['path'], $target['path']);
	}
	/**
	 * アプリケーションルートから指定ディレクトリへの相対パスを取得
	 *
	 * @param string $path	指定パス
	 */
	public function getRelativePathToSystemRootPath($path)
	{
		return $this->calcRelativePath($this->getSystemRootPath(), $path);
	}	
	// ##################### パス処理 #####################
	/**
	 * サーバのURLを取得
	 *
	 * 例) http://www.magic3.org
	 */
	public function getServerUrl()
	{
		return 'http://' . $_SERVER['SERVER_NAME'];
	}
	/**
	 * 現在実行中のスクリプトファイルのURLを取得
	 *
	 * 例) http://www.magic3.org/magic3/index.php
	 */
	public function getCurrentScriptUrl()
	{
		//return $_SERVER["SCRIPT_URI"];		// SCRIPT_URIはサーバによってはundefinedになる
		return self::getServerUrl() . $_SERVER["PHP_SELF"];
	}
	/**
	 * クライアントから要求されたURI(パラメータ付き)を取得
	 *
	 * 例) http://www.magic3.org/magic3/index.php?aaa=bbb
	 */
	public function getCurrentRequestUri()
	{
		return self::getServerUrl() . $_SERVER["REQUEST_URI"];
	}
	/**
	 * 現在実行中のスクリプトファイルのパスを取得
	 *
	 * 例) /var/www/html/magic3/index.php
	 */
	public function getCurrentScriptPath()
	{
		return realpath($_SERVER["SCRIPT_FILENAME"]);
	}	
	/**
	 * ドキュメントルートを取得
	 *
	 * 例) /var/www/html
	 */
	public function getDocumentRoot()
	{
		return $_SERVER["DOCUMENT_ROOT"];
	}
	/**
	 * ドキュメントルートURLを取得
	 *
	 * 例) http://www.magic3.org
	 */
	public function getDocumentRootUrl()
	{
		$rootUrl = parse_url($this->getRootUrl());
		return 'http://' . $rootUrl['host'];
	}
	/**
	 * システムルートURLを求める
	 *
	 * @return string		システムのルートURL。算出できなかったときは空文字列を返す。
	 */
	public function calcSystemRootUrl()
	{	
		// 相対パスを得る
		$base = explode(DIRECTORY_SEPARATOR, self::getSystemRootPath());
		$target = explode(DIRECTORY_SEPARATOR, self::getCurrentScriptPath());
		
		for ($i = 0; $i < count($base); $i++)
		{
			if ($base[$i] != $target[$i]) break;
		}
		$relativePath = '';
		for ($j = $i; $j < count($target); $j++)
		{
			$relativePath .= '/' . $target[$j];
		}
		// システムルートディレクトリ取得
		$sytemRootUrl = '';
		$pos = strrpos(self::getCurrentScriptUrl(), $relativePath);
		if (!($pos === false)){
			$sytemRootUrl = substr(self::getCurrentScriptUrl(), 0, $pos);
		}
		return $sytemRootUrl;
	}
	/**
	 * 相対パスを求める
	 *
	 * @param string $basePath		基点となるディレクトリの絶対パス
	 * @param string $targetPath	対象となるディレクトリの絶対パス
	 * @return string				相対パス
	 */
	public function calcRelativePath($basePath, $targetPath)
	{
		// 相対パスを得る
		$base = explode('/', $basePath);
		$target = explode('/', $targetPath);
		
		for ($i = 0; $i < count($base); $i++)
		{
			if ($base[$i] != $target[$i]) break;
		}
		$relativePath = '';
		for ($j = $i; $j < count($target); $j++)
		{
			$relativePath .= '/' . $target[$j];
		}
		return $relativePath;
	}
	/**
	 * 管理者用URLへのアクセスかどうか
	 *
	 * @return bool		管理者用ディレクトリへのアクセスのときは、true。それ以外の場合はfalse。
	 */
	public function isAdminDirAccess()
	{
		if (dirname($this->getCurrentScriptPath()) == $this->getAdminPath()){
			return true;
		} else {
			return false;
		}
	}
	// ##################### カレントのテンプレート関係 #####################
	/**
	 * 現在のテンプレート
	 */
	public function setCurrentTemplateId($name)
	{
		$this->currentTemplateId = $name;
		
		// テンプレートのタイプを取得
		$this->currentTemplateType = 0;
		if ($this->db->getTemplate($name, $row)){
			if ($row['tm_mobile']) $this->currentTemplateType = 1;		// 携帯の場合
		}
	}
	/**
	 * 現在テンプレート
	 */
	public function getCurrentTemplateId()
	{
		return $this->currentTemplateId;
	}
	/**
	 * 現在のテンプレートタイプ
	 *
	 * @return int		0=通常、1=携帯用
	 */
	public function getCurrentTemplateType()
	{
		return $this->currentTemplateType;
	}
	/**
	 * 現在のテンプレートへのパスを取得
	 *
	 * 例) /var/www/html/magic3/templates/menu
	 */
	public function getCurrentTemplatePath()
	{
		return M3_SYSTEM_ROOT_PATH . DIRECTORY_SEPARATOR . 'templates' . DIRECTORY_SEPARATOR . $this->currentTemplateId;
	}
	/**
	 * 現在のテンプレートへのURLを取得
	 *
	 * 例) http://www.magic3.org/magic3/templates/menu
	 */
	public function getCurrentTemplateUrl()
	{
		return M3_SYSTEM_ROOT_URL . '/templates/' . $this->currentTemplateId;
	}
	
	// ##################### カレントのウィジェット関係 #####################
	/**
	 * 現在処理中のウィジェットのルートディレクトリへのパスを取得
	 *
	 * 例) http://www.magic3.org/magic3/widgets/xxxxx
	 */
	public function getCurrentWidgetRootUrl()
	{
		return M3_SYSTEM_ROOT_URL . '/widgets/' . $this->currentWidgetId;
	}
	/**
	 * 現在処理中のウィジェットのルートディレクトリへのパスを取得
	 */
	public function getCurrentWidgetRootPath()
	{
		return M3_SYSTEM_ROOT_PATH . DIRECTORY_SEPARATOR . 'widgets' . DIRECTORY_SEPARATOR . $this->currentWidgetId;
	}
	/**
	 * 現在処理中のウィジェットのincludeディレクトリへのパスを取得
	 */
	public function getCurrentWidgetIncludePath()
	{
		return M3_SYSTEM_ROOT_PATH . '/widgets/' . $this->currentWidgetId . '/include';
	}
	/**
	 * 現在処理中のウィジェットのdbディレクトリへのパスを取得
	 *
	 * 例) /var/www/html/magic3/widgets/xxxxx/include/db
	 */
	public function getCurrentWidgetDbPath()
	{
		return $this->getWidgetsPath() . DIRECTORY_SEPARATOR . $this->currentWidgetId . DIRECTORY_SEPARATOR . 'include' . DIRECTORY_SEPARATOR . 'db';
	}
	/**
	 * 現在処理中のウィジェットのcontainerディレクトリへのパスを取得
	 *
	 * 例) /var/www/html/magic3/widgets/xxxxx/include/container
	 */
	public function getCurrentWidgetContainerPath()
	{
		return $this->getWidgetsPath() . DIRECTORY_SEPARATOR . $this->currentWidgetId . DIRECTORY_SEPARATOR . 'include' . DIRECTORY_SEPARATOR . 'container';
	}
	/**
	 * 現在処理中のウィジェットのtemplateディレクトリへのパスを取得
	 *
	 * 例) /var/www/html/magic3/widgets/xxxxx/include/template
	 */
	public function getCurrentWidgetTemplatePath()
	{
		return $this->getWidgetsPath() . DIRECTORY_SEPARATOR . $this->currentWidgetId . DIRECTORY_SEPARATOR . 'include' . DIRECTORY_SEPARATOR . 'template';
	}
	/**
	 * 現在処理中のウィジェットのsqlディレクトリへのパスを取得
	 *
	 * 例) /var/www/html/magic3/widgets/xxxxx/include/sql
	 */
	public function getCurrentWidgetSqlPath()
	{
		return $this->getWidgetsPath() . DIRECTORY_SEPARATOR . $this->currentWidgetId . DIRECTORY_SEPARATOR . 'include' . DIRECTORY_SEPARATOR . 'sql';
	}
	/**
	 * 現在処理中のウィジェットのscriptsディレクトリへURLを取得
	 */
	public function getCurrentWidgetScriptsUrl()
	{
		return M3_SYSTEM_ROOT_URL . '/widgets/' . $this->currentWidgetId . '/scripts';
	}
	/**
	 * 現在処理中のウィジェット
	 */
	public function setCurrentWidgetId($id)
	{
		$this->currentWidgetId = $id;
	}
	/**
	 * 現在処理中のウィジェット
	 */
	public function getCurrentWidgetId()
	{
		return $this->currentWidgetId;
	}
	/**
	 * 現在処理中のインナーウィジェット
	 */
	public function setCurrentIWidgetId($id)
	{
		$this->currentIWidgetId = $id;
	}
	/**
	 * 現在処理中のインナーウィジェット
	 */
	public function getCurrentIWidgetId()
	{
		return $this->currentIWidgetId;
	}
	/**
	 * 現在作成中のウィジェットの定義ID
	 *
	 * @param int $id		定義ID
	 */
	public function setCurrentWidgetConfigId($id)
	{
		$this->currentWidgetConfigId = $id;
	}
	/**
	 * 現在作成中のウィジェットの定義ID
	 *
	 * @return int			定義ID
	 */
	public function getCurrentWidgetConfigId()
	{
		return $this->currentWidgetConfigId;
	}
	/**
	 * 現在作成中のインナーウィジェットの定義ID
	 *
	 * @param int $id		定義ID
	 */
	public function setCurrentIWidgetConfigId($id)
	{
		$this->currentIWidgetConfigId = $id;
	}
	/**
	 * 現在作成中のインナーウィジェットの定義ID
	 *
	 * @return int			定義ID
	 */
	public function getCurrentIWidgetConfigId()
	{
		return $this->currentIWidgetConfigId;
	}
	/**
	 * 現在処理中のウィジェットのプレフィックス文字列
	 */
	public function setCurrentWidgetPrefix($val)
	{
		$this->currentWidgetPrefix = $val;
	}
	/**
	 * 現在処理中のウィジェットのプレフィックス文字列
	 */
	public function getCurrentWidgetPrefix()
	{
		return $this->currentWidgetPrefix;
	}
	/**
	 * 現在処理を行っているページ定義のレコードのシリアル番号
	 *
	 * @param int $serial		シリアル番号
	 */
	public function setCurrentPageDefSerial($serial)
	{
		$this->currentPageDefSerial = $serial;
	}
	/**
	 * 現在処理を行っているページ定義のレコードのシリアル番号
	 *
	 * @return int			シリアル番号
	 */
	public function getCurrentPageDefSerial()
	{
		return $this->currentPageDefSerial;
	}
	// ##################### ユーザ情報 #####################
	/**
	 * 現在アクセス中のユーザ情報取得
	 *
	 * @return UserInfo		ユーザ情報。設定されていない場合はnullを返す。
	 */
	public function getCurrentUserInfo()
	{
		global $gInstanceManager;

		return $gInstanceManager->getUserInfo();
	}
	/**
	 * 現在アクセス中のユーザID取得
	 *
	 * @return int		ユーザID,ユーザが確定できないときは0
	 */
	public function getCurrentUserId()
	{
		global $gInstanceManager;

		$userInfo = $gInstanceManager->getUserInfo();
		if (is_null($userInfo)){		// ログインしていない場合
			return 0;
		} else {
			return $userInfo->userId;
		}
	}
	/**
	 * 現在アクセス中のユーザの名前を取得
	 *
	 * @return string		ユーザ名,ユーザが確定できないときは空文字列
	 */
	public function getCurrentUserName()
	{
		global $gInstanceManager;

		$userInfo = $gInstanceManager->getUserInfo();
		if (is_null($userInfo)){		// ログインしていない場合
			return '';
		} else {
			return $userInfo->name;
		}
	}
	/**
	 * 管理者用一時キーを取得
	 *
	 * @return string		管理者キー、管理者でないときは空文字列
	 */
	public function getAdminKey()
	{
		global $gInstanceManager;

		$userInfo = $gInstanceManager->getUserInfo();
		if (is_null($userInfo)){		// ログインしていない場合
			return '';
		} else {
			return $userInfo->_adminKey;
		}
	}
	/**
	 * 現在アクセス中のユーザに管理者権限があるかどうかを返す
	 *
	 * @return bool		true=ログイン中かつ管理者権限あり、false=未ログインまたはログイン中であるが管理者権限なし
	 */
	public function isSystemAdmin()
	{
		$isAdmin = false;
		$userInfo = $this->getCurrentUserInfo();
		if (is_null($userInfo)){		// ログインしていない場合
		} else {
			if ($userInfo->isSystemAdmin()){	// システム管理者の場合
				$isAdmin = true;
			}
		}
		return $isAdmin;
	}
	/**
	 * 現在アクセス中のユーザがログインしているか確認
	 *
	 * @return bool		true=ログイン中、false=未ログイン
	 */
	public function isCurrentUserLogined()
	{
		$userInfo = $this->getCurrentUserInfo();
		if (is_null($userInfo)){		// ログインしていない場合
			return false;
		} else {
			return true;
		}
	}
	/**
	 * 指定のユーザに管理者権限があるかどうかを返す
	 *
	 * @param int $userId	ユーザID
	 * @return bool			true=管理者権限あり、false=管理者権限なし
	 */
	public function isSystemAdminUser($userId)
	{
		return $this->db->isSystemAdmin($userId);
	}
	// ##################### アクセスログ #####################
	/**
	 * 現在のアクセスログのシリアル番号を返す
	 *
	 * @return int			アクセスログシリアル番号
	 */
	public function getCurrentAccessLogSerial()
	{
		global $gAccessManager;
		return $gAccessManager->getAccessLogSerialNo();
	}
	// ##################### ページ制御 #####################
	/**
	 * デフォルトのページID取得
	 *
	 * @return string			デフォルトのページID
	 */
	public function getDefaultPageId()
	{
		return self::DEFAULT_PAGE_ID;
	}
	/**
	 * 携帯用デフォルトのページID取得
	 *
	 * @return string			携帯用デフォルトのページID
	 */
	public function getDefaultMobilePageId()
	{
		return self::DEFAULT_MOBILE_PAGE_ID;
	}
	/**
	 * デフォルトの管理機能用ページID取得
	 *
	 * @return string			デフォルトの管理機能用ページID
	 */
	public function getDefaultAdminPageId()
	{
		return self::DEFAULT_ADMIN_PAGE_ID;
	}
	/**
	 * 現在のページID
	 */
	public function setCurrentPageId($id)
	{
		$this->currentPageId = $id;
		
		// 現在のページIDが変更のときは、デフォルトのページサブIDを更新
		if ($this->canUseDb) $this->defaultPageSubId = $this->db->getDefaultPageSubId($this->currentPageId);
	}
	/**
	 * 現在のページID
	 */
	public function getCurrentPageId()
	{
		return $this->currentPageId;
	}
	/**
	 * 現在のページサブID
	 */
	public function setCurrentPageSubId($id)
	{
		$this->currentPageSubId = $id;
	}
	/**
	 * 現在のページサブID
	 */
	public function getCurrentPageSubId()
	{
		return $this->currentPageSubId;
	}
	/**
	 * 現在のページのデフォルトのページサブID
	 *
	 * @return string 		デフォルトのページサブID
	 */
	public function getDefaultPageSubId()
	{
		return $this->defaultPageSubId;
	}
	/**
	 * 指定ページのデフォルトのページサブID
	 *
	 * @param string $pageId	ページID
	 * @return string 			デフォルトのページサブID
	 */
	public function getDefaultPageSubIdByPageId($pageId)
	{
		return $this->db->getDefaultPageSubId($pageId);
	}
	/**
	 * 現在のページID、サブページIDのURLを作成
	 *
	 * @return string		作成したURL
	 */
	public function createCurrentPageUrl()
	{
		return $this->getCurrentScriptUrl() . '?sub=' . $this->getCurrentPageSubId();
	}
	/**
	 * デフォルト言語取得
	 */
	public function getDefaultLanguage()
	{
		return $this->defaultLanguage;
	}
	/**
	 * デフォルトの言語名をカレントの言語で表したもの
	 */
	public function getDefaultLanguageNameByCurrentLanguage()
	{
		return $this->db->getLanguageNameByDispLanguageId($this->defaultLanguage, $this->currentLanguage);
	}
	/**
	 * カレント言語取得
	 */
	public function getCurrentLanguage()
	{
		return $this->currentLanguage;
	}
	/**
	 * カレント言語設定
	 */
	public function setCurrentLanguage($value)
	{
		$this->currentLanguage = $value;
	}
	/**
	 * カレント言語の変更可否を取得
	 */
	public function getCanChangeLang()
	{
		return $this->canChangeLang;
	}
	/**
	 * 多言語対応かどうかを取得
	 */
	public function getMultiLanguage()
	{
		return $this->multiLanguage;
	}
	/**
	 * 携帯用の入出力エンコーディングを取得
	 *
	 * @return string		エンコーディング
	 */
	public function getMobileEncoding()
	{
		return $this->mobileEncoding;
	}
	/**
	 * DBセッションが使用できるかどうか
	 */
	public function canUseDbSession()
	{
		return $this->canUseDbSession;
	}
	/**
	 * DBが使用可能かどうか
	 */
	public function canUseDb()
	{
		return $this->canUseDb;
	}
	/**
	 * Timestamp型データの初期値を取得
	 *
	 * @param string Timestmp型初期データ文字列
	 */
	public function getInitValueOfTimestamp()
	{
		if ($this->db->getDbType() == M3_DB_TYPE_MYSQL){		// MySQLの場合
			return M3_TIMESTAMP_INIT_VALUE_MYSQL;
		} else if ($this->db->getDbType() == M3_DB_TYPE_PGSQL){
			return M3_TIMESTAMP_INIT_VALUE_PGSQL;
		} else {
			return '';
		}
	}
	/**
	 * サイトの名称を取得
	 *
	 * @param bool $reload	データを再取得するかどうか
	 * @return string		サイト名称
	 */
	public function getSiteName($reload = false)
	{
		// DBが使用不可のときは空文字列を返す
		if (!$this->canUseDb) return '';
		
		if ($reload || empty($this->siteName)){			// サイト名称
			$this->siteName = $this->db->getSiteDefValue(M3_TB_FIELD_SITE_NAME);
		}
		return $this->siteName;
	}
	/**
	 * サイトの所有者を取得
	 *
	 * @param bool $reload	データを再取得するかどうか
	 * @return string		サイト所有者
	 */
	public function getSiteOwner($reload = false)
	{
		// DBが使用不可のときは空文字列を返す
		if (!$this->canUseDb) return '';
		
		if ($reload || empty($this->siteOwner)){			// サイト所有者
			$this->siteOwner = $this->db->getSiteDefValue(M3_TB_FIELD_SITE_OWNER);
		}
		return $this->siteOwner;
	}
	/**
	 * サイトコピーライトを取得
	 *
	 * @param bool $reload	データを再取得するかどうか
	 * @return string		サイトコピーライト
	 */
	public function getSiteCopyRight($reload = false)
	{
		// DBが使用不可のときは空文字列を返す
		if (!$this->canUseDb) return '';
		
		if ($reload || empty($this->siteCopyRight)){			// サイトコピーライト
			$this->siteCopyRight = $this->db->getSiteDefValue(M3_TB_FIELD_SITE_COPYRIGHT);
		}
		return $this->siteCopyRight;
	}
	/**
	 * サイトEメールを取得
	 *
	 * @param bool $reload	データを再取得するかどうか
	 * @return string		サイトEメール
	 */
	public function getSiteEmail($reload = false)
	{
		// DBが使用不可のときは空文字列を返す
		if (!$this->canUseDb) return '';
		
		if ($reload || empty($this->siteEmail)){			// サイトEメール
			$this->siteEmail = $this->db->getSiteDefValue(M3_TB_FIELD_SITE_EMAIL);
		}
		return $this->siteEmail;
	}
	/**
	 * デフォルトCSV区切り文字コードを取得
	 *
	 * @return string		区切り文字
	 */
	public function getDefaultCsvDelimCode()
	{
		return $this->db->getSystemConfig(self::DEFAULT_CSV_DELIM_CODE);
	}
	/**
	 * デフォルトCSV改行コードを取得
	 *
	 * @return string		改行文字
	 */
	public function getDefaultCsvNLCode()
	{
		return $this->db->getSystemConfig(self::DEFAULT_CSV_NL_CODE);
	}
	/**
	 * デフォルトCSVファイル拡張子を取得
	 *
	 * @return string		改行文字
	 */
	public function getDefaultCsvFileSuffix()
	{
		return $this->db->getSystemConfig(self::DEFAULT_CSV_FILE_SUFFIX);
	}
	/**
	 * ウィジェット実行ログを残すかどうか
	 *
	 * @return bool		true=ログ出力、false=ログ出力しない
	 */
	public function getWidgetLog()
	{
		return $this->widgetLog;
	}
	
	/**
	 * 携帯アクセスかどうかを設定
	 *
	 * @param bool $status			true=携帯アクセス、false=通常アクセス
	 * @return 			なし
	 */
	public function setIsMobile($status)
	{
		$this->isMobile = $status;
		
		// ##### 携帯用の設定 #####
		// セッションをURLに保存
		ini_set('session.use_cookies', 0);
	}
	/**
	 * 携帯アクセスかどうか
	 *
	 * @return bool		true=携帯アクセス、false=通常アクセス
	 */
	public function getIsMobile()
	{
		return $this->isMobile;
	}
}
?>
