<?php
/**
 * DBクラス
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    Magic3 Framework
 * @author     平田直毅(Naoki Hirata) <naoki@aplo.co.jp>
 * @copyright  Copyright 2006-2008 Magic3 Project.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id: wiki_mainDb.php 1081 2008-10-16 05:08:59Z fishbone $
 * @link       http://www.magic3.org
 */
require_once($gEnvManager->getDbPath() . '/baseDb.php');

class wiki_mainDb extends BaseDb
{
	/**
	 * ページが存在するかチェック
	 *
	 * @param string $name	ページ名
	 * @param string $type	ページタイプ
	 * @return				true=存在する、false=存在しない
	 */
	function isExistsPage($name, $type='')
	{
		$queryStr = 'SELECT * from wiki_content ';
		$queryStr .=  'WHERE wc_type = ? ';
		$queryStr .=    'AND wc_id = ? ';
		$queryStr .=    'AND wc_deleted = false';
		return $this->isRecordExists($queryStr, array($type, $name));
	}
	/**
	 * Wikiページデータの取得
	 *
	 * @param string  $name			ウィキページ名
	 * @param array   $row			レコード
	 * @param string  $type			ページタイプ
	 * @return bool					取得 = true, 取得なし= false
	 */
	function getPage($name, &$row, $type='')
	{
		$queryStr  = 'SELECT * FROM wiki_content ';
		$queryStr .=   'WHERE wc_deleted = false ';	// 削除されていない
		$queryStr .=    'AND wc_type = ? ';
		$queryStr .=   'AND wc_id = ? ';
		$ret = $this->selectRecord($queryStr, array($type, $name), $row);
		return $ret;
	}
	/**
	 * 取得可能なWikiページ名の取得
	 *
	 * @param string  $type			ページタイプ
	 * @return array				ページ名
	 */
	function getAvailablePages($type='')
	{
		$retValue = array();
		$queryStr  = 'SELECT wc_id FROM wiki_content ';
		$queryStr .=   'WHERE wc_deleted = false ';	// 削除されていない
		$queryStr .=    'AND wc_type = ? ';
		$queryStr .=    'ORDER BY wc_id';
		$ret = $this->selectRecords($queryStr, array($type), $rows);
		if ($ret){
			for ($i = 0; $i < count($rows); $i++){
				$retValue[] = $rows[$i]['wc_id'];
			}
		}
		return $retValue;
	}
	/**
	 * Wikiページの更新
	 *
	 * @param string  $name			ウィキページ名
	 * @param string  $data			データ
	 * @param string  $type			ページタイプ
	 * @param bool $keepTime		更新日時を維持するかどうか
	 * @return bool					true = 成功、false = 失敗
	 */
	function updatePage($name, $data, $type='', $keepTime=false)
	{
		global $gEnvManager;
		
		$historyIndex = 0;		// 履歴番号
		$now = date("Y/m/d H:i:s");	// 現在日時
		$contentNow = $now;			// コンテンツ更新日時
		$locked = 1;		// コンテンツロック状態
		$userId = $gEnvManager->getCurrentUserId();	// 現在のユーザ
		
		// トランザクションスタート
		$this->startTransaction();
				
		// 前レコードの削除状態チェック
		$queryStr = 'SELECT * FROM wiki_content ';
		$queryStr .=  'WHERE wc_type = ? ';
		$queryStr .=    'AND wc_id = ? ';
		$queryStr .=  'ORDER BY wc_history_index DESC ';
		$ret = $this->selectRecord($queryStr, array($type, $name), $row);
		if ($ret){
			if (!$row['wc_deleted']){		// レコードが削除されていなければ削除
				// 古いレコードを削除
				$queryStr  = 'UPDATE wiki_content ';
				$queryStr .=   'SET wc_deleted = true, ';	// 削除
				$queryStr .=     'wc_update_user_id = ?, ';
				$queryStr .=     'wc_update_dt = ? ';
				$queryStr .=   'WHERE wc_serial = ?';
				$this->execStatement($queryStr, array($userId, $now, $row['wc_serial']));
			}
			$historyIndex = $row['wc_history_index'] + 1;
			$locked = $row['wc_locked'];
			
			// 更新日時を維持のときは更新日時取得
			if ($keepTime) $contentNow = $row['wc_content_dt'];
		}

		// 新規レコード追加
		$queryStr = 'INSERT INTO wiki_content ';
		$queryStr .=  '(wc_type, wc_id, wc_history_index, wc_data, wc_content_dt, wc_locked, wc_create_user_id, wc_create_dt) ';
		$queryStr .=  'VALUES ';
		$queryStr .=  '(?, ?, ?, ?, ?, ?, ?, ?)';
		$this->execStatement($queryStr, array($type, $name, $historyIndex, $data, $contentNow, $locked, $userId, $now));
		
		// トランザクション終了
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * Wikiページのロックの制御
	 *
	 * @param string  $name			ウィキページ名
	 * @param bool   $lock			true=ロック、false=ロック解除
	 * @param string  $type			ページタイプ
	 * @return bool					true=成功、false=失敗
	 */
	function lockPage($name, $lock, $type='')
	{
		global $gEnvManager;
		
		$now = date("Y/m/d H:i:s");	// 現在日時
		$userId = $gEnvManager->getCurrentUserId();	// 現在のユーザ
		
		// トランザクションスタート
		$this->startTransaction();
				
		// レコードを取得
		$queryStr = 'SELECT * from wiki_content ';
		$queryStr .=  'WHERE wc_type = ? ';
		$queryStr .=    'AND wc_id = ? ';
		$queryStr .=    'AND wc_deleted = false';
		$ret = $this->selectRecord($queryStr, array($type, $name), $row);
		if (!$ret){
			// トランザクション終了
			$ret = $this->endTransaction();
			return false;
		}
		
		// ロック状態を変更
		$lockValue = 0;		// ロック状態
		if ($lock) $lockValue = 1;
		
		$queryStr  = 'UPDATE wiki_content ';
		$queryStr .=   'SET wc_locked = ?, ';	// ロック状態
		$queryStr .=     'wc_check_out_user_id = ?, ';
		$queryStr .=     'wc_check_out_dt = ? ';
		$queryStr .=   'WHERE wc_serial = ?';
		$this->execStatement($queryStr, array($lockValue, $userId, $now, $row['wc_serial']));
		
		// トランザクション終了
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * Wikiページその他データの更新
	 *
	 * @param string  $name			ウィキページ名
	 * @param string  $data			データ
	 * @param string  $type			ページタイプ
	 * @return bool					true = 成功、false = 失敗
	 */
	function updatePageOther($name, $data, $type)
	{
		global $gEnvManager;
		
		$now = date("Y/m/d H:i:s");	// 現在日時
		$userId = $gEnvManager->getCurrentUserId();	// 現在のユーザ
		
		// トランザクションスタート
		$this->startTransaction();

		$queryStr = 'SELECT * FROM wiki_content ';
		$queryStr .=  'WHERE wc_type = ? ';
		$queryStr .=    'AND wc_id = ? ';
		$queryStr .=    'AND wc_history_index = 0';
		$ret = $this->selectRecord($queryStr, array($type, $name), $row);
		if ($ret){
			$queryStr  = 'UPDATE wiki_content ';
			$queryStr .=   'SET wc_data = ?, ';
			$queryStr .=     'wc_update_user_id = ?, ';
			$queryStr .=     'wc_update_dt = ? ';
			$queryStr .=   'WHERE wc_serial = ?';
			$this->execStatement($queryStr, array($data, $userId, $now, $row['wc_serial']));			
		} else {
			$queryStr  = 'INSERT INTO wiki_content (';
			$queryStr .=   'wc_type, ';
			$queryStr .=   'wc_id, ';
			$queryStr .=   'wc_data, ';
			$queryStr .=   'wc_create_user_id, ';
			$queryStr .=   'wc_create_dt ';
			$queryStr .= ') VALUES (';
			$queryStr .=   '?, ?, ?, ?, ?';
			$queryStr .= ')';
			$this->execStatement($queryStr, array($type, $name, $data, $userId, $now));	
		}
		// トランザクション終了
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * Wikiページその他データを取得
	 *
	 * @param string  $name			ウィキページ名
	 * @param string  $type			ページタイプ
	 * @return string				取得値
	 */
	function getPageOther($name, $type)
	{
		$queryStr = 'SELECT * FROM wiki_content ';
		$queryStr .=  'WHERE wc_type = ? ';
		$queryStr .=    'AND wc_id = ? ';
		$queryStr .=    'AND wc_history_index = 0';
		$ret = $this->selectRecord($queryStr, array($type, $name), $row);
		if ($ret){
			return $row['wc_data'];
		} else {
			return '';
		}
	}
	/**
	 * Wikiページその他データを削除
	 *
	 * @param string $name			ページ名(空文字列のときは同じタイプのデータをすべて削除)
	 * @param string  $type			ページタイプ
	 * @return bool					true = 成功、false = 失敗
	 */
	function clearPageOther($name, $type)
	{
		// トランザクションスタート
		$this->startTransaction();
		
		$param = array();
		$queryStr  = 'DELETE FROM wiki_content ';
		$queryStr .=  'WHERE wc_type = ? '; $param[] = $type;
		if (!empty($name)){
			$queryStr .=  'AND wc_id = ? '; $param[] = $name;
		}
		$this->execStatement($queryStr, $param);
		
		// トランザクション終了
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * 設定値が存在するかチェック
	 *
	 * @param string $id	定義ID
	 * @return				true=存在する、false=存在しない
	 */
	function isExistsConfig($id)
	{
		$queryStr = 'SELECT * from wiki_config ';
		$queryStr .=  'WHERE wg_id = ? ';
		return $this->isRecordExists($queryStr, array($id));
	}
	/**
	 * 設定値を定義IDで取得
	 *
	 * @param string $id		定義ID
	 * @return string			定義値
	 */
	function getConfig($id)
	{
		$queryStr  = 'SELECT * FROM wiki_config ';
		$queryStr .=   'WHERE wg_id = ? ';
		$ret = $this->selectRecord($queryStr, array($id), $row);
		if ($ret){
			return $row['wg_value'];
		} else {
			return '';
		}
	}
	/**
	 * 設定値の更新
	 *
	 * @param string $id			定義ID
	 * @param string  $data			データ
	 * @param string  $name			名前
	 * @param string  $desc			説明
	 * @return bool					true = 成功、false = 失敗
	 */
	function updateConfig($id, $data, $name='', $desc='')
	{
		// トランザクション開始
		$this->startTransaction();

		$queryStr  = 'SELECT * FROM wiki_config ';
		$queryStr .=   'WHERE wg_id = ? ';	
		$ret = $this->selectRecord($queryStr, array($id), $row);
		if ($ret){
			$queryStr  = 'UPDATE wiki_config ';
			$queryStr .=   'SET wg_value = ?, ';
			$queryStr .=     'wg_name = ?, ';
			$queryStr .=     'wg_description = ? ';
			$queryStr .=   'WHERE wg_id = ?';
			$this->execStatement($queryStr, array($data, $row['wg_name'], $row['wg_description'], $id));			
		} else {
			$queryStr = 'INSERT INTO wiki_config (';
			$queryStr .=  'wg_id, ';
			$queryStr .=  'wg_value, ';
			$queryStr .=  'wg_name, ';
			$queryStr .=  'wg_description ';
			$queryStr .=  ') VALUES (';
			$queryStr .=  '?, ?, ?, ?';
			$queryStr .=  ')';
			$this->execStatement($queryStr, array($id, $data, $name, $desc));	
		}
				
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
}
?>
