/*!
 * @file GLObject.cpp
 * @brief 3D Object
 */

#include "GLObject.h"


void OuterProduct(float *vec0,float *vec1,float *result){
  float xx,yy,zz,norm;
  xx=vec0[1]*vec1[2]-vec0[2]*vec1[1];
  yy=vec0[2]*vec1[0]-vec0[0]*vec1[2];
  zz=vec0[0]*vec1[1]-vec0[1]*vec1[0];

  norm=sqrt(xx*xx+yy*yy+zz*zz);
  result[0]=xx/norm;
  result[1]=yy/norm;
  result[2]=zz/norm;
}


void VectorMean(float *vlist,int length,float *result){
  int i,idx;
  float xsum=0.0,ysum=0.0,zsum=0.0;

  for(i=0;i<length;i++){
    idx=i*3;
    xsum+=vlist[idx]/length;
    ysum+=vlist[idx+1]/length;
    zsum+=vlist[idx+2]/length;
  }
  result[0]=xsum;
  result[1]=ysum;
  result[2]=zsum;
}


///////////
void GLObject::setTranslate(float tx,float ty,float tz){
  transx=tx;
  transy=ty;
  transz=tz;
}
void GLObject::setRotate(float rx,float ry,float rz){
  rotx=rx;
  roty=ry;
  rotz=rz;
}
void GLObject::translate(){
  glTranslatef(transx,transy,transz);
}
void GLObject::rotate(){
  glRotatef(rotz,0.0,0.0,1.0);
  glRotatef(roty,0.0,1.0,0.0);
  glRotatef(rotx,1.0,0.0,0.0);
}

//////////////////////////////

GLDummyObject::GLDummyObject(){
  mcolor=new float[4];
  mcolor[0]=1.0;
  mcolor[1]=1.0;
  mcolor[2]=1.0;
  mcolor[3]=1.0;
  count=0;


  listid=glGenLists(1);
  glNewList(listid,GL_COMPILE);

  glBegin(GL_TRIANGLES);
  mcolor[0]=1.0;
  mcolor[1]=0.0;
  mcolor[2]=0.0;
  glMaterialfv(GL_FRONT_AND_BACK , GL_DIFFUSE , mcolor);
  //glNormal3f(0.0,-0.706,-0.706);
  glNormal3f(-0.577,-0.577,-0.577);
  glVertex3f(-150,-150,0);
  glNormal3f(0.577,-0.577,-0.577);
  glVertex3f(150,-150,0);
  glNormal3f(0.0,0.0,-1.0);
  glVertex3f(0,0,-150);
  mcolor[0]=0.0;
  mcolor[1]=1.0;
  mcolor[2]=0.0;
  glMaterialfv(GL_FRONT_AND_BACK , GL_DIFFUSE , mcolor);
  //glNormal3f(0.706,0.0,-0.706);
  glNormal3f(0.577,-0.577,-0.577);
  glVertex3f(150,-150,0);
  glNormal3f(0.577,0.577,-0.577);
  glVertex3f(150,150,0);
  glNormal3f(0.0,0.0,-1.0);
  glVertex3f(0,0,-150);
  mcolor[0]=0.0;
  mcolor[1]=0.0;
  mcolor[2]=1.0;
  glMaterialfv(GL_FRONT_AND_BACK , GL_DIFFUSE , mcolor);
  //glNormal3f(0.0,0.706,-0.706);
  glNormal3f(0.577,0.577,-0.577);
  glVertex3f(150,150,0);
  glNormal3f(-0.577,0.577,-0.577);
  glVertex3f(-150,150,0);
  glNormal3f(0.0,0.0,-1.0);
  glVertex3f(0,0,-150);
  mcolor[0]=1.0;
  mcolor[1]=1.0;
  mcolor[2]=1.0;
  glMaterialfv(GL_FRONT_AND_BACK , GL_DIFFUSE , mcolor);
  //glNormal3f(-0.706,0.0,-0.706);
  glNormal3f(-0.577,0.577,-0.577);
  glVertex3f(-150,150,0);
  glNormal3f(-0.577,-0.577,-0.577);
  glVertex3f(-150,-150,0);
  glNormal3f(0.0,0.0,-1.0);
  glVertex3f(0,0,-150);
  glEnd();
  
  glEndList();

}

GLDummyObject::~GLDummyObject(){
  delete[] mcolor;
}


void GLDummyObject::reset(){
  count=0;
}

void GLDummyObject::glDraw(){
  glPushMatrix();

  glRotatef(count*1.5, 0.0, 0.0, 1.0);

  /*
  glBegin(GL_TRIANGLES);
  mcolor[0]=1.0;
  mcolor[1]=0.0;
  mcolor[2]=0.0;
  glMaterialfv(GL_FRONT_AND_BACK , GL_DIFFUSE , mcolor);
  //glNormal3f(0.0,-0.706,-0.706);
  glNormal3f(-0.577,-0.577,-0.577);
  glVertex3f(-150,-150,0);
  glNormal3f(0.577,-0.577,-0.577);
  glVertex3f(150,-150,0);
  glNormal3f(0.0,0.0,-1.0);
  glVertex3f(0,0,-150);
  mcolor[0]=0.0;
  mcolor[1]=1.0;
  mcolor[2]=0.0;
  glMaterialfv(GL_FRONT_AND_BACK , GL_DIFFUSE , mcolor);
  //glNormal3f(0.706,0.0,-0.706);
  glNormal3f(0.577,-0.577,-0.577);
  glVertex3f(150,-150,0);
  glNormal3f(0.577,0.577,-0.577);
  glVertex3f(150,150,0);
  glNormal3f(0.0,0.0,-1.0);
  glVertex3f(0,0,-150);
  mcolor[0]=0.0;
  mcolor[1]=0.0;
  mcolor[2]=1.0;
  glMaterialfv(GL_FRONT_AND_BACK , GL_DIFFUSE , mcolor);
  //glNormal3f(0.0,0.706,-0.706);
  glNormal3f(0.577,0.577,-0.577);
  glVertex3f(150,150,0);
  glNormal3f(-0.577,0.577,-0.577);
  glVertex3f(-150,150,0);
  glNormal3f(0.0,0.0,-1.0);
  glVertex3f(0,0,-150);
  mcolor[0]=1.0;
  mcolor[1]=1.0;
  mcolor[2]=1.0;
  glMaterialfv(GL_FRONT_AND_BACK , GL_DIFFUSE , mcolor);
  //glNormal3f(-0.706,0.0,-0.706);
  glNormal3f(-0.577,0.577,-0.577);
  glVertex3f(-150,150,0);
  glNormal3f(-0.577,-0.577,-0.577);
  glVertex3f(-150,-150,0);
  glNormal3f(0.0,0.0,-1.0);
  glVertex3f(0,0,-150);
  glEnd();
  */

  glCallList(listid);

  glPopMatrix();

  count=(count+1)%240;
}

/////////////////////////////////////

GLFeaturePoints::GLFeaturePoints(){
  count=0;
  color=new float[3];
  color[0]=1.0;
  color[1]=0.0;
  color[2]=0.0;

  pointsize=3.0;

  use_color_list=false;
}

GLFeaturePoints::~GLFeaturePoints(){
  delete[] color;
}

void GLFeaturePoints::glDraw(){
  glPushMatrix();
  translate();
  rotate();

  glPushAttrib(GL_ALL_ATTRIB_BITS);
  glDisable(GL_LIGHTING);

  glPointSize(pointsize);

  if(!use_color_list){
    glColor3fv(color);

    glBegin(GL_POINTS);

    for(int i=0;i<count;i++){
      glVertex3fv(&points3d[i*3]);
    }

    glEnd();
  }else{
    glBegin(GL_POINTS);

    for(int i=0;i<count;i++){
      glColor3fv(&colorlist[i*3]);
      glVertex3fv(&points3d[i*3]);
    }

    glEnd();
  }

  glEnable(GL_LIGHTING);
  glPopAttrib();

  glPopMatrix();
}

void GLFeaturePoints::setPoints(int num,float *pts){
  count=num;
  points3d.resize(count*3);
  memcpy(&points3d[0],pts,sizeof(float)*count*3);
}
void GLFeaturePoints::setPoints(vector<float> &pts){
  count=pts.size()/3;
  points3d.assign(pts.begin(),pts.end());
}

///////////////////////////////////////

GLGrid::GLGrid(){
  color=new float[3];
  color[0]=color[1]=color[2]=0.5;
  linewidth=3.0;
  gridnum=10;
  gridsize=50.0;
}
GLGrid::~GLGrid(){
  delete[] color;
}
void GLGrid::glDraw(){
  float xy0,xy1;
  float xy;

  xy0=-gridsize*(float)gridnum*0.5;
  xy1=-xy0;

  glPushMatrix();
  translate();
  rotate();

  glPushAttrib(GL_ALL_ATTRIB_BITS);
  glDisable(GL_LIGHTING);

  glColor3fv(color);
  glLineWidth(linewidth);
  
  glBegin(GL_LINES);

  for(int i=0;i<(gridnum+1);i++){
    xy=xy0+gridsize*i;
    glVertex3f(xy,xy0,0);
    glVertex3f(xy,xy1,0);
    glVertex3f(xy0,xy,0);
    glVertex3f(xy1,xy,0);
  }

  glEnd();

  glEnable(GL_LIGHTING);
  glPopAttrib();

  glPopMatrix();
}

void GLGrid::setGrid(int num,float size){
  gridnum=num;
  gridsize=size;
}

////////////////////////////////////

GLCoordinate::GLCoordinate(){
  asize=200.0;
  linewidth=4.0;
  red[0]=1.0; red[1]=0.0; red[2]=0.0;
  green[0]=0.0; green[1]=1.0; green[2]=0.0;
  blue[0]=0.0; blue[1]=0.0; blue[2]=1.0;
}

GLCoordinate::~GLCoordinate(){

}

void GLCoordinate::glDraw(){
  glPushMatrix();
  translate();
  rotate();

  glPushAttrib(GL_ALL_ATTRIB_BITS);
  glDisable(GL_LIGHTING);

  glColor3fv(red);
  glLineWidth(linewidth);
  glBegin(GL_LINES);
  glVertex3f(0,0,0);
  glVertex3f(-asize,0,0);
  glEnd();

  glColor3fv(green);
  glLineWidth(linewidth);
  glBegin(GL_LINES);
  glVertex3f(0,0,0);
  glVertex3f(0,-asize,0);
  glEnd();

  glColor3fv(blue);
  glLineWidth(linewidth);
  glBegin(GL_LINES);
  glVertex3f(0,0,0);
  glVertex3f(0,0,-asize);
  glEnd();

  glEnable(GL_LIGHTING);
  glPopAttrib();

  glPopMatrix();
}


////////////////////////////////////

/*
GLPolygon::GLPolygon(){
  isTexture=false;
  glGenTextures(1 , &texi);
  listid=glGenLists(1);
  color=new float[4];
  for(int i=0;i<4;i++) color[i]=1.0;
  norms.resize(0);

  rotomega=0.0;
}

GLPolygon::~GLPolygon(){
  glDeleteTextures(1 , &texi);
  glDeleteLists(listid,1);
  delete[] color;
}

void GLPolygon::setTexture(int width,int height,unsigned char depth,unsigned char *imgbuf){
  tex_img.resize(width*height*depth);
  memcpy(&tex_img[0],imgbuf,width*height*depth);
  glBindTexture(GL_TEXTURE_2D , texi);
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
  glTexImage2D(GL_TEXTURE_2D , 0 , depth , width , height ,
	       0 , (depth==3?GL_RGB:GL_RGBA) , GL_UNSIGNED_BYTE , &tex_img[0]);
  isTexture=true;
}

void GLPolygon::setGLList(){
  glNewList(listid,GL_COMPILE);

  cerr << "verts.size()= " << verts.size() << endl;
  cerr << "norms.size()= " << norms.size() << endl;
  cerr << "isTexture= " << isTexture << endl;

  glPushAttrib(GL_ALL_ATTRIB_BITS);

  if(isTexture){
    glEnable(GL_TEXTURE_2D);
    //glEnable(GL_ALPHA_TEST);
    //glAlphaFunc(GL_GREATER, 0);
    glEnable(GL_BLEND);
    //glBlendFunc(GL_SRC_ALPHA , GL_ONE);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glBindTexture(GL_TEXTURE_2D , texi);
  }
  glBegin(GL_TRIANGLE_STRIP);
  for(int i=0;i<verts.size()/3;i++){
    if(isTexture){
      glTexCoord2fv(&texco[i*2]);
    }else{
      glMaterialfv(GL_FRONT_AND_BACK , GL_DIFFUSE, color);
      if(norms.empty()) glNormal3f(0.0,0.0,-1.0);
      else glNormal3fv(&norms[i*3]);
    }
    glVertex3fv(&verts[i*3]);
  }
  glEnd();
  if(isTexture){
    glDisable(GL_TEXTURE_2D);
    //glDisable(GL_ALPHA_TEST);
    glDisable(GL_BLEND);
  }

  glPopAttrib();

  glEndList();
}

void GLPolygon::glDraw(){
  glPushMatrix();
  glPushAttrib(GL_ALL_ATTRIB_BITS);
  translate();
  rotate();
  glRotatef(rotangle,0.0,0.0,1.0);
  if(isTexture){
    glDisable(GL_LIGHTING);
    glEnable(GL_TEXTURE_2D);
    //glEnable(GL_ALPHA_TEST);
    //glAlphaFunc(GL_GREATER, 0);
    glEnable(GL_BLEND);
    //glBlendFunc(GL_SRC_ALPHA , GL_ONE);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
    glBindTexture(GL_TEXTURE_2D , texi);
  }
  glCallList(listid);
  if(isTexture){
    glDisable(GL_TEXTURE_2D);
    //glDisable(GL_ALPHA_TEST);
    glDisable(GL_BLEND);
    glEnable(GL_LIGHTING);
  }
  glPopAttrib();
  glPopMatrix();
  rotangle+=rotomega;
  if(rotangle>360.0) rotangle-=360.0;
}

void setRectanglePolygon(GLPolygon *glpoly,float width,float height){
  vector<float> verts;
  vector<float> texco;

  float hwidth=width*0.5;
  float hheight=height*0.5;
  
  verts.resize(12);
  texco.resize(8);

  verts[0]=-hwidth;verts[1]=-hheight;verts[2]=0.0;
  verts[3]=-hwidth;verts[4]=hheight;verts[5]=0.0;
  verts[6]=hwidth;verts[7]=-hheight;verts[8]=0.0;
  verts[9]=hwidth;verts[10]=hheight;verts[11]=0.0;

  texco[0]=0.0;texco[1]=0.0;
  texco[2]=0.0;texco[3]=1.0;
  texco[4]=1.0;texco[5]=0.0;
  texco[6]=1.0;texco[7]=1.0;

  glpoly->setVertices(verts);
  glpoly->setTexCoords(texco);
}
*/

/////////////////////////////////////

GLMaterial::GLMaterial(){
  use_texture=false;
  glGenTextures(1 , &texi);
  //cout << "texture id:" << texi << endl;
  mtlcolor.resize(4);
  for(int i=0;i<4;i++) mtlcolor[i]=1.0;
}

GLMaterial::~GLMaterial(){
  glDeleteTextures(1 , &texi);
}

void GLMaterial::setMaterialColor(float *color){
  memcpy(&mtlcolor[0],color,sizeof(float)*4);
}

void GLMaterial::setTexture(int width,int height,unsigned char depth,unsigned char *imgbuf){
  imagebuf.resize(width*height*depth);
  memcpy(&imagebuf[0],imgbuf,width*height*depth);
  glBindTexture(GL_TEXTURE_2D , texi);
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
  /*
  glTexImage2D(GL_TEXTURE_2D , 0 , depth , width , height ,
	       0 , (depth==3?GL_RGB:GL_RGBA) , GL_UNSIGNED_BYTE , &imagebuf[0]);
  */
  gluBuild2DMipmaps(GL_TEXTURE_2D, depth, width, height,
		    (depth==3?GL_RGB:GL_RGBA), GL_UNSIGNED_BYTE, &imagebuf[0]);
  use_texture=true;
}

void GLMaterial::subTexture(int xoff,int yoff,int width,int height,unsigned char depth,unsigned char *imgbuf){
  glBindTexture(GL_TEXTURE_2D , texi);
  glTexSubImage2D(GL_TEXTURE_2D , 0 , xoff, yoff , width , height ,
		  (depth==3?GL_RGB:GL_RGBA) , GL_UNSIGNED_BYTE , imgbuf);
}

void GLMaterial::glBeginMaterial(){
  if(use_texture){
    glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D , texi);
    glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
    glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
  }
  glMaterialfv(GL_FRONT_AND_BACK , GL_DIFFUSE , &mtlcolor[0]);
}

void GLMaterial::glEndMaterial(){
  if(use_texture){
    glDisable(GL_TEXTURE_2D);
  }
}

///////////////////////////

GLPolygon::GLPolygon(){
  listid=-1;
  vlength=0;
  flength=0;

  normtype=0;
  materials.clear();

  transx=transy=transz=0.0;
  rotx=roty=rotz=0.0;

  listid=glGenLists(1);

  handco=1.0;

  uselist=true;
  isvisible=true;
  type="GLPolygon";
}

GLPolygon::~GLPolygon(){
  glDeleteLists(listid,1);
}

void GLPolygon::setVertices(float *verts,int length){
  vlength=length;
  vertices.resize(vlength*3);
  memcpy(&vertices[0],verts,sizeof(float)*vlength*3);
}

void GLPolygon::setVertices(vector<float> &verts){
  vlength=verts.size()/3;
  vertices.assign(verts.begin(),verts.end());
}

void GLPolygon::setFaces(int *fcs,int length){
  flength=length;
  faces.resize(flength*3);
  memcpy(&faces[0],fcs,sizeof(int)*flength*3);
}

void GLPolygon::setFaces(vector<int> &fcs){
  flength=fcs.size()/3;
  faces.assign(fcs.begin(),fcs.end());
}

void GLPolygon::setMaterialList(GLMaterial **mtls,int length){
  materials.resize(length);
  memcpy(&materials[0],mtls,sizeof(GLMaterial *)*length);
}
void GLPolygon::setMaterialList(vector<GLMaterial *> &mtls){
  materials.assign(mtls.begin(),mtls.end());
}

void GLPolygon::destructMaterialList(){
  for(int i=0;i<materials.size();i++){
    delete materials[i];
  }
  materials.clear();
}


void GLPolygon::setMaterial(GLMaterial *mtl){
  materials.resize(1);
  materials[0]=mtl;
}

void GLPolygon::addMaterial(GLMaterial *mtl){
  materials.push_back(mtl);
}

void GLPolygon::setNormals(float *norms){
  normals.resize(flength*3);
  memcpy(&normals[0],norms,sizeof(float)*flength*3);
}

void GLPolygon::setNormals(vector<float> &norms){
  normals.assign(norms.begin(),norms.end());
}

void GLPolygon::calcNormals(){
  int i,idx0,idx1,idx2;
  float x0,y0,z0,x1,y1,z1,x2,y2,z2;
  float norm[3];
  float v0[3],v1[3];

  normals.resize(flength*3);

  for(i=0;i<flength;i++){
    idx0=faces[i*3]*3;
    idx1=faces[(i*3+1)]*3;
    idx2=faces[(i*3+2)]*3;
    norm[0]=0.0;
    norm[1]=0.0;
    norm[2]=0.0;
    if(idx0>=0 && idx1>=0 && idx2>=0){
      /*printf("%d/%d,%d/%d,%d/%d\n",
	     idx0/3,vlength,
	     idx1/3,vlength,
	     idx2/3,vlength);*/
      x0=vertices[idx0];
      y0=vertices[idx0+1];
      z0=vertices[idx0+2];
      x1=vertices[idx1];
      y1=vertices[idx1+1];
      z1=vertices[idx1+2];
      x2=vertices[idx2];
      y2=vertices[idx2+1];
      z2=vertices[idx2+2];
      v0[0]=x1-x0;
      v0[1]=y1-y0;
      v0[2]=z1-z0;
      v1[0]=x2-x0;
      v1[1]=y2-y0;
      v1[2]=z2-z0;
      OuterProduct(v0,v1,norm);
    }
    normals[i*3]=norm[0];
    normals[i*3+1]=norm[1];
    normals[i*3+2]=norm[2];
  }
}

void GLPolygon::setVertexNormals(float *vnorms){
  vnormals.resize(vlength*3);
  memcpy(&vnormals[0],vnorms,sizeof(float)*vlength*3);
}

void GLPolygon::setVertexNormals(vector<float> &vnorms){
  vnormals.assign(vnorms.begin(),vnorms.end());
}


void GLPolygon::calcVertexNormals(){
  vnormals.resize(vlength*3);

  int i,j,k,vidx,fidx;
  float xx,yy,zz,norm;

  for(i=0;i<vlength;i++){
    vidx=i*3;
    xx=yy=zz=0.0;
    for(j=0;j<flength;j++){
      fidx=j*3;
      if(faces[fidx]!=-1){
	for(k=0;k<3;k++){
	  if(faces[fidx+k]==i){
	    xx+=normals[fidx];
	    yy+=normals[fidx+1];
	    zz+=normals[fidx+2];
	    break;
	  }
	}
      }
    }
    norm=sqrt(xx*xx+yy*yy+zz*zz);
    if(norm>1e-5){
      vnormals[vidx]=xx/norm;
      vnormals[vidx+1]=yy/norm;
      vnormals[vidx+2]=zz/norm;
    }else{
      vnormals[vidx]=0.0;
      vnormals[vidx+1]=0.0;
      vnormals[vidx+2]=0.0;
    }
  }
}

void GLPolygon::drawFaces(){
  int i,idx,xidx;
  int f0,f1,f2;
  int tf0,tf1,tf2;
  float mcolor[4];
  mcolor[3]=1.0;
  GLMaterial *curmtl=materials[0];

  //glBegin(GL_TRIANGLES);
  curmtl->glBeginMaterial();
  for(i=0;i<flength;i++){
    idx=i*3;
    f0=faces[idx];
    f1=faces[idx+1];
    f2=faces[idx+2];

    if(f0==-1){
      curmtl->glEndMaterial();
      curmtl=materials[f2];
      curmtl->glBeginMaterial();
      /*
      xidx=f2*3;
      mcolor[0]=material_colors[xidx];
      mcolor[1]=material_colors[xidx+1];
      mcolor[2]=material_colors[xidx+2];
      glMaterialfv(GL_FRONT_AND_BACK , GL_DIFFUSE , mcolor);
      */      
    }else{
      glBegin(GL_TRIANGLES);
      if(curmtl->isTexture()){
	tf0=tfaces[idx];
	tf1=tfaces[idx+1];
	tf2=tfaces[idx+2];

	switch(normtype){
	case 0:// face normal
	  glNormal3f(normals[idx],normals[idx+1],handco*normals[idx+2]);

	  xidx=tf0*2;
	  glTexCoord2fv(&tvertices[xidx]);
	  xidx=f0*3;
	  glVertex3f(vertices[xidx],vertices[xidx+1],handco*vertices[xidx+2]);
	  //glVertex3fv(&vertices[xidx]);

	  xidx=tf1*2;
	  glTexCoord2fv(&tvertices[xidx]);
	  xidx=f1*3;
	  glVertex3f(vertices[xidx],vertices[xidx+1],handco*vertices[xidx+2]);

	  xidx=tf2*2;
	  glTexCoord2fv(&tvertices[xidx]);
	  xidx=f2*3;
	  glVertex3f(vertices[xidx],vertices[xidx+1],handco*vertices[xidx+2]);
	break;
	case 1:// vertex normal
	  xidx=tf0*2;
	  glTexCoord2fv(&tvertices[xidx]);
	  xidx=f0*3;
	  glNormal3f(vnormals[xidx],vnormals[xidx+1],handco*vnormals[xidx+2]);
	  glVertex3f(vertices[xidx],vertices[xidx+1],handco*vertices[xidx+2]);

	  xidx=tf1*2;
	  glTexCoord2fv(&tvertices[xidx]);
	  xidx=f1*3;
	  glNormal3f(vnormals[xidx],vnormals[xidx+1],handco*vnormals[xidx+2]);
	  glVertex3f(vertices[xidx],vertices[xidx+1],handco*vertices[xidx+2]);

	  xidx=tf2*2;
	  glTexCoord2fv(&tvertices[xidx]);
	  xidx=f2*3;
	  glNormal3f(vnormals[xidx],vnormals[xidx+1],handco*vnormals[xidx+2]);
	  glVertex3f(vertices[xidx],vertices[xidx+1],handco*vertices[xidx+2]);
	  break;
	}
      }else{
	switch(normtype){
	case 0:// face normal
	  glNormal3f(normals[idx],normals[idx+1],handco*normals[idx+2]);
	  xidx=f0*3;
	  glVertex3f(vertices[xidx],vertices[xidx+1],handco*vertices[xidx+2]);
	  xidx=f1*3;
	  glVertex3f(vertices[xidx],vertices[xidx+1],handco*vertices[xidx+2]);
	  xidx=f2*3;
	  glVertex3f(vertices[xidx],vertices[xidx+1],handco*vertices[xidx+2]);
	break;
	case 1:// vertex normal
	  xidx=f0*3;
	  glNormal3f(vnormals[xidx],vnormals[xidx+1],handco*vnormals[xidx+2]);
	  glVertex3f(vertices[xidx],vertices[xidx+1],handco*vertices[xidx+2]);
	  xidx=f1*3;
	  glNormal3f(vnormals[xidx],vnormals[xidx+1],handco*vnormals[xidx+2]);
	  glVertex3f(vertices[xidx],vertices[xidx+1],handco*vertices[xidx+2]);
	  xidx=f2*3;
	  glNormal3f(vnormals[xidx],vnormals[xidx+1],handco*vnormals[xidx+2]);
	  glVertex3f(vertices[xidx],vertices[xidx+1],handco*vertices[xidx+2]);
	  break;
	}
      }
      glEnd();
    }

  }
  curmtl->glEndMaterial();
  //glEnd();
}


void GLPolygon::setGLList(){
  glNewList(listid,GL_COMPILE);
  drawFaces();
  glEndList();
}


void GLPolygon::glDraw(){
  if(!isvisible) return;
  glPushMatrix();
  //glEnable(GL_TEXTURE_2D);
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

  translate();
  rotate();
  if(uselist){
    glCallList(listid);
  }else{
    //cout << "unuse list" <<endl;
    drawFaces();
  }
  glDisable(GL_TEXTURE_2D);
  glDisable(GL_BLEND);
  glPopMatrix();
}


//////

GLObjObject::GLObjObject(){
  materials.clear();
  glpoly=new GLPolygon();
  type="GLObjObject";
}

GLObjObject::~GLObjObject(){
  for(int i=0;i<materials.size();i++){
    delete materials[i];
  }
  delete glpoly;
}

void GLObjObject::setMaterialColors(float *mtlc,int length){
  mclength=length;
  memcpy(&mtlcolors[0],mtlc,sizeof(float)*3*mclength);
  createGLMaterials();
}
void GLObjObject::setMaterialColors(vector<float> &mtlc){
  mclength=mtlc.size()/3;
  mtlcolors.assign(mtlc.begin(),mtlc.end());
  createGLMaterials();
}

void GLObjObject::createGLMaterials(){
  GLMaterial *glmtl;
  float mcolor[4];
  mcolor[3]=1.0;

  materials.clear();

  for(int i=0;i<mclength;i++){
    glmtl=new GLMaterial();
    memcpy(mcolor,&mtlcolors[i*3],sizeof(float)*3);
    glmtl->setMaterialColor(mcolor);
    materials.push_back(glmtl);
  }
  glpoly->setMaterialList(materials);

}

void GLObjObject::glDraw(){
  glPushMatrix();
  translate();
  rotate();
  glpoly->glDraw();
  glPopMatrix();
}


/////////////////

GLBone::GLBone(){
  SetIdentity(init_tf);
  SetIdentity(current_tf);
  SetIdentity(world_tf);
  current_quat[0]=1.0;
  current_quat[1]=0.0;
  current_quat[2]=0.0;
  current_quat[3]=0.0;
  parrent_bone=NULL;
  type=BONE_LOCAL;

  sequence.clear();
  iksequence.clear();
}
GLBone::~GLBone(){
}

void GLBone::setInitPos(float *tv){
  init_tf[3]=tv[0];
  init_tf[7]=tv[1];
  init_tf[11]=tv[2];
}

void GLBone::setCurrentTrans(float *tv){
  memcpy(current_tv,tv,sizeof(float)*3);
  current_tf[3]=tv[0];
  current_tf[7]=tv[1];
  current_tf[11]=tv[2];
}
void GLBone::setCurrentQuaternion(float *quat){
  memcpy(current_quat,quat,sizeof(float)*4);
  SetQuaternionRotation(current_tf,current_quat);
}

void GLBone::calcWorldTransformation(){
  if(type==BONE_LOCAL){
    if(parrent_bone==NULL){
      TransformMatrix(init_tf,current_tf,world_tf);
    }else{
      float tmat[16];
      TransformMatrix(parrent_bone->getWorldTransformation(),init_tf,tmat);
      TransformMatrix(tmat,current_tf,world_tf);
    }
  }else{
    if(parrent_bone==NULL){
      TransformMatrix(current_tf,init_tf,world_tf);
    }else{
      float tmat[16];
      TransformMatrix(parrent_bone->getWorldTransformation(),init_tf,tmat);
      TransformMatrix(current_tf,tmat,world_tf);
    }
  }
}


void GLBone::clearSequence(){
  sequence.clear();
}
void GLBone::clearIKSequence(){
  iksequence.clear();
}
void GLBone::addSequence(unsigned int index,float *tv,float *quat){
  bone_sequence bseq;
  bseq.index=index;
  memcpy(bseq.tv,tv,sizeof(float)*3);
  memcpy(bseq.quat,quat,sizeof(float)*4);
  sequence.push_back(bseq);
}
void GLBone::addIKSequence(unsigned int index){
  bone_sequence bseq;
  bseq.index=index;

  memcpy(bseq.tv,current_tv,sizeof(float)*3);
  memcpy(bseq.quat,current_quat,sizeof(float)*4);
  iksequence.push_back(bseq);
}

void GLBone::setReturnSequence(unsigned int index){
  bone_sequence bseq;
  bseq.index=index;

  if(sequence.size()>1){
    //cout << " sequence.size() = " << sequence.size() << endl;
    //cout << "  index = " << index <<endl;
    memcpy(bseq.tv,sequence[0].tv,sizeof(float)*3);
    memcpy(bseq.quat,sequence[0].quat,sizeof(float)*4);
    sequence.push_back(bseq);
  }
  if(iksequence.size()>1){
    //cout << " iksequence.size() = " << iksequence.size() << endl;
    //cout << "  index = " << index <<endl;
    memcpy(bseq.tv,iksequence[0].tv,sizeof(float)*3);
    memcpy(bseq.quat,iksequence[0].quat,sizeof(float)*4);
    iksequence.push_back(bseq);
  }


}


void GLBone::selectSequence(unsigned int index){
  if(sequence.size()==1){
    setCurrentTrans(sequence[0].tv);
    setCurrentQuaternion(sequence[0].quat);
  }else if(sequence.size()>1){
    bone_sequence *bseq0,*bseq1;
    float ctv[3],cquat[4];

    int lastindex=sequence.size()-1;
    unsigned int last=sequence[lastindex].index;
    unsigned int nindex=index%last;

    for(int i=0;i<lastindex;i++){
      bseq0=&(sequence[i]);
      bseq1=&(sequence[i+1]);
      
      //cout << bseq0->index << " -- " << bseq1->index <<endl;

      if(bseq0->index<=nindex && bseq1->index>nindex){
	int iwidth=bseq1->index - bseq0->index;
	//int iw0=nindex-bseq0->index;
	int iw1=bseq1->index-nindex;
	float r0=(float)iw1/(float)iwidth;
	//float r1=(float)iw0/(float)iwidth;
	float r1=1.0-r0;
	
	for(int j=0;j<3;j++)
	  ctv[j]=(bseq0->tv[j]*r0)+(bseq1->tv[j]*r1);
	SlerpQuaternion(bseq0->quat,bseq1->quat,r0,cquat);
	
	setCurrentTrans(ctv);
	setCurrentQuaternion(cquat);
	
	//cout << " selected." << endl;
	break;
      }
    }
  }
}
void GLBone::selectIKSequence(unsigned int index){
  if(iksequence.size()==1){
    setCurrentTrans(iksequence[0].tv);
    setCurrentQuaternion(iksequence[0].quat);
  }else if(iksequence.size()>1){
    bone_sequence *bseq0,*bseq1;
    float ctv[3],cquat[4];

    int lastindex=iksequence.size()-1;
    unsigned int last=iksequence[lastindex].index;
    unsigned int nindex=index%last;

    for(int i=0;i<lastindex;i++){
      bseq0=&(iksequence[i]);
      bseq1=&(iksequence[i+1]);
      
      //cout << bseq0->index << " -- " << bseq1->index <<endl;

      if(bseq0->index<=nindex && bseq1->index>nindex){
	int iwidth=bseq1->index - bseq0->index;
	//int iw0=nindex-bseq0->index;
	int iw1=bseq1->index-nindex;
	float r0=(float)iw1/(float)iwidth;
	//float r1=(float)iw0/(float)iwidth;
	float r1=1.0-r0;
	
	for(int j=0;j<3;j++)
	  ctv[j]=(bseq0->tv[j]*r0)+(bseq1->tv[j]*r1);
	SlerpQuaternion(bseq0->quat,bseq1->quat,r0,cquat);
	
	setCurrentTrans(ctv);
	setCurrentQuaternion(cquat);
	
	//cout << " selected." << endl;
	break;
      }
    }
  }
}

/////////

GLPMDObject::GLPMDObject(){
  glpoly=new GLPolygon();
  glpoly->setNormalType(1);
  glpoly->setLeftHandCoordinate();
  type="GLPMDObject";
  isanime=false;
}
GLPMDObject::~GLPMDObject(){
  delete glpoly;
  for(int i=0;i<materials.size();i++)
    delete materials[i];
  for(int i=0;i<bones.size();i++)
    delete bones[i];
}

void GLPMDObject::createMaterials(int count){
  materials.clear();
  for(int i=0;i<count;i++)
    materials.push_back(new GLMaterial());
}

void GLPMDObject::setBones(vector<int> &parent,vector<float> &headpos,vector<string> &names){
  bones_parent.assign(parent.begin(),parent.end());
  bones_headpos.assign(headpos.begin(),headpos.end());
  bones_names.assign(names.begin(),names.end());

  float vv[3];

  bones.clear();
  GLBone *bone;
  for(int i=0;i<bones_parent.size();i++){
    bone=new GLBone();
    bone->setName(bones_names[i]);
    if(bones_parent[i]!=-1){
      vv[0]=bones_headpos[i*3]-bones_headpos[bones_parent[i]*3];
      vv[1]=bones_headpos[i*3+1]-bones_headpos[bones_parent[i]*3+1];
      vv[2]=bones_headpos[i*3+2]-bones_headpos[bones_parent[i]*3+2];
      bone->setInitPos(vv);
    }else{
      bone->setInitPos(&bones_headpos[i*3]);
    }
    bones.push_back(bone);
  }
  for(int i=0;i<bones_parent.size();i++){
    bone=bones[i];
    if(bones_parent[i]!=-1)
      bone->setParrent(bones[bones_parent[i]]);
  }

  updated.resize(bones.size());
  for(int i=0;i<updated.size();i++)
    updated[i]=0;
  

}
void GLPMDObject::setBonesTransformType(bone_transform_type type){
  for(int i=0;i<bones.size();i++)
    bones[i]->setTransformType(type);
}
void GLPMDObject::drawBones(){
  int par;
  float col[]={1.0,0.0,0.0};
  float vv0[3],vv1[3];

  glPushAttrib(GL_ALL_ATTRIB_BITS);
  glDisable(GL_LIGHTING);
  for(int i=0;i<bones_parent.size();i++){
    par=bones_parent[i];
    if(par!=-1){
      glLineWidth(1.0);
      glBegin(GL_LINES);
      glColor3fv(col);
      for(int j=0;j<3;j++){
	vv0[j]=bones[i]->getWorldTransformation()[j*4+3];
	vv1[j]=bones[par]->getWorldTransformation()[j*4+3];
      }
      vv0[2]=-vv0[2];
      vv1[2]=-vv1[2];
      //glVertex3fv(&bones_headpos[i*3]);
      //glVertex3fv(&bones_headpos[par*3]);
      glVertex3fv(vv0);
      glVertex3fv(vv1);
      glEnd();
    }
  }

  // bone heads
  glPointSize(4.0);
  glBegin(GL_POINTS);
  glColor3fv(col);
  for(int i=0;i<bones.size();i++){
    for(int j=0;j<3;j++){
      vv0[j]=bones[i]->getWorldTransformation()[j*4+3];
    }
    vv0[2]=-vv0[2];
    glVertex3fv(vv0);
  }
  glEnd();

  // ik bones
  col[0]=0.0;
  col[1]=1.0;
  for(int i=0;i<ik_bones.size();i++){
    glBegin(GL_LINES);
    glColor3fv(col);
    for(int j=0;j<3;j++){
      vv0[j]=bones[ik_bones[i]]->getWorldTransformation()[j*4+3];
      vv1[j]=bones[ik_target_bones[i]]->getWorldTransformation()[j*4+3];
    }
    vv0[2]=-vv0[2];
    vv1[2]=-vv1[2];
    glVertex3fv(vv0);
    glVertex3fv(vv1);
    glEnd();
  }

  // coordinate
  glLineWidth(4.0);
  vv0[0]=0.0; vv0[1]=0.0; vv0[2]=0.0;

  vv1[0]=scale*10.0; vv1[1]=0.0; vv1[2]=0.0;
  col[0]=1.0; col[1]=0.0; col[2]=0.0;
  glBegin(GL_LINES);
  glColor3fv(col);
  glVertex3fv(vv0);
  glVertex3fv(vv1);
  glEnd();

  vv1[0]=0.0; vv1[1]=scale*10.0; vv1[2]=0.0;
  col[0]=0.0; col[1]=1.0; col[2]=0.0;
  glBegin(GL_LINES);
  glColor3fv(col);
  glVertex3fv(vv0);
  glVertex3fv(vv1);
  glEnd();

  vv1[0]=0.0; vv1[1]=0.0; vv1[2]=-scale*10.0;
  col[0]=0.0; col[1]=0.0; col[2]=1.0;
  glBegin(GL_LINES);
  glColor3fv(col);
  glVertex3fv(vv0);
  glVertex3fv(vv1);
  glEnd();


  glEnable(GL_LIGHTING);
  glPopAttrib();
}


void GLPMDObject::setBonePose(string bname,float *tvec,float *quat){
  float stvec[3];
  bool isfound=false;
  for(int i=0;i<bones_names.size();i++){
    if(bones_names[i]==bname){
      stvec[0]=tvec[0]*scale;
      stvec[1]=tvec[1]*scale;
      stvec[2]=tvec[2]*scale;
      bones[i]->setCurrentTrans(stvec);
      bones[i]->setCurrentQuaternion(quat);
      isfound=true;
      updated[i]=1;
      break;
    }
  }
  if(!isfound) cerr << "not found: " << bname << endl;
}

void GLPMDObject::addBoneSequence(string bname,unsigned int index,float *tvec,float *quat){
  float stvec[3];
  bool isfound=false;
  for(int i=0;i<bones_names.size();i++){
    if(bones_names[i]==bname){
      stvec[0]=tvec[0]*scale;
      stvec[1]=tvec[1]*scale;
      stvec[2]=tvec[2]*scale;
      bones[i]->addSequence(index,stvec,quat);
      isfound=true;
      //updated[i]=1;
      break;
    }
  }
  if(!isfound) cerr << "not found: " << bname << endl;
}
void GLPMDObject::addBoneIKSequence(unsigned int index){
  for(int i=0;i<bones.size();i++){
    bones[i]->addIKSequence(index);
  }
}

void GLPMDObject::setReturnBoneSequence(unsigned int index){
  for(int i=0;i<bones.size();i++){
    bones[i]->setReturnSequence(index);
  }
}

void GLPMDObject::clearBoneSequence(){
  for(int i=0;i<bones.size();i++){
    bones[i]->clearSequence();
    bones[i]->clearIKSequence();
  }
}


void GLPMDObject::calcCurrentVertices(){
  for(int i=0;i<bones.size();i++)
    bones[i]->calcWorldTransformation();

  float vvx[3],vv0[3],vv1[3];
  int idx2,idx3;
  float sc0,sc1;
  for(int i=0;i<init_vertices.size()/3;i++){
    idx2=i*2;
    idx3=i*3;

    sc0=vertex_bones_weight[idx2];
    sc1=vertex_bones_weight[idx2+1];

    if(vertex_bones[idx2]<0 || vertex_bones[idx2+1]<0 ||
       vertex_bones[idx2]>=bones.size() || vertex_bones[idx2+1]>=bones.size()){
      //cout << "unknown index??? : " << vertex_bones[idx2] << ","  << vertex_bones[idx2+1] << endl;
      //return;
      continue;
    }

    for(int j=0;j<3;j++)
      vvx[j]=init_vertices[idx3+j]-bones_headpos[vertex_bones[idx2]*3+j];
    TransformVector(vvx,bones[vertex_bones[idx2]]->getWorldTransformation(),vv0);
    for(int j=0;j<3;j++)
      vvx[j]=init_vertices[idx3+j]-bones_headpos[vertex_bones[idx2+1]*3+j];
    TransformVector(vvx,bones[vertex_bones[idx2+1]]->getWorldTransformation(),vv1);

    for(int j=0;j<3;j++){
      current_vertices[idx3+j]=sc0*vv0[j]+sc1*vv1[j];
    }

    RotateVector(&init_normals[idx3],bones[vertex_bones[idx2]]->getWorldTransformation(),vv0);
    RotateVector(&init_normals[idx3],bones[vertex_bones[idx2+1]]->getWorldTransformation(),vv1);

    for(int j=0;j<3;j++){
      current_normals[idx3+j]=sc0*vv0[j]+sc1*vv1[j];
    }
  }

  glpoly->setVertices(current_vertices);
  glpoly->setVertexNormals(current_normals);
  glpoly->setGLList();
}


void GLPMDObject::glDraw(){
  if(isanime){
    cout << "animated: count = " << count << endl;
    selectSequence(count);
    count++;
  }

  glPushMatrix();
  translate();
  rotate();
  glpoly->glDraw();
  //drawBones();
  glPopMatrix();
}

void GLPMDObject::reset(){
  if(isanime){
    selectSequence(0);
    count=0;
  }
}

void GLPMDObject::animate(){
  cout << "animate" << endl;
  isanime=true;
  //glpoly->useList(false);
  reset();
}

void GLPMDObject::deanimate(){
  isanime=false;
  glpoly->useList(true);
}

void GLPMDObject::selectSequence(unsigned int index,bool ik){
  resetPose();
  for(int i=0;i<bones.size();i++){
    bones[i]->selectSequence(index);
  }
  if(ik){
    for(int i=0;i<bones.size();i++){
      bones[i]->selectIKSequence(index);
    }
  }
  calcCurrentVertices();
}

//////////////////////////////

GLMQOObject::GLMQOObject(){

}
GLMQOObject::~GLMQOObject(){
  for(int i=0;i<polygons.size();i++){
    delete polygons[i];
  }
}

void GLMQOObject::createMaterials(int num){
  materials.resize(num);
  for(int i=0;i<num;i++){
    materials[i]=new GLMaterial();
  }
}

void GLMQOObject::setMaterialsToPolygon(int idx){
  polygons[idx]->setMaterialList(materials);
}

void GLMQOObject::glDraw(){
  glPushMatrix();
  translate();
  rotate();
  for(int i=0;i<polygons.size();i++){
    polygons[i]->glDraw();
  }
  glPopMatrix();
}

///////////////////////////////

GLBox::GLBox(float x,float y,float z){
  color_mtl=new GLMaterial();
  setMaterial(color_mtl);
  setSize(x,y,z);
}
GLBox::~GLBox(){
  delete color_mtl;
}

void GLBox::setSize(float x,float y, float z){
  vector<float> vcs;
  vector<int> fcs;

  vcs.resize(24);

  float x0,y0,z0,x1,y1,z1;
  x0=x*-0.5;
  y0=y*-0.5;
  z0=z*-0.5;
  x1=-x0;
  y1=-y0;
  z1=-z0;

  vcs[ 0]=vcs[ 6]=vcs[12]=vcs[18]=x0;
  vcs[ 1]=vcs[ 4]=vcs[13]=vcs[16]=y0;
  vcs[ 2]=vcs[ 5]=vcs[ 8]=vcs[11]=z0;

  vcs[ 3]=vcs[ 9]=vcs[15]=vcs[21]=x1;
  vcs[ 7]=vcs[10]=vcs[19]=vcs[22]=y1;
  vcs[14]=vcs[17]=vcs[20]=vcs[23]=z1;

  fcs.resize(39);

  fcs[ 0]=-1;  fcs[ 1]= 0;  fcs[ 2]= 0; // face color

  fcs[ 3]= 0;  fcs[ 4]= 2;  fcs[ 5]= 1;
  fcs[ 6]= 1;  fcs[ 7]= 2;  fcs[ 8]= 3;

  fcs[ 9]= 0;  fcs[10]= 2;  fcs[11]= 4;
  fcs[12]= 2;  fcs[13]= 6;  fcs[14]= 4;

  fcs[15]= 1;  fcs[16]= 0;  fcs[17]= 4;
  fcs[18]= 1;  fcs[19]= 4;  fcs[20]= 5;

  fcs[21]= 4;  fcs[22]= 6;  fcs[23]= 5;
  fcs[24]= 5;  fcs[25]= 6;  fcs[26]= 7;

  fcs[27]= 5;  fcs[28]= 3;  fcs[29]= 1;
  fcs[30]= 5;  fcs[31]= 7;  fcs[32]= 3;

  fcs[33]= 2;  fcs[34]= 3;  fcs[35]= 6;
  fcs[36]= 3;  fcs[37]= 7;  fcs[38]= 6;

  setVertices(vcs);
  setFaces(fcs);

  calcNormals();
  setGLList();
}

void GLBox::setColor(float r,float g,float b,float a){
  float color[4]={r,g,b,a};
  setColor(color);
}
void GLBox::setColor(float *color){
  color_mtl->setMaterialColor(color);
}

///////////////////////////////////

GLVoxel::GLVoxel(){
  voxel_object=new GLBox(10.0,10.0,10.0);
  voxel_object->useList(false);
  width=0;
  height=0;
}

GLVoxel::GLVoxel(float x, float y, float z){
  voxel_object=new GLBox(x,y,z);
  voxel_object->useList(false);
  width=0;
  height=0;
}

GLVoxel::~GLVoxel(){
  delete voxel_object;
}

void GLVoxel::setVoxelSize(float x, float y, float z){
  vx=x;
  vy=y;
  vz=z;
  voxel_object->setSize(vx,vy,vz);
}

void GLVoxel::setMap(int w,int h,vector<float> &m){
  width=w;
  height=h;
  map.assign(m.begin(),m.end());
}

void GLVoxel::glDraw(){
  float xpos,ypos;
  float x0,y0;
  int idx;
  float val;

  glPushMatrix();
  translate();
  rotate();

  x0=(width/2+width%2)*(-vx);
  y0=(height/2+height%2)*(-vy);

  for(int i=0;i<height;i++){
    idx=i*width;
    ypos=y0+i*vy;
    for(int j=0;j<width;j++){
      xpos=x0+j*vx;
      val=map[idx+j];
      voxel_object->setTranslate(xpos,ypos,0.0);
      voxel_object->setColor(1.0,1.0,1.0,1.0-val);
      voxel_object->glDraw();
    }
  }

  glPopMatrix();

}

