#include "SURFConverter.h"

using namespace std;

/////////////////////////

double
compareSURFDescriptors( const float* d1, const float* d2, double best, int length )
{
    double total_cost = 0;
    assert( length % 4 == 0 );
    for( int i = 0; i < length; i += 4 )
    {
        double t0 = d1[i] - d2[i];
        double t1 = d1[i+1] - d2[i+1];
        double t2 = d1[i+2] - d2[i+2];
        double t3 = d1[i+3] - d2[i+3];
        total_cost += t0*t0 + t1*t1 + t2*t2 + t3*t3;
        if( total_cost > best )
            break;
    }
    return total_cost;
}

int
naiveNearestNeighbor( const float* vec, int laplacian,
                      const CvSeq* model_keypoints,
                      const CvSeq* model_descriptors )
{
    int length = (int)(model_descriptors->elem_size/sizeof(float));
    int i, neighbor = -1;
    double d, dist1 = 1e6, dist2 = 1e6;
    CvSeqReader reader, kreader;
    cvStartReadSeq( model_keypoints, &kreader, 0 );
    cvStartReadSeq( model_descriptors, &reader, 0 );

    for( i = 0; i < model_descriptors->total; i++ )
    {
        const CvSURFPoint* kp = (const CvSURFPoint*)kreader.ptr;
        const float* mvec = (const float*)reader.ptr;
        CV_NEXT_SEQ_ELEM( kreader.seq->elem_size, kreader );
        CV_NEXT_SEQ_ELEM( reader.seq->elem_size, reader );
        if( laplacian != kp->laplacian )
            continue;
        d = compareSURFDescriptors( vec, mvec, dist2, length );
        if( d < dist1 )
        {
            dist2 = dist1;
            dist1 = d;
            neighbor = i;
        }
        else if ( d < dist2 )
            dist2 = d;
    }
    if ( dist1 < 0.6*dist2 )
        return neighbor;
    return -1;
}

int
naiveNearestNeighbor2( const float* vec, const CvMat* model_descriptors )
{
  //int length = model_descriptors->cols;
  int length = 32;
  int i, neighbor = -1;
  double d, dist1 = 1e6, dist2 = 1e6;

  for( i = 0; i < model_descriptors->rows; i++ ){
    const float* mvec =
      (const float*)(model_descriptors->data.ptr+model_descriptors->step*i);
    d = compareSURFDescriptors( vec, mvec, dist2, length );
    if( d < dist1 )
      {
	dist2 = dist1;
	dist1 = d;
	neighbor = i;
      }
    else if ( d < dist2 )
      dist2 = d;
  }
  if ( dist1 < 0.6*dist2 )
    return neighbor;
  return -1;
}


void
findPairs( const CvSeq* objectKeypoints, const CvSeq* objectDescriptors,
           const CvSeq* imageKeypoints, const CvSeq* imageDescriptors, vector<int>& ptpairs )
{
    int i;
    CvSeqReader reader, kreader;
    cvStartReadSeq( objectKeypoints, &kreader );
    cvStartReadSeq( objectDescriptors, &reader );
    ptpairs.clear();

    for( i = 0; i < objectDescriptors->total; i++ )
    {
        const CvSURFPoint* kp = (const CvSURFPoint*)kreader.ptr;
        const float* descriptor = (const float*)reader.ptr;
        CV_NEXT_SEQ_ELEM( kreader.seq->elem_size, kreader );
        CV_NEXT_SEQ_ELEM( reader.seq->elem_size, reader );
        int nearest_neighbor = naiveNearestNeighbor( descriptor, kp->laplacian, imageKeypoints, imageDescriptors );
        if( nearest_neighbor >= 0 )
        {
            ptpairs.push_back(i);
            ptpairs.push_back(nearest_neighbor);
        }
    }
}

void
findPairs2( const CvMat* desc_pca_mat_ref_pl, const CvMat* desc_pca_mat_ref_nl, 
	    const CvMat* desc_pca_mat_ser_pl, const CvMat* desc_pca_mat_ser_nl, 
	    vector<int>& index_list_ref_pl, vector<int>& index_list_ref_nl,
	    vector<int>& index_list_ser_pl, vector<int>& index_list_ser_nl,
	    vector<int>& ptpairs )
{
  int i;
  float *mat_col;

  ////////////////
  
  ptpairs.clear();

  for(i=0;i<desc_pca_mat_ref_pl->rows;i++){// ???
    mat_col=(float *)(desc_pca_mat_ref_pl->data.ptr+desc_pca_mat_ref_pl->step*i);
    int nearest_neighbor=naiveNearestNeighbor2(mat_col,desc_pca_mat_ser_pl);
    if( nearest_neighbor >= 0 ){
      ptpairs.push_back(index_list_ref_pl[i]);
      ptpairs.push_back(index_list_ser_pl[nearest_neighbor]);
    }
  }
  for(i=0;i<desc_pca_mat_ref_nl->rows;i++){// ???
    mat_col=(float *)(desc_pca_mat_ref_nl->data.ptr+desc_pca_mat_ref_nl->step*i);
    int nearest_neighbor=naiveNearestNeighbor2(mat_col,desc_pca_mat_ser_nl);
    if( nearest_neighbor >= 0 ){
      ptpairs.push_back(index_list_ref_nl[i]);
      ptpairs.push_back(index_list_ser_nl[nearest_neighbor]);
    }
  }

  ////////////////
}

void
findPairs3( LSHTable* lsht_pl, LSHTable* lsht_nl,
	    const CvMat* desc_pca_mat_ser_pl, const CvMat* desc_pca_mat_ser_nl, 
	    vector<int>& index_list_ser_pl, vector<int>& index_list_ser_nl,
	    vector<int>& ptpairs ){
  int i;
  float *mat_col;
  float res[32];

  ptpairs.clear();

  for(i=0;i<desc_pca_mat_ser_pl->rows;i++){
    mat_col=(float *)(desc_pca_mat_ser_pl->data.ptr+desc_pca_mat_ser_pl->step*i);
    int nearest_neighbor=lsht_pl->searchFromHashTable(mat_col,res);
    if( nearest_neighbor >= 0 ){
      ptpairs.push_back(nearest_neighbor);
      ptpairs.push_back(index_list_ser_pl[i]);
    }
  }
  for(i=0;i<desc_pca_mat_ser_nl->rows;i++){
    mat_col=(float *)(desc_pca_mat_ser_nl->data.ptr+desc_pca_mat_ser_nl->step*i);
    int nearest_neighbor=lsht_nl->searchFromHashTable(mat_col,res);
    if( nearest_neighbor >= 0 ){
      ptpairs.push_back(nearest_neighbor);
      ptpairs.push_back(index_list_ser_nl[i]);
    }
  }

}

// using FLANN from OpenCV2.0 sample code
void
flannFindPairs( const CvSeq*, const CvSeq* objectDescriptors,
           const CvSeq*, const CvSeq* imageDescriptors, vector<int>& ptpairs )
{
	int length = (int)(objectDescriptors->elem_size/sizeof(float));

    cv::Mat m_object(objectDescriptors->total, length, CV_32F);
	cv::Mat m_image(imageDescriptors->total, length, CV_32F);


	// copy descriptors
    CvSeqReader obj_reader;
	float* obj_ptr = m_object.ptr<float>(0);
    cvStartReadSeq( objectDescriptors, &obj_reader );
    for(int i = 0; i < objectDescriptors->total; i++ )
    {
        const float* descriptor = (const float*)obj_reader.ptr;
        CV_NEXT_SEQ_ELEM( obj_reader.seq->elem_size, obj_reader );
        memcpy(obj_ptr, descriptor, length*sizeof(float));
        obj_ptr += length;
    }
    CvSeqReader img_reader;
	float* img_ptr = m_image.ptr<float>(0);
    cvStartReadSeq( imageDescriptors, &img_reader );
    for(int i = 0; i < imageDescriptors->total; i++ )
    {
        const float* descriptor = (const float*)img_reader.ptr;
        CV_NEXT_SEQ_ELEM( img_reader.seq->elem_size, img_reader );
        memcpy(img_ptr, descriptor, length*sizeof(float));
        img_ptr += length;
    }

    // find nearest neighbors using FLANN
    cv::Mat m_indices(objectDescriptors->total, 2, CV_32S);
    cv::Mat m_dists(objectDescriptors->total, 2, CV_32F);
    cv::flann::Index flann_index(m_image, cv::flann::KDTreeIndexParams(4));  // using 4 randomized kdtrees
    flann_index.knnSearch(m_object, m_indices, m_dists, 2, cv::flann::SearchParams(64) ); // maximum number of leafs checked

    int* indices_ptr = m_indices.ptr<int>(0);
    float* dists_ptr = m_dists.ptr<float>(0);
    for (int i=0;i<m_indices.rows;++i) {
    	if (dists_ptr[2*i]<0.6*dists_ptr[2*i+1]) {
    		ptpairs.push_back(i);
    		ptpairs.push_back(indices_ptr[2*i]);
    	}
    }
}

void
flannFindPairs2( cv::flann::Index *ann_index_pl,
		 cv::flann::Index *ann_index_nl,
		 CvMat* desc_pca_mat_ser_pl, CvMat* desc_pca_mat_ser_nl, 
		 vector<int>& index_list_ref_pl, vector<int>& index_list_ref_nl,
		 vector<int>& index_list_ser_pl, vector<int>& index_list_ser_nl,
		 vector<int>& ptpairs)
{
  cv::Mat desc_pca_mat_ser_pl_2(desc_pca_mat_ser_pl);
  cv::Mat desc_pca_mat_ser_nl_2(desc_pca_mat_ser_nl);

  // find nearest neighbors using FLANN
  cv::Mat m_indices_p(desc_pca_mat_ser_pl->rows, 2, CV_32S);
  cv::Mat m_dists_p(desc_pca_mat_ser_pl->rows, 2, CV_32F);
  cv::Mat m_indices_n(desc_pca_mat_ser_nl->rows, 2, CV_32S);
  cv::Mat m_dists_n(desc_pca_mat_ser_nl->rows, 2, CV_32F);
  
  ann_index_pl->knnSearch(desc_pca_mat_ser_pl_2, m_indices_p, m_dists_p, 2, cv::flann::SearchParams(64) );
  ann_index_nl->knnSearch(desc_pca_mat_ser_nl_2, m_indices_n, m_dists_n, 2, cv::flann::SearchParams(64) );
  // maximum number of leafs checked

  ptpairs.clear();
  int* indices_ptr_p = m_indices_p.ptr<int>(0);
  float* dists_ptr_p = m_dists_p.ptr<float>(0);
  for (int i=0;i<m_indices_p.rows;++i) {
    if (dists_ptr_p[2*i]<0.6*dists_ptr_p[2*i+1]) {
      ptpairs.push_back(index_list_ref_pl[indices_ptr_p[2*i]]);
      ptpairs.push_back(index_list_ser_pl[i]);
    }
  }
  int* indices_ptr_n = m_indices_n.ptr<int>(0);
  float* dists_ptr_n = m_dists_n.ptr<float>(0);
  for (int i=0;i<m_indices_n.rows;++i) {
    if (dists_ptr_n[2*i]<0.6*dists_ptr_n[2*i+1]) {
      ptpairs.push_back(index_list_ref_nl[indices_ptr_n[2*i]]);
      ptpairs.push_back(index_list_ser_nl[i]);
    }
  }
}



//


/* a rough implementation for object location */
int
locatePlanarObject( const CvSeq* objectKeypoints, const CvSeq* objectDescriptors,
                    const CvSeq* imageKeypoints, const CvSeq* imageDescriptors,
                    const CvPoint src_corners[4], CvPoint dst_corners[4] )
{
    double h[9];
    CvMat _h = cvMat(3, 3, CV_64F, h);
    vector<int> ptpairs;
    vector<CvPoint2D32f> pt1, pt2;
    CvMat _pt1, _pt2;
    int i, n;

    findPairs( objectKeypoints, objectDescriptors, imageKeypoints, imageDescriptors, ptpairs );
    n = ptpairs.size()/2;
    if( n < 4 )
        return 0;

    pt1.resize(n);
    pt2.resize(n);
    for( i = 0; i < n; i++ )
    {
        pt1[i] = ((CvSURFPoint*)cvGetSeqElem(objectKeypoints,ptpairs[i*2]))->pt;
        pt2[i] = ((CvSURFPoint*)cvGetSeqElem(imageKeypoints,ptpairs[i*2+1]))->pt;
    }

    _pt1 = cvMat(1, n, CV_32FC2, &pt1[0] );
    _pt2 = cvMat(1, n, CV_32FC2, &pt2[0] );
    if( !cvFindHomography( &_pt1, &_pt2, &_h, CV_RANSAC, 5 ))
        return 0;

    for( i = 0; i < 4; i++ )
    {
        double x = src_corners[i].x, y = src_corners[i].y;
        double Z = 1./(h[6]*x + h[7]*y + h[8]);
        double X = (h[0]*x + h[1]*y + h[2])*Z;
        double Y = (h[3]*x + h[4]*y + h[5])*Z;
        dst_corners[i] = cvPoint(cvRound(X), cvRound(Y));
    }

    return 1;
}


int
locatePlanarObject2( const int n, const float *src_points, const float *dst_points,
		     const float *src_corners, float *dst_corners )
{
    double h[9];
    CvMat _h = cvMat(3, 3, CV_64F, h);
    vector<CvPoint2D32f> pt1, pt2;
    CvMat _pt1, _pt2;
    int i;

    pt1.resize(n);
    pt2.resize(n);
    for( i = 0; i < n; i++ )
    {
      pt1[i].x=src_points[i*2];
      pt1[i].y=src_points[i*2+1];
      pt2[i].x=dst_points[i*2];
      pt2[i].y=dst_points[i*2+1];
    }

    _pt1 = cvMat(1, n, CV_32FC2, &pt1[0] );
    _pt2 = cvMat(1, n, CV_32FC2, &pt2[0] );
    if( !cvFindHomography( &_pt1, &_pt2, &_h, CV_RANSAC, 2 ))
        return 0;

    for( i = 0; i < 4; i++ )
    {
        double x = src_corners[i*2], y = src_corners[i*2+1];
        double Z = 1./(h[6]*x + h[7]*y + h[8]);
        double X = (h[0]*x + h[1]*y + h[2])*Z;
        double Y = (h[3]*x + h[4]*y + h[5])*Z;
        dst_corners[i*2] = X;
	dst_corners[i*2+1] = Y;
    }

    return 1;
}

/////////////////////////////////////////////


SURFConverter::SURFConverter(){
  storage_ref=cvCreateMemStorage(0);
  storage_ser=cvCreateMemStorage(0);

  refwidth=640;
  refheight=480;
  refimage=cvCreateImage(cvSize(refwidth,refheight),IPL_DEPTH_8U,1);

  serwidth=640;
  serheight=480;
  serimage=cvCreateImage(cvSize(serwidth,serheight),IPL_DEPTH_8U,1);

  refKeypoints=cvCreateSeq(0,sizeof(CvSeq),sizeof(CvSURFPoint),storage_ref);
  refDescriptors=cvCreateSeq(0,sizeof(CvSeq),128*CV_ELEM_SIZE(CV_32F),storage_ref);
  serKeypoints=cvCreateSeq(0,sizeof(CvSeq),sizeof(CvSURFPoint),storage_ser);
  serDescriptors=cvCreateSeq(0,sizeof(CvSeq),128*CV_ELEM_SIZE(CV_32F),storage_ser);

  eigenvec_pl=cvCreateMat(128,128,CV_32F);
  eigenval_pl=cvCreateMat(1,128,CV_32F);
  average_pl=cvCreateMat(1,128,CV_32F);
  eigenvec_nl=cvCreateMat(128,128,CV_32F);
  eigenval_nl=cvCreateMat(1,128,CV_32F);
  average_nl=cvCreateMat(1,128,CV_32F);

  desc_mat_ref_pl=cvCreateMat(100,128,CV_32F);
  desc_pca_mat_ref_pl=cvCreateMat(100,32,CV_32F);
  desc_mat_ref_nl=cvCreateMat(100,128,CV_32F);
  desc_pca_mat_ref_nl=cvCreateMat(100,32,CV_32F);
  desc_mat_ser_pl=cvCreateMat(100,128,CV_32F);
  desc_pca_mat_ser_pl=cvCreateMat(100,32,CV_32F);
  desc_mat_ser_nl=cvCreateMat(100,128,CV_32F);
  desc_pca_mat_ser_nl=cvCreateMat(100,32,CV_32F);

  storage_ref2=cvCreateMemStorage(0);
  desc_mat_reflist_pl=cvCreateMat(100,128,CV_32F);
  desc_pca_mat_reflist_pl=cvCreateMat(100,32,CV_32F);
  desc_mat_reflist_nl=cvCreateMat(100,128,CV_32F);
  desc_pca_mat_reflist_nl=cvCreateMat(100,32,CV_32F);

  surfparams=cvSURFParams(500,1);

  lsht_pl=new LSHTable(32,5);
  lsht_nl=new LSHTable(32,5);
  lsht_list_pl=new LSHTable(32,6);
  lsht_list_nl=new LSHTable(32,6);

  ann_index_ref_pl=ann_index_ref_nl=NULL;
  cv_lsh_pl=cv_lsh_nl=NULL;
}
SURFConverter::~SURFConverter(){
  cvRelease((void **)&refKeypoints);
  cvRelease((void **)&refDescriptors);
  cvReleaseImage(&refimage);

  cvRelease((void **)&serKeypoints);
  cvRelease((void **)&serDescriptors);
  cvReleaseImage(&serimage);

  cvReleaseMat(&eigenvec_pl);
  cvReleaseMat(&eigenval_pl);
  cvReleaseMat(&average_pl);
  cvReleaseMat(&eigenvec_nl);
  cvReleaseMat(&eigenval_nl);
  cvReleaseMat(&average_nl);

  if(desc_mat_ref_pl!=NULL) cvReleaseMat(&desc_mat_ref_pl);
  if(desc_pca_mat_ref_pl!=NULL) cvReleaseMat(&desc_pca_mat_ref_pl);
  if(desc_mat_ref_nl!=NULL) cvReleaseMat(&desc_mat_ref_nl);
  if(desc_pca_mat_ref_nl!=NULL) cvReleaseMat(&desc_pca_mat_ref_nl);

  if(desc_mat_ser_pl!=NULL) cvReleaseMat(&desc_mat_ser_pl);
  if(desc_pca_mat_ser_pl!=NULL) cvReleaseMat(&desc_pca_mat_ser_pl);
  if(desc_mat_ser_nl!=NULL) cvReleaseMat(&desc_mat_ser_nl);
  if(desc_pca_mat_ser_nl!=NULL) cvReleaseMat(&desc_pca_mat_ser_nl);

  cvReleaseMat(&desc_mat_reflist_pl);
  cvReleaseMat(&desc_pca_mat_reflist_pl);
  cvReleaseMat(&desc_mat_reflist_nl);
  cvReleaseMat(&desc_pca_mat_reflist_nl);

  cvReleaseMemStorage(&storage_ref);
  cvReleaseMemStorage(&storage_ser);
  cvReleaseMemStorage(&storage_ref2);

  delete lsht_pl;
  delete lsht_nl;
  delete lsht_list_pl;
  delete lsht_list_nl;

}

void SURFConverter::setRefImage(int width,int height,unsigned char *buf){
  if(refwidth!=width || refheight!=height){
    refwidth=width;
    refheight=height;
    cvReleaseImage(&refimage);
    //refimage=cvCreateImage(cvSize(refwidth,refheight),IPL_DEPTH_8U,3);
    refimage=cvCreateImage(cvSize(refwidth,refheight),IPL_DEPTH_8U,1);
  }
  //memcpy(refimage->imageData,buf,refwidth*refheight*3);
  unsigned char* ogbuf=(unsigned char *)refimage->imageData;
  int i,idx;
  for(i=0;i<refwidth*refheight;i++){
    idx=i*3;
    ogbuf[i]=(unsigned char)(((3*buf[idx]+6*buf[idx+1]+buf[idx+2])/10));
  }

}
void SURFConverter::setRefImage(IplImage *img){
  int width=img->width,height=img->height;
  if(refwidth!=width || refheight!=height){
    refwidth=width;
    refheight=height;
    cvReleaseImage(&refimage);
    //refimage=cvCreateImage(cvSize(refwidth,refheight),IPL_DEPTH_8U,3);
    refimage=cvCreateImage(cvSize(refwidth,refheight),IPL_DEPTH_8U,1);
  }

  switch(img->nChannels){
  case 1:
    cvCopy(img,refimage);
    break;
  case 3:
    cvCvtColor(img,refimage,CV_RGB2GRAY);
    break;
  default:
    cout << "image channel missmatch." << endl;
  }

}

void SURFConverter::convertRef(){
  //cvRelease((void **)&refKeypoints);
  //cvRelease((void **)&refDescriptors);
  cvClearMemStorage(storage_ref);
  cvExtractSURF(refimage,0,&refKeypoints,&refDescriptors,storage_ref,surfparams);
}

void SURFConverter::convertPCARef(){
  int i,j;

  CvSeqReader reader, kreader;
  cvStartReadSeq( refKeypoints, &kreader );
  cvStartReadSeq( refDescriptors, &reader );

  vector<float *> desc_pl,desc_nl;
  desc_pl.clear();
  desc_nl.clear();

  index_list_ref_pl.clear();
  index_list_ref_nl.clear();

  int len_desc=refDescriptors->total;
  cerr << "total points (ref):" << len_desc << endl;

  for(i=0;i<len_desc;i++){
    const CvSURFPoint* kp = (const CvSURFPoint*)kreader.ptr;
    float* descriptor = (float*)reader.ptr;
    CV_NEXT_SEQ_ELEM( kreader.seq->elem_size, kreader );
    CV_NEXT_SEQ_ELEM( reader.seq->elem_size, reader );
    switch(kp->laplacian){
    case 1:
      index_list_ref_pl.push_back(i);
      desc_pl.push_back(descriptor);
      break;
    case -1:
      index_list_ref_nl.push_back(i);
      desc_nl.push_back(descriptor);
      break;
    default:
      break;
    }
  }
  

  float *mat_col,*desc;

  if(desc_mat_ref_pl!=NULL) cvReleaseMat(&desc_mat_ref_pl);
  if(desc_pca_mat_ref_pl!=NULL) cvReleaseMat(&desc_pca_mat_ref_pl);
  cout << "in ConvertPcaRef: desc_pl.size() = " << desc_pl.size() << endl;
  //if(desc_pl.size()==0){
  if(desc_pl.size()<128){
    cerr << "WARNING: extracted SURF points <= 128 (Positive Laplacian)" << endl;

    desc_mat_ref_pl=NULL;
    desc_pca_mat_ref_pl=NULL;
  }else{
    desc_mat_ref_pl=cvCreateMat(desc_pl.size(),128,CV_32F);
    //desc_pca_mat_ref_pl=cvCreateMat(desc_pl.size(),32,CV_32F);
    desc_pca_mat_ref_pl=cvCreateMat(desc_pl.size(),128,CV_32F);

    for(i=0;i<desc_pl.size();i++){
      mat_col=(float *)(desc_mat_ref_pl->data.ptr+desc_mat_ref_pl->step*i);
      desc=desc_pl[i];
      memcpy(mat_col,desc,sizeof(float)*128);
    }

    cvCalcPCA(desc_mat_ref_pl,average_pl,eigenval_pl,eigenvec_pl,CV_PCA_DATA_AS_ROW);
    cvProjectPCA(desc_mat_ref_pl,average_pl,eigenvec_pl,desc_pca_mat_ref_pl);

    // create LSH
    lsht_pl->createHashTable(desc_pca_mat_ref_pl,index_list_ref_pl);

    // create ANN
    if(ann_index_ref_pl != NULL) delete ann_index_ref_pl;
    ann_index_ref_pl=
      new cv::flann::Index(cv::Mat(desc_pca_mat_ref_pl),cv::flann::KDTreeIndexParams(4));

    // create CvLSH
    if(cv_lsh_pl != NULL) cvReleaseLSH(&cv_lsh_pl);
    //cv_lsh_pl=cvCreateMemoryLSH(32,100000,4,16,CV_32FC1);
    //cvLSHAdd(cv_lsh_pl,desc_pca_mat_ref_pl);
  }

  if(desc_mat_ref_nl!=NULL) cvReleaseMat(&desc_mat_ref_nl);
  if(desc_pca_mat_ref_nl!=NULL) cvReleaseMat(&desc_pca_mat_ref_nl);
  cout << "in ConvertPcaRef: desc_nl.size() = " << desc_nl.size() << endl;
  //if(desc_nl.size()==0){
  if(desc_nl.size()<128){
    cerr << "WARNING: extracted SURF points <= 128 (Negative Laplacian)" << endl;

    desc_mat_ref_nl=NULL;
    desc_pca_mat_ref_nl=NULL;
  }else{
    desc_mat_ref_nl=cvCreateMat(desc_nl.size(),128,CV_32F);
    //desc_pca_mat_ref_nl=cvCreateMat(desc_nl.size(),32,CV_32F);
    desc_pca_mat_ref_nl=cvCreateMat(desc_nl.size(),128,CV_32F);

    for(i=0;i<desc_nl.size();i++){
      mat_col=(float *)(desc_mat_ref_nl->data.ptr+desc_mat_ref_nl->step*i);
      desc=desc_nl[i];
      memcpy(mat_col,desc,sizeof(float)*128);
    }

    cvCalcPCA(desc_mat_ref_nl,average_nl,eigenval_nl,eigenvec_nl,CV_PCA_DATA_AS_ROW);
    cvProjectPCA(desc_mat_ref_nl,average_nl,eigenvec_nl,desc_pca_mat_ref_nl);

    // create LSH
    lsht_nl->createHashTable(desc_pca_mat_ref_nl,index_list_ref_nl);

    // create ANN
    if(ann_index_ref_nl != NULL) delete ann_index_ref_nl;
    ann_index_ref_nl=
      new cv::flann::Index(cv::Mat(desc_pca_mat_ref_nl),cv::flann::KDTreeIndexParams(4));

    // create CvLSH
    if(cv_lsh_nl != NULL) cvReleaseLSH(&cv_lsh_nl);
    //cv_lsh_nl=cvCreateMemoryLSH(32,100000,4,16,CV_32FC1);
    //cvLSHAdd(cv_lsh_nl,desc_pca_mat_ref_nl);
  }

}

void SURFConverter::setSerImage(int width,int height,unsigned char *buf){
  if(serwidth!=width || serheight!=height){
    serwidth=width;
    serheight=height;
    cvReleaseImage(&serimage);
    //serimage=cvCreateImage(cvSize(serwidth,serheight),IPL_DEPTH_8U,3);
    serimage=cvCreateImage(cvSize(serwidth,serheight),IPL_DEPTH_8U,1);
  }
  //memcpy(serimage->imageData,buf,serwidth*serheight*3);
  unsigned char* ogbuf=(unsigned char *)serimage->imageData;
  int i,idx;
  for(i=0;i<serwidth*serheight;i++){
    idx=i*3;
    ogbuf[i]=(unsigned char)(((3*buf[idx]+6*buf[idx+1]+buf[idx+2])/10));
  }

}

void SURFConverter::setSerImage(IplImage *img){
  int width=img->width,height=img->height;
  if(serwidth!=width || serheight!=height){
    serwidth=width;
    serheight=height;
    cvReleaseImage(&serimage);
    //serimage=cvCreateImage(cvSize(serwidth,serheight),IPL_DEPTH_8U,3);
    serimage=cvCreateImage(cvSize(serwidth,serheight),IPL_DEPTH_8U,1);
  }

  switch(img->nChannels){
  case 1:
    cvCopy(img,serimage);
    break;
  case 3:
    cvCvtColor(img,serimage,CV_RGB2GRAY);
    break;
  default:
    cout << "image channel missmatch." << endl;
  }

}


void SURFConverter::convertSer(){
  //cvRelease((void **)&serKeypoints);
  //cvRelease((void **)&serDescriptors);
  cvClearMemStorage(storage_ser);
  cvExtractSURF(serimage,0,&serKeypoints,&serDescriptors,storage_ser,surfparams);
}

void SURFConverter::convertPCASer(){
  int i,j;

  CvSeqReader reader, kreader;
  cvStartReadSeq( serKeypoints, &kreader );
  cvStartReadSeq( serDescriptors, &reader );

  vector<float *> desc_pl,desc_nl;
  desc_pl.clear();
  desc_nl.clear();

  index_list_ser_pl.clear();
  index_list_ser_nl.clear();

  int len_desc=serDescriptors->total;
  cerr << "total points (ser):" << len_desc << endl;

  for(i=0;i<len_desc;i++){
    const CvSURFPoint* kp = (const CvSURFPoint*)kreader.ptr;
    float* descriptor = (float*)reader.ptr;
    CV_NEXT_SEQ_ELEM( kreader.seq->elem_size, kreader );
    CV_NEXT_SEQ_ELEM( reader.seq->elem_size, reader );
    switch(kp->laplacian){
    case 1:
      index_list_ser_pl.push_back(i);
      desc_pl.push_back(descriptor);
      break;
    case -1:
      index_list_ser_nl.push_back(i);
      desc_nl.push_back(descriptor);
      break;
    default:
      break;
    }
  }

  float *mat_col,*desc;

  if(desc_mat_ser_pl!=NULL) cvReleaseMat(&desc_mat_ser_pl);
  if(desc_pca_mat_ser_pl!=NULL) cvReleaseMat(&desc_pca_mat_ser_pl);
  // if desc_pl.size()==0 then matrices cannot be created 
  if(desc_pl.size()==0){
    desc_mat_ser_pl=NULL;
    desc_pca_mat_ser_pl=NULL;
  }else{
    desc_mat_ser_pl=cvCreateMat(desc_pl.size(),128,CV_32F);
    //desc_pca_mat_ser_pl=cvCreateMat(desc_pl.size(),32,CV_32F);
    desc_pca_mat_ser_pl=cvCreateMat(desc_pl.size(),128,CV_32F);
    
    for(i=0;i<desc_pl.size();i++){
      mat_col=(float *)(desc_mat_ser_pl->data.ptr+desc_mat_ser_pl->step*i);
      desc=desc_pl[i];
      memcpy(mat_col,desc,sizeof(float)*128);
    }
    cvProjectPCA(desc_mat_ser_pl,average_pl,eigenvec_pl,desc_pca_mat_ser_pl);
  }

  if(desc_mat_ser_nl!=NULL) cvReleaseMat(&desc_mat_ser_nl);
  if(desc_pca_mat_ser_nl!=NULL) cvReleaseMat(&desc_pca_mat_ser_nl);

  if(desc_nl.size()==0){
    desc_mat_ser_pl=NULL;
    desc_pca_mat_ser_pl=NULL;
  }else{
    desc_mat_ser_nl=cvCreateMat(desc_nl.size(),128,CV_32F);
    //desc_pca_mat_ser_nl=cvCreateMat(desc_nl.size(),32,CV_32F);
    desc_pca_mat_ser_nl=cvCreateMat(desc_nl.size(),128,CV_32F);

    for(i=0;i<desc_nl.size();i++){
      mat_col=(float *)(desc_mat_ser_nl->data.ptr+desc_mat_ser_nl->step*i);
      desc=desc_nl[i];
      memcpy(mat_col,desc,sizeof(float)*128);
    }
    cvProjectPCA(desc_mat_ser_nl,average_nl,eigenvec_nl,desc_pca_mat_ser_nl);
  }
}

/////////////////////////////////////////////

void SURFConverter::pushRefImageList(IplImage *img){
  IplImage *refimg;
  int width=img->width,height=img->height;
  refimg=cvCreateImage(cvSize(width,height),IPL_DEPTH_8U,1);
  refwidthlist.push_back(width);
  refheightlist.push_back(height);

  refwidthlistmm.push_back((float)width);
  refheightlistmm.push_back((float)height);

  switch(img->nChannels){
  case 1:
    cvCopy(img,refimg);
    break;
  case 3:
    cvCvtColor(img,refimg,CV_RGB2GRAY);
    break;
  default:
    cout << "image channel missmatch." << endl;
  }

  refimagelist.push_back(refimg);
}
void SURFConverter::pushRefImageList(IplImage *img,float widthmm,float heightmm){
  IplImage *refimg;
  int width=img->width,height=img->height;
  refimg=cvCreateImage(cvSize(width,height),IPL_DEPTH_8U,1);
  refwidthlist.push_back(width);
  refheightlist.push_back(height);

  refwidthlistmm.push_back(widthmm);
  refheightlistmm.push_back(heightmm);

  switch(img->nChannels){
  case 1:
    cvCopy(img,refimg);
    break;
  case 3:
    cvCvtColor(img,refimg,CV_RGB2GRAY);
    break;
  default:
    cout << "image channel missmatch." << endl;
  }

  refimagelist.push_back(refimg);
}
void SURFConverter::clearRefImageList(){
  for(int i=0;i<refimagelist.size();i++)
    cvReleaseImage(&refimagelist[i]);
  refimagelist.clear();
  refwidthlist.clear();
  refheightlist.clear();
  refwidthlistmm.clear();
  refheightlistmm.clear();
}
IplImage *SURFConverter::getRefImageListOf(int idx){
  if(idx>=0 && idx<refimagelist.size()){
    return refimagelist[idx];
  }else{
    return NULL;
  }
}


void SURFConverter::convertRefList(){
  int i,j;
  CvSeqReader reader, kreader;
  vector<float *> desc_pl,desc_nl;
  int len_desc;
  int nimages=refimagelist.size();
  int ofs;

  index_list_reflist_pl.clear();
  index_list_reflist_nl.clear();

  index_offset_reflist.resize(nimages+1);
  index_offset_reflist[0]=0;

  key_reflist.clear();

  desc_pl.clear();
  desc_nl.clear();

  for(int nn=0;nn<nimages;nn++){
    cvClearMemStorage(storage_ref2);
    cvExtractSURF(refimagelist[nn],0,&refKey2,&refDesc2,storage_ref2,surfparams);
    
    cvStartReadSeq( refKey2, &kreader );
    cvStartReadSeq( refDesc2, &reader );

    len_desc=refDesc2->total;
    index_offset_reflist[nn+1]=index_offset_reflist[nn]+len_desc;
    ofs=index_offset_reflist[nn];

    for(i=0;i<len_desc;i++){
      const CvSURFPoint* kp = (const CvSURFPoint*)kreader.ptr;
      float* descriptor = (float*)reader.ptr;
      float *ndesc=new float[128];
      CV_NEXT_SEQ_ELEM( kreader.seq->elem_size, kreader );
      CV_NEXT_SEQ_ELEM( reader.seq->elem_size, reader );
      switch(kp->laplacian){
      case 1:
	index_list_reflist_pl.push_back(i+ofs);
	//desc_pl.push_back(descriptor);
	memcpy(ndesc,descriptor,sizeof(float)*128);
	desc_pl.push_back(ndesc);
	break;
      case -1:
	index_list_reflist_nl.push_back(i+ofs);
	//desc_nl.push_back(descriptor);
	memcpy(ndesc,descriptor,sizeof(float)*128);
	desc_nl.push_back(ndesc);
	break;
      default:
	break;
      }
      key_reflist.push_back(kp->pt.x);
      key_reflist.push_back(kp->pt.y);
    }

  }// refimagelist

  float *mat_col,*desc;

  cvReleaseMat(&desc_mat_reflist_pl);
  cvReleaseMat(&desc_pca_mat_reflist_pl);
  desc_mat_reflist_pl=cvCreateMat(desc_pl.size(),128,CV_32F);
  //desc_pca_mat_reflist_pl=cvCreateMat(desc_pl.size(),32,CV_32F);
  desc_pca_mat_reflist_pl=cvCreateMat(desc_pl.size(),128,CV_32F);

  for(i=0;i<desc_pl.size();i++){
    mat_col=(float *)(desc_mat_reflist_pl->data.ptr+desc_mat_reflist_pl->step*i);
    desc=desc_pl[i];
    memcpy(mat_col,desc,sizeof(float)*128);
    delete[] desc;
  }

  cvCalcPCA(desc_mat_reflist_pl,average_pl,eigenval_pl,eigenvec_pl,CV_PCA_DATA_AS_ROW);
  cvProjectPCA(desc_mat_reflist_pl,average_pl,eigenvec_pl,desc_pca_mat_reflist_pl);

  cvReleaseMat(&desc_mat_reflist_nl);
  cvReleaseMat(&desc_pca_mat_reflist_nl);
  desc_mat_reflist_nl=cvCreateMat(desc_nl.size(),128,CV_32F);
  //desc_pca_mat_reflist_nl=cvCreateMat(desc_nl.size(),32,CV_32F);
  desc_pca_mat_reflist_nl=cvCreateMat(desc_nl.size(),128,CV_32F);

  for(i=0;i<desc_nl.size();i++){
    mat_col=(float *)(desc_mat_reflist_nl->data.ptr+desc_mat_reflist_nl->step*i);
    desc=desc_nl[i];
    memcpy(mat_col,desc,sizeof(float)*128);
    delete[] desc;
  }

  cvCalcPCA(desc_mat_reflist_nl,average_nl,eigenval_nl,eigenvec_nl,CV_PCA_DATA_AS_ROW);
  cvProjectPCA(desc_mat_reflist_nl,average_nl,eigenvec_nl,desc_pca_mat_reflist_nl);

  lsht_list_pl->createHashTable(desc_pca_mat_reflist_pl,index_list_reflist_pl);
  lsht_list_nl->createHashTable(desc_pca_mat_reflist_nl,index_list_reflist_nl);
}


/////////////////////////////////////////////

int SURFConverter::matching(float *refpoints,float *serpoints){
  vector<int> ptpairs;
  CvSURFPoint *r1,*r2;

  findPairs(refKeypoints,refDescriptors,serKeypoints,serDescriptors,ptpairs);
  fprintf(stderr,"ptpairs.size()=%d\n",ptpairs.size());
  for(int i=0;i<(int)ptpairs.size()/2;i++){
    r1 = (CvSURFPoint*)cvGetSeqElem( refKeypoints, ptpairs[i*2] );
    r2 = (CvSURFPoint*)cvGetSeqElem( serKeypoints, ptpairs[i*2+1] );
    refpoints[i*2]=r1->pt.x;
    refpoints[i*2+1]=r1->pt.y;
    serpoints[i*2]=r2->pt.x;
    serpoints[i*2+1]=r2->pt.y;
  }
  return (int)ptpairs.size()/2;
}

int SURFConverter::matchingANN(float *refpoints,float *serpoints){
  vector<int> ptpairs;
  CvSURFPoint *r1,*r2;

  flannFindPairs(refKeypoints,refDescriptors,serKeypoints,serDescriptors,ptpairs);
  fprintf(stderr,"ptpairs.size()=%d\n",ptpairs.size());
  for(int i=0;i<(int)ptpairs.size()/2;i++){
    r1 = (CvSURFPoint*)cvGetSeqElem( refKeypoints, ptpairs[i*2] );
    r2 = (CvSURFPoint*)cvGetSeqElem( serKeypoints, ptpairs[i*2+1] );
    refpoints[i*2]=r1->pt.x;
    refpoints[i*2+1]=r1->pt.y;
    serpoints[i*2]=r2->pt.x;
    serpoints[i*2+1]=r2->pt.y;
  }
  return (int)ptpairs.size()/2;
}



int SURFConverter::matchingPCA(float *refpoints,float *serpoints){
  vector<int> ptpairs;
  CvSURFPoint *r1,*r2;

  if(desc_pca_mat_ref_pl==NULL || desc_pca_mat_ref_nl==NULL
     || desc_pca_mat_ser_pl==NULL || desc_pca_mat_ser_nl==NULL
     )
    return 0;

  fprintf(stderr,"num of ser(pl) features = %d\n",desc_pca_mat_ser_pl->rows);
  fprintf(stderr,"num of ser(nl) features = %d\n",desc_pca_mat_ser_nl->rows);

  findPairs2(desc_pca_mat_ref_pl,desc_pca_mat_ref_nl, 
	     desc_pca_mat_ser_pl,desc_pca_mat_ser_nl,
	     index_list_ref_pl,index_list_ref_nl,
	     index_list_ser_pl,index_list_ser_nl,
	     ptpairs);
  fprintf(stderr,"ptpairs.size()=%d\n",ptpairs.size());
  for(int i=0;i<(int)ptpairs.size()/2;i++){
    r1 = (CvSURFPoint*)cvGetSeqElem( refKeypoints, ptpairs[i*2] );
    r2 = (CvSURFPoint*)cvGetSeqElem( serKeypoints, ptpairs[i*2+1] );
    refpoints[i*2]=r1->pt.x;
    refpoints[i*2+1]=r1->pt.y;
    serpoints[i*2]=r2->pt.x;
    serpoints[i*2+1]=r2->pt.y;
  }
  /*
  findPairs2(desc_pca_mat_ser_pl,desc_pca_mat_ser_nl,
	     desc_pca_mat_ref_pl,desc_pca_mat_ref_nl, 
	     index_list_ser_pl,index_list_ser_nl,
	     index_list_ref_pl,index_list_ref_nl,
	     ptpairs);
  fprintf(stderr,"ptpairs.size()=%d\n",ptpairs.size());
  for(int i=0;i<(int)ptpairs.size()/2;i++){
    r1 = (CvSURFPoint*)cvGetSeqElem( refKeypoints, ptpairs[i*2+1] );
    r2 = (CvSURFPoint*)cvGetSeqElem( serKeypoints, ptpairs[i*2] );
    refpoints[i*2]=r1->pt.x;
    refpoints[i*2+1]=r1->pt.y;
    serpoints[i*2]=r2->pt.x;
    serpoints[i*2+1]=r2->pt.y;
  }
  */
  return (int)ptpairs.size()/2;
}

int SURFConverter::matchingPCALSH(float *refpoints,float *serpoints){
  vector<int> ptpairs;
  CvSURFPoint *r1,*r2;

  if(desc_pca_mat_ser_pl==NULL || desc_pca_mat_ser_nl==NULL)
    return 0;
  fprintf(stderr,"num of ser(pl) features = %d\n",desc_pca_mat_ser_pl->rows);
  fprintf(stderr,"num of ser(nl) features = %d\n",desc_pca_mat_ser_nl->rows);

  findPairs3(lsht_pl,lsht_nl,
	     desc_pca_mat_ser_pl,desc_pca_mat_ser_nl,
	     index_list_ser_pl,index_list_ser_nl,
	     ptpairs);
  /*
  findPairs2(
	     desc_pca_mat_ser_pl,desc_pca_mat_ser_nl,
	     desc_pca_mat_ref_pl,desc_pca_mat_ref_nl, 
	     index_list_ser_pl,index_list_ser_nl,
	     index_list_ref_pl,index_list_ref_nl,
	     ptpairs);
  */

  fprintf(stderr,"ptpairs.size()=%d\n",ptpairs.size());
  for(int i=0;i<(int)ptpairs.size()/2;i++){
    r1 = (CvSURFPoint*)cvGetSeqElem( refKeypoints, ptpairs[i*2] );
    r2 = (CvSURFPoint*)cvGetSeqElem( serKeypoints, ptpairs[i*2+1] );
    refpoints[i*2]=r1->pt.x;
    refpoints[i*2+1]=r1->pt.y;
    serpoints[i*2]=r2->pt.x;
    serpoints[i*2+1]=r2->pt.y;
  }
  return (int)ptpairs.size()/2;
}


//////////////////////////////////

int SURFConverter::matchingAndLocate(float *param){
  // obsolete
  CvPoint src_corners[4]=
    {{0,0},
     {refwidth,0},
     {refwidth, refheight},
     {0, refheight}};
  CvPoint dst_corners[4];

  float pts2d[8]={
    0,0,
    refwidth,0,
    refwidth, refheight,
    0, refheight};
  float pts3d[12];
  convertTo3DPoints(4,refwidth,refheight,pts2d,
		    refwidth,refheight,pts3d);

  int ret=locatePlanarObject(refKeypoints,refDescriptors,
			     serKeypoints,serDescriptors,
			     src_corners,dst_corners);
  if(ret==0) return 0;

  for(int i=0;i<4;i++){
    pts2d[i*2]=(dst_corners[i].x-320.0)*0.006;
    pts2d[i*2+1]=(dst_corners[i].y-240.0)*0.006;
  }
  FindExtrinsicCameraParams(pts3d,pts2d,4,2.8,param);
  return 1;
}

int SURFConverter::matchingPCAAndLocate(float *param){
  // obsolete
  CvPoint src_corners[4]=
    {{0,0},
     {refwidth,0},
     {refwidth, refheight},
     {0, refheight}};
  CvPoint dst_corners[4];

  float pts2d[8]={
    0,0,
    refwidth,0,
    refwidth, refheight,
    0, refheight};
  float pts3d[12];
  convertTo3DPoints(4,refwidth,refheight,pts2d,
		    refwidth,refheight,pts3d);

  int ret=locatePlanarObject(refKeypoints,refDescriptors,
			     serKeypoints,serDescriptors,
			     src_corners,dst_corners);
  if(ret==0) return 0;

  for(int i=0;i<4;i++){
    pts2d[i*2]=(dst_corners[i].x-320.0)*0.006;
    pts2d[i*2+1]=(dst_corners[i].y-240.0)*0.006;
  }
  FindExtrinsicCameraParams(pts3d,pts2d,4,2.8,param);
  return 1;
}

///////////
int SURFConverter::matchingList(vector<float> &refpoints,vector<float> &serpoints){
  return matchingList(refpoints,serpoints,60);
}
int SURFConverter::matchingList(vector<float> &refpoints,vector<float> &serpoints,int thres){
  vector<int> ptpairs;
  CvSURFPoint *r1,*r2;

  if(desc_mat_ser_pl==NULL || desc_pca_mat_ser_pl==NULL ||
     desc_mat_ser_nl==NULL || desc_pca_mat_ser_nl==NULL){
    return -1;
  }

  refpoints.clear();
  serpoints.clear();
  
  int maxi,maxnum,nimages=refimagelist.size();
  vector<int> votetable;
  votetable.resize(nimages);
  for(int i=0;i<nimages;i++)
    votetable[i]=0;

  findPairs2(desc_pca_mat_ser_pl,desc_pca_mat_ser_nl,
	     desc_pca_mat_reflist_pl,desc_pca_mat_reflist_nl, 
	     index_list_ser_pl,index_list_ser_nl,
	     index_list_reflist_pl,index_list_reflist_nl,
	     ptpairs);

  fprintf(stderr,"ptpairs.size()=%d\n",ptpairs.size());

  for(int i=0;i<(int)ptpairs.size()/2;i++){
    maxi=nimages;
    for(int j=0;j<nimages;j++){
      if(ptpairs[i*2+1]<index_offset_reflist[j+1]){
	maxi=j;
	break;
      }
    }
    votetable[maxi]++;
  }

  maxi=-1;
  //maxnum=60;
  maxnum=thres;
  for(int i=0;i<nimages;i++){
    if(maxnum<votetable[i]){
      maxnum=votetable[i];
      maxi=i;
    }
  }
  if(maxi<0) return -1;

  int idx0=index_offset_reflist[maxi];
  int idx1=index_offset_reflist[maxi+1];


  for(int i=0;i<(int)ptpairs.size()/2;i++){
    if(idx0<=ptpairs[i*2+1] && idx1>ptpairs[i*2+1]){
      //r1 = (CvSURFPoint*)cvGetSeqElem( refKeypoints, ptpairs[i*2] );
      r2 = (CvSURFPoint*)cvGetSeqElem( serKeypoints, ptpairs[i*2] );
      refpoints.push_back(key_reflist[ptpairs[i*2+1]*2]);
      refpoints.push_back(key_reflist[ptpairs[i*2+1]*2+1]);
      serpoints.push_back(r2->pt.x);
      serpoints.push_back(r2->pt.y);
    }
  }

  //return (int)ptpairs.size()/2;
  return maxi;
}


int SURFConverter::matchingListLSH(vector<float> &refpoints,vector<float> &serpoints){
  return matchingListLSH(refpoints,serpoints,60);
}
int SURFConverter::matchingListLSH(vector<float> &refpoints,vector<float> &serpoints,int thres){
  vector<int> ptpairs;
  CvSURFPoint *r1,*r2;

  if(desc_mat_ser_pl==NULL || desc_pca_mat_ser_pl==NULL ||
     desc_mat_ser_nl==NULL || desc_pca_mat_ser_nl==NULL){
    return -1;
  }

  refpoints.clear();
  serpoints.clear();
  
  int maxi,maxnum,nimages=refimagelist.size();
  vector<int> votetable;
  votetable.resize(nimages);
  for(int i=0;i<nimages;i++)
    votetable[i]=0;

  if(desc_pca_mat_ser_pl==NULL | desc_pca_mat_ser_nl==NULL)
    return -1;

  findPairs3(lsht_list_pl,lsht_list_nl,
	     desc_pca_mat_ser_pl,desc_pca_mat_ser_nl,
	     index_list_ser_pl,index_list_ser_nl,
	     ptpairs);

  fprintf(stderr,"ptpairs.size()=%d\n",ptpairs.size());

  for(int i=0;i<(int)ptpairs.size()/2;i++){
    maxi=nimages;
    for(int j=0;j<nimages;j++){
      if(ptpairs[i*2]<index_offset_reflist[j+1]){
	maxi=j;
	break;
      }
    }
    votetable[maxi]++;
  }

  maxi=-1;
  //maxnum=60;
  maxnum=thres;
  for(int i=0;i<nimages;i++){
    //fprintf(stderr," %d: %d points\n",i,votetable[i]);
    if(maxnum<votetable[i]){
      maxnum=votetable[i];
      maxi=i;
    }
  }
  if(maxi<0) return -1;

  int idx0=index_offset_reflist[maxi];
  int idx1=index_offset_reflist[maxi+1];


  for(int i=0;i<(int)ptpairs.size()/2;i++){
    if(idx0<=ptpairs[i*2] && idx1>ptpairs[i*2]){
      r2 = (CvSURFPoint*)cvGetSeqElem( serKeypoints, ptpairs[i*2+1] );
      refpoints.push_back(key_reflist[ptpairs[i*2]*2]);
      refpoints.push_back(key_reflist[ptpairs[i*2]*2+1]);
      serpoints.push_back(r2->pt.x);
      serpoints.push_back(r2->pt.y);
    }
  }

  //return (int)ptpairs.size()/2;
  return maxi;
}

//////////////////////////
// using ann
int SURFConverter::matchingListANN(vector<float> &refpoints,vector<float> &serpoints){
  return matchingListANN(refpoints,serpoints,60);
}
int SURFConverter::matchingListANN(vector<float> &refpoints,vector<float> &serpoints,int thres){
  vector<int> ptpairs;
  CvSURFPoint *r1,*r2;

  if(desc_mat_ser_pl==NULL || desc_pca_mat_ser_pl==NULL ||
     desc_mat_ser_nl==NULL || desc_pca_mat_ser_nl==NULL){
    return -1;
  }

  refpoints.clear();
  serpoints.clear();
  
  int maxi,maxnum,nimages=refimagelist.size();
  vector<int> votetable;
  votetable.resize(nimages);
  for(int i=0;i<nimages;i++)
    votetable[i]=0;

  if(desc_pca_mat_ser_pl==NULL | desc_pca_mat_ser_nl==NULL)
    return -1;

  flannFindPairs2(ann_index_ref_pl,ann_index_ref_nl,
		  desc_pca_mat_ser_pl,desc_pca_mat_ser_nl,
		  index_list_ref_pl,index_list_ref_nl,
		  index_list_ser_pl,index_list_ser_nl,
		  ptpairs);

  fprintf(stderr,"ptpairs.size()=%d\n",ptpairs.size());

  for(int i=0;i<(int)ptpairs.size()/2;i++){
    maxi=nimages;
    for(int j=0;j<nimages;j++){
      if(ptpairs[i*2]<index_offset_reflist[j+1]){
	maxi=j;
	break;
      }
    }
    votetable[maxi]++;
  }

  maxi=-1;
  //maxnum=60;
  maxnum=thres;
  for(int i=0;i<nimages;i++){
    //fprintf(stderr," %d: %d points\n",i,votetable[i]);
    if(maxnum<votetable[i]){
      maxnum=votetable[i];
      maxi=i;
    }
  }
  if(maxi<0) return -1;

  int idx0=index_offset_reflist[maxi];
  int idx1=index_offset_reflist[maxi+1];


  for(int i=0;i<(int)ptpairs.size()/2;i++){
    if(idx0<=ptpairs[i*2] && idx1>ptpairs[i*2]){
      r2 = (CvSURFPoint*)cvGetSeqElem( serKeypoints, ptpairs[i*2+1] );
      refpoints.push_back(key_reflist[ptpairs[i*2]*2]);
      refpoints.push_back(key_reflist[ptpairs[i*2]*2+1]);
      serpoints.push_back(r2->pt.x);
      serpoints.push_back(r2->pt.y);
    }
  }

  //return (int)ptpairs.size()/2;
  return maxi;
}

// using CvLSH



////////////////////////////////
// for perftest
void SURFConverter::convertRefListEX(string &fname,int num,int usedim){
  delete lsht_list_pl;
  delete lsht_list_nl;

  lsht_list_pl=new LSHTable(32,usedim);
  lsht_list_nl=new LSHTable(32,usedim);

  int i,j;
  CvSeqReader reader, kreader;
  vector<float *> desc_pl,desc_nl;
  int len_desc;
  int nimages=(refimagelist.size()>num)?num:refimagelist.size();
  int ofs;

  index_list_reflist_pl.clear();
  index_list_reflist_nl.clear();

  index_offset_reflist.resize(nimages+1);
  index_offset_reflist[0]=0;

  key_reflist.clear();

  desc_pl.clear();
  desc_nl.clear();

  for(int nn=0;nn<nimages;nn++){
    cvClearMemStorage(storage_ref2);
    cvExtractSURF(refimagelist[nn],0,&refKey2,&refDesc2,storage_ref2,surfparams);
    
    cvStartReadSeq( refKey2, &kreader );
    cvStartReadSeq( refDesc2, &reader );

    len_desc=refDesc2->total;
    index_offset_reflist[nn+1]=index_offset_reflist[nn]+len_desc;
    ofs=index_offset_reflist[nn];

    for(i=0;i<len_desc;i++){
      const CvSURFPoint* kp = (const CvSURFPoint*)kreader.ptr;
      float* descriptor = (float*)reader.ptr;
      float *ndesc=new float[128];
      CV_NEXT_SEQ_ELEM( kreader.seq->elem_size, kreader );
      CV_NEXT_SEQ_ELEM( reader.seq->elem_size, reader );
      switch(kp->laplacian){
      case 1:
	index_list_reflist_pl.push_back(i+ofs);
	//desc_pl.push_back(descriptor);
	memcpy(ndesc,descriptor,sizeof(float)*128);
	desc_pl.push_back(ndesc);
	break;
      case -1:
	index_list_reflist_nl.push_back(i+ofs);
	//desc_nl.push_back(descriptor);
	memcpy(ndesc,descriptor,sizeof(float)*128);
	desc_nl.push_back(ndesc);
	break;
      default:
	break;
      }
      key_reflist.push_back(kp->pt.x);
      key_reflist.push_back(kp->pt.y);
    }

  }// refimagelist

  float *mat_col,*desc;

  cvReleaseMat(&desc_mat_reflist_pl);
  cvReleaseMat(&desc_pca_mat_reflist_pl);
  desc_mat_reflist_pl=cvCreateMat(desc_pl.size(),128,CV_32F);
  desc_pca_mat_reflist_pl=cvCreateMat(desc_pl.size(),32,CV_32F);

  for(i=0;i<desc_pl.size();i++){
    mat_col=(float *)(desc_mat_reflist_pl->data.ptr+desc_mat_reflist_pl->step*i);
    desc=desc_pl[i];
    memcpy(mat_col,desc,sizeof(float)*128);
    delete[] desc;
  }

  cvCalcPCA(desc_mat_reflist_pl,average_pl,eigenval_pl,eigenvec_pl,CV_PCA_DATA_AS_ROW);
  cvProjectPCA(desc_mat_reflist_pl,average_pl,eigenvec_pl,desc_pca_mat_reflist_pl);

  cvReleaseMat(&desc_mat_reflist_nl);
  cvReleaseMat(&desc_pca_mat_reflist_nl);
  desc_mat_reflist_nl=cvCreateMat(desc_nl.size(),128,CV_32F);
  desc_pca_mat_reflist_nl=cvCreateMat(desc_nl.size(),32,CV_32F);

  for(i=0;i<desc_nl.size();i++){
    mat_col=(float *)(desc_mat_reflist_nl->data.ptr+desc_mat_reflist_nl->step*i);
    desc=desc_nl[i];
    memcpy(mat_col,desc,sizeof(float)*128);
    delete[] desc;
  }

  cvCalcPCA(desc_mat_reflist_nl,average_nl,eigenval_nl,eigenvec_nl,CV_PCA_DATA_AS_ROW);
  cvProjectPCA(desc_mat_reflist_nl,average_nl,eigenvec_nl,desc_pca_mat_reflist_nl);

  lsht_list_pl->createHashTable(desc_pca_mat_reflist_pl,index_list_reflist_pl);
  lsht_list_nl->createHashTable(desc_pca_mat_reflist_nl,index_list_reflist_nl);

  {
    ofstream ofs(fname.c_str(),ios::app);
    ofs << desc_pl.size() << " " << desc_nl.size() << " " << (desc_pl.size()+desc_nl.size()) << " ";
  }
}


int SURFConverter::matchingListLSHEX(string &fname,int num,vector<float> &refpoints,vector<float> &serpoints){
  vector<int> ptpairs;
  CvSURFPoint *r1,*r2;

  refpoints.clear();
  serpoints.clear();
  
  int maxi,maxnum,nimages=(refimagelist.size()>num)?num:refimagelist.size();
  vector<int> votetable;
  votetable.resize(nimages);
  for(int i=0;i<nimages;i++)
    votetable[i]=0;

  findPairs3(lsht_list_pl,lsht_list_nl,
	     desc_pca_mat_ser_pl,desc_pca_mat_ser_nl,
	     index_list_ser_pl,index_list_ser_nl,
	     ptpairs);

  //fprintf(stderr,"ptpairs.size()=%d\n",ptpairs.size());

  for(int i=0;i<(int)ptpairs.size()/2;i++){
    maxi=nimages;
    for(int j=0;j<nimages;j++){
      if(ptpairs[i*2]<index_offset_reflist[j+1]){
	maxi=j;
	break;
      }
    }
    votetable[maxi]++;
  }

  maxi=-1;
  maxnum=0;
  for(int i=1;i<nimages;i++){
    //fprintf(stderr," %d: %d points\n",i,votetable[i]);
    if(maxnum<votetable[i]){
      maxnum=votetable[i];
      maxi=i;
    }
  }

  {
    ofstream ofs(fname.c_str(),ios::app);
    ofs << votetable[0] << " ";
    ofs << maxnum << " ";
  }
  if(maxi<0) return -1;

  int idx0=index_offset_reflist[maxi];
  int idx1=index_offset_reflist[maxi+1];


  for(int i=0;i<(int)ptpairs.size()/2;i++){
    if(idx0<=ptpairs[i*2] && idx1>ptpairs[i*2]){
      r2 = (CvSURFPoint*)cvGetSeqElem( serKeypoints, ptpairs[i*2+1] );
      refpoints.push_back(key_reflist[ptpairs[i*2]*2]);
      refpoints.push_back(key_reflist[ptpairs[i*2]*2+1]);
      serpoints.push_back(r2->pt.x);
      serpoints.push_back(r2->pt.y);
    }
  }

  //return (int)ptpairs.size()/2;
  return maxi;
}






///////////////////////////////////////////////////
void convertTo3DPoints(int npoints,float iwidth,float iheight,float *points2d,
		       float owidth,float oheight,float *points3d){
  float ix,iy,ox,oy,oz;
  for(int i=0;i<npoints;i++){
    ix=points2d[i*2];
    iy=points2d[i*2+1];

    ox=owidth*((ix/iwidth)-0.5);
    oy=oheight*((iy/iheight)-0.5);
    oz=0.0;

    points3d[i*3]=ox;
    points3d[i*3+1]=oy;
    points3d[i*3+2]=oz;
  }
}

////////////////////////////////////////////



