/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.m4hv.extensions.constraints;

import static java.lang.annotation.ElementType.CONSTRUCTOR;
import static java.lang.annotation.ElementType.METHOD;
import static java.lang.annotation.ElementType.FIELD;
import static java.lang.annotation.ElementType.ANNOTATION_TYPE;
import static java.lang.annotation.ElementType.PARAMETER;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

import java.lang.annotation.Documented;
import java.lang.annotation.Retention;
import java.lang.annotation.Target;

import javax.validation.Constraint;
import javax.validation.Payload;

import org.maru.m4hv.extensions.validator.TimeValidator;

/**
 * Time validator<br>
 *
 * This validator validates the time and the range of time.
 * You can define the time range using attributes of from and until on {@code Time} annotation.
 * The time range defaults from 00:00:00 until 23:59:59.
 * There are three types of default time pattern that we have already defined.
 * See below discription about default time pattern.<br>
 *
 * <pre>
 *    1. HH:mm:ss
 *       This is a default and standard time format.
 *    2. HH:mm
 *       In this case the second is omitted and defauls 0 second. For example if you define 12:45,
 *       then it means 12:45:00
 *    3. HH
 *       In this case the minute and the second are omitted and defauls both are 0s.
 *       For example if you define 12, then it means 12:00:00.
 * </pre>
 *
 * If you would like to add or redefine time patterns yourselves, just override the attribute of
 * patterns on {@code Time} annotation. You can define your own pattern by following the format
 * that simple date format class supports.
 */
@Target({METHOD, FIELD, ANNOTATION_TYPE, CONSTRUCTOR, PARAMETER})
@Retention(RUNTIME)
@Constraint(validatedBy = TimeValidator.class)
@Documented
public @interface Time {

    String message() default "{org.maru.m4hv.extensions.Time.message}";

    Class<?>[] groups() default {};

    Class<? extends Payload>[] payload() default {};

    String from() default "00:00:00";

    String until() default "23:59:59";

    String[] patterns() default {"HH:mm:ss", "HH:mm", "HH"};

    @Target({METHOD, FIELD, ANNOTATION_TYPE, CONSTRUCTOR, PARAMETER})
    @Retention(RUNTIME)
    @Documented
    @interface List {
        Time[] value();
    }

}
