/*
 * Copyright (c) 2003 Shinji Kashihara. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies
 * this distribution, and is available at cpl-v10.html.
 */
package mergedoc.core;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Javadoc API ドキュメントです。
 * @author Shinji Kashihara
 */
public class APIDocument {

    /** シグネチャをキーとしたコメントのテーブル */
    private final Map contextTable = new HashMap();

    /** see タグや link タグの埋め込みリンクパターン */
    private static final Pattern linkPattern = Pattern.compile(
        "<A HREF=\"(.+?)\"><CODE>.+?</CODE></A>");

    /**
     * コンストラクタです。
     * @param docDir API ドキュメントディレクトリ
     * @param className クラス名
     * @param charsetName 文字セット名
     * @throws IOException 入出力例外が発生した場合
     */
    public APIDocument(File docDir, String className, String charsetName) throws IOException {

        // API ドキュメント絶対パスを生成
        StringBuffer path = new StringBuffer();
        path.append( docDir.getPath() );
        path.append( File.separator );
        path.append( className.replaceAll("\\.", StringUtils.escapeMetaCharactor(File.separator)) );
        path.append( ".html" );

        // API ドキュメントファイルのロード        
        File docFile = new CachedFile(path.toString());
        load(docDir, docFile, charsetName);

        // インナークラス API ドキュメントファイルのロード
        String prefix = docFile.getName().replaceFirst("\\.html$", "");
        Pattern innerClass = Pattern.compile(prefix + "\\..+\\.html$");
        
        File[] fs = docFile.listFiles();
        for (int i = 0; i < fs.length; i++) {
            File f = fs[i];
            if (innerClass.matcher(f.getName()).matches()) {
                load(docDir, f, charsetName);
            }
        }
    }

    /**
     * ファイルクラスのプロキシです。<br>
     * リストキャッシュ機能を持ちます。
     */
    private static class CachedFile extends File {

        private static File cachedDir;
        private static File[] cachedFiles;
        private static final File[] EMPTY_FILES = new File[0]; 

        public CachedFile(String path) {
            super(path);
        }
        public File[] listFiles() {
            File dir = getParentFile();
            if (!dir.equals(cachedDir)) {
                cachedDir = dir;
                cachedFiles = dir.listFiles();
                if (cachedFiles == null) cachedFiles = EMPTY_FILES;
            }
            return cachedFiles;
        }
    }
    
    /**
     * API ドキュメントファイルを読み込みます。
     * <p>
     * 文字コードが ISO-2022-JP の場合は JDK1.4.2_02 以前の ISO-2022-JP コンバータは
     * バグ（Bug Id 4879522）があり無限ループになる場合があります。JISAutoDetect
     * を使用することで無限ループは回避できますが、不正な JIS 制御コードがある場合は、
     * それ以降が取得できないため、変換前のバイト列を操作することで対応しています。
     * <p>
     * 今のところ JDK1.4 の以下クラスの API ドキュメントファイルに問題となる不正な
     * JIS 制御コードが含まれています。
     * <pre>  
     *   java.util.logging.Logger
     *   javax.naming.OperationNotSupportedException
     *   javax.naming.PartialResultException
     * 
     * 無限ループするコード例（Bug Id 4879522 より）
     * MIME-code: =?ISO-2022-JP?B?GyRCGyRCIVobKEI=?=
     * 
     *   byte jisdata[] = {
     *      0x1b,0x24,0x42, // kanji-in      GyRC
     *      0x1b,0x24,0x42, // kanji-in
     *      0x21,0x5a,        // JIS-code
     *      0x1b,0x28,0x42  // kanji-out
     *   };
     *   String s = new String(jisdata, "JIS");  // infinite loop...
     * 
     * 不具合が発生する HTML ファイルには JIS 制御コードの一部である
     * 0x1b が 2 つ連続した箇所であることが確認できたため、コンバート前に
     * 以下の置換を行っています。
     * 
     *   0x1b, 0x1b -> 0x20, 0x1b
     * </pre>
     *   
     * @param docDir API ドキュメントディレクトリ
     * @param docFile API ドキュメントファイル
     * @param charsetName 文字セット名
     * @throws IOException 入出力例外が発生した場合
     */    
    private void load(File docDir, File docFile, String charsetName) throws IOException {

        String docHtml = null;
        if (docFile.canRead()) {

            // API ドキュメント読み込み
            InputStream is = new FileInputStream(docFile);
            byte[] buf = new byte[is.available()];
            is.read(buf);
            is.close();

            // ISO-2022-JP コンバータのバグ回避
            if (charsetName.equals("ISO-2022-JP")) {
                for (int i = 1; i < buf.length; i++) {
                    if (buf[i-1] == 27 && //0x1b
                        buf[i]   == 27)   //0x1b
                    {
                        buf[i-1] = 32;
                        buf[i]   = 27;
                    }
                }
            }
            docHtml = new String(buf, charsetName);
            docHtml = StringUtils.optimizeLineSeparator(docHtml);

            // WAVE DASH 文字化け回避（こんなんで良いのか？）
            char wavaDash = (char) Integer.decode("0x301c").intValue();
            docHtml = docHtml.replace(wavaDash, '～');
            
            // API ドキュメントファイルパスからクラス名取得
            String className = docFile.getPath().replaceFirst("\\.html$", "");
            className = className.replaceFirst(
                StringUtils.escapeMetaCharactor(docDir.getPath() + File.separator), "");
            className = className.replaceAll(
                StringUtils.escapeMetaCharactor(File.separator), ".");

            // API ドキュメントのコメント解析
            parseClassComment(className, docHtml);
            parseMethodComment(className, docHtml);
        }
    }

    /**
     * コンテキストが空か判定します。
     * @return コンテキストが空の場合は true
     */
    public boolean isEmpty() {
        return contextTable.isEmpty();
    }

    /**
     * 指定したシグネチャを持つ Javadoc コメントを取得します。
     * @param signature シグネチャ
     * @return Javadoc コメント
     */
    public Comment getComment(Signature signature) {
        Comment comment = (Comment)contextTable.get(signature);
        return comment;
    }

    /**
     * クラスの Javadoc コメント情報を作成します。
     * author, version タグは Javadoc デフォルトでは存在しないため解析しません。<br>
     * @param className クラス名
     * @param docHtml API ドキュメントソース
     */
    private void parseClassComment(String className, CharSequence docHtml) {

        String baseRegex =
            "<HR>\\n" +
            "(|<B>.+?<P>\\n)" +          //推奨されない
            "<DL>\\n" +
            "<DT>(.+?)</B>.*?</DL>\\n" + //シグネチャ
            "\\n" +
            "<P>\\n" +
            "(.+?\\n" + "<HR>\\n)" +     //評価コンテキスト
            "\\n" +
            "<P>\\n";

        Pattern pattern = Pattern.compile(baseRegex, Pattern.DOTALL);
        Matcher matcher = pattern.matcher(docHtml);

        if (matcher.find()) {

            // シグネチャの作成
            String sig = matcher.group(2);
            Signature signature = createSignature(className, sig);
            Comment comment = new Comment();

            // deprecated タグ
            String depre = matcher.group(1);
            parseDeprecatedTag(className, depre, comment);

            // 評価コンテキストの取得
            String context = matcher.group(3);

            // コメントの説明文
            // バージョンにより微妙に異なるので注意
            //   SDK 1.3: "(.+?)\\n<P>\\n<(DL|HR)>\\n"
            //   SDK 1.4: "(.+?)\\n<P>\\n\\n<P>\\n<(DL|HR)>\\n"
            Pattern pat = Pattern.compile(
                "(.+?)(|\\n<P>\\n)\\n<P>\\n<(DL|HR)>\\n",
                Pattern.DOTALL);
            Matcher mat = pat.matcher(context);
            if (mat.find()) {
                String body = mat.group(1);
                body = formatLinkTag(className, body);
                comment.setDescription(body);
            }

            // 共通タグ
            parseCommonTag(className, context, comment);

            contextTable.put(signature, comment);
        }
    }

    /**
     * メソッドやフィールドの Javadoc コメント情報を作成します。
     * @param className クラス名
     * @param docHtml API ドキュメントソース
     */
    private void parseMethodComment(String className, CharSequence docHtml) {

        String baseRegex =
            "<A NAME=.+?<!-- --></A><H3>\\n" +
            ".+?</H3>\\n" +
            "<PRE>\\n" +
            "(.*?)</PRE>\\n" +    //シグネチャ
            "(.+?)(<HR>|<!-- =)"; //評価コンテキスト

        Pattern pattern = Pattern.compile(baseRegex, Pattern.DOTALL);
        Matcher matcher = pattern.matcher(docHtml);

        while (matcher.find()) {

            // シグネチャの作成
            String sig = matcher.group(1);
            Signature signature = createSignature(className, sig);
            Comment comment = new Comment();

            // 評価コンテキストの取得
            String context = matcher.group(2);

            // コメント本文
            Pattern pat = Pattern.compile(
                "<DL>\\n" +
                "<DD>(.*?)" +
                "(<DD><DL>\\n|" +
                "</DL>\\n|" +
                "<DL>\\n|" +
                "\\n<P>\\n+(<DL>|</DL>|<DT>|<DD><DL>))",
                Pattern.DOTALL);
                
            Matcher mat = pat.matcher(context);
            if (mat.find()) {
                String body = mat.group(1);
                body = body.replaceFirst(
                    "(?s)<B>推奨されていません。.*?(\\n<P>\\n<DD>|$|</B> <DD>)", "");
                body = formatLinkTag(className, body);
                comment.setDescription(body);
            }

            // param タグ
            pat = Pattern.compile("<DT><B>パラメータ: </B>(.+?(</DL>|<DT>|\\n))");
            mat = pat.matcher(context);
            if (mat.find()) {
                String items = mat.group(1);
                Pattern p = Pattern.compile("<CODE>(.+?)</CODE> - (.*?)(<DD>|</DL>|<DT>|\\n)");
                Matcher m = p.matcher(items);
                while (m.find()) {
                    String name = m.group(1);
                    String desc = formatLinkTag(className, m.group(2));
                    String param = name + " " + desc;
                    comment.addParam(param);
                }
            }

            // return タグ
            //---------------------------------------------------------
            // 注意: 複数行の場合あり。String#startsWith(String,int) など
            //       他のタグもあるか？ 
            //---------------------------------------------------------
            //pat = Pattern.compile("<DT><B>戻り値: </B><DD>(.+?)(</DL>|<DT>|\\n)");
            pat = Pattern.compile("(?s)<DT><B>戻り値: </B><DD>(.+?)(</DL>|<DT>)");
            mat = pat.matcher(context);
            if (mat.find()) {
                String str = mat.group(1);
                str = formatLinkTag(className, str);
                comment.addReturn(str);
            }

            // throws (exception) タグ
            pat = Pattern.compile("(?s)<DT><B>例外: </B>\\s*?<DD>(.+?(</DL>|<DT>))");
            mat = pat.matcher(context);
            if (mat.find()) {
                String items = mat.group(1);
                Pattern p = Pattern.compile("<CODE>(.+?)</CODE> - (.*?)(<DD>|</DL>|<DT>|\\n)");
                Matcher m = p.matcher(items);
                while (m.find()) {
                    String name = m.group(1).replaceAll("(<A .+?>|</A>)", "");
                    String desc = m.group(2);
                    desc = formatLinkTag(className, desc);
                    comment.addThrows(name + " " + desc);
                }
            }

            // deprecated タグ
            parseDeprecatedTag(className, context, comment);

            // 共通タグ
            parseCommonTag(className, context, comment);

            contextTable.put(signature, comment);
        }
    }
    
    /**
     * シグネチャを作成します。
     * @param className クラス名
     * @param sig シグネチャ文字列
     */
    private Signature createSignature(String className, String sig) {
        sig = sig.replaceAll("\\&nbsp;", " ");
        sig = sig.replaceAll("<.+?>", " ");
        sig = sig.replaceFirst("(?s)\\sthrows\\s.*", "");
        Signature signature = new Signature(className, sig);
        return signature;
    }

    /**
     * Javadoc の 共通タグを解析しコメントに追加します。
     * @param className クラス名
     * @param context 評価コンテキスト
     * @param comment コメント
     */
    private void parseCommonTag(String className, String context, Comment comment) {

        // see タグ
        Pattern pat = Pattern.compile("(?s)<DT><B>関連項目:.+?<DD>(.+?)</DL>");
        Matcher mat = pat.matcher(context);

        if (mat.find()) {
            String items = mat.group(1);

            Matcher linkMatcher = linkPattern.matcher(items);
            while (linkMatcher.find()) {
                String path = linkMatcher.group(1);
                String ref = formatClassName(className, path);                
                comment.addSee(ref);
            }

            Pattern p = Pattern.compile("<a href=.+?>.+?</a>");
            Matcher m = p.matcher(items);
            while (m.find()) {
                comment.addSee(m.group());
            }
        }

        // since タグ
        pat = Pattern.compile("(?s)<DT><B>導入されたバージョン:.*?<DD>(.+?)(</DD>)");
        mat = pat.matcher(context);

        if (mat.find()) {
            comment.addSince(mat.group(1));
        }
    }

    /**
     * Javadoc の deprecated タグを解析しコメントに追加します。
     * @param context 評価コンテキスト
     * @param comment コメント
     */
    private void parseDeprecatedTag(String className, String context, Comment comment) {

        Pattern pat = Pattern.compile("<B>推奨されていません。.+?<I>(.+?)</I>");
        Matcher mat = pat.matcher(context);
        if (mat.find()) {
            String str = mat.group(1);
            str = formatLinkTag(className, str);
            comment.addDeprecated(str);
        }
    }

    /**
     * HTML の A タグを Javadoc の link タグにフォーマットします。
     * @param className クラス名
     * @param html HTML の A タグを含む文字列
     * @return Javadoc link タグ文字列
     */
    private String formatLinkTag(String className, String html) {

        StringBuffer sb = new StringBuffer();
        Matcher linkMatcher = linkPattern.matcher(html);

        while (linkMatcher.find()) {
            String path = linkMatcher.group(1);
            String ref = formatClassName(className, path);
            String link = "{@link " + ref + "}";                
            linkMatcher.appendReplacement(sb, link);
        }

        linkMatcher.appendTail(sb);
        return sb.toString();
    }

    /**
     * see タグや link タグの URL を package.class#member 形式にフォーマットします。
     * 同一パッケージの場合は package が省略され、同一クラスの場合は class も省略されます。
     * @param className クラス名
     * @param path パス
     * @return package.class#member 形式の文字列
     */
    private String formatClassName(String className, String path) {

        String lastClassName = className.replaceFirst(".+\\.", "");
        String packageName = className.replaceFirst("\\." + lastClassName, "");

        path = path.replaceAll("\\.html", "");
        path = path.replaceAll("/", ".");
        path = path.replaceAll("^\\.*", "");
        path = path.replaceAll("java\\.lang\\.", "");
        path = path.replaceAll(packageName + "\\.(\\w+($|#|,|\\)))", "$1");
        path = path.replaceFirst(lastClassName + "(#)", "$1");
        return path;
    }
}
