#include "common.h"
#include <unistd.h>
#include <string.h>

#include "config.h"

#if defined(HAVE_CURSES_H)
#include <curses.h>
#elif defined(HAVE_NCURSES_H)
#include <ncurses.h>
#elif defined(HAVE_NCURSES_NCURSES_H)
#include <ncurses/ncurses.h>
#endif

sMenu* gActiveMenu = NULL;
sObject* gMenu;
sObject* gOverridedMenu;

static sMenuItem* sMenuItem_new(char* name, int key, sObject* block, BOOL external) {
    sMenuItem* self = MALLOC(sizeof(sMenuItem));

    self->mName = STRDUP(name);
    self->mKey = key;
    self->mBlock = block_clone_malloc(block);
    self->mExternal = external;

    return self;
}

static void sMenuItem_delete(sMenuItem* self)
{
    FREE(self->mName);

    block_delete_malloc(self->mBlock);

    FREE(self);
}

void sMenu_delete(sMenu* self);

sMenu* sMenu_new(char* title)
{
    sMenu* self = MALLOC(sizeof(sMenu));

    xstrncpy(self->mTitle, title, 256);
    self->mScrollTop = 0;
    self->mCursor = 0;

    self->mMenuItems = VECTOR_NEW_MALLOC(5);

    sMenu* menu = hash_item(gMenu, title);
    hash_put(gMenu, title, self);

    if(menu) vector_add(gOverridedMenu, menu);

    return self;
}

void sMenu_delete(sMenu* self)
{
    int i;
    for(i=0; i<vector_count(self->mMenuItems); i++) {
        sMenuItem_delete(vector_item(self->mMenuItems, i));
    }
    vector_delete_malloc(self->mMenuItems);

    FREE(self);
}

///////////////////////////////////////////////////
// 初期化
///////////////////////////////////////////////////
void menu_init()
{
    gMenu = HASH_NEW_MALLOC(10);
    gOverridedMenu = VECTOR_NEW_MALLOC(10);
}

void menu_final()
{
    hash_it* it = hash_loop_begin(gMenu);
    while(it) {
        sMenu_delete(hash_loop_item(it));
        it = hash_loop_next(it);
    }
    hash_delete_malloc(gMenu);

    int i;
    for(i=0; i<vector_count(gOverridedMenu); i++) {
        sMenu_delete(vector_item(gOverridedMenu, i));
    }
    vector_delete_malloc(gOverridedMenu);
}

///////////////////////////////////////////////////
// メニューに追加
///////////////////////////////////////////////////
void menu_append(sMenu* self, char* name, int key, sObject* block, BOOL external)
{
    char buf[256];
    snprintf(buf, 256, " %-40s", name);
    vector_add(self->mMenuItems, sMenuItem_new(buf, key, block, external));
}

void menu_start(char* menu_name)
{
    gActiveMenu = hash_item(gMenu, menu_name);
    if(gActiveMenu) {
        gActiveMenu->mScrollTop = 0;
        gActiveMenu->mCursor = 0;
    }
}

///////////////////////////////////////////////////
// キー入力
///////////////////////////////////////////////////
static void menu_run(sMenuItem* item)
{
    mclear();
    view();
    mrefresh();
    
    if(item->mExternal) {
        const int maxy = mgetmaxy();
        mmove_immediately(maxy-2, 0);
        mendwin();

        mreset_tty();

        int rcode;
        if(xyzsh_run(&rcode, item->mBlock, item->mName, NULL, gStdin, gStdout, 0, NULL, gMFiler4))
        {
            if(rcode == 0) {
                minitscr();
                //(void)filer_reset_marks(adir());
            }
            else {
                char str[128];
                fprintf(stderr, "return code is %d", rcode);

                printf("\nHIT ANY KEY\n");

                minitscr();

                int meta;
                (void)mgetch(&meta);
            }
        }
        else {
            char str[128];
            fprintf(stderr, "runtime err\n%s", string_c_str(gErrMsg));
            printf("\nHIT ANY KEY\n");

            minitscr();

            int meta;
            (void)mgetch(&meta);
        }
    }
    else {
        int rcode;
        if(xyzsh_run(&rcode, item->mBlock, item->mName, NULL, gStdin, gStdout, 0, NULL, gMFiler4)) {
            if(rcode == 0) {
                //(void)filer_reset_marks(adir());
            }
            else {
                char str[128];
                snprintf(str, 128, "return code is %d", rcode);
                merr_msg(str);
            }
        }
        else {
            merr_msg(string_c_str(gErrMsg));
        }
    }
}

void menu_input(sMenu* self, int meta, int key)
{
    if(key == 1 || key == KEY_HOME) {        // CTRL-A
        self->mCursor = 0;
    }
    else if(key == 5 || key == KEY_END) {    // CTRL-E
        self->mCursor = vector_count(self->mMenuItems)-1;
    }
    else if(key == 14 || key == 6|| key == KEY_DOWN) {    // CTRL-N CTRL-F
        self->mCursor++;
    }
    else if(key == 16 || key == 2 || key == KEY_UP) {    // CTRL-P CTRL-B
        self->mCursor--;
    }
    else if(key == 4 || key == KEY_NPAGE) { // CTRL-D PAGE DOWN
        self->mCursor+=7;
    }
    else if(key == 21 || key == KEY_PPAGE) { // CTRL-U   PAGE UP
        self->mCursor-=7;
    }
    else if(key == 3 || key == 7 || key == 27) {    // CTRL-C CTRL-G Escape
        gActiveMenu = NULL;
        mclear();
    }
    else if(key == 10 || key == 13) {    // CTRL-M CTRL-J
        gActiveMenu = NULL;
        
        sMenuItem* item = vector_item(self->mMenuItems, self->mCursor);
        menu_run(item);
        mclear();
    }
    else if(key == 12) {// CTRL-L
        mclear_immediately();
    }
    else {
        gActiveMenu = NULL;
        
        int i;
        for(i=0; i<vector_count(self->mMenuItems); i++) {
            sMenuItem* item = (sMenuItem*)vector_item(self->mMenuItems, i);
            
            if(item->mKey == key) {
                menu_run(item);
            }
        }
        
        mclear();
    }

    while(self->mCursor < 0) {
        self->mCursor += vector_count(self->mMenuItems);
    }
    while(vector_count(self->mMenuItems) != 0 && self->mCursor >= vector_count(self->mMenuItems)) 
    {
       self->mCursor -= vector_count(self->mMenuItems);
    }

    const int maxy = mgetmaxy() -2;
    
    if(self->mCursor >= (self->mScrollTop+maxy)) {
        self->mScrollTop += self->mCursor - (self->mScrollTop+maxy) + 1;
    }
    if(self->mCursor < self->mScrollTop) {
        self->mScrollTop = self->mCursor;
    }
    sweep_overrided_menu();
}

///////////////////////////////////////////////////
// 描写
///////////////////////////////////////////////////
void menu_view(sMenu* self)
{
    const int maxx = mgetmaxx();
    const int maxy = mgetmaxy();

    int item_maxx = 0;
    int i;
    for(i=0; i<vector_count(self->mMenuItems); i++) {
        sMenuItem* item = (sMenuItem*)vector_item(self->mMenuItems, i);
        
        int len = str_termlen(gKanjiCode, item->mName);

        if(item_maxx < len) {
            item_maxx = len;
        }
    }
    item_maxx+=2;
    
    if(item_maxx < 43) {
        item_maxx = 43;
    }
    else if(item_maxx > maxx-2) {
        item_maxx = maxx-2;
    }
    
    const int size = vector_count(self->mMenuItems) + 2;
    mbox(0, 2, item_maxx, size < maxy ? size : maxy);

    mattron(kCABold);
    mmvprintw(0, 4, self->mTitle);
    mattroff();

    float f;
    if(vector_count(self->mMenuItems) == 0) {
        f = 0.0;
    }
    else {
        f = ((float)(self->mCursor+1)
                /(float)vector_count(self->mMenuItems))*100;
    }

    mmvprintw((size < maxy ? size: maxy) - 1, 2+item_maxx-8, "(%3d%%)", (int)f);
    for(i=self->mScrollTop;
        (i<vector_count(self->mMenuItems) && i-self->mScrollTop<maxy-2); i++) {
        sMenuItem* item = (sMenuItem*)vector_item(self->mMenuItems, i);
        
        char buf[1024];
        if(strlen(item->mName) > maxx-4) {
            str_cut2(gKanjiCode, item->mName, maxx-4, buf, 1024);
        }
        else {
            xstrncpy(buf, item->mName, 1024);
        }
        
        if(i == self->mCursor) {
            mattron(kCAReverse);
            mmvprintw(i-self->mScrollTop+1, 3, buf);
            mattroff();
        }
        else  {
            mmvprintw(i-self->mScrollTop+1, 3, buf);
        }
    }

    mmove(maxy-1, maxx-2);
}


void sweep_overrided_menu()
{
    int i;
    for(i=0; i<vector_count(gOverridedMenu); i++) {
        sMenu_delete(vector_item(gOverridedMenu, i));
    }
    vector_clear(gOverridedMenu);
}
