/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.motion;

import jp.sourceforge.mmd.motion.geo.Matrix;
import jp.sourceforge.mmd.motion.geo.Vector3D;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

/**
 * ボーン用のPose.
 * @author nazo
 */
public class BonePose extends Pose {
    /** 位置ベクトル(初期位置からのずれ) */
    public Vector3D v;
    /** 回転行列.
     * 角度で入力したい人は, {@link Matrix#rotation(double, double, double)}
     * を使用してください.
     */
    public Matrix mr;

    /** x軸に関する補間.
     * MMDの補間は,前キーフレームを(s,t)=(0,0),
     * 本キーフレームを(128,128)とするベージェ補間をする. t は時間軸.
     * [0] ベージェ補間の第1補間点のs, [1] ベージェ補間の第1補間点のt, 
     * [2] ベージェ補間の第2補間点のs, [3] ベージェ補間の第2補間点のt, 
     */
    public byte [] interpX =new byte[4];
    /** y軸に関する補間. */
    public byte [] interpY =new byte[4];
    /** z軸に関する補間. */
    public byte [] interpZ =new byte[4];
    /** 回転に関する補間. */
    public byte [] interpR =new byte[4];

    /**
     * 初期状態のポーズを作る.
     * 位置(0,0,0), 回転(0,0,0), 線形補間になる.
     */
    public BonePose(){
        super();
        v=new Vector3D();
        mr=new Matrix();
        interpX=new byte[]{0x14,0x14,0x6b,0x6b};
        interpY=new byte[]{0x14,0x14,0x6b,0x6b};
        interpZ=new byte[]{0x14,0x14,0x6b,0x6b};
        interpR=new byte[]{0x14,0x14,0x6b,0x6b};
    }

    /**
     * クローン.
     * @param other クローン元.
     */
    public BonePose(BonePose other){
        super(other);
        
        v=other.v.clone();
        mr=other.mr.clone();
        int i;
        interpX=new byte[4];
        interpY=new byte[4];
        interpZ=new byte[4];
        interpR=new byte[4];
        for(i=0;i<4;i++){
            interpX[i]=other.interpX[i];
            interpY[i]=other.interpY[i];
            interpZ[i]=other.interpZ[i];
            interpR[i]=other.interpR[i];
        }
    }

    /**
     * 等しいかを比較する. {@link Pose#equals(Pose)} 同様フレーム番の
     * 違いは無視される.
     * @param p 比較対象.
     * @return 等しければ true.
     */
    public boolean equals(BonePose p){
        return super.equals(p) && v.equals(p.v) && mr.equals(p.mr);
    }

    /**
     * CSV化されたMMDモーションを書き込む. CSVはVMDCSV上位互換.
     * @return CSV 1行
     */
    @Override
    public String toCSV(){
        StringBuilder line=new StringBuilder(512);
        line.append(nameOfBone).append(",").append(frame).append(",").append(v);
        double []r =mr.angles();

        line.append(",").append(r[0])
                .append(",").append(-r[1])
                .append(",").append(-r[2])
                .append(",0x");
        byte [] unknown=unknown();
        for(byte b:unknown){
            line.append(((b&0xf0)==0)?"0":"").append(Integer.toHexString(b).toUpperCase());
        }
        r=mr.getQuotanions();
        for(double d:r){
            line.append(",").append(d);
        }
        for(byte b:interpX){
            line.append(",").append(b);
        }
        for(byte b:interpY){
            line.append(",").append(b);
        }
        for(byte b:interpZ){
            line.append(",").append(b);
        }
        for(byte b:interpR){
            line.append(",").append(b);
        }
        return line.append("\n").toString();
    }

    /**
     * CSV化されたMMDモーションを読み込む. VMDCSV互換.
     * @param line CSV 1行.
     * @return 読み込んだBonePose.
     */
    static public BonePose fromCSV(String line){
        String [] column=CsvSpliter.split(line);
        if(column.length<9)return null;
        BonePose p=new BonePose();
        String hex;
        int i;
        double rx,ry,rz;

        p.nameOfBone=column[0];
        p.frame=Integer.parseInt(column[1]);
        p.v=new Vector3D(Double.parseDouble(column[2]),
                Double.parseDouble(column[3]),
                Double.parseDouble(column[4]));

        rx=Double.parseDouble(column[5]);
        ry=-Double.parseDouble(column[6]);
        rz=-Double.parseDouble(column[7]);

        for(i=0;i<32;i+=2){
            hex=column[8].substring(i+2,i+4);
            byte t=Byte.parseByte(hex,16);
            switch(i%8){
                case 0:
                    p.interpX[i/8]=t;
                    break;
                case 2:
                    p.interpY[i/8]=t;
                    break;
                case 4:
                    p.interpZ[i/8]=t;
                    break;
                case 6:
                    p.interpR[i/8]=t;
                    break;
            }
        }

        if(column.length>12){// extended
            double []q=new double [4];
            for(i=0;i<4;i++){
                q[i]=Double.parseDouble(column[9+i]);
            }
            p.mr=Matrix.rotationQ(q[0],q[1],q[2],q[3]);
        }else {
            p.mr=Matrix.rotation(rx, ry, rz);
        }

        if(column.length>16){// extended
            for(i=0;i<4;i++){
                p.interpX[i]=Byte.parseByte(column[13+i]);
            }
            for(i=0;i<4;i++){
                p.interpY[i]=Byte.parseByte(column[17+i]);
            }
            for(i=0;i<4;i++){
                p.interpZ[i]=Byte.parseByte(column[21+i]);
            }
            for(i=0;i<4;i++){
                p.interpR[i]=Byte.parseByte(column[25+i]);
            }
        }
        return p;
    }

    /**
     * VMDを書き出す.
     * @return VMDのBonePose 用バイナリー.
     */
    @Override
    protected byte[] toVMD() {
        int i;
        byte [] a;
        ByteBuffer ret=ByteBuffer.allocate(111).order(ByteOrder.LITTLE_ENDIAN);
        try {
            a=nameOfBone.getBytes("MS932");
            ret.put(a);
            for(i=a.length;i<15;i++){
                ret.put((byte)0);
            }
        } catch (UnsupportedEncodingException ex) {
            System.err.println("Syntax error in Pose.toVMD.");
            System.exit(-1);
        }

        ret.putInt(frame);

        double [] rv=v.toDouble();
        ret.putFloat((float) rv[0]);
        ret.putFloat((float) rv[1]);
        ret.putFloat((float) rv[2]);

        double []q=mr.getQuotanions();
        ret.putFloat((float) q[0]);
        ret.putFloat((float) q[1]);
        ret.putFloat((float) q[2]);
        ret.putFloat((float) q[3]);

        ret.put(unknown());
        return ret.array();
    }

    /**
     * 補間データのバイナリー化. ひどい名前だが,
     * VMDCSVがそう読んでいたのをそのまま継承した.
     * @return 補間データのバイナリーコード.
     */
    private byte []unknown(){
        byte [] ret=new byte[64];
        int i,j,l=0,flag;
        for(j=0;j<4;j++){
            for(i=j;i<16;i++){
                flag=i%4;

                ret[l]=(flag==0)?(interpX[i/4])
                        :(flag==1)?(interpY[i/4])
                        :(flag==2)?(interpZ[i/4])
                        :(interpR[i/4]);
                l++;
            }
            for(i=0;i<j;i++){
                ret[l]=0;l++;
            }
        }
        
        return ret;
    }

    /**
     * クローン.
     * {@link #BonePose(BonePose) new BonePose(this)} の方が使い勝手はいい.
     * @return クローンされたボーンポーズ.
     */
    @Override
    @SuppressWarnings({"CloneDoesntCallSuperClone", "CloneDeclaresCloneNotSupported"})
    public BonePose clone() {
        BonePose p=new BonePose(this);
        return p;
    }
}
