/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: BBSCommentVariableCreator.cpp 1866 2006-10-22 07:00:14Z svn $
 *****************************************************************************/

#include "Environment.h"

#include "BBSCommentVariableCreator.h"
#include "BBSCommentIterator.h"
#include "Auxiliary.h"
#include "Message.h"
#include "SourceInfo.h"

SOURCE_INFO_ADD("$Id: BBSCommentVariableCreator.cpp 1866 2006-10-22 07:00:14Z svn $");

/******************************************************************************
 * public メソッド
 *****************************************************************************/
BBSCommentVariableCreator::BBSCommentVariableCreator(const char **keys)
  : keys_(keys)
{
    init();
}

TemplateVariable::variable_t *
BBSCommentVariableCreator::create(apr_pool_t *pool,
                                 BBSCommentIterator *comment_iter) const
{
    void *memory;
    variable_t *comment_array_memory;
    scalar_t *comment_memory;
    variable_t *var;
    BBSThread::bbs_comment_p_t bcomment_p;

    // メモリを一気に確保
    APR_PALLOC(memory, void *, pool,
               sizeof(variable_t) +
               get_comment_array_memory_size(comment_iter) +
               get_comment_memory_size() * comment_iter->get_size());

    comment_array_memory = AS_VARIABLE(memory);
    comment_memory
        = AS_SCALAR(AS_CHAR(comment_array_memory) +
                    sizeof(variable_t) +
                    get_comment_array_memory_size(comment_iter));

    var = comment_array_memory++;
    var->type = TemplateVariable::ARRAY;
    var->v = comment_array_memory;

    do {
        comment_iter->get_comment(&bcomment_p);
        create(pool, &bcomment_p, comment_array_memory++, comment_memory);

        // +1 でインデックスから個数に変換
        comment_memory += comment_key_index_max_ + 1;
    } while (comment_iter->go_next());
    comment_array_memory->type = TemplateVariable::END;

    return var;
}

inline TemplateVariable::variable_t *
BBSCommentVariableCreator::create(apr_pool_t *pool,
                                  BBSThread::bbs_comment_p_t *bcomment_p,
                                  void *var_memory,
                                  void *sca_memory) const throw()
{
    variable_t *comment_var;
    scalar_t *comment_sca;
    scalar_t *sca;
    const char *escaped_message;
    apr_size_t escaped_length;

    comment_var = AS_VARIABLE(var_memory);
    comment_sca = AS_SCALAR(sca_memory);

    comment_var->type = TemplateVariable::HASH;
    comment_var->s = comment_sca;

    sca         = comment_sca + comment_key_index_.no.index;
    sca->type   = TemplateVariable::INTEGER;
    sca->i      = static_cast<int>(bcomment_p->no);

    sca         = comment_sca + comment_key_index_.name.index;
    sca->type   = TemplateVariable::STRING;
    sca->s      = bcomment_p->name;
    sca->l      = 0;

    sca         = comment_sca + comment_key_index_.date.index;
    sca->type   = TemplateVariable::STRING;
    sca->s      = bcomment_p->date;
    sca->l      = 0;

    sca         = comment_sca + comment_key_index_.trip.index;
    sca->type   = TemplateVariable::STRING;
    sca->s      = bcomment_p->trip;
    sca->l      = 0;

    sca         = comment_sca + comment_key_index_.email.index;
    sca->type   = TemplateVariable::STRING;
    sca->s      = bcomment_p->email;
    sca->l      = 0;

    sca         = comment_sca + comment_key_index_.message.index;
    sca->type   = TemplateVariable::STRING;
    if (LIKELY(bcomment_p->flag.is_need_html_escape == 0)) {
        sca->s  = bcomment_p->message;
        sca->l  = bcomment_p->message_length;
    } else {
        escape_html(pool, bcomment_p->message, bcomment_p->message_length,
                    &escaped_message, &escaped_length);
        sca->s  = escaped_message;
        sca->l  = escaped_length;
    }
    sca         = comment_sca + comment_key_index_.is_age.index;
    sca->type   = TemplateVariable::INTEGER;
    sca->i      = static_cast<int>(bcomment_p->flag.is_age);

    return comment_var;
}

BBSCommentVariableCreator *BBSCommentVariableCreator::get_instance(void *memory,
                                                                   const char **keys)
{
    new(memory) BBSCommentVariableCreator(keys);

    return reinterpret_cast<BBSCommentVariableCreator *>(memory);
}


/******************************************************************************
 * private メソッド
 *****************************************************************************/
void BBSCommentVariableCreator::init()
{
    comment_key_index_max_ = TemplateVariableCreator::calc_index
        (keys_, AS_KEY_INDEX(&comment_key_index_),
         sizeof(comment_key_index_t)/sizeof(key_index_t));
}

apr_size_t BBSCommentVariableCreator::get_comment_array_memory_size(BBSCommentIterator *comment_iter) const
{
    return sizeof(variable_t) * (comment_iter->get_size() + 1);
}

apr_size_t BBSCommentVariableCreator::get_comment_memory_size() const
{
    return sizeof(scalar_t) * (comment_key_index_max_ + 1);
}


/******************************************************************************
 * テスト
 *****************************************************************************/
#ifdef DEBUG_BBSCommentVariableCreator
#include "TestRunner.h"
#include "TemplateParser.h"
#include "Message.h"

void show_usage(const char *prog_name)
{
    cerr << "Usage: " << prog_name << " <DATA_DIR_PATH> <THREAD_ID> <TMPL_FILE_PATH>" << endl;
}

void run_create(apr_pool_t *pool, const char *data_dir_path,
                const char *tmpl_file_path, apr_size_t thread_id)
{
    BBSCommentIterator *comment_iter;
    const char **ids;
    const char **keys;

    show_test_name("create variable");

    TemplateParser::parse(pool, tmpl_file_path, &ids, &keys);
    comment_iter = BBSCommentIterator::read(pool, data_dir_path, thread_id);

    BBSCommentVariableCreator var_creator(keys);

    var_creator.create(pool, comment_iter);

    show_spacer();
}

void run_all(apr_pool_t *pool, int argc, const char * const *argv)
{
    const char *data_dir_path;
    const char *tmpl_file_path;
    apr_size_t thread_id;

    if (argc != 4) {
        THROW(MESSAGE_ARGUMENT_INVALID);
    }

    data_dir_path = argv[1];
    thread_id = atoi(argv[2]);
    tmpl_file_path = argv[3];

    if (!is_exist(pool, data_dir_path)) {
        THROW(MESSAGE_DAT_DIR_NOT_FOUND);
    }
    if (!is_exist(pool, tmpl_file_path)) {
        THROW(MESSAGE_TMPL_FILE_NOT_FOUND);
    }

    show_item("data dir", data_dir_path);
    show_item("thread id", thread_id);
    show_item("template file", tmpl_file_path);
    show_line();

    run_create(pool, data_dir_path, tmpl_file_path, thread_id);
}

#endif

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
